﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "visrv_ScreenShotServiceImpl.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#include <nn/capsrv/capsrv_Result.h>
#include "../capsrvServer_Config.h"
#include "../capsrvServer_ConvertResult.h"

#include "visrv_ScreenShotContextMacro.h"
#include "visrv_ScreenShotServerObject.h"
#include "../screenshot/visrv_SaveEditedScreenShot.h"
#include "../screenshot/visrv_SaveApplicationScreenShot.h"
#include "../detail/capsrvServer_ScreenShotAttributeUtility.h"
#include "../capsrvServer_ImplUtility.h"

#if defined(NN_CAPSRV_USE_HIPC)

#define NN_CAPSRV_SERVER_DO(Function)  \
    NN_RESULT_DO(g_ScreenShotErrorConverter.Convert(Function()));

#define NN_CAPSRV_SERVER_SYNC_DO(Function)  \
    NN_RESULT_DO(g_ScreenShotErrorConverter.Convert(g_ScreenShotRequestQueue.EnqueueSync(::nn::capsrv::server::MakeFsPriorityInheritedFunction(Function))));

#define NN_CAPSRV_SERVER_ASYNC_DO(Function) \
    NN_RESULT_DO(g_ScreenShotErrorConverter.Convert(g_ScreenShotRequestQueue.TryEnqueueAsync(Function)));

#elif defined(NN_CAPSRV_USE_DIRECT_FUNCTION_CALL)

#define NN_CAPSRV_SERVER_DO(Function)  \
    NN_RESULT_DO(g_ScreenShotErrorConverter.Convert(Function()));
#define NN_CAPSRV_SERVER_SYNC_DO  NN_CAPSRV_SERVER_DO
#define NN_CAPSRV_SERVER_ASYNC_DO NN_CAPSRV_SERVER_DO

#endif

namespace nn{ namespace capsrv{ namespace server{

    //--------------------------------------------------------------------
    // SaveScreenShot
    //--------------------------------------------------------------------
    nn::Result ScreenShotServiceImpl::SaveScreenShotImpl(
        nn::sf::Out<nn::capsrv::ApplicationAlbumEntry>& outEntry,
        const nn::sf::InBuffer& imageBuffer,
        const nn::capsrv::ScreenShotAttribute& attribute,
        const nn::capsrv::UserIdList& userIdList,
        nn::applet::AppletResourceUserId aruid,
        nn::capsrv::OverlayNotificationRequestType overlayRequest
        ) NN_NOEXCEPT
    {
        NN_CAPSRV_LOG_DEV("  aruid=%lld\n", aruid.lower);

        const void* pImageData = imageBuffer.GetPointerUnsafe();
        size_t imageDataSize   = imageBuffer.GetSize();
        NN_RESULT_THROW_UNLESS(
            pImageData != nullptr && imageDataSize > 0,
            ResultScreenShotInvalidData()
        );

        NN_RESULT_THROW_UNLESS(aruid != nn::applet::AppletResourceUserId::GetInvalidId(), ResultAlbumError());
        NN_RESULT_THROW_UNLESS(attribute.size == ScreenShotSize_1280x720, ResultScreenShotInvalidSize());
        NN_RESULT_THROW_UNLESS(imageDataSize >= 4 * ScreenShotWidth * ScreenShotHeight, ResultScreenShotInvalidData());

        ApplicationAlbumEntry entry = {};
        NN_CAPSRV_SERVER_SYNC_DO((
            [&]() -> nn::Result {
                NN_CAPSRV_SERVER_SCREENSHOT_CONTEXT(pContext);
                g_ScreenShotContextManager.SetupContextForSavingApplicationScreenShotAruid(
                    pContext, pImageData, imageDataSize, attribute, userIdList, aruid, overlayRequest
                );
                NN_RESULT_DO(screenshot::SaveApplicationScreenShotAruid(*pContext));
                {
                    auto& context = *pContext;
                    NN_CAPSRV_SCREENSHOT_CONTEXT_SCOPE(context);
                    NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_IN(context, ApplicationAlbumEntry);
                    NN_CAPSRV_SCREENSHOT_CONTEXT_GET(e, context, ApplicationAlbumEntry);
                    entry = e;
                }
                NN_RESULT_SUCCESS;
            }
        ));

        outEntry.Set(entry);
        NN_RESULT_SUCCESS;
    }

    nn::Result ScreenShotServiceImpl::SaveScreenShot(
        nn::sf::Out<nn::capsrv::ApplicationAlbumEntry> outEntry,
        const nn::sf::InBuffer& imageBuffer,
        nn::capsrv::ScreenShotSizeType size,
        nn::applet::AppletResourceUserId aruid,
        nn::capsrv::OverlayNotificationRequestType overlayRequest
        ) NN_NOEXCEPT
    {
        NN_CAPSRV_IPC_TRACE_SS(SaveScreenShot);
        ScreenShotAttribute attr = detail::ScreenShotAttributeUtility::GetScreenShotAttribute(
            nn::capsrv::AlbumFileDescription_ScreenShotSaved,
            size,
            nn::capsrv::ScreenShotOrientation_Default
        );
        capsrv::UserIdList userIdList = {};
        return SaveScreenShotImpl(outEntry, imageBuffer, attr, userIdList, aruid, overlayRequest);
    }

    nn::Result ScreenShotServiceImpl::SaveScreenShotEx0(
        nn::sf::Out<nn::capsrv::ApplicationAlbumEntry> outEntry,
        const nn::sf::InBuffer& imageBuffer,
        nn::capsrv::detail::ScreenShotAttributeEx0 attribute,
        nn::applet::AppletResourceUserId aruid,
        nn::capsrv::OverlayNotificationRequestType overlayRequest
        ) NN_NOEXCEPT
    {
        NN_CAPSRV_IPC_TRACE_SS(SaveScreenShotEx0);
        nn::capsrv::ScreenShotAttribute attr = {};
        NN_RESULT_DO(detail::ScreenShotAttributeUtility::ImportAttribute(
            &attr,
            attribute,
            nn::capsrv::AlbumFileDescription_ScreenShotSaved
        ));
        capsrv::UserIdList userIdList = {};
        return SaveScreenShotImpl(outEntry, imageBuffer, attr, userIdList, aruid, overlayRequest);
    }

    //--------------------------------------------------------------------
    // SaveEditedScreenShot(Applet 用)
    //--------------------------------------------------------------------

    nn::Result ScreenShotServiceImpl::SaveEditedScreenShotImpl(
        nn::sf::Out<nn::capsrv::AlbumEntry>& outFileEntry,
        const nn::sf::InBuffer& imageBuffer,
        const nn::sf::InBuffer& thumbnailBuffer,
        std::int64_t imageWidth,
        std::int64_t imageHeight,
        std::int64_t thumbnailWidth,
        std::int64_t thumbnailHeight,
        const nn::capsrv::ScreenShotAttribute& attribute,
        const nn::capsrv::AppletData* pAppletData,
        const nn::capsrv::AlbumFileId& originalFileId
        ) NN_NOEXCEPT
    {
        const void* pImageData = imageBuffer.GetPointerUnsafe();
        size_t imageDataSize   = imageBuffer.GetSize();
        NN_RESULT_THROW_UNLESS(
            pImageData != nullptr && imageDataSize > 0,
            ResultScreenShotInvalidData()
        );

        const void* pThumbData = thumbnailBuffer.GetPointerUnsafe();
        size_t thumbDataSize   = thumbnailBuffer.GetSize();
        NN_RESULT_THROW_UNLESS(
            pThumbData != nullptr && thumbDataSize > 0,
            ResultScreenShotInvalidData()
        );

        NN_RESULT_THROW_UNLESS(imageWidth == ScreenShotWidth, ResultScreenShotInvalidSize());
        NN_RESULT_THROW_UNLESS(imageHeight == ScreenShotHeight, ResultScreenShotInvalidSize());
        NN_RESULT_THROW_UNLESS(thumbnailWidth == ViewerThumbnailImageSize_Width, ResultScreenShotInvalidSize());
        NN_RESULT_THROW_UNLESS(thumbnailHeight == ViewerThumbnailImageSize_Height, ResultScreenShotInvalidSize());
        NN_RESULT_THROW_UNLESS(imageDataSize >= 4 * ScreenShotWidth * ScreenShotHeight, ResultScreenShotInvalidData());
        NN_RESULT_THROW_UNLESS(thumbDataSize >= ViewerThumbnailImageDataSize_Raw, ResultScreenShotInvalidData());

        AlbumEntry entry;
        NN_CAPSRV_SERVER_SYNC_DO((
            [&]() -> nn::Result {
                NN_CAPSRV_SERVER_SCREENSHOT_CONTEXT(pContext);
                g_ScreenShotContextManager.SetupContextForSavingEditedScreenShot(
                    pContext, pImageData, imageDataSize, pThumbData, thumbDataSize, attribute, pAppletData, originalFileId.applicationId
                );
                NN_RESULT_DO(screenshot::SaveEditedScreenShot(*pContext));
                {
                    auto& context = *pContext;
                    NN_CAPSRV_SCREENSHOT_CONTEXT_SCOPE(context);
                    NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_IN(context, AlbumFileId);
                    NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_IN(context, FileSize);
                    NN_CAPSRV_SCREENSHOT_CONTEXT_GET(fileId, context, AlbumFileId);
                    NN_CAPSRV_SCREENSHOT_CONTEXT_GET(fileSize, context, FileSize);
                    entry.fileId = fileId;
                    entry.size   = fileSize;
                }
                NN_RESULT_SUCCESS;
            }
        ));

        outFileEntry.Set(entry);
        NN_RESULT_SUCCESS;
    }
    nn::Result ScreenShotServiceImpl::SaveEditedScreenShot(
        nn::sf::Out<nn::capsrv::AlbumEntry> outFileEntry,
        const nn::sf::InBuffer& imageBuffer,
        const nn::sf::InBuffer& thumbnailBuffer,
        std::int64_t imageWidth,
        std::int64_t imageHeight,
        std::int64_t thumbnailWidth,
        std::int64_t thumbnailHeight,
        const nn::capsrv::AlbumFileId& originalFileId
        ) NN_NOEXCEPT
    {
        NN_CAPSRV_IPC_TRACE_SS(SaveEditedScreenShot);
        auto attr = detail::ScreenShotAttributeUtility::GetScreenShotAttribute(
            nn::capsrv::AlbumFileDescription_ScreenShotEdited,
            nn::capsrv::ScreenShotSize_1280x720,
            nn::capsrv::ScreenShotOrientation_Default
        );
        return SaveEditedScreenShotImpl(outFileEntry, imageBuffer, thumbnailBuffer, imageWidth, imageHeight, thumbnailWidth, thumbnailHeight, attr, nullptr, originalFileId);
    }

    nn::Result ScreenShotServiceImpl::SaveEditedScreenShotEx0(
        nn::sf::Out<nn::capsrv::AlbumEntry> outFileEntry,
        const nn::sf::InBuffer& imageBuffer,
        const nn::sf::InBuffer& thumbnailBuffer,
        std::int64_t imageWidth,
        std::int64_t imageHeight,
        std::int64_t thumbnailWidth,
        std::int64_t thumbnailHeight,
        nn::capsrv::detail::ScreenShotAttributeEx0 attribute,
        const nn::capsrv::AlbumFileId& originalFileId
        ) NN_NOEXCEPT
    {
        NN_CAPSRV_IPC_TRACE_SS(SaveEditedScreenShotEx0);
        nn::capsrv::ScreenShotAttribute attr = {};
        NN_RESULT_DO(detail::ScreenShotAttributeUtility::GetScreenShotAttributeFromOriginalAttribute(
            &attr,
            attribute,
            nn::capsrv::AlbumFileDescription_ScreenShotEdited
        ));
        return SaveEditedScreenShotImpl(outFileEntry, imageBuffer, thumbnailBuffer, imageWidth, imageHeight, thumbnailWidth, thumbnailHeight, attr, nullptr, originalFileId);
    }

    nn::Result ScreenShotServiceImpl::SaveEditedScreenShotEx1(
        nn::sf::Out<nn::capsrv::AlbumEntry> outFileEntry,
        const nn::sf::InBuffer& imageBuffer,
        const nn::sf::InBuffer& thumbnailBuffer,
        std::int64_t imageWidth,
        std::int64_t imageHeight,
        std::int64_t thumbnailWidth,
        std::int64_t thumbnailHeight,
        nn::capsrv::detail::ScreenShotAttributeEx0 attribute,
        const nn::capsrv::AppletData& appletData,
        const nn::capsrv::AlbumFileId& originalFileId
        ) NN_NOEXCEPT
    {
        NN_CAPSRV_IPC_TRACE_SS(SaveEditedScreenShotEx0);
        nn::capsrv::ScreenShotAttribute attr = {};
        NN_RESULT_DO(detail::ScreenShotAttributeUtility::GetScreenShotAttributeFromOriginalAttribute(
            &attr,
            attribute,
            nn::capsrv::AlbumFileDescription_ScreenShotEdited
        ));
        return SaveEditedScreenShotImpl(outFileEntry, imageBuffer, thumbnailBuffer, imageWidth, imageHeight, thumbnailWidth, thumbnailHeight, attr, &appletData, originalFileId);
    }

    nn::Result ScreenShotServiceImpl::SaveScreenShotOfMovieEx1(
        nn::sf::Out<nn::capsrv::AlbumEntry> outFileEntry,
        const nn::sf::InBuffer& imageBuffer,
        const nn::sf::InBuffer& thumbnailBuffer,
        std::int64_t imageWidth,
        std::int64_t imageHeight,
        std::int64_t thumbnailWidth,
        std::int64_t thumbnailHeight,
        nn::capsrv::detail::ScreenShotAttributeEx0 attribute,
        const nn::capsrv::AppletData& appletData,
        const nn::capsrv::AlbumFileId& originalFileId
        ) NN_NOEXCEPT
    {
        NN_CAPSRV_IPC_TRACE_SS(SaveScreenShotOfMovieEx1);
        nn::capsrv::ScreenShotAttribute attr = {};
        NN_RESULT_DO(detail::ScreenShotAttributeUtility::GetScreenShotAttributeFromOriginalAttribute(
            &attr,
            attribute,
            nn::capsrv::AlbumFileDescription_ScreenShotEdited
        ));
        return SaveEditedScreenShotImpl(outFileEntry, imageBuffer, thumbnailBuffer, imageWidth, imageHeight, thumbnailWidth, thumbnailHeight, attr, &appletData, originalFileId);
    }


}}}
