﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "visrv_ScreenShotContextManager.h"

#include <mutex>
#include <nn/result/result_HandlingUtility.h>

#include "visrv_ScreenShotServerObject.h"

namespace nn{ namespace capsrv{ namespace server{

    const int ScreenShotContextManager::ContextCount;

    void ScreenShotContextManager::Initialize() NN_NOEXCEPT
    {
        m_Mutex.Initialize();

        for(int i = 0; i < ContextCount; i++)
        {
            m_ContextStatusList[i] = ContextStatus_Free;
            m_ContextList[i].Clear();
        }
    }

    void ScreenShotContextManager::Finalize() NN_NOEXCEPT
    {
    }

    ScreenShotContext* ScreenShotContextManager::AcquireContext() NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

        for(int i = 0; i < ContextCount; i++)
        {
            if(m_ContextStatusList[i] == ContextStatus_Free)
            {
                m_ContextStatusList[i] = ContextStatus_Used;
                return &m_ContextList[i];
            }
        }

        return nullptr;
    }

    void ScreenShotContextManager::ReleaseContext(ScreenShotContext* pContext) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pContext);
        int index =static_cast<int>((reinterpret_cast<uintptr_t>(pContext) - reinterpret_cast<uintptr_t>(&m_ContextList[0])) / sizeof(ScreenShotContext));
        NN_SDK_REQUIRES_RANGE(index, 0, ContextCount);

        std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

        NN_SDK_REQUIRES_EQUAL(pContext, &m_ContextList[index]);
        NN_SDK_REQUIRES_EQUAL(m_ContextStatusList[index], ContextStatus_Used);
        ReleaseContextImpl(index);
    }

    void ScreenShotContextManager::SuspendContext(ScreenShotContext* pContext) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pContext);
        int index =static_cast<int>((reinterpret_cast<uintptr_t>(pContext) - reinterpret_cast<uintptr_t>(&m_ContextList[0])) / sizeof(ScreenShotContext));
        NN_SDK_REQUIRES_RANGE(index, 0, ContextCount);

        std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

        NN_SDK_REQUIRES_EQUAL(pContext, &m_ContextList[index]);
        NN_SDK_REQUIRES_EQUAL(m_ContextStatusList[index], ContextStatus_Used);
        if(pContext->IsRecoveryRequired())
        {
            NN_CAPSRV_LOG_DEV("[context] suspended for recovery.\n");
            m_ContextStatusList[index] = ContextStatus_SuspendedNeedRecovery;
        }
        else
        {
            NN_CAPSRV_LOG_DEV("[context] suspended. task = %d\n", pContext->GetSuspendedTask());
            m_ContextStatusList[index] = ContextStatus_Suspended;
        }
    }

    ScreenShotContext* ScreenShotContextManager::ResumeContext() NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

        for(int i = 0; i < ContextCount; i++)
        {
            if(m_ContextStatusList[i] == ContextStatus_Suspended || m_ContextStatusList[i] == ContextStatus_SuspendedNeedRecovery)
            {
                if(m_ContextStatusList[i] == ContextStatus_SuspendedNeedRecovery)
                {
                    NN_CAPSRV_LOG_DEV("[context] resuming for recovery.\n");
                }
                else if(m_ContextStatusList[i] == ContextStatus_Suspended)
                {
                    NN_CAPSRV_LOG_DEV("[context] resuming. task = %d\n", m_ContextList[i].GetSuspendedTask());
                }
                m_ContextStatusList[i] = ContextStatus_Used;
                return &m_ContextList[i];
            }
        }

        return nullptr;
    }

    ScreenShotContext* ScreenShotContextManager::ResumeContext(ScreenShotContextSuspendedTask task) NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

        for(int i = 0; i < ContextCount; i++)
        {
            if(m_ContextStatusList[i] == ContextStatus_Suspended && m_ContextList[i].GetSuspendedTask() == task)
            {
                NN_CAPSRV_LOG_DEV("[context] resuming. task = %d\n", task);
                m_ContextStatusList[i] = ContextStatus_Used;
                return &m_ContextList[i];
            }
        }

        return nullptr;
    }

    bool ScreenShotContextManager::HasSuspendedContext() NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

        for(int i = 0; i < ContextCount; i++)
        {
            if(m_ContextStatusList[i] == ContextStatus_Suspended || m_ContextStatusList[i] == ContextStatus_SuspendedNeedRecovery)
            {
                return true;
            }
        }
        return false;
    }

    bool ScreenShotContextManager::HasSuspendedContext(ScreenShotContextSuspendedTask task) NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

        for(int i = 0; i < ContextCount; i++)
        {
            if(m_ContextStatusList[i] == ContextStatus_Suspended && m_ContextList[i].GetSuspendedTask() == task)
            {
                return true;
            }
        }
        return false;
    }

    bool ScreenShotContextManager::TryRecoveryAll() NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

        for(int i = 0; i < ContextCount; i++)
        {
            if(m_ContextStatusList[i] == ContextStatus_SuspendedNeedRecovery)
            {
                auto result = m_ContextList[i].TryRecovery();
                if(result == ScreenShotContextRecoveryResult_Failure)
                {
                    return false;
                }
                else if(result == ScreenShotContextRecoveryResult_SuccessToSuspend)
                {
                    m_ContextStatusList[i] = ContextStatus_Suspended;
                }
                else /* ScreenShotContextRecoveryResult_SuccessToRelease */
                {
                    ReleaseContextImpl(i);
                }
            }
        }

        return true;
    }

    int ScreenShotContextManager::GetIndex(ScreenShotContext* pContext) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pContext);
        int index =static_cast<int>((reinterpret_cast<uintptr_t>(pContext) - reinterpret_cast<uintptr_t>(&m_ContextList[0])) / sizeof(ScreenShotContext));
        NN_SDK_REQUIRES_RANGE(index, 0, ContextCount);
        return index;
    }

    void ScreenShotContextManager::ReleaseContextImpl(int index) NN_NOEXCEPT
    {
        m_ContextStatusList[index] = ContextStatus_Free;
        m_ContextList[index].Clear();
    }

    //---------------------------------------------------------------------------------------

    void ScreenShotContextManager::SetupContextForTakingScreenShot(
        ScreenShotContext* pContext,
        uint64_t seqNo,
        nn::ncm::ApplicationId applicationId,
        nn::vi::LayerStack layerStack,
        const nn::capsrv::ScreenShotAttribute& attribute,
        const nn::capsrv::UserIdList& userIdList,
        nn::TimeSpan timeout
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pContext);
        auto& context = *pContext;
        auto& workSafe = g_ScreenShotWorkMemory.cpuOnly.screenShot;
        auto& workUnsafe = g_ScreenShotWorkMemory.deviceShared.screenShot;
        context.SetContextNameForDevelop("TakingScreenShot");
        NN_CAPSRV_SCREENSHOT_CONTEXT_SCOPE(context);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, ScreenShotSequenceNumber);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, ScreenShotSequenceManager);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, ApplicationId);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, ContentsType);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, MakerNoteVersion);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, ScreenShotAttribute);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, AppletData);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, SystemReservedInfo);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, CapturingLayerStack);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, Timeout);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, OverlayNotificationRequest);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, TakingScreenShotReason);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, FileSize);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, CaptureModule);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, DisplayCaptureYuv);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, OverlayNotifier);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, OverlayThumbnailImageBuffer);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, ViewerThumbnailImageBuffer);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_NOMEMORY(context, EncoderWorkMemory);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_NOMEMORY(context, RawOverlayThumbnailMemory);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_NOMEMORY(context, RawViewerThumbnailMemory);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_NOMEMORY(context, EncodedViewerThumbnailMemory);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_NOMEMORY(context, ExifMemory);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_NOMEMORY(context, FileDataMemory);

        // シーケンス
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, ScreenShotSequenceNumber, seqNo);
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, ScreenShotSequenceManager, &g_ScreenShotSequenceManager);
        // ファイル ID
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, ApplicationId, applicationId);
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, ContentsType, AlbumFileContents_ScreenShot);
        // MakerNote
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, MakerNoteVersion, g_ScreenShotEnvironmentInfo.GetMakerNoteVersion());
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, ScreenShotAttribute, attribute);
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, AppletData, {});
        {
            SystemReservedInfo info;
            std::memset(info._reserved, 0, sizeof(info._reserved));
            info.uidList = userIdList;
            NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, SystemReservedInfo, info);
        }
        // キャプチャ
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, CaptureModule, g_ScreenShotCaptureModule.GetModule());
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, DisplayCaptureYuv, g_ScreenShotCaptureModule.GetScreenShotFullSizeCapture());
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, CapturingLayerStack, layerStack);
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, Timeout, timeout);
        // データ生成
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, FileSize, -1); // あとで上書きされる
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, OverlayThumbnailImageBuffer, g_ScreenShotCaptureModule.GetScreenShotOverlayThumbnailImage());
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, ViewerThumbnailImageBuffer, g_ScreenShotCaptureModule.GetScreenShotViewerThumbnailImage());
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET_MEMORY(context, EncoderWorkMemory, workSafe.jpegEncoderWorkMemory, sizeof(workSafe.jpegEncoderWorkMemory));
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET_MEMORY(context, RawOverlayThumbnailMemory, workUnsafe.overlayThumbnailRawData, sizeof(workUnsafe.overlayThumbnailRawData));
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET_MEMORY(context, RawViewerThumbnailMemory, workUnsafe.viewerThumbnailRawData, sizeof(workUnsafe.viewerThumbnailRawData));
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET_MEMORY(context, EncodedViewerThumbnailMemory, workUnsafe.viewerThumbnailJpegData, sizeof(workUnsafe.viewerThumbnailJpegData));
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET_MEMORY(context, ExifMemory, workSafe.exifWorkMemory, sizeof(workSafe.exifWorkMemory));
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET_MEMORY(context, FileDataMemory, workUnsafe.jpegData, sizeof(workUnsafe.jpegData));
        // オーバーレイ通知
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, OverlayNotifier, &g_OverlayNotifier);
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, OverlayNotificationRequest, OverlayNotificationRequest_All);
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, TakingScreenShotReason, nn::ovln::format::ScreenShotReason::CaptureButtonPressed);
    }

    void ScreenShotContextManager::SetupContextForSavingEditedScreenShot(
        ScreenShotContext* pContext,
        const void* pFullData,
        size_t fullDataSize,
        const void* pThumbData,
        size_t thumbDataSize,
        const nn::capsrv::ScreenShotAttribute& attribute,
        const nn::capsrv::AppletData* pAppletData,
        nn::ncm::ApplicationId applicationId
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pContext);
        auto& context = *pContext;
        auto& workSafe   = g_ScreenShotWorkMemory.cpuOnly.screenShot;
        auto& workUnsafe = g_ScreenShotWorkMemory.deviceShared.screenShot;
        nn::capsrv::AppletData appletData = {};
        if(pAppletData)
        {
            appletData = *pAppletData;
        }
        context.SetContextNameForDevelop("SavingEditedScreenShot");
        NN_CAPSRV_SCREENSHOT_CONTEXT_SCOPE(context);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, ApplicationId);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, ContentsType);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, MakerNoteVersion);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, ScreenShotAttribute);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, AppletData);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, SystemReservedInfo);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, FileSize);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_NOMEMORY(context, EncoderWorkMemory);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_NOMEMORY(context, RawDisplayCaptureRgbaMemory);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_NOMEMORY(context, RawViewerThumbnailMemory);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_NOMEMORY(context, EncodedViewerThumbnailMemory);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_NOMEMORY(context, ExifMemory);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_NOMEMORY(context, FileDataMemory);

        // ファイル ID
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, ApplicationId, applicationId);
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, ContentsType, AlbumFileContents_ScreenShot);
        // MakerNote
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, MakerNoteVersion, g_ScreenShotEnvironmentInfo.GetMakerNoteVersion());
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, ScreenShotAttribute, attribute);
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, AppletData, appletData);
        {
            SystemReservedInfo info;
            std::memset(info._reserved, 0, sizeof(info._reserved));
            // TORIAEZU:
            //  本当は編集元ファイルの UidList を引き継ぐべき
            info.uidList = {};
            NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, SystemReservedInfo, info);
        }
        // データ生成
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, FileSize, -1);
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET_MEMORY(context, EncoderWorkMemory, workSafe.jpegEncoderWorkMemory, sizeof(workSafe.jpegEncoderWorkMemory));
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET_MEMORY(context, RawDisplayCaptureRgbaMemory, const_cast<void*>(pFullData), fullDataSize);
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET_MEMORY(context, RawViewerThumbnailMemory, const_cast<void*>(pThumbData), thumbDataSize);
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET_MEMORY(context, EncodedViewerThumbnailMemory, workUnsafe.viewerThumbnailJpegData, sizeof(workUnsafe.viewerThumbnailJpegData));
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET_MEMORY(context, ExifMemory, workSafe.exifWorkMemory, sizeof(workSafe.exifWorkMemory));
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET_MEMORY(context, FileDataMemory, workUnsafe.jpegData, sizeof(workUnsafe.jpegData));
    }

    void ScreenShotContextManager::SetupContextForSavingScreenShotOfMovie(
        ScreenShotContext* pContext,
        const void* pFullData,
        size_t fullDataSize,
        const void* pThumbData,
        size_t thumbDataSize,
        const nn::capsrv::ScreenShotAttribute& attribute,
        const nn::capsrv::AppletData* pAppletData,
        nn::ncm::ApplicationId applicationId
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pContext);
        auto& context = *pContext;
        auto& workSafe   = g_ScreenShotWorkMemory.cpuOnly.screenShot;
        auto& workUnsafe = g_ScreenShotWorkMemory.deviceShared.screenShot;
        nn::capsrv::AppletData appletData = {};
        if(pAppletData)
        {
            appletData = *pAppletData;
        }
        context.SetContextNameForDevelop("SavingScreenShotOfMovie");
        NN_CAPSRV_SCREENSHOT_CONTEXT_SCOPE(context);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, ApplicationId);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, ContentsType);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, MakerNoteVersion);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, ScreenShotAttribute);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, AppletData);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, SystemReservedInfo);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, FileSize);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_NOMEMORY(context, EncoderWorkMemory);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_NOMEMORY(context, RawDisplayCaptureRgbaMemory);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_NOMEMORY(context, RawViewerThumbnailMemory);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_NOMEMORY(context, EncodedViewerThumbnailMemory);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_NOMEMORY(context, ExifMemory);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_NOMEMORY(context, FileDataMemory);

        // ファイル ID
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, ApplicationId, applicationId);
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, ContentsType, AlbumFileContents_ScreenShot);
        // MakerNote
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, MakerNoteVersion, g_ScreenShotEnvironmentInfo.GetMakerNoteVersion());
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, ScreenShotAttribute, attribute);
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, AppletData, appletData);
        {
            SystemReservedInfo info;
            std::memset(info._reserved, 0, sizeof(info._reserved));
            // TORIAEZU:
            //  本当は編集元ファイルの UidList を引き継ぐべき
            info.uidList = {};
            NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, SystemReservedInfo, info);
        }
        // データ生成
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, FileSize, -1);
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET_MEMORY(context, EncoderWorkMemory, workSafe.jpegEncoderWorkMemory, sizeof(workSafe.jpegEncoderWorkMemory));
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET_MEMORY(context, RawDisplayCaptureRgbaMemory, const_cast<void*>(pFullData), fullDataSize);
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET_MEMORY(context, RawViewerThumbnailMemory, const_cast<void*>(pThumbData), thumbDataSize);
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET_MEMORY(context, EncodedViewerThumbnailMemory, workUnsafe.viewerThumbnailJpegData, sizeof(workUnsafe.viewerThumbnailJpegData));
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET_MEMORY(context, ExifMemory, workSafe.exifWorkMemory, sizeof(workSafe.exifWorkMemory));
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET_MEMORY(context, FileDataMemory, workUnsafe.jpegData, sizeof(workUnsafe.jpegData));
    }

    void ScreenShotContextManager::SetupContextForSavingApplicationScreenShotAruid(
        ScreenShotContext* pContext,
        const void* pFullData,
        size_t fullDataSize,
        const nn::capsrv::ScreenShotAttribute& attribute,
        const nn::capsrv::UserIdList& userIdList,
        nn::applet::AppletResourceUserId aruid,
        OverlayNotificationRequestType overlayRequest
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pContext);
        auto& context = *pContext;
        auto& workSafe   = g_ScreenShotWorkMemory.cpuOnly.screenShot;
        auto& workUnsafe = g_ScreenShotWorkMemory.deviceShared.screenShot;
        context.SetContextNameForDevelop("SavingApplicationScreenShotAruid");
        NN_CAPSRV_SCREENSHOT_CONTEXT_SCOPE(context);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, Aruid);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, ContentsType);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, MakerNoteVersion);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, ScreenShotAttribute);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, AppletData);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, SystemReservedInfo);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, FileSize);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, OverlayNotificationRequest);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, OverlayNotifier);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, TakingScreenShotReason);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_NOMEMORY(context, EncoderWorkMemory);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_NOMEMORY(context, RawDisplayCaptureRgbaMemory);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_NOMEMORY(context, RawViewerThumbnailMemory);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_NOMEMORY(context, EncodedViewerThumbnailMemory);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_NOMEMORY(context, ExifMemory);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_NOMEMORY(context, FileDataMemory);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_NOMEMORY(context, RawOverlayThumbnailMemory);

        // ファイル ID
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, Aruid, aruid);
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, ContentsType, AlbumFileContents_ScreenShot);
        // MakerNote
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, MakerNoteVersion, g_ScreenShotEnvironmentInfo.GetMakerNoteVersion());
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, ScreenShotAttribute, attribute);
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, AppletData, {});
        {
            SystemReservedInfo info;
            std::memset(info._reserved, 0, sizeof(info._reserved));
            info.uidList = userIdList;
            NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, SystemReservedInfo, info);
        }
        // データ生成
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, FileSize, -1);
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET_MEMORY(context, EncoderWorkMemory, workSafe.jpegEncoderWorkMemory, sizeof(workSafe.jpegEncoderWorkMemory));
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET_MEMORY(context, RawDisplayCaptureRgbaMemory, const_cast<void*>(pFullData), fullDataSize);
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET_MEMORY(context, RawViewerThumbnailMemory, workUnsafe.viewerThumbnailRawData, sizeof(workUnsafe.viewerThumbnailRawData));
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET_MEMORY(context, EncodedViewerThumbnailMemory, workUnsafe.viewerThumbnailJpegData, sizeof(workUnsafe.viewerThumbnailJpegData));
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET_MEMORY(context, ExifMemory, workSafe.exifWorkMemory, sizeof(workSafe.exifWorkMemory));
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET_MEMORY(context, FileDataMemory, workUnsafe.jpegData, sizeof(workUnsafe.jpegData));
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET_MEMORY(context, RawOverlayThumbnailMemory, workUnsafe.overlayThumbnailRawData, sizeof(workUnsafe.overlayThumbnailRawData));
        // オーバーレイ通知
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, OverlayNotifier, &g_OverlayNotifier);
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, OverlayNotificationRequest, overlayRequest);
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, TakingScreenShotReason, nn::ovln::format::ScreenShotReason::SaveApiCalledByApplication);
    }

    void ScreenShotContextManager::SetupContextForOverlayMovieThumbnail(
        ScreenShotContext* pContext,
        capture::ImageBuffer* pImageBufferY,
        capture::ImageBuffer* pImageBufferUv,
        const nn::capsrv::AlbumFileId& fileId
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pContext);
        auto& context = *pContext;
        auto& workUnsafe = g_ScreenShotWorkMemory.deviceShared.screenShot;
        context.SetContextNameForDevelop("OverlayMovieThumbnail");
        NN_CAPSRV_SCREENSHOT_CONTEXT_SCOPE(context);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, ImageBufferNv12Y);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, ImageBufferNv12Uv);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, AlbumFileId);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, OverlayThumbnailImageBuffer);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, CaptureModule);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_NOMEMORY(context, RawOverlayThumbnailMemory);

        // 入力
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, ImageBufferNv12Y, pImageBufferY);
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, ImageBufferNv12Uv, pImageBufferUv);
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, AlbumFileId, fileId);
        // 出力・中間
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, OverlayThumbnailImageBuffer, g_ScreenShotCaptureModule.GetScreenShotOverlayThumbnailImage());
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET_MEMORY(context, RawOverlayThumbnailMemory, workUnsafe.overlayThumbnailRawData, sizeof(workUnsafe.overlayThumbnailRawData));
        // VIC 用
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, CaptureModule, g_ScreenShotCaptureModule.GetModule());


#if defined(NN_CAPSRV_DEBUG_SEND_SCREENSHOT_OVLN_IN_MOVIE_THUMB_CREATION)
        // デバッグ用: 静止画のオーバーレイ通知を出させる
        pContext->SetOverlayNotificationRequest(OverlayNotificationRequest_All);
        pContext->SetTakingScreenShotReason(nn::ovln::format::ScreenShotReason::SaveApiCalledByApplication);
        pContext->SetOverlayNotifier(&g_OverlayNotifier);
#endif
    }

    void ScreenShotContextManager::SetupContextForProtoMovieMetaDataNv12(
        ScreenShotContext* pContext,
        void* pOutBuffer,
        size_t outBufferSize,
        capture::ImageBuffer* pImageBufferY,
        capture::ImageBuffer* pImageBufferUv,
        const nn::capsrv::AlbumFileId& fileId,
        const nn::capsrv::ScreenShotAttribute& attribute,
        const nn::capsrv::AppletData* pAppletData,
        const nn::capsrv::ApplicationData* pApplicationData
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pContext);
        auto& context = *pContext;
        auto& workSafe = g_ScreenShotWorkMemory.cpuOnly.screenShot;
        auto& workUnsafe = g_ScreenShotWorkMemory.deviceShared.screenShot;
        context.SetContextNameForDevelop("ProtoMovieMetaData");
        NN_CAPSRV_SCREENSHOT_CONTEXT_SCOPE(context);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, ImageBufferNv12Y);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, ImageBufferNv12Uv);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, AlbumFileId);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, FileSize);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, CaptureModule);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, MakerNoteVersion);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, ScreenShotAttribute);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, AppletData);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, ApplicationData);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, SystemReservedInfo);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, ViewerThumbnailImageBuffer);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_NOMEMORY(context, FileDataMemory);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_NOMEMORY(context, EncoderWorkMemory);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_NOMEMORY(context, RawViewerThumbnailMemory);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_NOMEMORY(context, EncodedViewerThumbnailMemory);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_NOMEMORY(context, ExifMemory);

        // 入力
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, AlbumFileId, fileId);
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, ImageBufferNv12Y, pImageBufferY);
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, ImageBufferNv12Uv, pImageBufferUv);
        // MakerNote
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, MakerNoteVersion, g_ScreenShotEnvironmentInfo.GetMakerNoteVersion());
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, ScreenShotAttribute, attribute);
        if(pAppletData)
        {
            NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, AppletData, *pAppletData);
        }
        else
        {
            NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, AppletData, {});
        }
        if(pApplicationData)
        {
            NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, ApplicationData, *pApplicationData);
        }
        else
        {
            NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, ApplicationData, {});
        }
        {
            SystemReservedInfo info;
            std::memset(info._reserved, 0, sizeof(info._reserved));
            // TODO:
            //  am→grc から渡された UidList を保存すべき
            info.uidList = {};
            NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, SystemReservedInfo, info);
        }
        // VIC 用
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, CaptureModule, g_ScreenShotCaptureModule.GetModule());
        // データ生成
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, FileSize, -1);
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, ViewerThumbnailImageBuffer, g_ScreenShotCaptureModule.GetScreenShotViewerThumbnailImage());
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET_MEMORY(context, EncoderWorkMemory, workSafe.jpegEncoderWorkMemory, sizeof(workSafe.jpegEncoderWorkMemory));
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET_MEMORY(context, RawViewerThumbnailMemory, workUnsafe.viewerThumbnailRawData, sizeof(workUnsafe.viewerThumbnailRawData));
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET_MEMORY(context, EncodedViewerThumbnailMemory, workUnsafe.viewerThumbnailJpegData, sizeof(workUnsafe.viewerThumbnailJpegData));
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET_MEMORY(context, ExifMemory, workSafe.exifWorkMemory, sizeof(workSafe.exifWorkMemory));
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET_MEMORY(context, FileDataMemory, pOutBuffer, outBufferSize);
    }

    void ScreenShotContextManager::SetupContextForProtoMovieMetaDataRgba(
        ScreenShotContext* pContext,
        void* pOutBuffer,
        size_t outBufferSize,
        const void* pFullData,
        size_t fullDataSize,
        const nn::capsrv::AlbumFileId& fileId,
        const nn::capsrv::ScreenShotAttribute& attribute,
        const nn::capsrv::AppletData* pAppletData,
        const nn::capsrv::ApplicationData* pApplicationData
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pContext);
        auto& context = *pContext;
        auto& workSafe   = g_ScreenShotWorkMemory.cpuOnly.screenShot;
        auto& workUnsafe = g_ScreenShotWorkMemory.deviceShared.screenShot;
        context.SetContextNameForDevelop("ProtoMovieMetaRgba");
        NN_CAPSRV_SCREENSHOT_CONTEXT_SCOPE(context);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, AlbumFileId);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, MakerNoteVersion);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, AppletData);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, ApplicationData);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, SystemReservedInfo);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, ScreenShotAttribute);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, FileSize);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_NOMEMORY(context, EncoderWorkMemory);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_NOMEMORY(context, RawDisplayCaptureRgbaMemory);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_NOMEMORY(context, RawViewerThumbnailMemory);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_NOMEMORY(context, EncodedViewerThumbnailMemory);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_NOMEMORY(context, ExifMemory);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_NOMEMORY(context, FileDataMemory);

        // ファイル ID
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, AlbumFileId, fileId);
        // MakerNote
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, MakerNoteVersion, g_ScreenShotEnvironmentInfo.GetMakerNoteVersion());
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, ScreenShotAttribute, attribute);
        if(pAppletData)
        {
            NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, AppletData, *pAppletData);
        }
        else
        {
            NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, AppletData, {});
        }
        if(pApplicationData)
        {
            NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, ApplicationData, *pApplicationData);
        }
        else
        {
            NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, ApplicationData, {});
        }
        {
            SystemReservedInfo info;
            std::memset(info._reserved, 0, sizeof(info._reserved));
            // TODO:
            //  am→grc から渡された UidList を保存すべき
            info.uidList = {};
            NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, SystemReservedInfo, info);
        }
        // データ生成
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, FileSize, -1);
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET_MEMORY(context, EncoderWorkMemory, workSafe.jpegEncoderWorkMemory, sizeof(workSafe.jpegEncoderWorkMemory));
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET_MEMORY(context, RawDisplayCaptureRgbaMemory, const_cast<void*>(pFullData), fullDataSize);
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET_MEMORY(context, RawViewerThumbnailMemory, workUnsafe.viewerThumbnailRawData, sizeof(workUnsafe.viewerThumbnailRawData));
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET_MEMORY(context, EncodedViewerThumbnailMemory, workUnsafe.viewerThumbnailJpegData, sizeof(workUnsafe.viewerThumbnailJpegData));
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET_MEMORY(context, ExifMemory, workSafe.exifWorkMemory, sizeof(workSafe.exifWorkMemory));
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET_MEMORY(context, FileDataMemory, pOutBuffer, outBufferSize);
    }

    void ScreenShotContextManager::SetupContextForRawScreenShotReadStream(
        ScreenShotContext* pContext,
        nn::vi::LayerStack layerStack,
        nn::TimeSpan timeout
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pContext);
        auto& context = *pContext;
        context.SetContextNameForDevelop("RawScreenShotReadStream");
        NN_CAPSRV_SCREENSHOT_CONTEXT_SCOPE(context);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, CapturingLayerStack);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, Timeout);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, CaptureModule);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, DisplayCaptureRgba);

        // キャプチャ
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, CapturingLayerStack, layerStack);
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, DisplayCaptureRgba, g_ScreenShotCaptureModule.GetRawScreenShotCapture());
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, Timeout, timeout);
        NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, CaptureModule, g_ScreenShotCaptureModule.GetModule());
    }

}}}
