﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/util/util_Optional.h>
#include <nn/vi/vi_LayerStack.h>
#include <nn/applet/applet_Types.h>
#include <nn/capsrv/capsrv_AlbumEntry.h>
#include <nn/capsrv/capsrv_ApplicationAlbumEntry.h>
#include <nn/capsrv/capsrv_ScreenShotOrientation.h>
#include <nn/capsrv/capsrv_OverlayNotificationRequest.h>
#include <nn/capsrv/capsrv_ScreenShotAttribute.h>
#include "../../capsrv_Assert.h"
#include "../../capture/capsrv_CaptureModule.h"
#include "../../capture/capsrv_DisplayCapture.h"
#include "../../capture/capsrv_DisplayBuffer.h"
#include "../../capture/capsrv_ImageBuffer.h"
#include "../screenshot/visrv_ScreenShotSequenceManager.h"
#include "../detail/visrv_OverlayNotifier.h"
#include "../detail/capsrvServer_MakerNoteInfo.h"

// ScreenShotContext 使い方
// なんでもかんでも Context に放り込んでいるので入出力関係を毎回宣言して使うようにする。
//
// ScreenShotContext& context;
// {
//   // 必ず最初に SCOPE を宣言
//   NN_CAPSRV_SCREENSHOT_CONTEXT_SCOPE(context);
//
//   // このスコープで使用する context の値を宣言
//   NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_IN(context, CaptureModule); // context の値を読込む
//   NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, ImageBuffer);  // context に値を保存する
//
//   // アクセスするときもマクロ経由
//   NN_CAPSRV_SCREENSHOT_CONTEXT_GET(pModule, context, CaptureModule); // auto pModule = context.GetCaptureModule();
//
//   NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, ImageBuffer, pImageBuffer); // context.SetImageBuffer(pImageBuffer);
//
// }
//
// 使い方がおかしい場合はコンパイルエラーで止まる。

//----------------------------------------
#define NN_CAPSRV_SCREENSHOT_CONTEXT_SCOPE(context) \
    nn::capsrv::server::ScreenShotContextTokenType contextToken = 0;


#define NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_IN(context, Name)   \
    bool is##Name##MarkedAsInput = true;    \
    NN_ABORT_UNLESS(context.Is##Name##Valid(contextToken), "[in](%s)%s is not set", context.GetContextNameForDevelop(), #Name);

#define NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_IN_OPTIONAL(context, Name)   \
    bool is##Name##MarkedAsOptionalInput = true;

#define NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OUT(context, Name)   \
    bool is##Name##MarkedAsOutput = true;   \
    NN_ABORT_UNLESS(!context.Is##Name##Valid(contextToken), "[out](%s)%s is already set", context.GetContextNameForDevelop(), #Name);

#define NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_INOUT(context, Name)   \
    bool is##Name##MarkedAsInput = true;    \
    bool is##Name##MarkedAsOutput = true;   \
    NN_ABORT_UNLESS(context.Is##Name##Valid(contextToken), "[inout](%s)%s is not set", context.GetContextNameForDevelop(), #Name);

#define NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OVERWRITEABLE(context, Name)   \
    bool is##Name##MarkedAsOverwritable = true;   \
    NN_ABORT_UNLESS(context.Is##Name##Valid(contextToken), "[overwrite](%s)%s is not set", context.GetContextNameForDevelop(), #Name);

#define NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_MEMORY(context, Name, requiredSize)   \
    bool is##Name##MarkedAsMemory = true;   \
    NN_ABORT_UNLESS(context.Is##Name##ValidMemory(contextToken, requiredSize), "[memory](%s)%s is nullptr or too small", context.GetContextNameForDevelop(), #Name);

#define NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_NOMEMORY(context, Name)   \
    bool is##Name##MarkedAsNoMemory = true;   \
    NN_ABORT_UNLESS(!context.Is##Name##ValidMemory(contextToken, 0), "[nomemory](%s)%s should be nullptr", context.GetContextNameForDevelop(), #Name);
//---------------------------------------

#define NN_CAPSRV_SCREENSHOT_CONTEXT_GET(varName, context, Name)    \
    NN_UNUSED(is##Name##MarkedAsInput); \
    auto varName = context.Get##Name(contextToken);

#define NN_CAPSRV_SCREENSHOT_CONTEXT_GET_OPTIONAL(varName, context, Name)    \
    NN_UNUSED(is##Name##MarkedAsOptionalInput); \
    auto varName = context.Get##Name##Optional(contextToken);

#define NN_CAPSRV_SCREENSHOT_CONTEXT_SET(context, Name, value)  \
    NN_UNUSED(is##Name##MarkedAsOutput);    \
    context.Set##Name(value, contextToken);

#define NN_CAPSRV_SCREENSHOT_CONTEXT_RESET(context, Name)  \
    NN_UNUSED(is##Name##MarkedAsOutput);    \
    context.Reset##Name(contextToken);

#define NN_CAPSRV_SCREENSHOT_CONTEXT_OVERWRITE(context, Name, value)  \
    NN_UNUSED(is##Name##MarkedAsOverwritable);    \
    context.Set##Name(value, contextToken);

#define NN_CAPSRV_SCREENSHOT_CONTEXT_GET_MEMORY(varNamePointer, varNameSize, context, Name) \
    NN_UNUSED(is##Name##MarkedAsMemory);    \
    void* varNamePointer = nullptr; \
    size_t varNameSize = 0; \
    context.Get##Name(&varNamePointer, &varNameSize, contextToken);

#define NN_CAPSRV_SCREENSHOT_CONTEXT_GET_MEMORY(varNamePointer, varNameSize, context, Name) \
    NN_UNUSED(is##Name##MarkedAsMemory);    \
    void* varNamePointer = nullptr; \
    size_t varNameSize = 0; \
    context.Get##Name(&varNamePointer, &varNameSize, contextToken);

#define NN_CAPSRV_SCREENSHOT_CONTEXT_SET_MEMORY(context, Name, pMemory, size) \
    NN_UNUSED(is##Name##MarkedAsNoMemory);    \
    context.Set##Name(pMemory, size, contextToken);

//--------------------------------------

#define NN_CAPSRV_SCREENSHOT_CONTEXT_DEFINE_VALUE_PROPERTY(value_type, Name)    \
    public: \
        bool Is##Name##Valid(nn::capsrv::server::ScreenShotContextTokenType token) const NN_NOEXCEPT \
        {   \
            NN_UNUSED(token);   \
            return static_cast<bool>(m_##Name); \
        }   \
        value_type Get##Name(nn::capsrv::server::ScreenShotContextTokenType token) const NN_NOEXCEPT    \
        {   \
            NN_UNUSED(token);   \
            return *m_##Name;   \
        }   \
        nn::util::optional<value_type> Get##Name##Optional(nn::capsrv::server::ScreenShotContextTokenType token) const NN_NOEXCEPT    \
        {   \
            NN_UNUSED(token);   \
            return m_##Name;   \
        }   \
        void Set##Name(const value_type& value, nn::capsrv::server::ScreenShotContextTokenType token) NN_NOEXCEPT   \
        {   \
            NN_UNUSED(token);   \
            m_##Name = value;   \
        }   \
        void Reset##Name(nn::capsrv::server::ScreenShotContextTokenType token) NN_NOEXCEPT  \
        {   \
            NN_UNUSED(token);   \
            m_##Name = nn::util::nullopt;   \
        }   \
    private:    \
        nn::util::optional<value_type> m_##Name;

#define NN_CAPSRV_SCREENSHOT_CONTEXT_DEFINE_POINTER_PROPERTY(pointer_type, Name)    \
    public: \
        bool Is##Name##Valid(nn::capsrv::server::ScreenShotContextTokenType token) const NN_NOEXCEPT \
        {   \
            NN_UNUSED(token);   \
            return m_p##Name != nullptr;    \
        }   \
        pointer_type Get##Name(nn::capsrv::server::ScreenShotContextTokenType token) const NN_NOEXCEPT  \
        {   \
            NN_UNUSED(token);   \
            return m_p##Name;   \
        }   \
        void Set##Name(pointer_type pValue, nn::capsrv::server::ScreenShotContextTokenType token) NN_NOEXCEPT \
        {   \
            NN_UNUSED(token);   \
            m_p##Name = pValue; \
        }    \
        void Reset##Name(nn::capsrv::server::ScreenShotContextTokenType token) NN_NOEXCEPT \
        {   \
            NN_UNUSED(token);   \
            m_p##Name = nullptr; \
        }    \
    private:    \
        pointer_type m_p##Name;

#define NN_CAPSRV_SCREENSHOT_CONTEXT_DEFINE_MEMORY_PROPERTY(Name)   \
    public: \
        bool Is##Name##ValidMemory(size_t requiredSize, nn::capsrv::server::ScreenShotContextTokenType token) const NN_NOEXCEPT \
        {   \
            NN_UNUSED(token);   \
            return m_p##Name != nullptr && m_##Name##Size >= requiredSize;  \
        }   \
        void Get##Name(void** pOutValue, size_t* pOutSize, nn::capsrv::server::ScreenShotContextTokenType token) const NN_NOEXCEPT  \
        {   \
            NN_UNUSED(token);   \
            *pOutValue = m_p##Name; \
            *pOutSize = m_##Name##Size; \
        }   \
        void Set##Name(void* memory, size_t size, nn::capsrv::server::ScreenShotContextTokenType token) NN_NOEXCEPT   \
        {   \
            NN_UNUSED(token);   \
            m_p##Name = memory; \
            m_##Name##Size = size;  \
        }   \
    private:    \
        void*   m_p##Name;  \
        size_t  m_##Name##Size;


namespace nn{ namespace image{
    class ExifBuilder;
}}

namespace nn{ namespace capsrv{ namespace server{
    typedef int ScreenShotContextTokenType;

    enum ScreenShotContextSuspendedTask
    {
        ScreenShotContextSuspendedTask_None,
        ScreenShotContextSuspendedTask_RawScreenShotReadStream,
    };

    enum ScreenShotContextRecoveryResult
    {
        ScreenShotContextRecoveryResult_Failure,
        ScreenShotContextRecoveryResult_SuccessToSuspend,
        ScreenShotContextRecoveryResult_SuccessToRelease,
    };

    class ScreenShotContext
    {
    public:
        ScreenShotContext() NN_NOEXCEPT
        {
            Clear();
        }

        void Clear() NN_NOEXCEPT
        {
            std::memset(this, 0, sizeof(*this));
        }

        NN_CAPSRV_SCREENSHOT_CONTEXT_DEFINE_VALUE_PROPERTY(uint64_t,                            ScreenShotSequenceNumber);
        NN_CAPSRV_SCREENSHOT_CONTEXT_DEFINE_VALUE_PROPERTY(nn::applet::AppletResourceUserId,    Aruid);
        NN_CAPSRV_SCREENSHOT_CONTEXT_DEFINE_VALUE_PROPERTY(nn::ncm::ApplicationId,              ApplicationId);
        NN_CAPSRV_SCREENSHOT_CONTEXT_DEFINE_VALUE_PROPERTY(AlbumFileContentsType,               ContentsType);
        NN_CAPSRV_SCREENSHOT_CONTEXT_DEFINE_VALUE_PROPERTY(detail::MakerNoteVersionType,        MakerNoteVersion);
        NN_CAPSRV_SCREENSHOT_CONTEXT_DEFINE_VALUE_PROPERTY(AppletData,                          AppletData);
        NN_CAPSRV_SCREENSHOT_CONTEXT_DEFINE_VALUE_PROPERTY(ApplicationData,                     ApplicationData);
        NN_CAPSRV_SCREENSHOT_CONTEXT_DEFINE_VALUE_PROPERTY(SystemReservedInfo,                  SystemReservedInfo);
        NN_CAPSRV_SCREENSHOT_CONTEXT_DEFINE_VALUE_PROPERTY(ScreenShotAttribute,                 ScreenShotAttribute);
        NN_CAPSRV_SCREENSHOT_CONTEXT_DEFINE_VALUE_PROPERTY(nn::vi::LayerStack,                  CapturingLayerStack);
        NN_CAPSRV_SCREENSHOT_CONTEXT_DEFINE_VALUE_PROPERTY(nn::TimeSpan,                        Timeout);
        NN_CAPSRV_SCREENSHOT_CONTEXT_DEFINE_VALUE_PROPERTY(AlbumFileId,                         AlbumFileId);
        NN_CAPSRV_SCREENSHOT_CONTEXT_DEFINE_VALUE_PROPERTY(ApplicationAlbumEntry,               ApplicationAlbumEntry);
        NN_CAPSRV_SCREENSHOT_CONTEXT_DEFINE_VALUE_PROPERTY(int64_t,                             FileSize);
        NN_CAPSRV_SCREENSHOT_CONTEXT_DEFINE_VALUE_PROPERTY(int64_t,                             MakerNoteOffset);
        NN_CAPSRV_SCREENSHOT_CONTEXT_DEFINE_VALUE_PROPERTY(int64_t,                             MakerNoteSize);
        NN_CAPSRV_SCREENSHOT_CONTEXT_DEFINE_VALUE_PROPERTY(nn::Result,                          Result);
        NN_CAPSRV_SCREENSHOT_CONTEXT_DEFINE_VALUE_PROPERTY(size_t,                              EncodedViewerThumbnailSize);
        NN_CAPSRV_SCREENSHOT_CONTEXT_DEFINE_VALUE_PROPERTY(int,                                 SharedTextureIndex);
        NN_CAPSRV_SCREENSHOT_CONTEXT_DEFINE_POINTER_PROPERTY(nn::image::ExifBuilder*,           ExifBuilder);
        NN_CAPSRV_SCREENSHOT_CONTEXT_DEFINE_POINTER_PROPERTY(capture::CaptureModule*,           CaptureModule);
        NN_CAPSRV_SCREENSHOT_CONTEXT_DEFINE_POINTER_PROPERTY(capture::DisplayCapture*,          DisplayCaptureYuv);
        NN_CAPSRV_SCREENSHOT_CONTEXT_DEFINE_POINTER_PROPERTY(capture::DisplayCapture*,          DisplayCaptureRgba);
        NN_CAPSRV_SCREENSHOT_CONTEXT_DEFINE_POINTER_PROPERTY(capture::DisplayBuffer*,           DisplayBufferYuv);
        NN_CAPSRV_SCREENSHOT_CONTEXT_DEFINE_POINTER_PROPERTY(capture::DisplayBuffer*,           DisplayBufferRgba);
        NN_CAPSRV_SCREENSHOT_CONTEXT_DEFINE_POINTER_PROPERTY(capture::ImageBuffer*,             ImageBufferNv12Y);
        NN_CAPSRV_SCREENSHOT_CONTEXT_DEFINE_POINTER_PROPERTY(capture::ImageBuffer*,             ImageBufferNv12Uv);
        NN_CAPSRV_SCREENSHOT_CONTEXT_DEFINE_POINTER_PROPERTY(capture::ImageBuffer*,             OverlayThumbnailImageBuffer);
        NN_CAPSRV_SCREENSHOT_CONTEXT_DEFINE_POINTER_PROPERTY(capture::ImageBuffer*,             ViewerThumbnailImageBuffer);
        NN_CAPSRV_SCREENSHOT_CONTEXT_DEFINE_POINTER_PROPERTY(screenshot::ScreenShotSequenceManager*,  ScreenShotSequenceManager);
        NN_CAPSRV_SCREENSHOT_CONTEXT_DEFINE_POINTER_PROPERTY(OverlayNotifier*,                  OverlayNotifier);
        NN_CAPSRV_SCREENSHOT_CONTEXT_DEFINE_VALUE_PROPERTY(OverlayNotificationRequestType,      OverlayNotificationRequest);
        NN_CAPSRV_SCREENSHOT_CONTEXT_DEFINE_VALUE_PROPERTY(nn::ovln::format::ScreenShotReason,  TakingScreenShotReason);
        NN_CAPSRV_SCREENSHOT_CONTEXT_DEFINE_VALUE_PROPERTY(AlbumFileId,                         OverlayAlbumFileId);
        NN_CAPSRV_SCREENSHOT_CONTEXT_DEFINE_MEMORY_PROPERTY(ExifMemory);
        NN_CAPSRV_SCREENSHOT_CONTEXT_DEFINE_MEMORY_PROPERTY(EncoderWorkMemory);
        NN_CAPSRV_SCREENSHOT_CONTEXT_DEFINE_MEMORY_PROPERTY(RawOverlayThumbnailMemory);
        NN_CAPSRV_SCREENSHOT_CONTEXT_DEFINE_MEMORY_PROPERTY(RawViewerThumbnailMemory);
        NN_CAPSRV_SCREENSHOT_CONTEXT_DEFINE_MEMORY_PROPERTY(EncodedViewerThumbnailMemory);
        NN_CAPSRV_SCREENSHOT_CONTEXT_DEFINE_MEMORY_PROPERTY(FileDataMemory);
        NN_CAPSRV_SCREENSHOT_CONTEXT_DEFINE_MEMORY_PROPERTY(RawDisplayCaptureRgbaMemory);

    public:
        bool IsRecoveryRequired() const NN_NOEXCEPT { return m_IsRecoveryRequired; }

        void MarkRecoveryRequired(
            ScreenShotContextRecoveryResult (*pRecovertyFunction)(void*),
            void* pRecovertyParameter
        ) NN_NOEXCEPT
        {
            m_IsRecoveryRequired = true;
            m_pRecoveryFunction  = pRecovertyFunction;
            m_pRecoveryParameter = pRecovertyParameter;
        }

        ScreenShotContextRecoveryResult TryRecovery() const NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(m_pRecoveryFunction);
            return m_pRecoveryFunction(m_pRecoveryParameter);
        }

        bool IsSuspendRequired() const NN_NOEXCEPT { return m_SuspendedTask != ScreenShotContextSuspendedTask_None; }

        void MarkSuspendRequired(ScreenShotContextSuspendedTask task) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES(m_SuspendedTask == ScreenShotContextSuspendedTask_None || m_SuspendedTask == task);
            m_SuspendedTask = task;
        }

        void UnmarkSuspendRequired() NN_NOEXCEPT
        {
            m_SuspendedTask = ScreenShotContextSuspendedTask_None;
        }

        ScreenShotContextSuspendedTask GetSuspendedTask() const NN_NOEXCEPT
        {
            return m_SuspendedTask;
        }

        const char* GetContextNameForDevelop() const NN_NOEXCEPT
        {
#ifndef NN_SDK_BUILD_RELEASE
            return m_pContextNameForDevelop;
#else
            return "";
#endif
        }

        void SetContextNameForDevelop(const char* name) NN_NOEXCEPT
        {
#ifndef NN_SDK_BUILD_RELEASE
            m_pContextNameForDevelop = name;
#else
            NN_UNUSED(name);
            m_pContextNameForDevelop = "";
#endif
        }

    private:
        ScreenShotContextSuspendedTask m_SuspendedTask;

        bool m_IsRecoveryRequired;
        ScreenShotContextRecoveryResult (*m_pRecoveryFunction)(void*);
        void* m_pRecoveryParameter;

        const char* m_pContextNameForDevelop;
    };

}}}
