﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "visrv_CaptureModuleSharedTexture.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

namespace nn{ namespace capsrv{ namespace server{ namespace screenshot{

    CaptureModuleSharedTexture::CaptureModuleSharedTexture() NN_NOEXCEPT
    {
    }

    void CaptureModuleSharedTexture::Initialize(
        capture::CaptureModule* pModule,
        const char* captureName
    ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(!IsInitialized());

        m_Capture.Initialize(pModule, captureName, nn::vi::LayerStack_LastFrame, 1280, 720, nn::capsrv::capture::ImageFormat_SharedTexture);

        m_pModule = pModule;
        m_Layout = {};
        m_CurrentBufferIndex = -1;
    }

    void CaptureModuleSharedTexture::Finalize() NN_NOEXCEPT
    {
        DeactivateBufferImpl();

        m_Capture.Finalize();
        if(m_DisplayBuffer.IsInitialized())
        {
            m_DisplayBuffer.Finalize();
        }
        if(m_ImageBuffer.IsInitialized())
        {
            m_ImageBuffer.Finalize();
        }
        if(m_MemoryPool.IsInitialized())
        {
            m_MemoryPool.Finalize();
        }

        m_pModule = nullptr;
        m_Layout = {};
        m_CurrentBufferIndex = {};
    }

    bool CaptureModuleSharedTexture::IsInitialized() const NN_NOEXCEPT
    {
        return m_Capture.IsInitialized();
    }

    void CaptureModuleSharedTexture::AttachSharedMemoryPool(
        const capture::MemoryPoolImportData& src,
        const nn::vi::fbshare::SharedMemoryPoolLayout& layout
    ) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_MemoryPool.InitializeAsImported(src));
        m_Layout = layout;
        m_CurrentBufferIndex = -1;
    }

    void CaptureModuleSharedTexture::ActivateBufferImpl(int iBuf) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_LESS(m_CurrentBufferIndex, 0);
        NN_SDK_REQUIRES_RANGE(iBuf, 0, m_Layout.count);

        auto& e = m_Layout.entries[iBuf];

        capture::ImageBufferInfo info = {};
        info.width = e.width;
        info.height = e.height;
        info.format = capture::ImageFormat_SharedTexture;
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_ImageBuffer.Initialize(m_pModule, info, &m_MemoryPool, e.offset, e.size));

        m_DisplayBuffer.InitializeRgba(m_pModule, capture::ImageFormat_SharedTexture, &m_ImageBuffer);

        m_Capture.SetCaptureBuffer(0, &m_DisplayBuffer);

        m_CurrentBufferIndex = iBuf;
    }

    void CaptureModuleSharedTexture::DeactivateBufferImpl() NN_NOEXCEPT
    {
        if(m_CurrentBufferIndex < 0)
        {
            return;
        }

        m_Capture.UnsetCaptureBuffer(0);
        m_DisplayBuffer.Finalize();
        m_ImageBuffer.Finalize();

        m_CurrentBufferIndex = -1;
    }

    capture::DisplayBuffer* CaptureModuleSharedTexture::AcquireCaptureBuffer(int iBuf, nn::TimeSpan timeout, nn::vi::LayerStack layerStack) NN_NOEXCEPT
    {
        if(iBuf < 0 || iBuf >= m_Layout.count)
        {
            return nullptr;
        }

        // 復帰が必要なら何もしない。
        if(IsRecoveryRequired())
        {
            return nullptr;
        }

        if(m_CurrentBufferIndex != iBuf)
        {
            DeactivateBufferImpl();
            ActivateBufferImpl(iBuf);
        }

        auto& e = m_Layout.entries[m_CurrentBufferIndex];
        return m_Capture.AcquireCaptureBuffer(timeout, layerStack, e.width, e.height);
    }

    void CaptureModuleSharedTexture::ReleaseCaptureBuffer(capture::DisplayBuffer* pBuffer) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_EQUAL(pBuffer, &m_DisplayBuffer);
        m_Capture.ReleaseCaptureBuffer(pBuffer);
    }

    bool CaptureModuleSharedTexture::IsRecoveryRequired() const NN_NOEXCEPT
    {
        return m_Capture.IsRecoveryRequired();
    }

    bool CaptureModuleSharedTexture::RecoverCapture(nn::TimeSpan timeout) NN_NOEXCEPT
    {
        return m_Capture.RecoverCapture(timeout);
    }

}}}}

