﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/capsrv/capsrv_Result.h>
#include <nn/capsrv/capsrv_AlbumFileSizeLimit.h>
#include "../visrv_ScreenShotContext.h"
#include "visrv_ActionTimeMeasure.h"

#include <nn/capsrv/capsrv_ViewerThumbnailFormat.h>
#include "../../capsrvServer_Config.h"
#include "../visrv_ScreenShotUtility.h"

namespace nn{ namespace capsrv{ namespace server{ namespace screenshot{ namespace action{

    // 画面写真を JPEG エンコードする
    // @param[in]  context.ImageBufferNv12Y
    // @param[in]  context.ImageBufferNv12Uv
    // @param[in]  context.ExifBuilder
    // @param[out] context.FileDataMemory
    // @param[out] context.FileSize
    // @param      context.EncoderWorkMemory
    static nn::Result EncodeScreenShotJpegNv12(ScreenShotContext& context) NN_NOEXCEPT
    {
        NN_CAPSRV_SCREENSHOT_CONTEXT_SCOPE(context);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_IN(context, ImageBufferNv12Y);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_IN(context, ImageBufferNv12Uv);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_IN(context, ExifBuilder);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_MEMORY(context, FileDataMemory, AlbumFileSizeLimit_ScreenShot);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_MEMORY(context, EncoderWorkMemory, SoftwareJpegEncoderWorkMemorySize);
        NN_CAPSRV_SCREENSHOT_CONTEXT_REQUIRES_OVERWRITEABLE(context, FileSize);

        NN_CAPSRV_SCREENSHOT_ACTION_TIMEMEASURE("encodeNV12");

        NN_CAPSRV_SCREENSHOT_CONTEXT_GET_MEMORY(pEncodedBuffer, encodedBufferSize, context, FileDataMemory);
        std::memset(pEncodedBuffer, 0, encodedBufferSize);

        NN_CAPSRV_SCREENSHOT_CONTEXT_GET_MEMORY(pWork, workSize, context, EncoderWorkMemory);
        std::memset(pWork, 0, workSize);

        NN_CAPSRV_SCREENSHOT_CONTEXT_GET(pImageBufferY, context, ImageBufferNv12Y);
        NN_CAPSRV_SCREENSHOT_CONTEXT_GET(pImageBufferUv, context, ImageBufferNv12Uv);
        NN_CAPSRV_SCREENSHOT_CONTEXT_GET(pExifBuilder, context, ExifBuilder);

        class ImageSubRegion
        {
        public:
            ImageSubRegion(const capture::ImageBuffer* pY, const capture::ImageBuffer* pUv) NN_NOEXCEPT
                : m_pImageBufferY(pY), m_pImageBufferUv(pUv)
            {
            }

        private:
            static void ConvertExpandImage(
                uint8_t* dstBufY,
                uint8_t* dstBufU,
                uint8_t* dstBufV,
                const uint8_t* srcBufY,
                const uint8_t* srcBufU,
                const uint8_t* srcBufV,
                int width,
                int height
            ) NN_NOEXCEPT
            {
                // 色空間変換:
                //   - src: BT.709 limited-range
                //   - dst: BT.601 full-range
                // サイズ変換:
                //   - U,V 面を横方向に 2 倍に拡大
                //
                // 色変換: SIGLO-68565
                //
                //  ( Y_601Ful )   [ mYY, mYU, mYV ] ( Y_709Lim )   ( tY )
                //  ( U_601Ful ) = [ mUY, mUU, mUV ] ( U_709Lim ) + ( tU )
                //  ( V_601Ful )   [ mVY, mVU, mVV ] ( V_709Lim )   ( tV )
                //
                //    NOTE: U,V の変換は Y に依存しない(mUY = 0, mVY = 0)
                static const float mYY =   1.1644f;
                static const float mYU =   0.11564f;
                static const float mYV =   0.22321f;
                //static const float mUY = 0;
                static const float mUU =   1.1268f;
                static const float mUV = - 0.12597f;
                //static const float mVY = 0;
                static const float mVU = - 0.082480f;
                static const float mVV =   1.1195f;

                static const float tY  = -62.003f;
                static const float tU  = - 0.11219f;
                static const float tV  = -4.7377f;

                auto clamp = [](float v) -> uint8_t{
                    if(v < 0)
                    {
                        return 0;
                    }
                    if(v >= 255)
                    {
                        return 255;
                    }
                    return static_cast<uint8_t>(v);
                };

                for(int y = 0; y < height; y += 2)
                {
                    // 縦方向は 2 行ずつ処理する。
                    auto pDstYLine0 = dstBufY +  width      * (y + 0);
                    auto pDstYLine1 = dstBufY +  width      * (y + 1);
                    auto pDstULine  = dstBufU +  width      * (y / 2);
                    auto pDstVLine  = dstBufV +  width      * (y / 2);
                    auto pSrcYLine0 = srcBufY +  width      * (y + 0);
                    auto pSrcYLine1 = srcBufY +  width      * (y + 1);
                    auto pSrcULine  = srcBufU + (width / 2) * (y / 2);
                    auto pSrcVLine  = srcBufV + (width / 2) * (y / 2);
                    for(int x = 0; x < width; x += 2)
                    {
                        // 横方向は 2 ピクセルずつ処理する。
                        const uint8_t srcY00 = pSrcYLine0[x + 0];
                        const uint8_t srcY01 = pSrcYLine0[x + 1];
                        const uint8_t srcY10 = pSrcYLine1[x + 0];
                        const uint8_t srcY11 = pSrcYLine1[x + 1];
                        const uint8_t srcU = pSrcULine[x / 2];
                        const uint8_t srcV = pSrcVLine[x / 2];

                        const float yTmp = mYU * srcU + mYV * srcV + tY;
                        const float y00 = mYY * srcY00 + yTmp;
                        const float y01 = mYY * srcY01 + yTmp;
                        const float y10 = mYY * srcY10 + yTmp;
                        const float y11 = mYY * srcY11 + yTmp;
                        const float u = mUU * srcU + mUV * srcV + tU;
                        const float v = mVU * srcU + mVV * srcV + tV;

                        pDstYLine0[x + 0] = clamp(y00);
                        pDstYLine0[x + 1] = clamp(y01);
                        pDstYLine1[x + 0] = clamp(y10);
                        pDstYLine1[x + 1] = clamp(y11);
                        pDstULine[x + 0] = pDstULine[x + 1] = clamp(u);
                        pDstVLine[x + 0] = pDstVLine[x + 1] = clamp(v);
                    }
                }

            }

        public:
            static nn::Result Invoke(const jpeg::SoftwareJpegEncoderYuvBuffer& outBuffer, capture::ImageFormat format, uint32_t x, uint32_t y, uint32_t width, uint32_t height, void* userPtr) NN_NOEXCEPT
            {
                auto pArg = reinterpret_cast<ImageSubRegion*>(userPtr);

                const uint32_t hSubsamplingShift = GetHorizontalSubsamplingShift(format);
                const uint32_t vSubsamplingShift = GetVerticalSubsamplingShift(format);
                NN_SDK_ASSERT_MINMAX(hSubsamplingShift, 0u, 1u);
                NN_SDK_ASSERT_MINMAX(vSubsamplingShift, 0u, 1u);

                // 色変換のロジックの都合で dst は YUV440 に固定
                NN_SDK_ASSERT_EQUAL(format, nn::capsrv::capture::ImageFormat_Yuv440_Bt601_Planar);
                NN_SDK_ASSERT_EQUAL(hSubsamplingShift, 0u); // 水平方向は等倍
                NN_SDK_ASSERT_EQUAL(vSubsamplingShift, 1u); // 垂直方向は 1/2
                NN_RESULT_THROW_UNLESS(hSubsamplingShift == 0u, ResultInternalError());
                NN_RESULT_THROW_UNLESS(vSubsamplingShift == 1u, ResultInternalError());

                NN_SDK_ASSERT_EQUAL(width % 2, 0);
                NN_RESULT_THROW_UNLESS(width % 2 == 0, ResultInternalError());

                uint8_t* bufY = reinterpret_cast<uint8_t*>(outBuffer.pBufferY);
                uint8_t* bufU = reinterpret_cast<uint8_t*>(outBuffer.pBufferU);
                uint8_t* bufV = reinterpret_cast<uint8_t*>(outBuffer.pBufferV);
                size_t bufSizeY = outBuffer.bufferSizeY;
                size_t bufSizeU = outBuffer.bufferSizeU;
                size_t bufSizeV = outBuffer.bufferSizeV;
                NN_SDK_ASSERT_EQUAL(bufSizeU % 2, 0);
                NN_SDK_ASSERT_EQUAL(bufSizeV % 2, 0);

                std::memset(bufY, 0, bufSizeY);
                std::memset(bufU, 0, bufSizeU);
                std::memset(bufV, 0, bufSizeV);

                // src からコピー。
                // - Y 面: dst も src も縦横等倍
                // - U, Y 面: dst は横方向等倍、縦方向 1/2 倍。src は横方向 1/2 倍、縦方向 1/2 倍。
                //   横方向を 2 倍に引き延ばす必要がある。
                //   後で色変換しながら引き延ばすので、一旦 dst バッファの後半部分にコピー。
                {
                    const capture::Rectangle  rectY = capture::MakeRectangle(x, y, width, height);
                    const capture::Rectangle  rectU = capture::MakeRectangle(x / 2 , y / 2, width / 2, height / 2);
                    const capture::Rectangle& rectV = rectU;
                    size_t sizeY = 0;
                    size_t sizeU = 0;
                    size_t sizeV = 0;
                    NN_RESULT_DO(pArg->m_pImageBufferY ->WriteToMemory(&sizeY, bufY               , bufSizeY    , nn::capsrv::capture::ImageFormat_Y_Bt601, &rectY));
                    NN_RESULT_DO(pArg->m_pImageBufferUv->WriteToMemory(&sizeU, bufU + bufSizeU / 2, bufSizeU / 2, nn::capsrv::capture::ImageFormat_U_Bt601, &rectU));
                    NN_RESULT_DO(pArg->m_pImageBufferUv->WriteToMemory(&sizeV, bufV + bufSizeV / 2, bufSizeV / 2, nn::capsrv::capture::ImageFormat_V_Bt601, &rectV));
                }

                // 変換
                ConvertExpandImage(
                    bufY,
                    bufU,
                    bufV,
                    bufY,
                    bufU + bufSizeU / 2,
                    bufV + bufSizeV / 2,
                    width,
                    height
                );

                NN_RESULT_SUCCESS;
            }
        private:
            const capture::ImageBuffer* m_pImageBufferY;
            const capture::ImageBuffer* m_pImageBufferUv;
        };

        auto subRegion = ImageSubRegion(pImageBufferY, pImageBufferUv);

        jpeg::SoftwareJpegEncoderStreamInputInfoYuv inputInfo;
        inputInfo.width  = ScreenShotWidth;
        inputInfo.height = ScreenShotHeight;
        inputInfo.format = capture::ImageFormat_Yuv440_Bt601_Planar;
        inputInfo.pGetSubRegionFunction = ImageSubRegion::Invoke;
        inputInfo.pGetSubRegionUserPtr  = &subRegion;

        size_t encodedSize = 0;
        jpeg::SoftwareJpegEncoderBufferOutputInfo outputInfo;
        outputInfo.pOutSize   = &encodedSize;
        outputInfo.pBuffer    = pEncodedBuffer;
        outputInfo.bufferSize = (encodedBufferSize < AlbumFileSizeLimit_ScreenShot) ? encodedBufferSize : AlbumFileSizeLimit_ScreenShot;

        NN_RESULT_DO(ScreenShotUtility::EncodeYuvStreamToBuffer(outputInfo, inputInfo, pExifBuilder, pWork, workSize));
        NN_ABORT_UNLESS_GREATER_EQUAL(encodedBufferSize, encodedSize);

        NN_CAPSRV_SCREENSHOT_CONTEXT_OVERWRITE(context, FileSize, static_cast<int64_t>(encodedSize));
        NN_RESULT_SUCCESS;
    }// NOLINT(impl/function_size)

}}}}}

