﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "visrv_OverlayNotifier.h"

#include <mutex>
#include <nn/nn_Abort.h>
#include <nn/result/result_HandlingUtility.h>

#if defined(NN_CAPSRV_USE_OVERLAY_NOTIFICATION)
#include <nn/ovln/ovln_SenderForOverlay.h>
#include <nn/ovln/format/ovln_CapsrvMessage.h>
#endif

#include "../capsrvServer_ResultPrivate.h"

namespace nn{ namespace capsrv{ namespace server{

    void OverlayNotifier::Initialize() NN_NOEXCEPT
    {
#if defined(NN_CAPSRV_USE_OVERLAY_NOTIFICATION)
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ovln::InitializeSenderLibraryForOverlay());
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ovln::InitializeSenderForValue(&m_Sender));
        m_Mutex.Initialize();
#endif
    }

    void OverlayNotifier::Finalize() NN_NOEXCEPT
    {
#if defined(NN_CAPSRV_USE_OVERLAY_NOTIFICATION)
        nn::ovln::FinalizeSender(&m_Sender);
        nn::ovln::FinalizeSenderLibraryForOverlay();
#endif
    }

    nn::Result OverlayNotifier::SendToOverlay(const nn::ovln::format::ScreenShotCapturedData& data) NN_NOEXCEPT
    {
#if defined(NN_CAPSRV_USE_OVERLAY_NOTIFICATION)
        std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

        nn::ovln::Message message = {};
        switch(data.result)
        {
        case nn::ovln::format::ScreenShotResult::Success:
            {
                message.tag = nn::ovln::format::ScreenShotSuccessDataTag;
                break;
            }
        case nn::ovln::format::ScreenShotResult::NotPermitted:
            {
                message.tag = nn::ovln::format::ScreenShotIsNotPermittedDataTag;
                break;
            }
        case nn::ovln::format::ScreenShotResult::ThumbnailIsReady:
            {
                message.tag = nn::ovln::format::ScreenShotThumbnailIsReadyDataTag;
                break;
            }
        default:
            {
                message.tag = nn::ovln::format::ScreenShotFailureDataTag;
                break;
            }
        }
        message.dataSize = sizeof(data);
        std::memcpy(&message.data, &data, sizeof(data));
        NN_RESULT_THROW_UNLESS(
            nn::ovln::Send(&m_Sender, message),
            ResultInternalSendToOverlayFailed()
        );
        NN_RESULT_SUCCESS;
#else
        NN_UNUSED(data);
        NN_RESULT_SUCCESS;
#endif
    }

}}}
