﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "visrv_DownsampleImage.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include "../../capsrv_Assert.h"

namespace nn{ namespace capsrv{ namespace server{ namespace detail{

    namespace {

        class DownsampleImageRgbaAverage4x4Impl
        {
        public:
            typedef uint32_t PixelType;
            // ユニット。 2 ピクセルを並べたもの。{A1 B1 G1 R1 A0 B0 G0 R0}
            typedef uint64_t UnitType;
            static const UnitType MaskOBOR  = 0x00FF00FF00FF00FF;
            static const UnitType MaskAOGO  = 0xFF00FF00FF00FF00;
            static const UnitType MaskUpper = 0xFFFFFFFF00000000;
            static const UnitType MaskLower = 0x00000000FFFFFFFF;

            // 拡張ピクセル。16bit RGBA 2 つ分に相当。
            struct PixelEx
            {
                UnitType bbrr; // 0xBBBB-RRRR-BBBB-RRRR
                UnitType aagg; // 0xAAAA-GGGG-AAAA-GGGG
            };

            typedef void (*DownsampleFunction)(uint8_t*, const uint8_t*, const uint8_t*, const uint8_t*, const uint8_t*, int);

        public:
            NN_FORCEINLINE static PixelEx CombineUnit(UnitType src0, UnitType src1) NN_NOEXCEPT
            {
                const UnitType obor0 =  src0 & MaskOBOR;
                const UnitType oaog0 = (src0 & MaskAOGO) >> 8;
                const UnitType obor1 =  src1 & MaskOBOR;
                const UnitType oaog1 = (src1 & MaskAOGO) >> 8;
                return { obor0 + obor1, oaog0 + oaog1 };
            }

            template<bool IsOpaqueForced>
            static void Downsample(
                uint8_t* pDstLine,
                const uint8_t* pSrcLine0,
                const uint8_t* pSrcLine1,
                const uint8_t* pSrcLine2,
                const uint8_t* pSrcLine3,
                int srcWidth
            ) NN_NOEXCEPT
            {
                PixelType* dst = reinterpret_cast<PixelType*>(pDstLine);
                const UnitType* src0 = reinterpret_cast<const UnitType*>(pSrcLine0);
                const UnitType* src1 = reinterpret_cast<const UnitType*>(pSrcLine1);
                const UnitType* src2 = reinterpret_cast<const UnitType*>(pSrcLine2);
                const UnitType* src3 = reinterpret_cast<const UnitType*>(pSrcLine3);

                for(int x = 0; x < srcWidth; x += 4)
                {
                    // Unit 単位で横方向に足して拡張ピクセルを作る
                    const PixelEx v0 = CombineUnit(src0[0], src0[1]);
                    const PixelEx v1 = CombineUnit(src1[0], src1[1]);
                    const PixelEx v2 = CombineUnit(src2[0], src2[1]);
                    const PixelEx v3 = CombineUnit(src3[0], src3[1]);

                    // 拡張ピクセル単位で縦方向に足す
                    const PixelEx w0 = { v0.bbrr + v1.bbrr, v0.aagg + v1.aagg }; // L0 + L1
                    const PixelEx w1 = { v2.bbrr + v3.bbrr, v2.aagg + v3.aagg }; // L2 + L3
                    const PixelEx w2 = { w0.bbrr + w1.bbrr, w0.aagg + w1.aagg }; // (L0 + L1) + (L2 + L3)

                    // 拡張ピクセル内で足す
                    const UnitType bbrr = (w2.bbrr >> 32) + (w2.bbrr);
                    const UnitType aagg = (w2.aagg >> 32) + (w2.aagg);
                    // NOTE: マスクしなくても結果に影響はない
                    //const UnitType bbrr = ((w2.bbrr & MaskUpper) >> 32) + (w2.bbrr & MaskLower);
                    //const UnitType aagg = ((w2.aagg & MaskUpper) >> 32) + (w2.aagg & MaskLower);
                    NN_UNUSED(MaskUpper);
                    NN_UNUSED(MaskLower);

                    // 1/16 を掛けながら argb に戻す
                    const PixelType obor = static_cast<PixelType>((bbrr >> 4) & MaskOBOR);
                    const PixelType aogo = static_cast<PixelType>((aagg << 4) & MaskAOGO);
                    const PixelType abgr = obor | aogo;

                    if(NN_STATIC_CONDITION(IsOpaqueForced))
                    {
                        *dst = abgr | 0xFF000000;
                    }
                    else
                    {
                        *dst = abgr;
                    }

                    dst  += 1;
                    src0 += 2;
                    src1 += 2;
                    src2 += 2;
                    src3 += 2;
                }
            }

        };

    }

    void DownsampleImageRgbaAverage4x4(
        int* pOutDstWidth,
        int* pOutDstHeight,
        void* pDstBuffer,
        size_t dstBufferSize,
        const void* pSrcBuffer,
        size_t srcBufferSize,
        int srcWidth,
        int srcHeight,
        DownsampleImageOption option
    ) NN_NOEXCEPT
    {
        NN_CAPSRV_REQUIRES_NOT_NULL(pOutDstWidth);
        NN_CAPSRV_REQUIRES_NOT_NULL(pOutDstHeight);
        NN_CAPSRV_REQUIRES_NOT_NULL(pDstBuffer);
        NN_CAPSRV_REQUIRES_GREATER(dstBufferSize, static_cast<size_t>(0));
        NN_CAPSRV_REQUIRES_NOT_NULL(pSrcBuffer);
        NN_CAPSRV_REQUIRES_GREATER(srcBufferSize, static_cast<size_t>(0));
        NN_CAPSRV_REQUIRES_GREATER(srcWidth, 0);
        NN_CAPSRV_REQUIRES_GREATER(srcHeight, 0);

        NN_CAPSRV_REQUIRES_EQUAL(srcWidth % 4, 0);
        NN_CAPSRV_REQUIRES_EQUAL(srcHeight % 4, 0);
        NN_CAPSRV_REQUIRES_GREATER_EQUAL(srcBufferSize, static_cast<size_t>(4 * srcWidth * srcHeight));
        NN_CAPSRV_REQUIRES_GREATER_EQUAL(dstBufferSize, static_cast<size_t>(srcWidth * srcHeight / 4));
        NN_UNUSED(srcBufferSize);
        NN_UNUSED(dstBufferSize);

        static const int PixelSize = 4;
        static const int DstLineCount = 1;
        static const int SrcLineCount = 4;

        uint8_t* dst = reinterpret_cast<uint8_t*>(pDstBuffer);
        const uint8_t* src = reinterpret_cast<const uint8_t*>(pSrcBuffer);

        int dstAdvance = DstLineCount * PixelSize * (srcWidth / 4);
        int srcAdvance = SrcLineCount * PixelSize * srcWidth;

        DownsampleImageRgbaAverage4x4Impl::DownsampleFunction downsample = nullptr;
        if(option & DownsampleImageOption_Opaque)
        {
            downsample = DownsampleImageRgbaAverage4x4Impl::Downsample<true>;
        }
        else
        {
            downsample = DownsampleImageRgbaAverage4x4Impl::Downsample<false>;
        }

        for(int y = 0; y < srcHeight; y += SrcLineCount)
        {
            downsample(
                dst,
                src,
                src + PixelSize * srcWidth,
                src + 2 * PixelSize * srcWidth,
                src + 3 * PixelSize * srcWidth,
                srcWidth
            );

            dst += dstAdvance;
            src += srcAdvance;
        }

        *pOutDstWidth = srcWidth / 4;
        *pOutDstHeight = srcHeight / 4;
    }


}}}}

