﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "capsrvServer_ScreenShotCaptureModule.h"

#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include "capsrvServer_ParseLayerStackName.h"

namespace nn{ namespace capsrv{ namespace server{ namespace detail{

    namespace {

        ptrdiff_t GetOffset(WorkMemory::ScreenShotWorkMemory* pWorkMemory, char* pMemory) NN_NOEXCEPT
        {
            char* pBase = pWorkMemory->value.all.memory;
            return pMemory - pBase;
        }

    }

    void ScreenShotCaptureModule::Initialize(WorkMemory::ScreenShotWorkMemory* pWorkMemory) NN_NOEXCEPT
    {
        const capture::ImageFormat captureFormat = capture::ImageFormat_Yuv440_Planar;

        capture::InitializeCaptureModule();
        m_pModule = capture::GetCaptureModule();
        m_MemoryPool.Initialize(m_pModule, pWorkMemory->value.all.memory, sizeof(WorkMemory::ScreenShotWorkMemory));

        // ScreenShot
        {
            uint32_t horizontalSubsamplingShift = capture::GetHorizontalSubsamplingShift(captureFormat);
            uint32_t verticalSubsamplingShift   = capture::GetVerticalSubsamplingShift(captureFormat);
            NN_SDK_REQUIRES_EQUAL(ScreenShotWidth  % (1 << horizontalSubsamplingShift), 0);
            NN_SDK_REQUIRES_EQUAL(ScreenShotHeight % (1 << verticalSubsamplingShift),   0);

            // レイヤースタックの設定を取得
            nn::vi::LayerStack layerStack = DefaultScreenShotLayerStack;
            {
                char name[LayerStackNameSize + 1] = {};
                nn::settings::fwdbg::GetSettingsItemValue(name, LayerStackNameSize, "capsrv", "screenshot_layerstack");
                layerStack = ParseLayerStackName(name, CaptureUsage_ScreenShot);
            }

            // Y plane
            {
                capture::ImageBufferInfo info;
                info.width  = ScreenShotWidth;
                info.height = ScreenShotHeight;
                info.format = capture::ImageFormat_Y;
                m_ScreenShot.fullSizeY.Initialize(m_pModule, info, &m_MemoryPool, GetOffset(pWorkMemory, pWorkMemory->value.screenShot.fullSizeRawImageY), WorkMemorySize_ScreenShotFullSizeRawImageY);
            }
            // U plane
            {
                capture::ImageBufferInfo info;
                info.width  = ScreenShotWidth  >> horizontalSubsamplingShift;
                info.height = ScreenShotHeight >> verticalSubsamplingShift;
                info.format = capture::ImageFormat_U;
                m_ScreenShot.fullSizeU.Initialize(m_pModule, info, &m_MemoryPool, GetOffset(pWorkMemory, pWorkMemory->value.screenShot.fullSizeRawImageU), WorkMemorySize_ScreenShotFullSizeRawImageU);
            }
            // V plane
            {
                capture::ImageBufferInfo info;
                info.width  = ScreenShotWidth  >> horizontalSubsamplingShift;
                info.height = ScreenShotHeight >> verticalSubsamplingShift;
                info.format = capture::ImageFormat_V;
                m_ScreenShot.fullSizeV.Initialize(m_pModule, info, &m_MemoryPool, GetOffset(pWorkMemory, pWorkMemory->value.screenShot.fullSizeRawImageV), WorkMemorySize_ScreenShotFullSizeRawImageV);
            }
            m_ScreenShot.fullSizeBuffer.InitializeYuv(m_pModule, captureFormat, &m_ScreenShot.fullSizeY, &m_ScreenShot.fullSizeU, &m_ScreenShot.fullSizeV);
            {
                m_ScreenShot.fullSizeCapture.Initialize(m_pModule, "screenshot", layerStack, ScreenShotWidth, ScreenShotHeight, captureFormat);
                m_ScreenShot.fullSizeCapture.SetCaptureBuffer(0, &m_ScreenShot.fullSizeBuffer);
            }
            {
                capture::ImageBufferInfo info;
                info.width = ThumbnailImageSize_Width;
                info.height = ThumbnailImageSize_Height;
                info.format = capture::ImageFormat_Rgba8;
                m_ScreenShot.overlayThumbnail.Initialize(m_pModule, info, &m_MemoryPool, GetOffset(pWorkMemory, pWorkMemory->value.screenShot.overlayThumbnailRawImage), WorkMemorySize_ScreenShotOverlayThumbnailRawImage);
            }
            {
                capture::ImageBufferInfo info;
                info.width = ViewerThumbnailImageSize_Width;
                info.height = ViewerThumbnailImageSize_Height;
                info.format = capture::ImageFormat_Rgba8;
                m_ScreenShot.viewerThumbnail.Initialize(m_pModule, info, &m_MemoryPool, GetOffset(pWorkMemory, pWorkMemory->value.screenShot.viewerThumbnailRawImage), WorkMemorySize_ScreenShotViewerThumbnailRawImage);
            }
        }

        // KeepPresentation
        {
            nn::vi::LayerStack layerStack = DefaultLastFrameLayerStack;
            {
                capture::ImageBufferInfo info;
                info.width  = ScreenShotWidth;
                info.height = ScreenShotHeight;
                info.format = capture::ImageFormat_Rgba8;
                m_KeepPresentation.presentationRgba.Initialize(m_pModule, info, &m_MemoryPool, GetOffset(pWorkMemory, pWorkMemory->value.keepPresentation.fullSizeRawImage), WorkMemorySize_KeepPresentationFullSizeRawImage);
            }
            m_KeepPresentation.presentationBuffer.Initialize(m_pModule, capture::ImageFormat_Rgba8, &m_KeepPresentation.presentationRgba);
            {
                m_KeepPresentation.presentationCapture.Initialize(m_pModule, "lastframe", layerStack, ScreenShotWidth, ScreenShotHeight, capture::ImageFormat_Rgba8);
                m_KeepPresentation.presentationCapture.SetCaptureBuffer(0, &m_KeepPresentation.presentationBuffer);
            }
        }

    };

    void ScreenShotCaptureModule::Finalize() NN_NOEXCEPT
    {
        m_KeepPresentation.presentationCapture.Finalize();
        m_KeepPresentation.presentationBuffer.Finalize();
        m_KeepPresentation.presentationRgba.Finalize();

        m_ScreenShot.fullSizeCapture.Finalize();
        m_ScreenShot.fullSizeBuffer.Finalize();
        m_ScreenShot.fullSizeY.Finalize();
        m_ScreenShot.fullSizeU.Finalize();
        m_ScreenShot.fullSizeV.Finalize();
        m_ScreenShot.overlayThumbnail.Finalize();
        m_ScreenShot.viewerThumbnail.Finalize();

        m_MemoryPool.Finalize();
        capture::FinalizeCaptureModule();
    }

}}}}
