﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_Endian.h>
#include <nn/capsrv/capsrv_Result.h>

#include "capsrvServer_MakerNoteEntry.h"

namespace nn{ namespace capsrv{ namespace server{ namespace detail{

    template<typename Tag>
    class MakerNoteEntryParser
    {
        // TryParse 系関数共通
        // @param[out] pOutNext 解析対象の次の位置を受け取る変数のポインタ
        // @param[out] pOutValue 解析結果を受け取る変数のポインタ
        // @param[in] p 解析の開始位置のポインタ
        // @param[in] size 解析対象の大きさ。p の位置から解析対象の末尾までの長さ（バイト）。
        // @retval nn::ResultSuccess 解析に成功
        // @retval nn::capsrv::ResultAlbumInvalidFileData 解析に失敗
        // @details
        //   解析に失敗した場合、 *pOutNext と *pOutValue の値は変更されません。

    public:
        // 整数値用。入力を LittleEndian で解釈して読み込む。
        template<typename T>
        static nn::Result TryParseInteger(const char** pOutNext, T* pOutValue, const char* p, ptrdiff_t size) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(pOutNext);
            NN_SDK_REQUIRES_NOT_NULL(pOutValue);
            NN_RESULT_THROW_UNLESS(
                static_cast<size_t>(size) >= sizeof(T),
                ResultAlbumInvalidFileData()
            );
            T tmp;
            std::memcpy(&tmp, p, sizeof(T));
            *pOutValue = nn::util::LoadLittleEndian(&tmp);
            *pOutNext = p + sizeof(T);
            NN_RESULT_SUCCESS;
        }

        static nn::Result TryParseVersion(const char** pOutNext, uint32_t* pOutValue, const char* p, ptrdiff_t size) NN_NOEXCEPT
        {
            NN_RESULT_THROW(TryParseInteger<uint32_t>(pOutNext, pOutValue, p, size));
        }

        static nn::Result TryParseEntryTag(const char** pOutNext, uint16_t* pOutValue, const char* p, ptrdiff_t size) NN_NOEXCEPT
        {
            NN_RESULT_THROW(TryParseInteger<uint16_t>(pOutNext, pOutValue, p, size));
        }

        static nn::Result TryParseEntrySize(const char** pOutNext, uint16_t* pOutValue, const char* p, ptrdiff_t size) NN_NOEXCEPT
        {
            NN_RESULT_THROW(TryParseInteger<uint16_t>(pOutNext, pOutValue, p, size));
        }

        static nn::Result TryParseEntry(const char** pOutNext, MakerNoteEntry* pOutValue, const char* pBuffer, ptrdiff_t size) NN_NOEXCEPT
        {
            MakerNoteEntry entry;
            const char* ptr = pBuffer;
            const char* const ptrEnd = pBuffer + size;
            NN_RESULT_DO(TryParseEntryTag(&ptr, &entry.tag, ptr, ptrEnd - ptr));
            NN_RESULT_DO(TryParseEntrySize(&ptr, &entry.size, ptr, ptrEnd - ptr));

            NN_RESULT_THROW_UNLESS(entry.size > 0, ResultAlbumInvalidFileData());

            entry.body = ptr;
            ptr += entry.size;
            NN_RESULT_THROW_UNLESS(
                ptr <= ptrEnd,
                ResultAlbumInvalidFileData()
            );
            *pOutValue = entry;
            *pOutNext = ptr;
            NN_RESULT_SUCCESS;
        }
    };

}}}}
