﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "capsrvServer_ExtractJpegMakerNoteRange.h"

#include <limits>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_IntUtil.h>
#include <nn/image/image_JpegDecoder.h>
#include <nn/image/image_ExifExtractor.h>
#include "../capsrvServer_Config.h"
#include "../capsrvServer_ResultPrivate.h"

namespace nn{ namespace capsrv{ namespace server{ namespace detail{

    nn::Result ExtractJpegMakerNoteRange(
        int64_t* pOutOffset,
        int64_t* pOutSize,
        const void* pJpegData,
        size_t jpegDataSize,
        void* pWorkMemory,
        size_t workMemorySize
        ) NN_NOEXCEPT
    {
        NN_CAPSRV_LOG_DEV("extracting maker-note range\n");
        NN_SDK_REQUIRES_NOT_NULL(pOutOffset);
        NN_SDK_REQUIRES_NOT_NULL(pOutSize);
        NN_SDK_REQUIRES_NOT_NULL(pJpegData);
        NN_SDK_REQUIRES_NOT_NULL(pWorkMemory);
        NN_SDK_REQUIRES_ALIGNED(pWorkMemory, NN_ALIGNOF(std::max_align_t));
        NN_SDK_REQUIRES_GREATER_EQUAL(workMemorySize, nn::image::ExifExtractor::GetWorkBufferSize());

        NN_RESULT_THROW_UNLESS(pJpegData != nullptr, ResultInternalSignatureMakerNoteExtractionFailed());
        NN_RESULT_THROW_UNLESS(nn::util::IsIntValueRepresentable<int64_t>(jpegDataSize), ResultInternalSignatureMakerNoteExtractionFailed());

        // Exif を取得
        const void* pExifData = nullptr;
        size_t exifDataSize = 0;
        NN_RESULT_THROW_UNLESS(
            nn::image::JpegDecoder::GetExifData(&pExifData, &exifDataSize, pJpegData, jpegDataSize) == nn::image::JpegStatus_Ok,
            ResultInternalSignatureExifExtractionFailed()
        );

        // MakerNote を取得
        const void* pMakerNote = nullptr;
        size_t makerNoteSize = 0;
        {
            nn::image::ExifExtractor exifExtractor(pWorkMemory, workMemorySize);
            exifExtractor.SetExifData(pExifData, exifDataSize);
            NN_RESULT_THROW_UNLESS(
                exifExtractor.Analyze() == nn::image::JpegStatus_Ok,
                ResultInternalSignatureMakerNoteExtractionFailed()
            );

            pMakerNote = exifExtractor.ExtractMakerNote(&makerNoteSize);

            NN_RESULT_THROW_UNLESS(
                pMakerNote != nullptr,
                ResultInternalSignatureMakerNoteExtractionFailed()
            );
        }

        int64_t offset = reinterpret_cast<const char*>(pMakerNote) - reinterpret_cast<const char*>(pJpegData);
        int64_t size   = static_cast<int64_t>(makerNoteSize);

        NN_CAPSRV_LOG_DEV("  offset = %lld, size = %lld\n", offset, size);
        // 取得された領域が渡された jpeg data 内であることを確認
        NN_RESULT_THROW_UNLESS(offset >= 0, ResultAlbumInvalidFileData());
        NN_RESULT_THROW_UNLESS(size >= 0, ResultAlbumInvalidFileData());
        NN_RESULT_THROW_UNLESS(size <= std::numeric_limits<int64_t>::max() - offset, ResultAlbumInvalidFileData());
        NN_RESULT_THROW_UNLESS(offset + size <= static_cast<int64_t>(jpegDataSize), ResultAlbumInvalidFileData());

        NN_CAPSRV_LOG_DEV("  -> success\n");
        *pOutOffset = offset;
        *pOutSize = size;
        NN_RESULT_SUCCESS;
    }

}}}}
