﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "capsrvServer_ApplicationResourceManager.h"

#include <mutex>
#include <nn/result/result_HandlingUtility.h>
#include <nn/os.h>
#include <nn/capsrv/capsrv_Result.h>

#include "../capsrvServer_Config.h"
#include "../../capsrv_Assert.h"

#include "capsrvServer_GetApplicationAlbumEntry.h"

namespace nn{ namespace capsrv{ namespace server{ namespace detail{

    ApplicationResource::ApplicationResource() NN_NOEXCEPT
        : m_IsInitialized(false)
        , m_Aruid(nn::applet::AppletResourceUserId::GetInvalidId())
        , m_ApplicationId(nn::ncm::ApplicationId::GetInvalidId())
        , m_ApplicationAlbumEntryKey(ApplicationAlbumEntryKey::GetInvalidValue())
    {
    }

    void ApplicationResource::Initialize(
        nn::applet::AppletResourceUserId aruid,
        nn::ncm::ApplicationId applicationId
    ) NN_NOEXCEPT
    {
        NN_CAPSRV_REQUIRES(!m_IsInitialized);

        m_IsInitialized = true;
        m_Aruid = aruid;
        m_ApplicationId = applicationId;

        InitializeApplicationAlbumEntryKey();
    }

    void ApplicationResource::Finalize() NN_NOEXCEPT
    {
        NN_CAPSRV_REQUIRES(m_IsInitialized);

        FinalizeApplicationAlbumEntryKey();

        m_IsInitialized = false;
        m_Aruid = nn::applet::AppletResourceUserId::GetInvalidId();
        m_ApplicationId = nn::ncm::ApplicationId::GetInvalidId();
    }

    void ApplicationResource::InitializeApplicationAlbumEntryKey() NN_NOEXCEPT
    {
        for(;;)
        {
            nn::os::GenerateRandomBytes(&m_ApplicationAlbumEntryKey, sizeof(m_ApplicationAlbumEntryKey));

            // 無効な鍵ならやりなおし
            if(m_ApplicationAlbumEntryKey == ApplicationAlbumEntryKey::GetInvalidValue())
            {
                continue;
            }

            break;
        }
    }

    void ApplicationResource::FinalizeApplicationAlbumEntryKey() NN_NOEXCEPT
    {
        m_ApplicationAlbumEntryKey = ApplicationAlbumEntryKey::GetInvalidValue();
    }

    bool ApplicationResource::IsInitialized() const NN_NOEXCEPT
    {
        return m_IsInitialized;
    }

    nn::applet::AppletResourceUserId ApplicationResource::GetAruid() const NN_NOEXCEPT
    {
        NN_CAPSRV_REQUIRES(m_IsInitialized);
        return m_Aruid;
    }

    nn::ncm::ApplicationId ApplicationResource::GetApplicationId() const NN_NOEXCEPT
    {
        NN_CAPSRV_REQUIRES(m_IsInitialized);
        return m_ApplicationId;
    }

    const ApplicationAlbumEntryKey& ApplicationResource::GetApplicationAlbumEntryKey() const NN_NOEXCEPT
    {
        NN_CAPSRV_REQUIRES(m_IsInitialized);
        return m_ApplicationAlbumEntryKey;
    }

    //----------------------------------------------------------------------------------

    ApplicationResourceManager::ApplicationResourceManager() NN_NOEXCEPT
        : m_IsInitialized(false)
    {
    }

    void ApplicationResourceManager::Initialize() NN_NOEXCEPT
    {
        NN_CAPSRV_REQUIRES(!m_IsInitialized);
        m_Mutex.Initialize();

        auto seed = 0;
        nn::os::GenerateRandomBytes(&seed, sizeof(seed));
        m_Rand.seed(seed);

        m_IsInitialized = true;
    }

    void ApplicationResourceManager::Finalize() NN_NOEXCEPT
    {
        NN_CAPSRV_REQUIRES(m_IsInitialized);

        for(int i = 0; i < ApplicationCountMax; i++)
        {
            if(m_ApplicationResourceList[i].IsInitialized())
            {
                m_ApplicationResourceList[i].Finalize();
            }
        }

        m_IsInitialized = false;
    }

    nn::Result ApplicationResourceManager::GetApplicationResourceFromApplicationIdImpl(ApplicationResource** pOutValue, nn::ncm::ApplicationId applicationId) NN_NOEXCEPT
    {
        NN_CAPSRV_REQUIRES(m_IsInitialized);
        for(int i = 0; i < ApplicationCountMax; i++)
        {
            auto& e = m_ApplicationResourceList[i];
            if(e.IsInitialized() && e.GetApplicationId() == applicationId)
            {
                *pOutValue = &e;
                NN_RESULT_SUCCESS;
            }
        }
        NN_RESULT_THROW(ResultControlNotOpened());
    }

    nn::Result ApplicationResourceManager::GetApplicationResourceFromAruidImpl(ApplicationResource** pOutValue, nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
    {
        NN_CAPSRV_REQUIRES(m_IsInitialized);
        for(int i = 0; i < ApplicationCountMax; i++)
        {
            auto& e = m_ApplicationResourceList[i];
            if(e.IsInitialized() && e.GetAruid() == aruid)
            {
                *pOutValue = &e;
                NN_RESULT_SUCCESS;
            }
        }
        NN_RESULT_THROW(ResultControlNotOpened());
    }

    nn::Result ApplicationResourceManager::Register(nn::applet::AppletResourceUserId aruid, nn::ncm::ApplicationId applicationId) NN_NOEXCEPT
    {
        NN_CAPSRV_REQUIRES(m_IsInitialized);
        std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

        for(int i = 0; i < ApplicationCountMax; i++)
        {
            auto& e = m_ApplicationResourceList[i];
            if(e.IsInitialized() && e.GetAruid() == aruid)
            {
                NN_CAPSRV_LOG_ERR("ApplicationResource already registered (aruid=%lld(0x%016llX), applicationId=0x%016llX)\n", aruid.lower, aruid.lower, applicationId.value);
                NN_RESULT_THROW(ResultControlAlreadyOpen());
            }
        }

        for(int i = 0; i < ApplicationCountMax; i++)
        {
            if(!m_ApplicationResourceList[i].IsInitialized())
            {
                m_ApplicationResourceList[i].Initialize(aruid, applicationId);
                NN_CAPSRV_LOG_DEV("Registered (aruid=%lld, appId=0x%016llX) to slot %d\n", aruid.lower, applicationId.value, i);
                NN_RESULT_SUCCESS;
            }
        }
        NN_RESULT_THROW(ResultControlResourceLimit());
    }

    void ApplicationResourceManager::Unregister(nn::applet::AppletResourceUserId aruid, nn::ncm::ApplicationId applicationId) NN_NOEXCEPT
    {
        NN_CAPSRV_REQUIRES(m_IsInitialized);
        NN_UNUSED(applicationId);
        std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

        for(int i = 0; i < ApplicationCountMax; i++)
        {
            auto& e = m_ApplicationResourceList[i];
            if(e.IsInitialized())
            {
                if(e.GetAruid() == aruid)
                {
                    NN_CAPSRV_LOG_DEV("Unregistered (aruid=%lld, appId=0x%016llX) from slot %d\n", aruid.lower, applicationId.value, i);
                    e.Finalize();
                    return;
                }
            }
        }
    }

    nn::Result ApplicationResourceManager::GetApplicationIdFromAruid(nn::ncm::ApplicationId* pOutApplicationId, nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
    {
        NN_CAPSRV_REQUIRES(m_IsInitialized);
        std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

        ApplicationResource* pResource = nullptr;
        NN_RESULT_DO(GetApplicationResourceFromAruidImpl(&pResource, aruid));

        *pOutApplicationId = pResource->GetApplicationId();
        NN_RESULT_SUCCESS;
    }

    nn::Result ApplicationResourceManager::CheckApplicationIdRegistered(nn::ncm::ApplicationId applicationId) NN_NOEXCEPT
    {
        NN_CAPSRV_REQUIRES(m_IsInitialized);
        std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

        ApplicationResource* pResource = nullptr;
        NN_RESULT_DO(GetApplicationResourceFromApplicationIdImpl(&pResource, applicationId));

        NN_RESULT_SUCCESS;
    }

    nn::Result ApplicationResourceManager::GetApplicationAlbumEntryKeyFromApplicationId(ApplicationAlbumEntryKey* pOutValue, nn::ncm::ApplicationId applicationId) NN_NOEXCEPT
    {
        NN_CAPSRV_REQUIRES(m_IsInitialized);
        std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

        ApplicationResource* pResource = nullptr;
        NN_RESULT_DO(GetApplicationResourceFromApplicationIdImpl(&pResource, applicationId));

        *pOutValue = pResource->GetApplicationAlbumEntryKey();
        NN_RESULT_SUCCESS;
    }

    nn::Result ApplicationResourceManager::GetApplicationAlbumEntryKeyFromAruid(ApplicationAlbumEntryKey* pOutValue, nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
    {
        NN_CAPSRV_REQUIRES(m_IsInitialized);
        std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

        ApplicationResource* pResource = nullptr;
        NN_RESULT_DO(GetApplicationResourceFromAruidImpl(&pResource, aruid));

        *pOutValue = pResource->GetApplicationAlbumEntryKey();
        NN_RESULT_SUCCESS;
    }

}}}}

