﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "capsrvServer_ConvertFileSystemResult.h"

#include <nn/nn_Abort.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultPrivate.h>
#include "capsrvServer_ResultPrivate.h"

namespace nn{ namespace capsrv{ namespace server{

    namespace {
        nn::Result ConvertFileSystemResult_I(nn::Result result) NN_NOEXCEPT
        {
            NN_RESULT_TRY(result)
                NN_RESULT_CATCH(nn::fs::ResultPathNotFound           ) { NN_RESULT_THROW(ResultAlbumFileNotFound()); }
                NN_RESULT_CATCH(nn::fs::ResultPathAlreadyExists      ) { NN_RESULT_RETHROW; } // 別途ハンドリングされる
                NN_RESULT_CATCH(nn::fs::ResultUsableSpaceNotEnough   ) { NN_RESULT_THROW(ResultAlbumIsFull()); }
                NN_RESULT_CATCH(nn::fs::ResultAllocationMemoryFailed ) { NN_RESULT_THROW(ResultAlbumWorkMemoryError()); }
                NN_RESULT_CATCH(nn::fs::ResultTargetLocked           ) { NN_RESULT_THROW(ResultAlbumAccessCorrupted()); }
                NN_RESULT_CATCH(nn::fs::ResultDirectoryNotEmpty      ) { NN_RESULT_THROW(ResultAlbumAccessCorrupted()); }
                NN_RESULT_CATCH(nn::fs::ResultMountNameAlreadyExists ) { NN_RESULT_THROW(ResultAlbumFileSystemCorrupted()); }
                NN_RESULT_CATCH(nn::fs::ResultDataCorrupted          ) { NN_RESULT_THROW(ResultAlbumFileSystemCorrupted()); }
                NN_RESULT_CATCH(nn::fs::ResultOutOfRange             ) { NN_RESULT_THROW(ResultAlbumAccessCorrupted()); }
                NN_RESULT_CATCH(nn::fs::ResultDirectoryStatusChanged ) { NN_RESULT_THROW(ResultAlbumAccessCorrupted()); }
                NN_RESULT_CATCH(nn::fs::ResultPartitionNotFound      ) { NN_RESULT_THROW(ResultAlbumFileSystemCorrupted()); }
                NN_RESULT_CATCH(nn::fs::ResultTargetNotFound         ) { NN_RESULT_THROW(ResultAlbumFileSystemCorrupted()); }
                NN_RESULT_CATCH(nn::fs::ResultMmcAccessFailed        ) { NN_RESULT_THROW(ResultAlbumFileSystemCorrupted()); }
                NN_RESULT_CATCH(nn::fs::ResultSdCardAccessFailed     ) { NN_RESULT_THROW(ResultAlbumFileSystemCorrupted()); }
                NN_RESULT_CATCH(nn::fs::ResultGameCardAccessFailed   ) { NN_RESULT_THROW(ResultAlbumFileSystemCorrupted()); }
                NN_RESULT_CATCH(nn::fs::ResultUnexpected             ) { NN_RESULT_THROW(ResultAlbumUnknownFileSystemError()); }
                NN_RESULT_CATCH(nn::fs::ResultUnsupportedSdkVersion  ) { NN_RESULT_THROW(ResultAlbumFileSystemCorrupted()); }
                NN_RESULT_CATCH(nn::fs::ResultNotImplemented         ) { NN_RESULT_THROW(ResultAlbumFileSystemCorrupted()); }
                NN_RESULT_CATCH(nn::fs::ResultHandledByAllProcess    ) { NN_RESULT_THROW(ResultAlbumUnknownFileSystemError()); }
                NN_RESULT_CATCH(nn::fs::ResultHandledBySystemProcess ) { NN_RESULT_THROW(ResultAlbumUnknownFileSystemError()); }
                NN_RESULT_CATCH_ALL { NN_RESULT_THROW(ResultAlbumUnknownFileSystemError()); }
            NN_RESULT_END_TRY;
            NN_RESULT_SUCCESS;
        }
    }

    nn::Result ConvertFileSystemResult(nn::Result result, AlbumStorageDirectionType direction) NN_NOEXCEPT
    {
        // FS のエラーでなければ無視
        if(!nn::fs::ResultHandledByAllProcess::Includes(result) &&
            !nn::fs::ResultHandledBySystemProcess::Includes(result) &&
            !nn::fs::ResultInternal::Includes(result)
            )
        {
            NN_RESULT_THROW(result);
        }

        auto r = ConvertFileSystemResult_I(result);
        if(direction == AlbumStorageDirection_Source)
        {
            NN_RESULT_TRY(r)
                NN_RESULT_CATCH(ResultAlbumFileSystemCorrupted    ) { NN_RESULT_THROW(ResultAlbumSourceFileSystemCorrupted()); }
                NN_RESULT_CATCH(ResultAlbumAccessCorrupted        ) { NN_RESULT_THROW(ResultAlbumSourceAccessCorrupted()); }
                NN_RESULT_CATCH(ResultAlbumUnknownFileSystemError ) { NN_RESULT_THROW(ResultAlbumSourceUnknownFileSystemError()); }
            NN_RESULT_END_TRY;
            NN_RESULT_SUCCESS;
        }
        else if(direction == AlbumStorageDirection_Destination)
        {
            NN_RESULT_TRY(r)
                NN_RESULT_CATCH(ResultAlbumFileSystemCorrupted    ) { NN_RESULT_THROW(ResultAlbumDestinationFileSystemCorrupted()); }
                NN_RESULT_CATCH(ResultAlbumAccessCorrupted        ) { NN_RESULT_THROW(ResultAlbumDestinationAccessCorrupted()); }
                NN_RESULT_CATCH(ResultAlbumUnknownFileSystemError ) { NN_RESULT_THROW(ResultAlbumDestinationUnknownFileSystemError()); }
            NN_RESULT_END_TRY;
            NN_RESULT_SUCCESS;
        }
        else
        {
            NN_RESULT_THROW(r);
        }
    }

}}}

