﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "capsrvServer_VerifyScreenShotFileData.h"

#include <type_traits>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_StringUtil.h>
#include <nn/image/image_JpegDecoder.h>
#include <nn/image/image_ExifExtractor.h>
#include <nn/capsrv/capsrv_SystemReservedInfo.h>

#include "../capsrvServer_ResultPrivate.h"
#include "capsrvServer_AlbumPathUtility.h"
#include "../detail/capsrvServer_ParseMakerNote.h"
#include "../detail/capsrvServer_EncryptApplicationId.h"
#include "../detail/capsrvServer_ExifWorkStorage.h"
#include "../detail/capsrvServer_CalculateJpegMac.h"
#include "../detail/capsrvServer_ExtractJpegMakerNoteRange.h"
#include "../detail/capsrvServer_ScreenShotAttributeUtility.h"


namespace nn{ namespace capsrv{ namespace server{ namespace album{

    namespace {

        nn::capsrv::ScreenShotOrientationType GetScreenShotOrientation(nn::image::ExifOrientation src) NN_NOEXCEPT
        {
            switch(src)
            {
            case nn::image::ExifOrientation_Normal:
                return ScreenShotOrientation_Default;
            case nn::image::ExifOrientation_Rotate90:
                return ScreenShotOrientation_Rotate90;
            case nn::image::ExifOrientation_Rotate180:
                return ScreenShotOrientation_Rotate180;
            case nn::image::ExifOrientation_Rotate270:
                return ScreenShotOrientation_Rotate270;
            default:
                return ScreenShotOrientation_Default;
            }
        }

        //------------------------------------------------------------------------------------

        struct ExtractedExifInfo
        {
            const char* pDateTime;
            size_t dateTimeSize;
            nn::image::ExifOrientation orientation;
            detail::MakerNoteInfo makerNoteInfo;
        };


        nn::Result ExtractExifInfo(
            ExtractedExifInfo* pOutValue,
            const void* pFileData,
            size_t fileDataSize
            ) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(pOutValue);
            NN_SDK_REQUIRES_NOT_NULL(pFileData);
            NN_SDK_REQUIRES_GREATER(fileDataSize, 0u);

            // Exif を抽出
            const void* pExif = nullptr;
            size_t exifSize = 0;
            {
                auto exifResult = nn::image::JpegDecoder::GetExifData(&pExif, &exifSize, pFileData, fileDataSize);
                NN_RESULT_THROW_UNLESS(
                    exifResult == nn::image::JpegStatus_Ok &&
                    pExif != nullptr &&
                    exifSize > 0,
                    ResultInternalFileDataVerificationExifExtractionFailed()
                );
            }

            // 抽出
            const char* pDateTime = nullptr;
            size_t dateTimeSize = 0;
            nn::image::ExifOrientation orientation = nn::image::ExifOrientation_Normal;
            const void* pMakerNote = nullptr;
            size_t makerNoteSize = 0;
            {
                detail::ExifExtractWorkStorage workBuffer;
                NN_SDK_ASSERT_GREATER_EQUAL(sizeof(workBuffer), nn::image::ExifExtractor::GetWorkBufferSize());
                nn::image::ExifExtractor extractor(&workBuffer, sizeof(workBuffer));
                extractor.SetExifData(pExif, exifSize);
                auto analyzeResult =  extractor.Analyze();
                NN_RESULT_THROW_UNLESS(
                    analyzeResult == nn::image::JpegStatus_Ok,
                    ResultInternalFileDataVerificationExifAnalyzationFailed()
                );
                pDateTime = extractor.ExtractDateTime(&dateTimeSize);
                NN_RESULT_THROW_UNLESS(
                    pDateTime != nullptr && dateTimeSize > 0,
                    ResultInternalFileDataVerificationDateTimeExtractionFailed()
                );

                if(!extractor.ExtractOrientation(&orientation))
                {
                    orientation = nn::image::ExifOrientation_Normal;
                }

                pMakerNote = extractor.ExtractMakerNote(&makerNoteSize);
                NN_RESULT_THROW_UNLESS(
                    pMakerNote != nullptr && makerNoteSize > 0,
                    ResultInternalFileDataVerificationMakerNoteExtractionFailed()
                );
            }

            // MakerNote を解析
            detail::MakerNoteInfo makerNoteInfo = {};
            NN_RESULT_DO(TryParseMakerNote(&makerNoteInfo, pMakerNote, makerNoteSize));

            pOutValue->pDateTime     = pDateTime;
            pOutValue->dateTimeSize  = dateTimeSize;
            pOutValue->orientation   = orientation;
            pOutValue->makerNoteInfo = makerNoteInfo;
            NN_RESULT_SUCCESS;
        }

        // MakerNote の修正を行う。
        // MakerNote の解析直後（値のチェックの前）に実行される。
        // エントリーのデフォルト値を他の値を使って設定する場合に使用する。
        nn::Result ModifyMakerNote(ExtractedExifInfo& exifInfo, const AlbumFileId& fileId) NN_NOEXCEPT
        {
            auto& makerNoteInfo = exifInfo.makerNoteInfo;

            if(makerNoteInfo.version == detail::MakerNoteVersion_Version0)
            {
                makerNoteInfo.dateTime = fileId.time;
                makerNoteInfo.dataOrientation = GetScreenShotOrientation(exifInfo.orientation);
            }

            // movieDataWidth
            if(detail::MakerNoteValueProperty::MovieDataWidthProperty::IsDefaultValue(makerNoteInfo.movieDataWidth, makerNoteInfo.version))
            {
                makerNoteInfo.movieDataWidth = makerNoteInfo.dataWidth;
            }

            // movieDataHeight
            if(detail::MakerNoteValueProperty::MovieDataHeightProperty::IsDefaultValue(makerNoteInfo.movieDataHeight, makerNoteInfo.version))
            {
                makerNoteInfo.movieDataHeight = makerNoteInfo.dataHeight;
            }

            NN_RESULT_SUCCESS;
        }

        //---------------------------------------------------------------------------------

        nn::Result VerifySignature(const void* pFileData, size_t fileDataSize) NN_NOEXCEPT
        {
            NN_CAPSRV_LOG_DEV("verifying signature\n");
            NN_SDK_REQUIRES_NOT_NULL(pFileData);
            NN_SDK_REQUIRES_GREATER(fileDataSize, 0u);

            detail::ExifExtractWorkStorage work;

            int64_t makerNoteOffset = 0;
            int64_t makerNoteSize = 0;
            NN_RESULT_DO(
                detail::ExtractJpegMakerNoteRange(&makerNoteOffset, &makerNoteSize, pFileData, fileDataSize, &work, sizeof(work))
            );

            NN_RESULT_DO(
                detail::CalculateAndCheckJpegMac(pFileData, fileDataSize, makerNoteOffset, makerNoteSize)
            );

            NN_CAPSRV_LOG_DEV("  -> success\n");
            NN_RESULT_SUCCESS;
        }

        nn::Result VerifyDateTime(const ExtractedExifInfo& exifInfo, const AlbumFileId* pFileId) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(pFileId);
            NN_SDK_REQUIRES(AlbumPathUtility::ValidateTime(&pFileId->time).IsSuccess());

            // Exif と AlbumFileId の DateTime が一致するか検査
            {
                const char* pDateTime = exifInfo.pDateTime;
                size_t dateTimeSize = exifInfo.dateTimeSize;
                NN_SDK_ASSERT_NOT_NULL(pDateTime);
                NN_SDK_ASSERT_GREATER(dateTimeSize, 0u);

                const size_t DateTimeStringSize = 20;

                NN_RESULT_THROW_UNLESS(
                    dateTimeSize == DateTimeStringSize,
                    ResultInternalFileDataVerificationInvalidDateTimeLength()
                );

                const AlbumFileDateTime& time = pFileId->time;
                char str[DateTimeStringSize] = {};

                int length = nn::util::SNPrintf(
                    str,
                    DateTimeStringSize,
                    "%04d:%02d:%02d %02d:%02d:%02d",
                    static_cast<int>(time.year),
                    static_cast<int>(time.month),
                    static_cast<int>(time.day),
                    static_cast<int>(time.hour),
                    static_cast<int>(time.minute),
                    static_cast<int>(time.second)
                    );
                NN_SDK_ASSERT_EQUAL(length, static_cast<int>(DateTimeStringSize - 1));
                NN_UNUSED(length);

                NN_RESULT_THROW_UNLESS(
                    nn::util::Strncmp(str, pDateTime, DateTimeStringSize) == 0,
                    ResultInternalFileDataVerificationInconsistentDateTime()
                );
            }

            // MakerNote と AlbumFileId の DateTime が一致するか検査
            {
                NN_STATIC_ASSERT(sizeof(exifInfo.makerNoteInfo.dateTime) == sizeof(pFileId->time));
                NN_RESULT_THROW_UNLESS(
                    0 == std::memcmp(&exifInfo.makerNoteInfo.dateTime, &pFileId->time, sizeof(AlbumFileDateTime)),
                    ResultInternalFileDataVerificationInconsistentDateTime()
                );
            }

            NN_RESULT_SUCCESS;
        }

        nn::Result VerifyOrientation(const ExtractedExifInfo& exifInfo) NN_NOEXCEPT
        {
            // Exif の値が適切か検査
            switch(exifInfo.orientation)
            {
            case nn::image::ExifOrientation_Normal:
            case nn::image::ExifOrientation_Rotate90:
            case nn::image::ExifOrientation_Rotate180:
            case nn::image::ExifOrientation_Rotate270:
                // ok
                break;
            default:
                NN_RESULT_THROW(ResultInternalFileDataVerificationUnsupportedOrientation());
            }

            // MakerNote と Exif の回転が一致するか検査
            {
                NN_RESULT_THROW_UNLESS(
                    exifInfo.makerNoteInfo.dataOrientation == GetScreenShotOrientation(exifInfo.orientation),
                    ResultInternalFileDataVerificationInconsistentOrientation()
                );
            }

            NN_RESULT_SUCCESS;
        }

        nn::Result VerifyApplicationId(const detail::MakerNoteInfo* pMakerNoteInfo, const AlbumFileId* pFileId) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(pMakerNoteInfo);
            NN_SDK_REQUIRES_NOT_NULL(pFileId);

            nn::ncm::ApplicationId appId;
            bool isExtra;
            NN_RESULT_DO(TryDecryptApplicationId(&appId, &isExtra, pMakerNoteInfo->encryptedApplicationId));
            NN_RESULT_THROW_UNLESS(IsExtraAlbumFileContents(pFileId->contents) == isExtra, ResultAlbumInvalidApplicationId());

            NN_RESULT_THROW_UNLESS(
                appId == pFileId->applicationId,
                ResultInternalFileDataVerificationInconsistentApplicationId()
            );
            NN_RESULT_SUCCESS;
        }

        nn::Result VerifyScreenShotDataWidthAndHeight(const detail::MakerNoteInfo& info) NN_NOEXCEPT
        {
            NN_RESULT_THROW_UNLESS(info.dataWidth == ScreenShotWidth, ResultInternalFileDataVerificationInvalidDataDimension());
            NN_RESULT_THROW_UNLESS(info.dataHeight == ScreenShotHeight, ResultInternalFileDataVerificationInvalidDataDimension());
            NN_RESULT_SUCCESS;
        }

        nn::Result VerifyMovieDataWidthAndHeight(const detail::MakerNoteInfo& info, const AlbumFileId& fileId) NN_NOEXCEPT
        {
            if(fileId.contents == AlbumFileContents_Movie)
            {
                NN_RESULT_THROW_UNLESS(
                    (info.movieDataWidth == 1280 && info.movieDataHeight == 720),
                    ResultInternalFileDataVerificationInvalidDataDimension()
                );
            }
            else if(fileId.contents == AlbumFileContents_ExtraMovie)
            {
                NN_RESULT_THROW_UNLESS(
                    (info.movieDataWidth == 1280 && info.movieDataHeight == 720) ||
                    (info.movieDataWidth == 1920 && info.movieDataHeight == 1080),
                    ResultInternalFileDataVerificationInvalidDataDimension()
                );
            }
            else
            {
                NN_RESULT_THROW_UNLESS(
                    (info.movieDataWidth == ScreenShotWidth && info.movieDataHeight == ScreenShotHeight),
                    ResultInternalFileDataVerificationInvalidDataDimension()
                );
            }
            NN_RESULT_SUCCESS;
        }

        nn::Result VerifyDescription(const detail::MakerNoteInfo& info, AlbumFileContentsType contents) NN_NOEXCEPT
        {
            bool isOk = false;
            auto version = info.version;
            switch(info.fileDescription)
            {
            case AlbumFileDescription_ScreenShot:
                {
                    if(version == 0 && contents == AlbumFileContents_ScreenShot)
                    {
                        isOk = true;
                    }
                    break;
                }
            case AlbumFileDescription_ScreenShotCaptured:
            case AlbumFileDescription_ScreenShotEdited:
            case AlbumFileDescription_ScreenShotSaved:
                {
                    if(version >= 1 && contents == AlbumFileContents_ScreenShot)
                    {
                        isOk = true;
                    }
                    break;
                }
            case AlbumFileDescription_MovieContinuous:
            case AlbumFileDescription_MovieTrimmed:
                {
                    if(version >= 1 && contents == AlbumFileContents_Movie)
                    {
                        isOk = true;
                    }
                    break;
                }
            case AlbumFileDescription_MovieMakerSaved:
                {
                    if(version >= 2 && contents == AlbumFileContents_ExtraMovie)
                    {
                        isOk = true;
                    }
                    break;
                }
            default: break;
            }
            NN_RESULT_THROW_UNLESS(isOk, ResultInternalFileDataVerificationError());
            NN_RESULT_SUCCESS;
        }

        nn::Result VerifyMovieAttribute(const detail::MakerNoteInfo& info, AlbumFileContentsType contents) NN_NOEXCEPT
        {
            bool isOk = false;
            if(contents == AlbumFileContents_ScreenShot || contents == AlbumFileContents_ExtraScreenShot)
            {
                isOk =
                    detail::MakerNoteValueProperty::FrameCountProperty::IsDefaultValue(info.frameCount, info.version) &&
                    detail::MakerNoteValueProperty::FrameRateNumeratorProperty::IsDefaultValue(info.frameRateNumerator, info.version) &&
                    detail::MakerNoteValueProperty::FrameRateDenominatorProperty::IsDefaultValue(info.frameRateDenominator, info.version) &&
                    detail::MakerNoteValueProperty::DataDurationProperty::IsDefaultValue(info.dataDurationMilliseconds, info.version) &&
                    detail::MakerNoteValueProperty::KeyFrameIntervalProperty::IsDefaultValue(info.keyFrameInterval, info.version)
                    ;
            }
            else if(contents == AlbumFileContents_Movie || contents == AlbumFileContents_ExtraMovie)
            {
                // TODO: 妥当な範囲で制限
                isOk = true;
            }
            NN_RESULT_THROW_UNLESS(isOk, ResultInternalFileDataVerificationError());
            NN_RESULT_SUCCESS;
        }

        // MakerNoteInfo 内の SystemReservedInfo の内容についてチェックする。
        // 今後、SystemReservedInfo に項目を追加する場合には、
        // ここにもその内容をチェックするコードを追加する。
        nn::Result VerifySystemReservedInfo(const detail::MakerNoteInfo& info) NN_NOEXCEPT
        {
            // UA 情報のチェック
            {
                const auto& uidList = info.systemReservedInfo.uidList;
                auto count = uidList.uidCount;

                NN_RESULT_THROW_UNLESS(0 <= count && count <= UserIdListCountMax, ResultInternalFileDataVerificationError());
                for (int i=0; i<count; ++i)
                {
                    NN_RESULT_THROW_UNLESS(uidList.uid[i] != account::InvalidUid, ResultInternalFileDataVerificationError());
                }
            }

            // 次の項目のチェック
            {
                // _reserved[] はチェックしない
            }

            // 全て PASS すれば Success を返す
            NN_RESULT_SUCCESS;

        }   // NOLINT(impl/function_size)

    }   // namespace

        nn::Result VerifyAndLoadScreenShotMakerNoteInfoImpl(
            nn::capsrv::server::detail::MakerNoteInfo* pOutMakerNoteInfo,
            const AlbumFileId* pFileId,
            const void* pFileData,
            size_t fileDataSize,
            const EnvironmentInfo& env,
            FileDataRange fileDataRange
            ) NN_NOEXCEPT
        {
            NN_CAPSRV_LOG_DEV("verifying screenshot file data...\n");
            NN_SDK_REQUIRES_NOT_NULL(pFileId);
            NN_SDK_REQUIRES(AlbumPathUtility::ValidateFileId(pFileId, env).IsSuccess());
            NN_SDK_REQUIRES(fileDataSize == 0 || pFileData != nullptr);
            NN_SDK_REQUIRES(fileDataRange == FileDataRange_WholeFile || fileDataRange == FileDataRange_HeaderOnly);

            const bool isVerifying = env.IsAlbumScreenShotFileDataVerificationEnabled();
            if(!isVerifying)
            {
                NN_CAPSRV_LOG_WARN("screenshot file data verification is disabled\n");
            }

            NN_RESULT_THROW_UNLESS(
                env.IsAlbumFileContentsSupported(pFileId->contents),
                ResultAlbumInvalidFileContents()
            );

            // ファイルサイズが正であることを検査
            NN_CAPSRV_LOG_DEV("  checking data size\n");
            NN_RESULT_THROW_UNLESS(
                fileDataSize > 0,
                ResultInternalFileDataVerificationEmptyFileData()
            );

            // Signature を検査
            if(isVerifying && fileDataRange == FileDataRange_WholeFile)
            {
                NN_CAPSRV_LOG_DEV("  checking signature\n");
                NN_RESULT_DO(VerifySignature(pFileData, fileDataSize));
            }

            // Exif を解析
            NN_CAPSRV_LOG_DEV("  extracting exif\n");
            ExtractedExifInfo exifInfo = {};
            NN_RESULT_DO(ExtractExifInfo(&exifInfo, pFileData, fileDataSize));
            NN_RESULT_DO(ModifyMakerNote(exifInfo, *pFileId));

            // DateTime を検査
            if(isVerifying)
            {
                NN_CAPSRV_LOG_DEV("  checking datetime\n");
                NN_RESULT_DO(VerifyDateTime(exifInfo, pFileId));
            }

            // ScreenShotDataWidth, ScreenShotDataHeight を検査
            if(isVerifying)
            {
                NN_CAPSRV_LOG_DEV("  checking screenshot width, height\n");
                NN_RESULT_DO(VerifyScreenShotDataWidthAndHeight(exifInfo.makerNoteInfo));
            }

            // MovieDataWidth, MovieDataHeight を検査
            if(isVerifying)
            {
                NN_CAPSRV_LOG_DEV("  checking movie width, height\n");
                NN_RESULT_DO(VerifyMovieDataWidthAndHeight(exifInfo.makerNoteInfo, *pFileId));
            }

            // Orientation を検査
            if(isVerifying)
            {
                NN_CAPSRV_LOG_DEV("  checking orientation\n");
                NN_RESULT_DO(VerifyOrientation(exifInfo));
            }

            // ApplicationID を検査
            if(isVerifying)
            {
                NN_CAPSRV_LOG_DEV("  checking applicationId\n");
                NN_RESULT_DO(VerifyApplicationId(&exifInfo.makerNoteInfo, pFileId));
            }

            // Description を検査
            if(isVerifying)
            {
                NN_CAPSRV_LOG_DEV("  checking description\n");
                NN_RESULT_DO(VerifyDescription(exifInfo.makerNoteInfo, pFileId->contents));
            }

            // MovieAttribute を検査
            if(isVerifying)
            {
                NN_CAPSRV_LOG_DEV("  checking movie attribute\n");
                NN_RESULT_DO(VerifyMovieAttribute(exifInfo.makerNoteInfo, pFileId->contents));
            }

            // SystemReservedInfo を検査
            if(isVerifying)
            {
                NN_CAPSRV_LOG_DEV("  checking system reserved info\n");
                NN_RESULT_DO(VerifySystemReservedInfo(exifInfo.makerNoteInfo));
            }

            *pOutMakerNoteInfo = exifInfo.makerNoteInfo;
            NN_RESULT_SUCCESS;
        }

    namespace {

        nn::Result VerifyScreenShotFileDataImpl(
            ScreenShotAttribute* pOutAttribute,
            nn::capsrv::AppletData* pOutAppletData,
            nn::capsrv::ApplicationData* pOutApplicationData,
            nn::capsrv::SystemReservedInfo* pOutSystemReservedInfo,
            const AlbumFileId* pFileId,
            const void* pFileData,
            size_t fileDataSize,
            const EnvironmentInfo& env,
            FileDataRange fileDataRange
            ) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(pFileId);

            detail::MakerNoteInfo makerNoteInfo = {};
            NN_RESULT_DO(VerifyAndLoadScreenShotMakerNoteInfoImpl(&makerNoteInfo, pFileId, pFileData, fileDataSize, env, fileDataRange));

            // 属性を取得
            if(pOutAttribute)
            {
                NN_CAPSRV_LOG_DEV("  extracting attributes\n");
                // 値は検査済なので失敗してはいけない。
                NN_RESULT_DO(detail::ScreenShotAttributeUtility::ConvertFromMakerNoteInfo(pOutAttribute, makerNoteInfo));
            }
            if(pOutAppletData)
            {
                NN_CAPSRV_LOG_DEV("  extracting applet data\n");
                std::memcpy(pOutAppletData, &makerNoteInfo.appletData, sizeof(AppletData));
            }
            if(pOutApplicationData)
            {
                NN_CAPSRV_LOG_DEV("  extracting application data\n");
                std::memcpy(pOutApplicationData, &makerNoteInfo.applicationData, sizeof(ApplicationData));
            }
            if(pOutSystemReservedInfo)
            {
                NN_CAPSRV_LOG_DEV("  extracting system reserved info\n");
                std::memcpy(pOutSystemReservedInfo, &makerNoteInfo.systemReservedInfo, sizeof(SystemReservedInfo));
            }

            NN_CAPSRV_LOG_DEV("  -> success\n");
            NN_RESULT_SUCCESS;
        }

    }// anonymous namespace


    nn::Result VerifyScreenShotFileData(
        const AlbumFileId* pFileId,
        const void* pFileData,
        size_t fileDataSize,
        const EnvironmentInfo& env
        ) NN_NOEXCEPT
    {
        return VerifyScreenShotFileDataImpl(nullptr, nullptr, nullptr, nullptr, pFileId, pFileData, fileDataSize, env, FileDataRange_WholeFile);
    }

    nn::Result VerifyScreenShotFileDataEx(
        ScreenShotAttribute* pOutAttribute,
        nn::capsrv::AppletData* pOutAppletData,
        nn::capsrv::ApplicationData* pOutApplicationData,
        nn::capsrv::SystemReservedInfo* pOutSystemReservedInfo,
        const AlbumFileId* pFileId,
        const void* pFileData,
        size_t fileDataSize,
        const EnvironmentInfo& env
        ) NN_NOEXCEPT
    {
        return VerifyScreenShotFileDataImpl(pOutAttribute, pOutAppletData, pOutApplicationData, pOutSystemReservedInfo, pFileId, pFileData, fileDataSize, env, FileDataRange_WholeFile);
    }

    nn::Result VerifyScreenShotFileHeader(
        const AlbumFileId* pFileId,
        const void* pFileHeader,
        size_t fileHeaderSize,
        const EnvironmentInfo& env
        ) NN_NOEXCEPT
    {
        return VerifyScreenShotFileDataImpl(nullptr, nullptr, nullptr, nullptr, pFileId, pFileHeader, fileHeaderSize, env, FileDataRange_HeaderOnly);
    }

    nn::Result VerifyScreenShotFileHeaderEx(
        ScreenShotAttribute* pOutAttribute,
        nn::capsrv::AppletData* pOutAppletData,
        nn::capsrv::ApplicationData* pOutApplicationData,
        nn::capsrv::SystemReservedInfo* pOutSystemReservedInfo,
        const AlbumFileId* pFileId,
        const void* pFileHeader,
        size_t fileHeaderSize,
        const EnvironmentInfo& env
        ) NN_NOEXCEPT
    {
        return VerifyScreenShotFileDataImpl(pOutAttribute, pOutAppletData, pOutApplicationData, pOutSystemReservedInfo, pFileId, pFileHeader, fileHeaderSize, env, FileDataRange_HeaderOnly);
    }

}}}}
