﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/capsrv/capsrv_AlbumFileId.h>
#include <nn/util/util_BitFlagSet.h>
#include "../capsrvServer_Config.h"
#include "../capsrvServer_SessionId.h"
#include "capsrvServer_AlbumFileManipulator.h"
#include "capsrvServer_MovieStreamId.h"
#include "capsrvServer_MovieWriteHashSignature.h"

namespace nn{ namespace capsrv{ namespace server{ namespace album{

    enum MovieWriteStreamState
    {
        MovieWriteStreamState_Unused = 0,
        MovieWriteStreamState_Empty,
        MovieWriteStreamState_WritingData,
        MovieWriteStreamState_DataComplete,
        MovieWriteStreamState_WritingMeta,
        MovieWriteStreamState_MetaComplete,
        MovieWriteStreamState_Complete,
        MovieWriteStreamState_Broken,
    };

    class MovieWriteStreamEntry
    {
    private:
        // 自動的にファイルを拡大する。
        // SIGLO-68867
        //   ハッシュの計算単位で拡大すると著しくファイルが断片化する可能性があるため、
        //   より大きな単位でファイルを作成・伸長する。
        class WriteFileStream
        {
        public:
            WriteFileStream() NN_NOEXCEPT;

            bool IsOpened() const NN_NOEXCEPT;

            // 初期サイズでファイルを作成する。
            // 不足した際にファイルは自動で拡大される。
            nn::Result Open(AlbumCacheDelta* pOutDelta, const AlbumFileId& fileId, const EnvironmentInfo& env) NN_NOEXCEPT;

            // ファイルをクローズする。
            // 正常終了の場合には先に SetFileSize を呼んでファイルサイズを確定させること。
            void Close() NN_NOEXCEPT;

            // ファイルから読込む
            nn::Result ReadFile(size_t* pOutSize, int64_t offset, void* buffer, size_t size) NN_NOEXCEPT;

            // ファイルに書き出す。
            // ファイルサイズを超える場合、自動的に拡大される。
            nn::Result WriteFile(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT;

            // ファイルサイズを設定する。
            nn::Result SetFileSize(int64_t size) NN_NOEXCEPT;

        private:
            AlbumFileHandle m_FileHandle;
            int64_t         m_FileSize;
            bool            m_IsFileOpened;
        };

    public:
        MovieWriteStreamEntry() NN_NOEXCEPT;

        static int64_t GetFileSizeLimit() NN_NOEXCEPT;

        MovieWriteStreamState GetState() const NN_NOEXCEPT;
        SessionId GetClientSessionId() const NN_NOEXCEPT;
        MovieStreamId GetStreamId() const NN_NOEXCEPT;
        const AlbumFileId& GetAlbumFileId() const NN_NOEXCEPT;
        bool IsBroken() const NN_NOEXCEPT;
        nn::Result GetBrokenReason() const NN_NOEXCEPT;

        void NotifyStorageUnmounted(AlbumStorageType storage) NN_NOEXCEPT;

        nn::Result OpenFile(
            AlbumCacheDelta* pOutCacheDelta,
            SessionId clientSessionId,
            MovieStreamId streamId,
            const AlbumFileId& fileId,
            const EnvironmentInfo& env
        ) NN_NOEXCEPT;
        nn::Result StartDataSection() NN_NOEXCEPT;
        nn::Result EndDataSection() NN_NOEXCEPT;
        nn::Result StartMetaSection() NN_NOEXCEPT;
        nn::Result EndMetaSection() NN_NOEXCEPT;
        nn::Result FinishFile(void* workMemory, size_t workMemorySize) NN_NOEXCEPT;
        nn::Result CommitFile() NN_NOEXCEPT;
        void DiscardFile(
            AlbumCacheDelta* pOutCacheDelta,
            bool isDeleteSuppressed,
            const EnvironmentInfo& env
        ) NN_NOEXCEPT;

        nn::Result ReadData(size_t* pOutReadSize, void* buffer, size_t size, int64_t offset) NN_NOEXCEPT;
        nn::Result GetDataSize(int64_t* pOutValue) NN_NOEXCEPT;
        nn::Result WriteData(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT;
        nn::Result SetDataSize(int64_t size) NN_NOEXCEPT;
        nn::Result WriteMeta(const void* meta, size_t size, uint64_t makerNoteVersion, int64_t makerNoteOffset, int64_t makerNoteSize, void* workMemory, size_t workMemorySize) NN_NOEXCEPT;

    private:
        void MarkAsBroken(nn::Result brokenReason) NN_NOEXCEPT;
        // result.IsFailure() なら MarkBroken を呼び出す
        // @return result の値をそのまま返す。
        nn::Result MarkAsBrokenIfFailure(nn::Result result) NN_NOEXCEPT;

        // @post FileHandle が無効
        // @post IsFileOpened == false
        // @details IsFileOpened == false の場合、何もしない
        void CloseFileInternal() NN_NOEXCEPT;

        nn::Result WriteHashSignImpl(void* workMemory, size_t workMemorySize) NN_NOEXCEPT;
        nn::Result WriteVersionImpl() NN_NOEXCEPT;

    private:
        MovieWriteStreamState m_State;
        nn::Result      m_BrokenReason;

        SessionId       m_ClientSessionId;
        MovieStreamId   m_StreamId;
        AlbumFileId     m_FileId;
        uint64_t        m_Version;

        int64_t         m_DataOffset;
        int64_t         m_DataSize;
        int64_t         m_MetaOffset;
        int64_t         m_MetaSize;
        int64_t         m_HashSignOffset;
        int64_t         m_HashSignSize;
        int64_t         m_VersionOffset;
        int64_t         m_VersionSize;
        nn::util::BitFlagSet<MovieDataBlockCountLimit> m_WrittenDataBlockFlag;

        WriteFileStream m_FileStream;
        MovieWriteHashSignature m_HashSignature{};
    };

}}}}
