﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/fs.h>
#include <nn/ncm/ncm_ContentMetaId.h>
#include <nn/account/account_Types.h>
#include <nn/capsrv/capsrv_ApplicationAlbumFileEntry.h>
#include <nn/capsrv/capsrv_AlbumFileContents.h>
#include <nn/capsrv/capsrv_AlbumStorage.h>
#include <nn/capsrv/capsrv_AlbumEntry.h>
#include <nn/capsrv/capsrv_AlbumUsage.h>
#include <nn/capsrv/capsrv_AlbumCacheData.h>
#include "../capsrvServer_Config.h"
#include "../capsrvServer_EnvironmentInfo.h"
#include "capsrvServer_AlbumStorageCache.h"
#include "capsrvServer_AlbumFileLockTable.h"
#include "capsrvServer_AlbumFileAttribute.h"
#include "capsrvServer_MovieStreamManager.h"
#include "../detail/capsrvServer_MakerNoteInfo.h"

namespace nn{ namespace capsrv{ namespace server{ namespace album{

    class AlbumManager
    {
    private:

        enum class StorageMountStatus
        {
            // ストレージが無効です。
            Unavailable,
            // マウントを試みていません。
            AvailableNotTried,
            // マウント済です。
            AvailableMounted,
        };

        struct StorageStatus
        {
        public:
            void SetAvailableNotTried() NN_NOEXCEPT;
            void SetAvailable() NN_NOEXCEPT;
            void SetUnavailable() NN_NOEXCEPT;
            void SetUnavailable(nn::Result result) NN_NOEXCEPT;

        public:
            StorageMountStatus mountStatus;
            nn::Result mountResult;
        };

    public:
        AlbumManager() NN_NOEXCEPT;

        // @brief 初期化します。
        // @details
        //   すべてのストレージは無効状態になります。
        //   pEnvironmentInfo は参照を保持します。
        void Initialize(const EnvironmentInfo* pEnvironmentInfo, ResourceIdManager* pResourceIdMgr) NN_NOEXCEPT;
        // @brief 破棄します。
        // @details
        //   すべてのマウント済ストレージをアンマウントします。
        void Finalize() NN_NOEXCEPT;

        void SetWorkMemory(void* memory, size_t size) NN_NOEXCEPT;

        // @retval nn::ResultSuccess
        // @retval nn::capsrv::ResultAlbumInvalidStorage
        // @retval nn::capsrv::ResultAlbumIsNotMounted
        // @retval nn::capsrv::ResultAlbumStructureCorrupted
        nn::Result EnsureMounted(AlbumStorageType storage) NN_NOEXCEPT;
        nn::Result EnsureMounted(AlbumStorageType storage, AlbumStorageDirection direction) NN_NOEXCEPT;

        // @brief 指定したストレージを強制的にアンマウント状態にします。
        // @details
        //   EnsureMounted() が ResultAlbumIsNotMounted を返すようになります。
        //   この関数で設定した状態は一度 Finalize() して Initialize() するとリセットされます。
        //   マウント済の場合アンマウントしてマウント失敗状態にします。
        //   マウントを試行していない場合はマウント失敗状態にします。
        nn::Result ForceUnmounted(AlbumStorageType storage) NN_NOEXCEPT;

        // @brief 指定したストレージをマウント未試行状態にします。
        // @details
        //   次の EnsureMounted() の呼び出しによってマウントが試みられます。
        //   マウント済の場合アンマウントしてマウント未試行状態にします。
        //   マウントに失敗していた場合はマウント未試行状態にします。
        nn::Result ResetMountStatus(AlbumStorageType storage) NN_NOEXCEPT;

        // @brief 指定したストレージのキャッシュを再構築します。
        // @pre ワークメモリが設定されている
        nn::Result RefreshStorageCache(AlbumStorageType storage) NN_NOEXCEPT;

        // @brief 指定したストレージのキャッシュを取得します。
        nn::Result GetStorageCache(AlbumCacheData* pOutValue, AlbumStorageType storage, AlbumFileContentsType contents) NN_NOEXCEPT;

        // @brief 指定したストレージのファイル上限をチェックする。
        // @pre fileCountDelta >= 0
        nn::Result CheckAlbumLimitation(AlbumStorageType storage, AlbumFileContentsType contents, int64_t fileCountDelta) const NN_NOEXCEPT;


        // @retval nn::ResultSuccess
        // @retval nn::capsrv::ResultAlbumError
        // @retval nn::capsrv::ResultAlbumInvalidStorage
        // @retval nn::capsrv::ResultAlbumIsNotMounted
        // @retval その他
        nn::Result GetFileCount(int* pOutCount, AlbumStorageType storage, AlbumFileContentsFlag contentsMask) NN_NOEXCEPT;

        // @retval nn::ResultSuccess
        // @retval nn::capsrv::ResultAlbumInvalidStorage
        // @retval nn::capsrv::ResultAlbumIsNotMounted
        // @retval nn::capstv::ResultAlbumError
        // @retval その他
        nn::Result GetFileList(int* pOutCount, AlbumEntry* pOutList, int listCapacity, AlbumStorageType storage, AlbumFileContentsFlag contentsMask) NN_NOEXCEPT;

        // @retval nn::ResultSuccess
        // @retval nn::capsrv::ResultAlbumInvalidStorage
        // @retval nn::capsrv::ResultAlbumIsNotMounted
        // @retval nn::capstv::ResultAlbumError
        // @retval その他
        nn::Result GetFileListForApplication(int* pOutCount, ApplicationAlbumFileEntry* pOutList, int listCapacity, AlbumStorageType storage, AlbumFileContentsFlag contentsMask, const account::Uid* pUid, AlbumFileDateTime beginDateTime, AlbumFileDateTime endDateTime, nn::ncm::ApplicationId applicationId) NN_NOEXCEPT;

        // @retval nn::ResultSuccess
        // @retval nn::capsrv::ResultAlbumError
        // @retval nn::capsrv::ResultAlbumInvalidFileId
        // @retval nn::capsrv::ResultAlbumIsNotMounted
        // @retval nn::capsrv::ResultPathNotFound
        // @retval その他
        nn::Result GetFileSize(size_t* pOutValue, const AlbumFileId* pFileId) NN_NOEXCEPT;

        // @retval nn::ResultSuccess
        // @retval nn::capsrv::ResultAlbumError
        // @retval nn::capsrv::ResultAlbumInvalidFileId
        // @retval nn::capsrv::ResultAlbumIsNotMounted
        // @retval nn::capsrv::ResultPathNotFound
        // @retval nn::capsrv::ResultAlbumReadBufferShortage
        // @retval その他
        // pOutAttribute == nullptr の場合、値を書き込まない。
        // pOutAppletData == nullptr の場合、値を書き込まない。
        nn::Result LoadFile(size_t* pOutFileSize, void* pBuffer, size_t bufferSize, const AlbumFileId* pFileId) NN_NOEXCEPT;
        nn::Result LoadFileEx(AlbumFileAttribute* pOutAttribute, AppletData* pOutAppletData, ApplicationData* pOutApplicationData, SystemReservedInfo* pOutSystemReservedInfo, size_t* pOutFileSize, void* pBuffer, size_t bufferSize, const AlbumFileId* pFileId) NN_NOEXCEPT;

        // @retval nn::ResultSuccess
        // @retval nn::capsrv::ResultAlbumError
        // @retval nn::capsrv::ResultAlbumInvalidFileId
        // @retval nn::capsrv::ResultAlbumIsNotMounted
        // @retval nn::capsrv::ResultPathNotFound
        // @retval nn::capsrv::ResultAlbumReadBufferShortage
        // @retval その他
        // pOutAttribute == nullptr の場合、値を書き込まない。
        // pOutAppletData == nullptr の場合、値を書き込まない。
        nn::Result LoadFileThumbnail(size_t* pOutSize, void* pBuffer, size_t bufferSize, const AlbumFileId* pFileId) NN_NOEXCEPT;
        nn::Result LoadFileThumbnailEx(AlbumFileAttribute* pOutAttribute, AppletData* pOutAppletData, ApplicationData* pOutApplicationData, SystemReservedInfo* pOutSystemReservedInfo, size_t* pOutSize, void* pBuffer, size_t bufferSize, const AlbumFileId* pFileId) NN_NOEXCEPT;

        // @retval nn::ResultSuccess
        // @retval nn::capsrv::ResultAlbumInvalidFileId
        // @retval nn::capsrv::ResultAlbumIsNotMounted
        // @retval nn::capsrv::ResultPathNotFound
        // @retval その他
        nn::Result DeleteFile(const AlbumFileId* pFileId) NN_NOEXCEPT;

        // @retval nn::ResultSuccess
        // @retval nn::capsrv::ResultAlbumInvalidFileId
        // @retval nn::capsrv::ResultAlbumInvalidStorage
        // @retval nn::capsrv::ResultAlbumIsFull
        // @retval nn::capsrv::ResultAlbumIsNotMounted
        // @retval nn::capsrv::ResultPathNotFound
        // @retval その他
        nn::Result StorageCopyFile(const AlbumFileId* pSourceFileId, AlbumStorageType destinationStorage) NN_NOEXCEPT;

        // @retval nn::ResultSuccess
        // @retval nn::capsrv::ResultAlbumInvalidStorage
        // @retval nn::capsrv::ResultAlbumIsNotMounted
        // @retval nn::capstv::ResultAlbumError
        // @retval その他
        nn::Result GetUsage(
            nn::capsrv::AlbumContentsUsage* pOutUnknownUsage,
            nn::capsrv::AlbumContentsUsage* pOutUsageArray,
            int usageArrayLength,
            AlbumStorageType storage,
            AlbumFileContentsFlag requestedContentsMask
        ) NN_NOEXCEPT;

        nn::Result GetRequiredStorageSpaceSizeToCopyAll(size_t* pOutSize, AlbumStorageType destinationStorage, AlbumStorageType sourceStorage) NN_NOEXCEPT;

    //------------
    // 静止画用
    //------------
    public:
        // @retval nn::ResultSuccess
        // @retval nn::capsrv::ResultAlbumError
        // @retval nn::capsrv::ResultAlbumInvalidStorage
        // @retval nn::capsrv::ResultAlbumIsFull
        // @retval nn::capsrv::ResultAlbumIsNotMounted
        // @retval その他
        nn::Result SaveScreenShotFile(const AlbumFileId* pFileId, const void* data, size_t dataSize, uint64_t makerNoteVersion, int64_t makerNoteOffset, int64_t makerNoteSize) NN_NOEXCEPT;

    private:
        nn::Result LoadScreenShotImageFileImpl(
            AlbumFileAttribute* pOutAttribute,
            AppletData* pOutAppletData,
            ApplicationData* pOutApplicationData,
            SystemReservedInfo* pOutSystemReservedInfo,
            size_t* pOutFileSize,
            void* pBuffer,
            size_t bufferSize,
            const AlbumFileId& fileId
            ) NN_NOEXCEPT;

        nn::Result LoadScreenShotThumbnailFileImpl(
            AlbumFileAttribute* pOutAttribute,
            AppletData* pOutAppletData,
            ApplicationData* pOutApplicationData,
            SystemReservedInfo* pOutSystemReservedInfo,
            size_t* pOutSize,
            void* pBuffer,
            size_t bufferSize,
            const AlbumFileId& fileId,
            void* workMemory,
            size_t workSize
            ) NN_NOEXCEPT;

        nn::Result VerifyAndLoadScreenShotMakerNoteInfo(
            uint64_t* pOutFileSize,
            nn::capsrv::server::detail::MakerNoteInfo* pOutMakerNoteInfo,
            const AlbumFileId& fileId,
            void* pWorkBuffer,
            size_t workBufferSize
            ) NN_NOEXCEPT;

    //------------
    // 動画用
    //------------
    private:
        nn::Result LoadMovieImageFileImpl(
            AlbumFileAttribute* pOutAttribute,
            AppletData* pOutAppletData,
            ApplicationData* pOutApplicationData,
            SystemReservedInfo* pOutSystemReservedInfo,
            size_t* pOutFileSize,
            void* pBuffer,
            size_t bufferSize,
            const AlbumFileId& fileId,
            void* workMemory,
            size_t workSize
            ) NN_NOEXCEPT;

        nn::Result LoadMovieThumbnailFileImpl(
            AlbumFileAttribute* pOutAttribute,
            AppletData* pOutAppletData,
            ApplicationData* pOutApplicationData,
            SystemReservedInfo* pOutSystemReservedInfo,
            size_t* pOutSize,
            void* pBuffer,
            size_t bufferSize,
            const AlbumFileId& fileId,
            void* workMemory,
            size_t workSize
            ) NN_NOEXCEPT;

        nn::Result VerifyAndLoadMovieMakerNoteInfo(
            uint64_t* pOutFileSize,
            nn::capsrv::server::detail::MakerNoteInfo* pOutMakerNoteInfo,
            const AlbumFileId& fileId,
            void* pWorkBuffer,
            size_t workBufferSize
            ) NN_NOEXCEPT;

    public:
        // ReadStream
        nn::Result OpenMovieReadStream(MovieStreamId* pOutStreamId, const AlbumFileId& fileId, SessionId sessionId) NN_NOEXCEPT;
        nn::Result CloseMovieReadStream(MovieStreamId streamId) NN_NOEXCEPT;
        nn::Result ReadMovieDataFromMovieReadStream(size_t* pOutReadSize, void* buffer, size_t size, MovieStreamId streamId, int64_t offset) NN_NOEXCEPT;
        nn::Result GetMovieReadStreamMovieDataSize(int64_t* pOutValue, MovieStreamId streamId) NN_NOEXCEPT;
        nn::Result ReadImageDataFromMovieReadStream(size_t* pOutReadSize, void* buffer, size_t size, MovieStreamId streamId, int64_t offset) NN_NOEXCEPT;
        nn::Result GetMovieReadStreamImageDataSize(int64_t* pOutValue, MovieStreamId streamId) NN_NOEXCEPT;
        nn::Result ReadFileAttributeFromMovieReadStream(AlbumFileAttribute* pOutAttribute, AppletData* pOutAppletData, ApplicationData* pOutApplicationData, MovieStreamId streamId) NN_NOEXCEPT;
        nn::Result GetMovieReadStreamBrokenReason(nn::Result* pOutValue, MovieStreamId streamId) NN_NOEXCEPT;

        // WriteStream
        nn::Result OpenMovieWriteStream(MovieStreamId* pOutStreamId, const AlbumFileId& fileId, SessionId sessionId) NN_NOEXCEPT;
        nn::Result FinishMovieWriteStream(MovieStreamId streamId) NN_NOEXCEPT;
        nn::Result CommitMovieWriteStream(MovieStreamId streamId) NN_NOEXCEPT;
        nn::Result DiscardMovieWriteStream(MovieStreamId streamId) NN_NOEXCEPT;
        nn::Result DiscardMovieWriteStreamNoDelete(MovieStreamId streamId) NN_NOEXCEPT;
        nn::Result StartMovieWriteStreamDataSection(MovieStreamId streamId) NN_NOEXCEPT;
        nn::Result EndMovieWriteStreamDataSection(MovieStreamId streamId) NN_NOEXCEPT;
        nn::Result StartMovieWriteStreamMetaSection(MovieStreamId streamId) NN_NOEXCEPT;
        nn::Result EndMovieWriteStreamMetaSection(MovieStreamId streamId) NN_NOEXCEPT;
        nn::Result ReadDataFromMovieWriteStream(size_t* pOutReadSize, void* buffer, size_t size, MovieStreamId streamId, int64_t offset) NN_NOEXCEPT;
        nn::Result WriteDataToMovieWriteStream(MovieStreamId streamId, int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT;
        nn::Result WriteMetaToMovieWriteStream(MovieStreamId streamId, const void* buffer, size_t size, uint64_t makerNoteVersion, int64_t makerNoteOffset, int64_t makerNoteSize) NN_NOEXCEPT;
        nn::Result GetMovieWriteStreamBrokenReason(nn::Result* pOutValue, MovieStreamId streamId) NN_NOEXCEPT;
        nn::Result GetMovieWriteStreamDataSize(int64_t* pOutValue, MovieStreamId streamId) NN_NOEXCEPT;
        nn::Result SetMovieWriteStreamDataSize(MovieStreamId streamId, int64_t size) NN_NOEXCEPT;

    //--------------
    // デバッグ機能
    //--------------
    public:
        nn::Result LoadMakerNoteInfo(uint64_t* pOutFileSize, nn::capsrv::server::detail::MakerNoteInfo* pOutInfo, const AlbumFileId& fileId, void* pWorkBuffer, size_t workBufferSize) NN_NOEXCEPT;

    private:
        void* m_Memory;
        size_t m_MemorySize;

        StorageStatus m_StorageStatusList[AlbumStorageCount];

        // ストレージのキャッシュ情報
        AlbumStorageCache m_AlbumStorageCache;

        // ファイルのロック状態のテーブル
        AlbumFileLockTable m_ReadLockFileTable;
        AlbumFileLockTable m_WriteLockFileTable;

        const EnvironmentInfo* m_pEnvironmentInfo;
        ResourceIdManager* m_pResourceIdManager;

        MovieStreamManager m_MovieStreamManager;
    };

}}}}
