﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/fs.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/capsrv/capsrv_AlbumFileDateTime.h>
#include "../capsrvServer_ConvertFileSystemResult.h"
#include "../capsrvServer_EnvironmentInfo.h"
#include "capsrvServer_AlbumStorageCache.h"
#include "capsrvServer_AlbumPathUtility.h"

namespace nn{ namespace capsrv{ namespace server{ namespace album{

    class AlbumFileManipulator;

    struct AlbumVisitorType
    {
        // 管理下のディレクトリが見つかった際に呼ばれるコールバック関数
        nn::Result (*pOnManagedDirectoryFoundCallbackFunction)(const AlbumFileId& fileId, int depth, void* pUserParam);
        // 管理外のディレクトリが見つかった際に呼ばれるコールバック関数
        nn::Result (*pOnUnmanagedDirectoryFoundCallbackFunction)(void* pUserParam);
        // 管理下のファイルが見つかった際に呼ばれるコールバック関数
        nn::Result (*pOnManagedFileFoundCallbackFunction)(int64_t fileSize, const AlbumFileId& fileId, void* pUserParam);
        // 管理外のファイルが見つかった際に呼ばれるコールバック関数
        nn::Result (*pOnUnmanagedFileFoundCallbackFuncton)(int64_t fileSize, void* pUserParam);

        // 各コールバックの pUserParam に渡されるパラメータ
        void* pUserParam;
    };

    namespace detail
    {
        template<typename F0, typename F1, typename F2, typename F3>
        class AlbumVisitor
            : public AlbumVisitorType
        {
        public:
            AlbumVisitor(F0 onManagedDirectory, F1 onUnmanagedDirectory, F2 onManagedFile, F3 onUnmanagedFile) NN_NOEXCEPT
                : m_OnManagedDirectoryFunction  (onManagedDirectory)
                , m_OnUnmanagedDirectoryFunction(onUnmanagedDirectory)
                , m_OnManagedFileFunction       (onManagedFile)
                , m_OnUnmanagedFileFunction     (onUnmanagedFile)
            {
                pOnManagedDirectoryFoundCallbackFunction   = InvokeOnManagedDirectoryFunction;
                pOnUnmanagedDirectoryFoundCallbackFunction = InvokeOnUnmanagedDirectoryFunction;
                pOnManagedFileFoundCallbackFunction        = InvokeOnManagedFileFunction;
                pOnUnmanagedFileFoundCallbackFuncton       = InvokeOnUnmanagedFileFunction;
                pUserParam = this;
            }

            AlbumVisitor(const AlbumVisitor& v) NN_NOEXCEPT
                : AlbumVisitorType(v)
                , m_OnManagedDirectoryFunction   (v.m_OnManagedDirectoryFunction)
                , m_OnUnmanagedDirectoryFunction (v.m_OnUnmanagedDirectoryFunction)
                , m_OnManagedFileFunction        (v.m_OnManagedFileFunction)
                , m_OnUnmanagedFileFunction      (v.m_OnUnmanagedFileFunction)
            {
                pUserParam = this;
            }

            AlbumVisitor(AlbumVisitor&& v) NN_NOEXCEPT
                : AlbumVisitorType(std::move(v))
                , m_OnManagedDirectoryFunction  (std::move(v.m_OnManagedDirectoryFunction))
                , m_OnUnmanagedDirectoryFunction(std::move(v.m_OnUnmanagedDirectoryFunction))
                , m_OnManagedFileFunction       (std::move(v.m_OnManagedFileFunction))
                , m_OnUnmanagedFileFunction     (std::move(v.m_OnUnmanagedFileFunction))
            {
                pUserParam = this;
            }

        private:
            static nn::Result InvokeOnManagedDirectoryFunction(const AlbumFileId& fileId, int depth, void* pUserParam) NN_NOEXCEPT
            {
                auto pSelf = reinterpret_cast<AlbumVisitor*>(pUserParam);
                return pSelf->m_OnManagedDirectoryFunction(fileId, depth);
            }

            static nn::Result InvokeOnUnmanagedDirectoryFunction(void* pUserParam) NN_NOEXCEPT
            {
                auto pSelf = reinterpret_cast<AlbumVisitor*>(pUserParam);
                return pSelf->m_OnUnmanagedDirectoryFunction();
            }

            static nn::Result InvokeOnManagedFileFunction(int64_t fileSize, const AlbumFileId& fileId, void* pUserParam) NN_NOEXCEPT
            {
                auto pSelf = reinterpret_cast<AlbumVisitor*>(pUserParam);
                return pSelf->m_OnManagedFileFunction(fileSize, fileId);
            }

            static nn::Result InvokeOnUnmanagedFileFunction(int64_t fileSize, void* pUserParam) NN_NOEXCEPT
            {
                auto pSelf = reinterpret_cast<AlbumVisitor*>(pUserParam);
                return pSelf->m_OnUnmanagedFileFunction(fileSize);
            }

        private:
            F0 m_OnManagedDirectoryFunction;
            F1 m_OnUnmanagedDirectoryFunction;
            F2 m_OnManagedFileFunction;
            F3 m_OnUnmanagedFileFunction;
        };
    }

    template<typename F0, typename F1, typename F2, typename F3>
    static inline detail::AlbumVisitor<F0, F1, F2, F3> MakeAlbumVisitor(F0 onManagedDir, F1 onUnmanagedDir, F2 onManagedFile, F3 onUnmanagedFile) NN_NOEXCEPT
    {
        return detail::AlbumVisitor<F0, F1, F2, F3>(onManagedDir, onUnmanagedDir, onManagedFile, onUnmanagedFile);
    }



    class AlbumFilePath
    {
        friend AlbumFileManipulator;

    public:
        static const int Size = AlbumPathUtility::ExtraPathSize;

    public:
        AlbumFilePath() NN_NOEXCEPT;

        const char* Get() const NN_NOEXCEPT;
        AlbumStorageType GetStorage() const NN_NOEXCEPT;
        AlbumFileContentsType GetContents() const NN_NOEXCEPT;
        AlbumStorageDirectionType GetDirection() const NN_NOEXCEPT;

    private:
        char m_Value[Size];
        AlbumStorageType m_Storage;
        AlbumFileContentsType m_Contents;
        AlbumStorageDirectionType m_Direction;
    };

    struct AlbumFileHandle
    {
        friend AlbumFileManipulator;

    public:
        AlbumFileHandle() NN_NOEXCEPT;

        nn::fs::FileHandle GetHandle() const NN_NOEXCEPT;
        AlbumStorageType GetStorage() const NN_NOEXCEPT;
        AlbumFileContentsType GetContents() const NN_NOEXCEPT;
        AlbumStorageDirectionType GetDirection() const NN_NOEXCEPT;
        bool IsFlushOnCloseRequired() const NN_NOEXCEPT;

    private:
        nn::fs::FileHandle        m_Handle;
        AlbumStorageType          m_Storage;
        AlbumFileContentsType     m_Contents;
        AlbumStorageDirectionType m_Direction;
        bool                      m_IsFlushOnCloseRequired;
    };

    class AlbumFileManipulator
    {
    public:
        static nn::Result GetFilePath(
            AlbumFilePath* pOutValue,
            const AlbumFileId& fileId,
            AlbumStorageDirectionType direction,
            const EnvironmentInfo& env
        ) NN_NOEXCEPT;

        static nn::Result GetSubDirectoryPath(
            AlbumFilePath* pOutValue,
            const AlbumFileId& directoryId,
            int depth,
            bool isExtra,
            AlbumStorageDirectionType direction
        ) NN_NOEXCEPT;

        static nn::Result GetStorageRootPath(
            AlbumFilePath* pOutValue,
            AlbumStorageType storage,
            AlbumStorageDirectionType direction
        ) NN_NOEXCEPT;

        static nn::Result OpenAlbumFileForSave(
            AlbumFileHandle* pOutHandle,
            AlbumCacheDelta* pOutCacheDelta,
            AlbumFilePath& filepath,
            int64_t fileSize
        ) NN_NOEXCEPT;

        static nn::Result OpenAlbumFileForWriteStream(
            AlbumFileHandle* pOutHandle,
            AlbumCacheDelta* pOutCacheDelta,
            AlbumFilePath& filepath,
            int64_t fileSize
        ) NN_NOEXCEPT;

        static nn::Result OpenAlbumFileForLoad(
            AlbumFileHandle* pOutHandle,
            const AlbumFilePath& filepath
        ) NN_NOEXCEPT;

        static void CloseAlbumFile(const AlbumFileHandle& handle) NN_NOEXCEPT;

        static nn::Result DeleteAlbumFile(AlbumCacheDelta* pOutCacheDelta, const AlbumFilePath& filepath) NN_NOEXCEPT;

        static nn::Result GetFileSize(int64_t* pOutValue, const AlbumFileHandle& handle) NN_NOEXCEPT;
        static nn::Result ReadFile(size_t* pOutReadSize, const AlbumFileHandle& handle, int64_t offset, void* buffer, size_t size) NN_NOEXCEPT;
        static nn::Result WriteFile(const AlbumFileHandle& handle, int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT;
        static nn::Result ResizeFile(const AlbumFileHandle& handle, int64_t size) NN_NOEXCEPT;

        static nn::Result CheckFileExist(bool* pOutValue, const AlbumFileId& fileId, AlbumStorageDirectionType direction, const EnvironmentInfo& env) NN_NOEXCEPT;
        static nn::Result CheckSubDirectoryExist(bool* pOutValue, const AlbumFileId& directoryId, int depth, bool isExtra, AlbumStorageDirectionType direction) NN_NOEXCEPT;

        static nn::Result VisitAlbumStorage(
            AlbumStorageType storage,
            const AlbumVisitorType& callback,
            nn::fs::DirectoryEntry* pWorkspace,
            int workSize,
            AlbumStorageDirectionType direction,
            const EnvironmentInfo& env
        ) NN_NOEXCEPT;

    private:
        static nn::Result OpenAlbumFileForSaveImpl(
            AlbumFileHandle* pOutHandle,
            AlbumCacheDelta* pOutCacheDelta,
            AlbumFilePath& filepath,
            int64_t fileSize,
            int openMode
        ) NN_NOEXCEPT;
    };

}}}}
