﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "capsrv_VicOperation-module.nvnflinger.h"

#include <nvgr.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_StringUtil.h>
#include "../capsrv_CaptureConfig.h"
#include "../capsrv_ResultCapture.h"
#include "../capsrv_CaptureModule-module.nvnflinger.h"
#include "../capsrv_ImageBuffer-module.nvnflinger.h"
#if defined(NN_CAPSRV_CAPTURE_ENABLE_DUMP_VIC_PARAMETER)
#include "capsrv_DumpNvRmSurface-module.nvnflinger.h"
#endif

namespace nn{ namespace capsrv{ namespace capture{ namespace detail{

    static void DumpVicConfig(const NvDdkVicConfigParameters& v) NN_NOEXCEPT;
    static void DumpVicExec(const NvDdkVicExecParameters& v, int dstSurfaceCount) NN_NOEXCEPT;

    nn::Result VicOperation::CopyToSurface(
        ImageBuffer* pDstBuffer,
        CaptureModule* pModule,
        const VicCopyDestinationRectangle* pDstRect,
        const VicCopySourceInfo& srcInfo,
        const VicCopyOption& option
    ) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(
            pDstBuffer->IsInitialized(),
            ResultCaptureOperationNotReady()
        );

        VicCopyDestinationInfo dstInfo = {};
        if(pDstRect)
        {
            dstInfo.rectangle = *pDstRect;
        }
        else
        {
            dstInfo.rectangle.left   = 0;
            dstInfo.rectangle.top    = 0;
            dstInfo.rectangle.right  = static_cast<NvS32>(pDstBuffer->GetData()->GetWidth());
            dstInfo.rectangle.bottom = static_cast<NvS32>(pDstBuffer->GetData()->GetHeight());
        }

        dstInfo.pSurfaceList = pDstBuffer->GetData()->GetSurface();
        dstInfo.surfaceCount = 1;

        return CopyToSurfaceList(pModule, dstInfo, srcInfo, option);
    }

    nn::Result VicOperation::CopyToSurfaceList(
        CaptureModule* pModule,
        const VicCopyDestinationInfo& dstInfo,
        const VicCopySourceInfo& srcInfo,
        const VicCopyOption& option
    ) NN_NOEXCEPT
    {
        const NvU32 InputSlotIndex = 0;

        NvError err = {};

        NvDdkVicConfigParameters config;
        std::memset(&config, 0, sizeof(config));
        config.BackgroundColor = NvDdkVicFloatColorRec{0, 0, 0, 1};
        config.BackgroundColorSpace = NvDdkVicColorSpace_LegacyRGB;

        NvRectF32 sRect = {};
        sRect.left   = 0;
        sRect.top    = 0;
        sRect.right  = static_cast<float>(srcInfo.width);
        sRect.bottom = static_cast<float>(srcInfo.height);

        NvRect dRect = {};
        dRect.left   = static_cast<NvS32>(dstInfo.rectangle.left);
        dRect.top    = static_cast<NvS32>(dstInfo.rectangle.top);
        dRect.right  = static_cast<NvS32>(dstInfo.rectangle.right);
        dRect.bottom = static_cast<NvS32>(dstInfo.rectangle.bottom);

        //NN_SDK_LOG("[capsrv][vic]NvDdkVicConfigureTargetSurface\n");
        err = NvDdkVicConfigureTargetSurface(
            pModule->GetVicSession(),
            &config,
            dstInfo.pSurfaceList,
            dstInfo.surfaceCount,
            &dRect
        );
        NN_RESULT_THROW_UNLESS(
            err == NvSuccess,
            ResultCaptureOperationFailed()
        );

        //NN_SDK_LOG("[capsrv][vic]NvDdkVicConfigureSourceSurface\n");
        err = NvDdkVicConfigureSourceSurface(
            pModule->GetVicSession(),
            &config,
            InputSlotIndex,
            const_cast<NvRmSurface*>(srcInfo.pSurfaceList),
            srcInfo.surfaceCount,
            &sRect,
            &dRect
        );
        NN_RESULT_THROW_UNLESS(
            err == NvSuccess,
            ResultCaptureOperationFailed()
        );

        // setup downsample method
        switch(option.filter)
        {
        case VicCopyFilter_Nearest:
            {
                config.Slot[InputSlotIndex].FilterX = NvDdkVicFilter_Nearest;
                config.Slot[InputSlotIndex].FilterY = NvDdkVicFilter_Nearest;
                break;
            }
        case VicCopyFilter_Nicest:
            {
                config.Slot[InputSlotIndex].FilterX = NvDdkVicFilter_Nicest;
                config.Slot[InputSlotIndex].FilterY = NvDdkVicFilter_Nicest;
                break;
            }
        default:
            ;
        }

        // setup blend parameter
        {
            auto& blend = config.Slot[InputSlotIndex].Blend;
            blend.K1 = 1;
            blend.SrcFactC = NvDdkVicBlendSrcFactC_K1;
            blend.DstFactC = NvDdkVicBlendDstFactC_Zero;
            blend.SrcFactA = NvDdkVicBlendSrcFactA_K1;
            blend.DstFactA = NvDdkVicBlendDstFactA_Zero;
            blend.UseOverrideA = NV_TRUE;
        }

        //config.OutputInfo.PixelFormat = NvDdkVicPixelFormat_R8G8B8A8;
        DumpVicConfig(config);


        //NN_SDK_LOG("[capsrv][vic]NvDdkVicConfigure\n");
        NvDdkVicConfigure(pModule->GetVicSession(), &config);

        NvDdkVicExecParameters params;
        std::memset(&params, 0, sizeof(params));
        params.OutputSurface = dstInfo.pSurfaceList;
        for(size_t p = 0; p < srcInfo.surfaceCount; p++)
        {
            params.InputSurfaces[InputSlotIndex][p] = const_cast<NvRmSurface*>(&srcInfo.pSurfaceList[p]);
        }

        DumpVicExec(params, dstInfo.surfaceCount);

        NvRmFence fence;
        //NN_SDK_LOG("[capsrv][vic]NvDdkVicExecute\n");
        err = NvDdkVicExecute(pModule->GetVicSession(), &params, nullptr, 0, &fence);

        NN_RESULT_THROW_UNLESS(
            err == NvSuccess,
            ResultCaptureOperationFailed()
        );

        NvRmFenceWait(pModule->GetVicDevice(), &fence, NV_WAIT_INFINITE);
        NN_RESULT_SUCCESS;

    }

    //------------------------------------------------------------------------------------

#if defined(NN_CAPSRV_CAPTURE_ENABLE_DUMP_VIC_PARAMETER)

#define NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT(...)  \
    {   \
        int advance = nn::util::SNPrintf(p, static_cast<size_t>(pEnd - pBeg), __VA_ARGS__); \
        p += advance;   \
        if(p >= pEnd)   \
        {   \
            *pOutNext = pEnd;   \
            return; \
        }   \
    }

#define NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_BOOLEAN(name) \
    if(v.name){ NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT( #name ","); }
#define NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_INTEGER(name) \
    NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT( #name "=%d,", static_cast<int>(v.name));
#define NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_FLOAT(name) \
    NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT( #name "=%f,", static_cast<float>(v.name));

#define NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_STRUCT(name, stringizer)  \
    NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT( #name "={"); \
    stringizer(&p, p, static_cast<size_t>(pEnd - p), v.name);   \
    NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT("},");

    namespace {
        void StringizeNvDdkVicSurfaceInfo(char** pOutNext, char* pBeg, size_t size, const NvDdkVicSurfaceInfo& v) NN_NOEXCEPT
        {
            char* p = pBeg;
            char* pEnd = pBeg + size;
            NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_INTEGER(PixelFormat);
            NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_INTEGER(SurfaceLayout);
            NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_INTEGER(FrameFormat);
            NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_INTEGER(SubSamplingType);
            NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_INTEGER(ColorSpace);
            NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_INTEGER(LumaSurfaceWidth);
            NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_INTEGER(LumaSurfaceHeight);
            NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_INTEGER(ChromaSurfaceWidth);
            NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_INTEGER(ChromaSurfaceHeight);
            NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_INTEGER(CompBitsOffset);
            NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_INTEGER(ZBCColor);
            NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_INTEGER(ChromaPlanes);
            NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_INTEGER(ChromaLocHoriz);
            NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_INTEGER(ChromaLocVert);
            NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_INTEGER(CacheWidth);
            NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_INTEGER(BlkHeight);
            NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_BOOLEAN(ForOutput);
            NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_BOOLEAN(HasAlpha);
            *pOutNext = p;
        }

        void StringizeNvDdkVicSurfaceVideoInfo(char** pOutNext, char* pBeg, size_t size, const NvDdkVicSurfaceVideoInfo& v) NN_NOEXCEPT
        {
            char* p = pBeg;
            char* pEnd = pBeg + size;
            NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_INTEGER(Format);
            NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_BOOLEAN(EvenField);
            NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_BOOLEAN(SubStream);
            *pOutNext = p;
        }

        void StringizeNvDdkVicBlend(char** pOutNext, char* pBeg, size_t size, const NvDdkVicBlend& v) NN_NOEXCEPT
        {
            char* p = pBeg;
            char* pEnd = pBeg + size;
            NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_FLOAT(K1);
            NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_FLOAT(K2);
            NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_INTEGER(SrcFactC);
            NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_INTEGER(DstFactC);
            NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_INTEGER(SrcFactA);
            NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_INTEGER(DstFactA);
            NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_FLOAT(OverrideR);
            NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_FLOAT(OverrideG);
            NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_FLOAT(OverrideB);
            NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_FLOAT(OverrideA);
            NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_BOOLEAN(UseOverrideR);
            NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_BOOLEAN(UseOverrideG);
            NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_BOOLEAN(UseOverrideB);
            NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_BOOLEAN(UseOverrideA);
            NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_BOOLEAN(MaskR);
            NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_BOOLEAN(MaskG);
            NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_BOOLEAN(MaskB);
            NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_BOOLEAN(MaskA);
            *pOutNext = p;
        }

        void StringizeNvRectF32(char** pOutNext, char* pBeg, size_t size, const NvRectF32& v) NN_NOEXCEPT
        {
            char* p = pBeg;
            char* pEnd = pBeg + size;
            NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_FLOAT(left);
            NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_FLOAT(top);
            NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_FLOAT(right);
            NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_FLOAT(bottom);
            *pOutNext = p;
        }

        void StringizeNvRect(char** pOutNext, char* pBeg, size_t size, const NvRect& v) NN_NOEXCEPT
        {
            char* p = pBeg;
            char* pEnd = pBeg + size;
            NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_INTEGER(left);
            NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_INTEGER(top);
            NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_INTEGER(right);
            NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_INTEGER(bottom);
            *pOutNext = p;
        }

        void StringizeNvDdkVicSlot(char** pOutNext, char* pBeg, size_t size, const NvDdkVicSlot& v) NN_NOEXCEPT
        {
            char* p = pBeg;
            char* pEnd = pBeg + size;
            NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_BOOLEAN(Enable);
            NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_BOOLEAN(DeNoise);
            NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_BOOLEAN(CadenceDetect);
            NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_BOOLEAN(MotionMap);
            NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_BOOLEAN(IsEven);
            NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_BOOLEAN(MMapCombine);
            NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_BOOLEAN(DoInitialMotionCalc);
            NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_STRUCT(Info, StringizeNvDdkVicSurfaceInfo);
            NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_STRUCT(VideoInfo, StringizeNvDdkVicSurfaceVideoInfo);
            NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_STRUCT(Blend, StringizeNvDdkVicBlend);
            NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_STRUCT(SourceRect, StringizeNvRectF32);
            NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_STRUCT(DestRect, StringizeNvRect);
            NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_INTEGER(ClearRectMask);
            NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_FLOAT(PanoramicScaling);
            NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_INTEGER(FilterX);
            NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_INTEGER(FilterY);
            NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_FLOAT(FilterNoiseWeight);
            NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_FLOAT(FilterDetailWeight);
            NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_FLOAT(ChromaNoiseWeight);
            NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_FLOAT(ChromaDetailWeight);
            *pOutNext = p;
        }

        void StringizeNvDdkVicFloatColor(char** pOutNext, char* pBeg, size_t size, const NvDdkVicFloatColor& v) NN_NOEXCEPT
        {
            char* p = pBeg;
            char* pEnd = pBeg + size;
            NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_INTEGER(R);
            NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_INTEGER(G);
            NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_INTEGER(B);
            NN_DETAIL_CAPSRV_VIC_STRINGIZE_CONCAT_INTEGER(A);
            *pOutNext = p;
        }

    }

    static void DumpVicConfig(const NvDdkVicConfigParameters& v) NN_NOEXCEPT
    {
        static char buf[4096] = {};
        char* p = nullptr;

        NN_CAPSRV_CAPTURE_LOG_VIC_DUMP("ConfigParameters:\n");
        for(int i = 0; i < NVDDK_VIC_MAX_SLOTS_RESERVED; i++)
        {
            if(!v.Slot[i].Enable)
            {
                continue;
            }
            StringizeNvDdkVicSlot(&p, buf, sizeof(buf), v.Slot[i]);
            NN_CAPSRV_CAPTURE_LOG_VIC_DUMP("Slot[%d]={%s}\n", i, buf);
        }
        {
            StringizeNvDdkVicSurfaceInfo(&p, buf, sizeof(buf), v.OutputInfo);
            NN_CAPSRV_CAPTURE_LOG_VIC_DUMP("OutputInfo={%s}\n", buf);
        }
        {
            StringizeNvRect(&p, buf, sizeof(buf), v.TargetRect);
            NN_CAPSRV_CAPTURE_LOG_VIC_DUMP("TargetRect={%s}\n", buf);
        }
        {
            StringizeNvDdkVicFloatColor(&p, buf, sizeof(buf), v.BackgroundColor);
            NN_CAPSRV_CAPTURE_LOG_VIC_DUMP("BackgroundColor={%s}\n", buf);
        }
        NN_CAPSRV_CAPTURE_LOG_VIC_DUMP("BackgroundColorSpace=%d\n", static_cast<int>(v.BackgroundColorSpace));
        NN_CAPSRV_CAPTURE_LOG_VIC_DUMP("AlphaFillMode=%d\n", static_cast<int>(v.AlphaFillMode));
        for(int i = 0; i < NVDDK_VIC_MAX_CLEAR_RECTS_RESERVED; i++)
        {
            StringizeNvRect(&p, buf, sizeof(buf), v.ClearRects[i]);
            NN_CAPSRV_CAPTURE_LOG_VIC_DUMP("ClearRects[%d]={%s}\n", i, buf);
        }
        if(v.OutputFlipX){ NN_CAPSRV_CAPTURE_LOG_VIC_DUMP("OutputFlipX\n"); }
        if(v.OutputFlipY){ NN_CAPSRV_CAPTURE_LOG_VIC_DUMP("OutputFlipY\n"); }
        if(v.OutputTranspose){ NN_CAPSRV_CAPTURE_LOG_VIC_DUMP("OutputTranspose\n"); }
        if(v.EnableCrcs){ NN_CAPSRV_CAPTURE_LOG_VIC_DUMP("EnableCrcs\n"); }
        if(v.EnableGammaCorrection){ NN_CAPSRV_CAPTURE_LOG_VIC_DUMP("EnableGammaCorrection\n"); }
        if(v.DisableChromaUpsampling){ NN_CAPSRV_CAPTURE_LOG_VIC_DUMP("DisableChromaUpsampling\n"); }
        if(v.LegacyIsLinearRGB){ NN_CAPSRV_CAPTURE_LOG_VIC_DUMP("LegacyIsLinearRGB\n"); }
        if(v.ForceSRGBGammaForNonLinear){ NN_CAPSRV_CAPTURE_LOG_VIC_DUMP("ForceSRGBGammaForNonLinear\n"); }
        if(v.EnableFilterOverride){ NN_CAPSRV_CAPTURE_LOG_VIC_DUMP("EnableFilterOverride\n"); }
    }

    static void DumpVicExec(const NvDdkVicExecParameters& v, int dstSurfaceCount) NN_NOEXCEPT
    {
        NN_CAPSRV_CAPTURE_LOG_VIC_DUMP("ExecParameters(dst=%d):\n", dstSurfaceCount);
        for(int i = 0; i < dstSurfaceCount; i++)
        {
            NN_CAPSRV_CAPTURE_LOG_VIC_DUMP("  OutputSurface[%d]:\n", i);
            DumpNvRmSurface(v.OutputSurface[i]);
        }

        for(int i = 0; i < NVDDK_VIC_MAX_SLOTS_RESERVED; i++)
        {
            for(int j = 0; j < NVDDK_VIC_MAX_SURFACES_PER_SLOT; j++)
            {
                if(v.InputSurfaces[i][j] == nullptr)
                {
                    continue;
                }
                NN_CAPSRV_CAPTURE_LOG_VIC_DUMP("  InputSurfaces[%d:%d]:\n", i, j);
                DumpNvRmSurface(*v.InputSurfaces[i][j]);
            }
        }
    }
#else
    static void DumpVicConfig(const NvDdkVicConfigParameters&) NN_NOEXCEPT
    {
    }
    static void DumpVicExec(const NvDdkVicExecParameters&, int) NN_NOEXCEPT
    {
    }
#endif

}}}}
