﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/capsrv/capsrv_AlbumControl.h>

#include <nn/capsrv/capsrv_ThumbnailFormat.h>
#include <nn/capsrv/capsrv_AlbumFileSizeLimit.h>
#include <nn/capsrv/capsrv_Result.h>
#include "capsrv_AlbumControlServiceHolder.h"
#include "capsrv_Macro.h"
#include "capsrv_LibraryState.h"

#if defined(NN_CAPSRV_USE_DIRECT_FUNCTION_CALL)
#include "capsrv_InitializeForLibrary.h"
#endif

namespace nn{ namespace capsrv{

    nn::Result InitializeAlbumControl() NN_NOEXCEPT
    {
        NN_RESULT_DO(g_AlbumControlServiceHolder.Initialize(nullptr));
        NN_RESULT_SUCCESS;
    }

    nn::Result InitializeAlbumControlExtension() NN_NOEXCEPT
    {
        NN_CAPSRV_GET_SERVICE_POINTER(pService, g_AlbumControlServiceHolder);

        NN_RESULT_THROW_UNLESS(g_AlbumControlServiceHolder.GetSession() == nullptr, ResultAlbumAlreadyOpened());

        AlbumControlServiceHolder::SessionPointerType pControlSession;
        NN_RESULT_DO(pService->OpenControlSession(&pControlSession, LibraryState::GetMyAruid()));
        g_AlbumControlServiceHolder.StoreSession(pControlSession);
        g_LibraryState.SetMovieReadStreamSession(pControlSession, LibraryState::ServiceLevel_AlbumControl);
        g_LibraryState.SetMovieWriteStreamSession(pControlSession, LibraryState::ServiceLevel_AlbumControl);
        NN_RESULT_SUCCESS;
    }

    void FinalizeAlbumControl() NN_NOEXCEPT
    {
        // NOTE:
        //   Holder の Finalize で ServiceObject 用のアロケータが破棄されるので
        //   先にすべてのオブジェクトを始末しておく。
        if(g_AlbumControlServiceHolder.GetInitializeCount() == 1)
        {
            g_LibraryState.SetMovieReadStreamSession(nullptr, LibraryState::ServiceLevel_AlbumControl);
            g_LibraryState.SetMovieWriteStreamSession(nullptr, LibraryState::ServiceLevel_AlbumControl);
        }

        g_AlbumControlServiceHolder.Finalize();
    }

    nn::Result RegisterAppletResourceUserId(
        nn::applet::AppletResourceUserId aruid,
        nn::ncm::ApplicationId applicationId
        ) NN_NOEXCEPT
    {
        NN_CAPSRV_GET_SERVICE_POINTER(pService, g_AlbumControlServiceHolder);
        NN_RESULT_DO(pService->RegisterAppletResourceUserId(aruid, applicationId));
        NN_RESULT_SUCCESS;
    }

    nn::Result UnregisterAppletResourceUserId(
        nn::applet::AppletResourceUserId aruid,
        nn::ncm::ApplicationId applicationId
        ) NN_NOEXCEPT
    {
        NN_CAPSRV_GET_SERVICE_POINTER(pService, g_AlbumControlServiceHolder);
        NN_RESULT_DO(pService->UnregisterAppletResourceUserId(aruid, applicationId));
        NN_RESULT_SUCCESS;
    }

    nn::Result NotifyAlbumStorageIsAvailable(AlbumStorageType storage) NN_NOEXCEPT
    {
        NN_CAPSRV_REQUIRES_VALID_STORAGE(storage);

        NN_CAPSRV_GET_SERVICE_POINTER(pService, g_AlbumControlServiceHolder);
        return pService->NotifyAlbumStorageIsAvailable(storage);
    }

    nn::Result NotifyAlbumStorageIsUnavailable(AlbumStorageType storage) NN_NOEXCEPT
    {
        NN_CAPSRV_REQUIRES_VALID_STORAGE(storage);

        NN_CAPSRV_GET_SERVICE_POINTER(pService, g_AlbumControlServiceHolder);
        return pService->NotifyAlbumStorageIsUnavailable(storage);
    }

    nn::Result GetApplicationIdFromAruid(nn::ncm::ApplicationId* pOutValue, nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
    {
        NN_CAPSRV_GET_SERVICE_POINTER(pService, g_AlbumControlServiceHolder);
        nn::ncm::ApplicationId applicationId = {};
        NN_RESULT_DO(pService->GetApplicationIdFromAruid(&applicationId, aruid));
        *pOutValue = applicationId;
        NN_RESULT_SUCCESS;
    }

    nn::Result CheckApplicationIdRegistered(nn::ncm::ApplicationId applicationId) NN_NOEXCEPT
    {
        NN_CAPSRV_GET_SERVICE_POINTER(pService, g_AlbumControlServiceHolder);
        return pService->CheckApplicationIdRegistered(applicationId);
    }

    nn::Result GenerateCurrentAlbumFileId(AlbumFileId* pOutValue, nn::ncm::ApplicationId applicationId, AlbumFileContentsType contents) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutValue);
        NN_CAPSRV_REQUIRES_VALID_CONTENTS(contents);

        NN_CAPSRV_GET_SERVICE_POINTER(pService, g_AlbumControlServiceHolder);
        return pService->GenerateCurrentAlbumFileId(pOutValue, {applicationId.value}, contents);
    }

    nn::Result GenerateApplicationAlbumEntry(ApplicationAlbumEntry* pOutValue, const AlbumEntry& srcEntry, nn::ncm::ApplicationId applicationId) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutValue);

        NN_CAPSRV_GET_SERVICE_POINTER(pService, g_AlbumControlServiceHolder);
        return pService->GenerateApplicationAlbumEnrty(pOutValue, srcEntry, applicationId);
    }

    nn::Result SetOverlayScreenShotThumbnailData(const AlbumFileId& fileId, const void* pData, size_t dataSize) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pData);
        NN_SDK_REQUIRES_GREATER_EQUAL(dataSize, static_cast<size_t>(ThumbnailImageDataSize_Default));

        NN_CAPSRV_GET_SERVICE_POINTER(pService, g_AlbumControlServiceHolder);
        return pService->SetOverlayScreenShotThumbnailData(fileId, nn::sf::InBuffer(reinterpret_cast<const char*>(pData), dataSize));
    }

    nn::Result SetOverlayMovieThumbnailData(const AlbumFileId& fileId, const void* pData, size_t dataSize) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pData);
        NN_SDK_REQUIRES_GREATER_EQUAL(dataSize, static_cast<size_t>(ThumbnailImageDataSize_Default));

        NN_CAPSRV_GET_SERVICE_POINTER(pService, g_AlbumControlServiceHolder);
        return pService->SetOverlayMovieThumbnailData(fileId, nn::sf::InBuffer(reinterpret_cast<const char*>(pData), dataSize));
    }

    nn::Result SaveAlbumScreenShotFile(const AlbumFileId& fileId, const void* pData, size_t dataSize, uint64_t makerNoteVersion, int64_t makerNoteOffset, int64_t makerNoteSize) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pData);
        NN_SDK_REQUIRES_EQUAL(fileId.contents, AlbumFileContents_ScreenShot);
        NN_SDK_REQUIRES_MINMAX(dataSize, 0u, static_cast<size_t>(AlbumFileSizeLimit_ScreenShot));

        NN_CAPSRV_GET_SERVICE_POINTER(pService, g_AlbumControlServiceHolder);
        return pService->SaveAlbumScreenShotFileEx(fileId, nn::sf::InBuffer(reinterpret_cast<const char*>(pData), dataSize), makerNoteVersion, makerNoteOffset, makerNoteSize);
    }

    //-----------------------------
    // 動画 API
    //-----------------------------
#define NN_CAPSRV_GET_MOVIEWRITE_SESSION_POINTER(varName)   \
    auto varName = g_LibraryState.GetMovieWriteStreamSession(); \
    NN_ABORT_UNLESS(varName != nullptr, "Call nn::capsrv::InitializeAlbumControlExtension() to use this function.")

    nn::Result OpenAlbumMovieWriteStream(AlbumMovieWriteStreamHandle* pOutHandle, const AlbumFileId& fileId) NN_NOEXCEPT
    {
        NN_CAPSRV_GET_MOVIEWRITE_SESSION_POINTER(pSession);

        detail::AlbumMovieWriteStreamHandleType handleValue = {};
        NN_RESULT_DO(pSession->OpenAlbumMovieWriteStream(&handleValue, fileId));

        *pOutHandle = AlbumMovieWriteStreamHandle(handleValue);
        NN_RESULT_SUCCESS;
    }

    nn::Result FinishAlbumMovieWriteStream(AlbumMovieWriteStreamHandle handle) NN_NOEXCEPT
    {
        NN_CAPSRV_GET_MOVIEWRITE_SESSION_POINTER(pSession);
        NN_RESULT_DO(pSession->FinishAlbumMovieWriteStream(handle.GetInnerValue()));
        NN_RESULT_SUCCESS;
    }

    nn::Result CommitAlbumMovieWriteStream(AlbumMovieWriteStreamHandle handle) NN_NOEXCEPT
    {
        NN_CAPSRV_GET_MOVIEWRITE_SESSION_POINTER(pSession);
        NN_RESULT_DO(pSession->CommitAlbumMovieWriteStream(handle.GetInnerValue()));
        NN_RESULT_SUCCESS;
    }

    nn::Result DiscardAlbumMovieWriteStreamImpl(AlbumMovieWriteStreamHandle handle) NN_NOEXCEPT
    {
        NN_CAPSRV_GET_MOVIEWRITE_SESSION_POINTER(pSession);
        NN_RESULT_DO(pSession->DiscardAlbumMovieWriteStream(handle.GetInnerValue()));
        NN_RESULT_SUCCESS;
    }

    nn::Result DiscardAlbumMovieWriteStreamNoDeleteImpl(AlbumMovieWriteStreamHandle handle) NN_NOEXCEPT
    {
        NN_CAPSRV_GET_MOVIEWRITE_SESSION_POINTER(pSession);
        NN_RESULT_DO(pSession->DiscardAlbumMovieWriteStreamNoDelete(handle.GetInnerValue()));
        NN_RESULT_SUCCESS;
    }

    void DiscardAlbumMovieWriteStream(AlbumMovieWriteStreamHandle handle) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(DiscardAlbumMovieWriteStreamImpl(handle));
    }

    void DiscardAlbumMovieWriteStreamNoDelete(AlbumMovieWriteStreamHandle handle) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(DiscardAlbumMovieWriteStreamNoDeleteImpl(handle));
    }

    nn::Result StartAlbumMovieWriteStreamDataSection(AlbumMovieWriteStreamHandle handle) NN_NOEXCEPT
    {
        NN_CAPSRV_GET_MOVIEWRITE_SESSION_POINTER(pSession);
        NN_RESULT_DO(pSession->StartAlbumMovieWriteStreamDataSection(handle.GetInnerValue()));
        NN_RESULT_SUCCESS;
    }

    nn::Result EndAlbumMovieWriteStreamDataSection(AlbumMovieWriteStreamHandle handle) NN_NOEXCEPT
    {
        NN_CAPSRV_GET_MOVIEWRITE_SESSION_POINTER(pSession);
        NN_RESULT_DO(pSession->EndAlbumMovieWriteStreamDataSection(handle.GetInnerValue()));
        NN_RESULT_SUCCESS;
    }

    nn::Result StartAlbumMovieWriteStreamMetaSection(AlbumMovieWriteStreamHandle handle) NN_NOEXCEPT
    {
        NN_CAPSRV_GET_MOVIEWRITE_SESSION_POINTER(pSession);
        NN_RESULT_DO(pSession->StartAlbumMovieWriteStreamMetaSection(handle.GetInnerValue()));
        NN_RESULT_SUCCESS;
    }

    nn::Result EndAlbumMovieWriteStreamMetaSection(AlbumMovieWriteStreamHandle handle) NN_NOEXCEPT
    {
        NN_CAPSRV_GET_MOVIEWRITE_SESSION_POINTER(pSession);
        NN_RESULT_DO(pSession->EndAlbumMovieWriteStreamMetaSection(handle.GetInnerValue()));
        NN_RESULT_SUCCESS;
    }

    nn::Result ReadDataFromAlbumMovieWriteStream(size_t* pOutReadSize, void* buffer, size_t size, AlbumMovieWriteStreamHandle handle, int64_t offset) NN_NOEXCEPT
    {
        NN_CAPSRV_GET_MOVIEWRITE_SESSION_POINTER(pSession);

        int64_t readSize = {};
        NN_RESULT_DO(pSession->ReadDataFromAlbumMovieWriteStream(&readSize, nn::sf::OutBuffer(reinterpret_cast<char*>(buffer), size), handle.GetInnerValue(), offset));

        *pOutReadSize = static_cast<size_t>(readSize);
        NN_RESULT_SUCCESS;
    }

    nn::Result WriteDataToAlbumMovieWriteStream(AlbumMovieWriteStreamHandle handle, int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT
    {
        NN_CAPSRV_GET_MOVIEWRITE_SESSION_POINTER(pSession);
        NN_RESULT_DO(pSession->WriteDataToAlbumMovieWriteStream(handle.GetInnerValue(), offset, nn::sf::InBuffer(reinterpret_cast<const char*>(buffer), size)));
        NN_RESULT_SUCCESS;
    }

    nn::Result WriteMetaToAlbumMovieWriteStream(AlbumMovieWriteStreamHandle handle, const void* buffer, size_t size, uint64_t makerNoteVersion, int64_t makerNoteOffset, int64_t makerNoteSize) NN_NOEXCEPT
    {
        NN_CAPSRV_GET_MOVIEWRITE_SESSION_POINTER(pSession);
        NN_RESULT_DO(pSession->WriteMetaToAlbumMovieWriteStream(handle.GetInnerValue(), nn::sf::InBuffer(reinterpret_cast<const char*>(buffer), size), makerNoteVersion, makerNoteOffset, makerNoteSize));
        NN_RESULT_SUCCESS;
    }

    nn::Result GetAlbumMovieWriteStreamBrokenReason(AlbumMovieWriteStreamHandle handle) NN_NOEXCEPT
    {
        NN_CAPSRV_GET_MOVIEWRITE_SESSION_POINTER(pSession);
        return pSession->GetAlbumMovieWriteStreamBrokenReason(handle.GetInnerValue());
    }

    nn::Result GetAlbumMovieWriteStreamDataSize(int64_t* pOutValue, AlbumMovieWriteStreamHandle handle) NN_NOEXCEPT
    {
        NN_CAPSRV_GET_MOVIEWRITE_SESSION_POINTER(pSession);
        int64_t size = {};
        NN_RESULT_DO(pSession->GetAlbumMovieWriteStreamDataSize(&size, handle.GetInnerValue()));
        *pOutValue = size;
        NN_RESULT_SUCCESS;
    }

    nn::Result SetAlbumMovieWriteStreamDataSize(AlbumMovieWriteStreamHandle handle, int64_t value) NN_NOEXCEPT
    {
        NN_CAPSRV_GET_MOVIEWRITE_SESSION_POINTER(pSession);
        NN_RESULT_DO(pSession->SetAlbumMovieWriteStreamDataSize(handle.GetInnerValue(), value));
        NN_RESULT_SUCCESS;
    }

    //-----------------------------
    // テスト用 API
    //-----------------------------
    nn::Result OpenAlbumControlSession() NN_NOEXCEPT
    {
        return InitializeAlbumControlExtension();
    }

    void CloseAlbumControlSession() NN_NOEXCEPT
    {
        g_LibraryState.SetMovieReadStreamSession(nullptr, LibraryState::ServiceLevel_AlbumControl);
        g_LibraryState.SetMovieWriteStreamSession(nullptr, LibraryState::ServiceLevel_AlbumControl);
        g_AlbumControlServiceHolder.ReleaseSession();
    }


}}
