﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/album/album_SaveScreenshot.h>
#include <nn/album/album_SaveAndShareScreenshot.h>
#include <nn/album/album_ScreenshotApi.private.h>
#include <nn/album/album_AlbumFileAccessWithUserId.private.h>

#include <mutex>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_Optional.h>
#include <nn/util/util_StringUtil.h>
#include <nn/svc/svc_Result.h>
#include <nn/album/album_Result.h>

#include <nn/capsrv/capsrv_ScreenShotForApplication.h>

#include <nn/web/share_ApiPrivate.h>

#include <nn/fs/detail/fs_AccessLog.h>

#include "album_Log.h"
#include "album_LibraryState.h"
#include "album_ShareIdInternal.h"
#include "album_ConvertCapsrvResult.h"

#define NN_ALBUM_REQUIRES(cond) \
    do{                         \
        NN_SDK_REQUIRES(cond);  \
        NN_ABORT_UNLESS(cond);  \
    }while(NN_STATIC_CONDITION(false))

#define NN_ALBUM_REQUIRES_NOT_NULL(p)   \
    do{                                 \
        NN_SDK_REQUIRES_NOT_NULL(p);    \
        NN_ABORT_UNLESS_NOT_NULL(p);    \
    }while(NN_STATIC_CONDITION(false))

#define NN_ALBUM_REQUIRES_VALID_IMAGE_DATA(p,dataSize,imageSize)            \
    do{                                                                     \
        size_t requiredImageDataSize = GetRequiredImageDataSize(imageSize); \
        NN_SDK_REQUIRES_NOT_NULL(p);                                        \
        NN_ABORT_UNLESS_NOT_NULL(p);                                        \
        NN_SDK_REQUIRES_GREATER_EQUAL(dataSize, requiredImageDataSize);     \
        NN_ABORT_UNLESS_GREATER_EQUAL(dataSize, requiredImageDataSize);     \
    }while(NN_STATIC_CONDITION(false))

NN_STATIC_ASSERT(nn::album::PresetPositingMessageSizeMax <= nn::web::ShowSharePageArg::AdditionalCommentTextMaxLength);

namespace nn{ namespace album{

#ifndef NN_ALBUM_IS_STUB_IMPLEMENTATION

    namespace {

        struct SharePageWorkspace
        {
            nn::util::optional<nn::web::ShowSharePageArg> argument;
            nn::util::optional<nn::web::SharePageReturnValue> returnValue;
        };

        //------------------------------------

        bool IsValidImageSize(ImageSize size) NN_NOEXCEPT
        {
            return size == ImageSize_1280x720;
        }

        bool IsValidImageOrientation(ImageOrientation ori) NN_NOEXCEPT
        {
            switch(ori)
            {
            case ImageOrientation_None:
            case ImageOrientation_Rotate90:
            case ImageOrientation_Rotate180:
            case ImageOrientation_Rotate270:
                return true;
            default:
                return false;
            };
        }

        bool IsValidPresetPostingMessage(const char* message) NN_NOEXCEPT
        {
            if(message == nullptr)
            {
                return true;
            }
            if(nn::util::Strnlen(message, static_cast<int>(PresetPositingMessageSizeMax)) >= static_cast<int>(PresetPositingMessageSizeMax))
            {
                return false;
            }
            return true;
        }

        //------------------------------------

        size_t GetRequiredImageDataSize(ImageSize size) NN_NOEXCEPT
        {
            return static_cast<size_t>(4 * GetImageWidth(size) * GetImageHeight(size));
        }

        nn::capsrv::ScreenShotSizeType GetCapsrvScreenShotSize(ImageSize size) NN_NOEXCEPT
        {
            switch(size)
            {
            case ImageSize_1280x720:
                return nn::capsrv::ScreenShotSize_1280x720;
            default: NN_UNEXPECTED_DEFAULT;
            }
        }

        nn::capsrv::ScreenShotOrientationType GetCapsrvScreenShotOrientation(ImageOrientation src) NN_NOEXCEPT
        {
            switch(src)
            {
            case ImageOrientation_None:
                return nn::capsrv::ScreenShotOrientation_Default;
            case ImageOrientation_Rotate90:
                return nn::capsrv::ScreenShotOrientation_Rotate90;
            case ImageOrientation_Rotate180:
                return nn::capsrv::ScreenShotOrientation_Rotate180;
            case ImageOrientation_Rotate270:
                return nn::capsrv::ScreenShotOrientation_Rotate270;
            default: NN_UNEXPECTED_DEFAULT;
            }
        }

        nn::capsrv::OverlayNotificationRequestType GetCapsrvOverlayNotificationRequest(AlbumReportOption option) NN_NOEXCEPT
        {
            nn::capsrv::OverlayNotificationRequestType value = 0;
            if(option & AlbumReportOption_ReportOnSuccess)
            {
                value |= nn::capsrv::OverlayNotificationRequest_OnSuccess;
            }
            if(option & AlbumReportOption_ReportOnFailure)
            {
                value |= nn::capsrv::OverlayNotificationRequest_OnFailure;
            }
            return value;
        }

        //---------------------------------------------------------------

        nn::Result ConvertShareResult(nn::Result result) NN_NOEXCEPT
        {
            NN_ALBUM_LOG_DETAIL("[share] Result: %d-%d\n", result.GetModule(), result.GetDescription());
            NN_RESULT_TRY(result)
                NN_RESULT_CATCH_ALL
                {
                    NN_RESULT_THROW(ResultShareCanceled());
                }
            NN_RESULT_END_TRY;
            NN_RESULT_SUCCESS;
        }

        nn::Result ConvertShareExitReason(const nn::web::SharePageReturnValue& result) NN_NOEXCEPT
        {
            NN_ALBUM_LOG_DETAIL("[share] Result:%d, Reason: %d, Service: '%s'\n", result.GetSharePostResult(), result.GetShareExitReason(), result.GetPostServiceName());

            switch(result.GetSharePostResult())
            {
            case nn::web::SharePostResult_Success:
                NN_RESULT_SUCCESS;
            case nn::web::SharePostResult_Cancel:
                NN_RESULT_THROW(ResultShareCanceled());
            default:
                NN_RESULT_THROW(ResultShareFailed());
            }
        }

        void MakeShareId(ShareId* pOutValue, const nn::web::SharePageReturnValue& result) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(pOutValue);
            std::memset(pOutValue, 0, sizeof(ShareId));
            auto pInternal = reinterpret_cast<SharedIdInternal*>(pOutValue);
            nn::util::Strlcpy(pInternal->serviceName, result.GetPostServiceName(), sizeof(pInternal->serviceName));
            nn::util::Strlcpy(pInternal->entryId, result.GetPostId(), sizeof(pInternal->entryId));
            nn::util::Strlcpy(pInternal->lastUrl, result.GetLastUrl(), sizeof(pInternal->lastUrl));
        }


        //---------------------------------------------------------------

        nn::Result SaveScreenshotImpl(
            nn::capsrv::ApplicationAlbumEntry* pOutEntry,
            const void* pImageData,
            size_t imageDataSize,
            ImageSize size,
            ImageOrientation orientation,
            AlbumReportOption reportOption
            ) NN_NOEXCEPT
        {

            // 画面写真の保存(ApplicationAlbumEntry が返る)
            nn::capsrv::ApplicationAlbumEntry entry = {};
            nn::capsrv::ScreenShotAttribute attr = {};
            attr.SetDefault();
            attr.size = GetCapsrvScreenShotSize(size);
            attr.orientation = GetCapsrvScreenShotOrientation(orientation);
            NN_RESULT_DO(ConvertCapsrvResult(
                nn::capsrv::SaveScreenShotForApplication(
                    &entry,
                    pImageData,
                    imageDataSize,
                    attr,
                    GetCapsrvOverlayNotificationRequest(reportOption)
                )
            ));

            *pOutEntry = entry;
            NN_RESULT_SUCCESS;
        }

        nn::Result ShowSharePageImpl(
            ShareId* pOutShareId,
            const nn::capsrv::ApplicationAlbumEntry& entry,
            const nn::account::Uid& uid,
            const char* presetPostingMessage,
            SharePageWorkspace& workspace
            ) NN_NOEXCEPT
        {
            // NOTE:
            //   投稿 ID を返さない版の API 呼び出しの場合、 pOutShareId == nullptr であって合法。

            NN_ALBUM_LOG_DETAIL("[album] showing SharePage\n");
            NN_ALBUM_LOG_DETAIL("[album]   outShareId : %s\n", pOutShareId ? "enabled" : "disabled");
            NN_ALBUM_LOG_DETAIL("[album]   entry      : ...\n");
            NN_ALBUM_LOG_DETAIL("[album]   uid        : %016llX-%016llX\n", uid._data[0], uid._data[1]);
            NN_ALBUM_LOG_DETAIL("[album]   presetMsg  : %s\n", presetPostingMessage);

            if(pOutShareId)
            {
                std::memset(pOutShareId->_data, 0, sizeof(pOutShareId->_data));
            }

            workspace.argument.emplace(nn::web::ShareStartPage_SharePage);
            workspace.returnValue.emplace();
            NN_UTIL_SCOPE_EXIT{
                workspace.argument = nn::util::nullopt;
                workspace.returnValue = nn::util::nullopt;
            };
            workspace.argument->SetApplicationAlbumEntry(entry);
            workspace.argument->SetUid(uid);
            if(presetPostingMessage != nullptr)
            {
                workspace.argument->SetAdditionalCommentText(presetPostingMessage);
            }

            NN_RESULT_DO(ConvertShareResult(
                nn::web::ShowSharePage(&workspace.returnValue.value(), workspace.argument.value())
            ));

            NN_RESULT_DO(ConvertShareExitReason(
                *workspace.returnValue
            ));

            if(pOutShareId)
            {
                NN_ALBUM_LOG_DETAIL("[album] -> ShareId:\n");
                NN_ALBUM_LOG_DETAIL("[album]    ServiceName: '%s'\n", workspace.returnValue->GetPostServiceName());
                NN_ALBUM_LOG_DETAIL("[album]    Id         : '%s'\n", workspace.returnValue->GetPostId());
                NN_ALBUM_LOG_DETAIL("[album]    Url        : '%s'\n", workspace.returnValue->GetLastUrl());
                MakeShareId(pOutShareId, *workspace.returnValue);
            }

            NN_RESULT_SUCCESS;
        }

        //---------------------------------------------------------------

        SharePageWorkspace g_SharePageWorkspace;
    }

    // ImageOrientation 無指定
    nn::Result SaveScreenshot(
        const void* pImageData,
        size_t imageDataSize,
        ImageSize screenshotSize,
        AlbumReportOption reportOption
        ) NN_NOEXCEPT
    {
        NN_ALBUM_LOG_DETAIL("SaveScreenshot(4)\n");
        NN_SDK_REQUIRES(g_LibraryState.IsInitialized());
        NN_ALBUM_REQUIRES(IsValidImageSize(screenshotSize));
        NN_ALBUM_REQUIRES_VALID_IMAGE_DATA(pImageData, imageDataSize, screenshotSize);

        auto& lk = *g_LibraryState.GetLibraryMutex();
        std::lock_guard<decltype(lk)> lock(lk);

        g_LibraryState.EnsureAlbumAvailable();

        nn::capsrv::ApplicationAlbumEntry entry = {};
        ImageOrientation imageOrientation = ImageOrientation_None;
        // 以下のコードを SaveScreenshot() から移動する場合は注意
        // NN_DETAIL_FS_ACCESS_LOG_EXPLICIT_FUNCTION_NAME を使用して、明示的に "SaveScreenshot" を指定してください
        NN_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG(SaveScreenshotImpl(&entry, pImageData, imageDataSize, screenshotSize, imageOrientation, reportOption),
            nullptr, NN_DETAIL_FS_ACCESS_LOG_FORMAT_ALBUM_SCREENSHOT(imageDataSize, screenshotSize, reportOption)));

        NN_RESULT_SUCCESS;
    }

    // ImageOrientation 指定
    nn::Result SaveScreenshot(
        const void* pImageData,
        size_t imageDataSize,
        ImageSize screenshotSize,
        ImageOrientation imageOrientation,
        AlbumReportOption reportOption
        ) NN_NOEXCEPT
    {
        NN_ALBUM_LOG_DETAIL("SaveScreenshot(5)\n");
        NN_SDK_REQUIRES(g_LibraryState.IsInitialized());
        NN_ALBUM_REQUIRES(IsValidImageSize(screenshotSize));
        NN_ALBUM_REQUIRES_VALID_IMAGE_DATA(pImageData, imageDataSize, screenshotSize);

        auto& lk = *g_LibraryState.GetLibraryMutex();
        std::lock_guard<decltype(lk)> lock(lk);

        g_LibraryState.EnsureAlbumAvailable();

        nn::capsrv::ApplicationAlbumEntry entry = {};
        // 以下のコードを SaveScreenshot() から移動する場合は注意
        // NN_DETAIL_FS_ACCESS_LOG_EXPLICIT_FUNCTION_NAME を使用して、明示的に "SaveScreenshot" を指定してください
        NN_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG(SaveScreenshotImpl(&entry, pImageData, imageDataSize, screenshotSize, imageOrientation, reportOption),
            nullptr, NN_DETAIL_FS_ACCESS_LOG_FORMAT_ALBUM_SCREENSHOT_ORIENTATION(imageDataSize, screenshotSize, imageOrientation, reportOption)));

        NN_RESULT_SUCCESS;
    }

    // ImageOrientation および UserIdList 指定
    nn::Result SaveScreenshot(
        const void* pImageData,
        size_t imageDataSize,
        ImageSize screenshotSize,
        ImageOrientation imageOrientation,
        AlbumReportOption reportOption,
        const nn::account::Uid* pUidList,
        int uidListCount
        ) NN_NOEXCEPT
    {
        NN_ALBUM_LOG_DETAIL("SaveScreenshot(7)\n");
        NN_SDK_REQUIRES(g_LibraryState.IsInitialized());
        NN_ALBUM_REQUIRES(IsValidImageSize(screenshotSize));
        NN_ALBUM_REQUIRES_VALID_IMAGE_DATA(pImageData, imageDataSize, screenshotSize);
        NN_SDK_REQUIRES_NOT_NULL(pUidList);
        NN_SDK_REQUIRES(uidListCount >= 0 && uidListCount <= album::UidListCountMax);

        auto& lk = *g_LibraryState.GetLibraryMutex();
        std::lock_guard<decltype(lk)> lock(lk);

        g_LibraryState.EnsureAlbumAvailable();

        // 画面写真の保存(ApplicationAlbumEntry が返る)
        nn::capsrv::ApplicationAlbumEntry entry = {};
        nn::capsrv::ScreenShotAttribute attr = {};
        attr.SetDefault();
        attr.size = GetCapsrvScreenShotSize(screenshotSize);
        attr.orientation = GetCapsrvScreenShotOrientation(imageOrientation);
        nn::capsrv::UserIdList userIdList = {};
        std::memcpy(userIdList.uid, pUidList, sizeof(account::Uid) * uidListCount);
        userIdList.uidCount = uidListCount;

        NN_RESULT_DO(ConvertCapsrvResult(
            nn::capsrv::SaveScreenShotForApplication(
                &entry,
                pImageData,
                imageDataSize,
                attr,
                userIdList,
                GetCapsrvOverlayNotificationRequest(reportOption)
            )
        ));

        NN_UNUSED(entry);
        NN_RESULT_SUCCESS;
    }

    namespace {
        nn::Result SaveAndShareScreenshotImpl(
            ShareId* pOutShareId,
            const void* pImageData,
            size_t imageDataSize,
            ImageSize screenshotSize,
            ImageOrientation imageOrientation,
            AlbumReportOption reportOption,
            const account::Uid& uid,
            const char* presetPostingMessage
        ) NN_NOEXCEPT
        {
            auto& lk = *g_LibraryState.GetLibraryMutex();
            std::lock_guard<decltype(lk)> lock(lk);

            g_LibraryState.EnsureAlbumAvailable();
            g_LibraryState.EnsureShareAvailable();

            nn::capsrv::ApplicationAlbumEntry entry = {};
            NN_RESULT_DO(
                NN_DETAIL_FS_ACCESS_LOG_EXPLICIT_FUNCTION_NAME(
                    SaveScreenshotImpl(&entry, pImageData, imageDataSize, screenshotSize, imageOrientation, reportOption),
                    nullptr, "SaveAndShareScreenshot",
                    NN_DETAIL_FS_ACCESS_LOG_FORMAT_ALBUM_SCREENSHOT_UID(imageDataSize, screenshotSize, imageOrientation, reportOption, uid)));

            SharePageWorkspace& shareWorkspace = g_SharePageWorkspace;
            NN_RESULT_DO(ShowSharePageImpl(pOutShareId, entry, uid, presetPostingMessage, shareWorkspace));

            NN_RESULT_SUCCESS;
        }
    }

#endif // NN_ALBUM_IS_STUB_IMPLEMENTATION


//*******************************************************
// API の実装
//   引数のバリエーションを一括生成する。
//   引数のチェックだけ行って実処理は Impl 関数で行う。
//*******************************************************

// 関数の引数。 IMAGEDATA より前にあるか後ろにあるかでカンマの位置が違うので注意。
#define NN_DETAIL_ALBUM_SCREENSHOTAPI_DECLARG_OUTSHAREID_Y ShareId* pOutShareId,
#define NN_DETAIL_ALBUM_SCREENSHOTAPI_DECLARG_OUTSHAREID_N
#define NN_DETAIL_ALBUM_SCREENSHOTAPI_DECLARG_IMAGEDATA_Y  const void* pImageData, size_t imageDataSize, ImageSize screenshotSize
//#define NN_DETAIL_ALBUM_SCREENSHOTAPI_DECLARG_IMAGEDATA_N
#define NN_DETAIL_ALBUM_SCREENSHOTAPI_DECLARG_IMAGEORI_Y   ,ImageOrientation imageOrientation
#define NN_DETAIL_ALBUM_SCREENSHOTAPI_DECLARG_IMAGEORI_N
#define NN_DETAIL_ALBUM_SCREENSHOTAPI_DECLARG_REPORTOPT_Y  ,AlbumReportOption reportOption
#define NN_DETAIL_ALBUM_SCREENSHOTAPI_DECLARG_REPORTOPT_N
#define NN_DETAIL_ALBUM_SCREENSHOTAPI_DECLARG_POSTINGUID_Y ,const nn::account::Uid& uid
#define NN_DETAIL_ALBUM_SCREENSHOTAPI_DECLARG_POSTINGUID_N
#define NN_DETAIL_ALBUM_SCREENSHOTAPI_DECLARG_POSTINGMSG_Y ,const char* presetPostingMessage
#define NN_DETAIL_ALBUM_SCREENSHOTAPI_DECLARG_POSTINGMSG_N

// 引数のチェック
#define NN_DETAIL_ALBUM_SCREENSHOTAPI_ARGCHECK_OUTSHAREID_Y NN_ALBUM_REQUIRES_NOT_NULL(pOutShareId);
#define NN_DETAIL_ALBUM_SCREENSHOTAPI_ARGCHECK_OUTSHAREID_N
#define NN_DETAIL_ALBUM_SCREENSHOTAPI_ARGCHECK_IMAGEDATA_Y  NN_ALBUM_REQUIRES(IsValidImageSize(screenshotSize)); NN_ALBUM_REQUIRES_VALID_IMAGE_DATA(pImageData, imageDataSize, screenshotSize);
#define NN_DETAIL_ALBUM_SCREENSHOTAPI_ARGCHECK_IMAGEDATA_N
#define NN_DETAIL_ALBUM_SCREENSHOTAPI_ARGCHECK_IMAGEORI_Y   NN_ALBUM_REQUIRES(IsValidImageOrientation(imageOrientation));
#define NN_DETAIL_ALBUM_SCREENSHOTAPI_ARGCHECK_IMAGEORI_N
#define NN_DETAIL_ALBUM_SCREENSHOTAPI_ARGCHECK_REPORTOPT_Y  // チェックなし
#define NN_DETAIL_ALBUM_SCREENSHOTAPI_ARGCHECK_REPORTOPT_N
#define NN_DETAIL_ALBUM_SCREENSHOTAPI_ARGCHECK_POSTINGUID_Y // チェックなし
#define NN_DETAIL_ALBUM_SCREENSHOTAPI_ARGCHECK_POSTINGUID_N
#define NN_DETAIL_ALBUM_SCREENSHOTAPI_ARGCHECK_POSTINGMSG_Y NN_ALBUM_REQUIRES(IsValidPresetPostingMessage(presetPostingMessage));
#define NN_DETAIL_ALBUM_SCREENSHOTAPI_ARGCHECK_POSTINGMSG_N

// 引数の UNUSED 指定（stub 用）
#define NN_DETAIL_ALBUM_SCREENSHOTAPI_UNUSED_OUTSHAREID_Y NN_UNUSED(pOutShareId);
#define NN_DETAIL_ALBUM_SCREENSHOTAPI_UNUSED_OUTSHAREID_N
#define NN_DETAIL_ALBUM_SCREENSHOTAPI_UNUSED_IMAGEDATA_Y  NN_UNUSED(pImageData); NN_UNUSED(imageDataSize); NN_UNUSED(screenshotSize);
#define NN_DETAIL_ALBUM_SCREENSHOTAPI_UNUSED_IMAGEDATA_N
#define NN_DETAIL_ALBUM_SCREENSHOTAPI_UNUSED_IMAGEORI_Y   NN_UNUSED(imageOrientation);
#define NN_DETAIL_ALBUM_SCREENSHOTAPI_UNUSED_IMAGEORI_N
#define NN_DETAIL_ALBUM_SCREENSHOTAPI_UNUSED_REPORTOPT_Y  NN_UNUSED(reportOption);
#define NN_DETAIL_ALBUM_SCREENSHOTAPI_UNUSED_REPORTOPT_N
#define NN_DETAIL_ALBUM_SCREENSHOTAPI_UNUSED_POSTINGUID_Y NN_UNUSED(uid);
#define NN_DETAIL_ALBUM_SCREENSHOTAPI_UNUSED_POSTINGUID_N
#define NN_DETAIL_ALBUM_SCREENSHOTAPI_UNUSED_POSTINGMSG_Y NN_UNUSED(presetPostingMessage);
#define NN_DETAIL_ALBUM_SCREENSHOTAPI_UNUSED_POSTINGMSG_N

// Impl 呼び出しの引数。こちらはカンマなし。
#define NN_DETAIL_ALBUM_SCREENSHOTAPI_CALLARG_OUTSHAREID_Y pOutShareId
#define NN_DETAIL_ALBUM_SCREENSHOTAPI_CALLARG_OUTSHAREID_N nullptr
#define NN_DETAIL_ALBUM_SCREENSHOTAPI_CALLARG_IMAGEDATA_Y  pImageData, imageDataSize, screenshotSize
#define NN_DETAIL_ALBUM_SCREENSHOTAPI_CALLARG_IMAGEDATA_N  /* 必須引数 */
#define NN_DETAIL_ALBUM_SCREENSHOTAPI_CALLARG_IMAGEORI_Y   imageOrientation
#define NN_DETAIL_ALBUM_SCREENSHOTAPI_CALLARG_IMAGEORI_N   nn::album::ImageOrientation_None
#define NN_DETAIL_ALBUM_SCREENSHOTAPI_CALLARG_REPORTOPT_Y  reportOption
#define NN_DETAIL_ALBUM_SCREENSHOTAPI_CALLARG_REPORTOPT_N  AlbumReportOption_ReportAlways
#define NN_DETAIL_ALBUM_SCREENSHOTAPI_CALLARG_POSTINGUID_Y uid
#define NN_DETAIL_ALBUM_SCREENSHOTAPI_CALLARG_POSTINGUID_N nn::account::InvalidUid
#define NN_DETAIL_ALBUM_SCREENSHOTAPI_CALLARG_POSTINGMSG_Y presetPostingMessage
#define NN_DETAIL_ALBUM_SCREENSHOTAPI_CALLARG_POSTINGMSG_N nullptr

#ifndef NN_ALBUM_IS_STUB_IMPLEMENTATION
#define NN_DETAIL_ALBUM_SAVEANDSHARESCREENSHOT_DEFINEFUNCTION(useOutShareId, useImageData, useImageOri, useReportOpt, usePostingUid, usePostingMsg) \
    nn::Result SaveAndShareScreenshot(  \
        NN_DETAIL_ALBUM_SCREENSHOTAPI_DECLARG_OUTSHAREID_##useOutShareId    \
        NN_DETAIL_ALBUM_SCREENSHOTAPI_DECLARG_IMAGEDATA_ ##useImageData     \
        NN_DETAIL_ALBUM_SCREENSHOTAPI_DECLARG_IMAGEORI_  ##useImageOri      \
        NN_DETAIL_ALBUM_SCREENSHOTAPI_DECLARG_REPORTOPT_ ##useReportOpt     \
        NN_DETAIL_ALBUM_SCREENSHOTAPI_DECLARG_POSTINGUID_##usePostingUid    \
        NN_DETAIL_ALBUM_SCREENSHOTAPI_DECLARG_POSTINGMSG_##usePostingMsg    \
    ) NN_NOEXCEPT   \
    {   \
        NN_ALBUM_LOG_DETAIL("SaveAndShareScreenshot("   \
            #useOutShareId   \
            #useImageData    \
            #useImageOri     \
            #useReportOpt    \
            #usePostingUid   \
            #usePostingMsg   \
            ")\n");    \
        NN_SDK_REQUIRES(g_LibraryState.IsInitialized());    \
        NN_DETAIL_ALBUM_SCREENSHOTAPI_ARGCHECK_OUTSHAREID_##useOutShareId   \
        NN_DETAIL_ALBUM_SCREENSHOTAPI_ARGCHECK_IMAGEDATA_ ##useImageData    \
        NN_DETAIL_ALBUM_SCREENSHOTAPI_ARGCHECK_IMAGEORI_  ##useImageOri     \
        NN_DETAIL_ALBUM_SCREENSHOTAPI_ARGCHECK_REPORTOPT_ ##useReportOpt    \
        NN_DETAIL_ALBUM_SCREENSHOTAPI_ARGCHECK_POSTINGUID_##usePostingUid   \
        NN_DETAIL_ALBUM_SCREENSHOTAPI_ARGCHECK_POSTINGMSG_##usePostingMsg   \
        \
        return SaveAndShareScreenshotImpl(  \
            NN_DETAIL_ALBUM_SCREENSHOTAPI_CALLARG_OUTSHAREID_##useOutShareId ,  \
            NN_DETAIL_ALBUM_SCREENSHOTAPI_CALLARG_IMAGEDATA_ ##useImageData  ,  \
            NN_DETAIL_ALBUM_SCREENSHOTAPI_CALLARG_IMAGEORI_  ##useImageOri   ,  \
            NN_DETAIL_ALBUM_SCREENSHOTAPI_CALLARG_REPORTOPT_ ##useReportOpt  ,  \
            NN_DETAIL_ALBUM_SCREENSHOTAPI_CALLARG_POSTINGUID_##usePostingUid ,  \
            NN_DETAIL_ALBUM_SCREENSHOTAPI_CALLARG_POSTINGMSG_##usePostingMsg ); \
    }
#else
    // スタブ用
#define NN_DETAIL_ALBUM_SAVEANDSHARESCREENSHOT_DEFINEFUNCTION(useOutShareId, useImageData, useImageOri, useReportOpt, usePostingUid, usePostingMsg) \
    nn::Result SaveAndShareScreenshot(  \
        NN_DETAIL_ALBUM_SCREENSHOTAPI_DECLARG_OUTSHAREID_##useOutShareId    \
        NN_DETAIL_ALBUM_SCREENSHOTAPI_DECLARG_IMAGEDATA_ ##useImageData     \
        NN_DETAIL_ALBUM_SCREENSHOTAPI_DECLARG_IMAGEORI_  ##useImageOri      \
        NN_DETAIL_ALBUM_SCREENSHOTAPI_DECLARG_REPORTOPT_ ##useReportOpt     \
        NN_DETAIL_ALBUM_SCREENSHOTAPI_DECLARG_POSTINGUID_##usePostingUid    \
        NN_DETAIL_ALBUM_SCREENSHOTAPI_DECLARG_POSTINGMSG_##usePostingMsg    \
    ) NN_NOEXCEPT   \
    {   \
        NN_DETAIL_ALBUM_SCREENSHOTAPI_UNUSED_OUTSHAREID_##useOutShareId   \
        NN_DETAIL_ALBUM_SCREENSHOTAPI_UNUSED_IMAGEDATA_ ##useImageData    \
        NN_DETAIL_ALBUM_SCREENSHOTAPI_UNUSED_IMAGEORI_  ##useImageOri     \
        NN_DETAIL_ALBUM_SCREENSHOTAPI_UNUSED_REPORTOPT_ ##useReportOpt    \
        NN_DETAIL_ALBUM_SCREENSHOTAPI_UNUSED_POSTINGUID_##usePostingUid   \
        NN_DETAIL_ALBUM_SCREENSHOTAPI_UNUSED_POSTINGMSG_##usePostingMsg   \
        NN_RESULT_THROW(ResultNotSupported());  \
    }
#endif

    // SaveAndShareScreenshot() の全組合せを定義
    //                                                    pOutShareId
    //                                                    |  pImageData, imageDataSize, screenshotSize
    //                                                    |  |  imageOrientation
    //                                                    |  |  |  reportOption
    //                                                    |  |  |  |  uid
    //                                                    |  |  |  |  |  presetPostingMessage
    //                                                    |  |  |  |  |  |
    //                                                    *  Y  *  Y  Y  *
    NN_DETAIL_ALBUM_SAVEANDSHARESCREENSHOT_DEFINEFUNCTION(N, Y, N, Y, Y, N);
    NN_DETAIL_ALBUM_SAVEANDSHARESCREENSHOT_DEFINEFUNCTION(Y, Y, N, Y, Y, N);
    NN_DETAIL_ALBUM_SAVEANDSHARESCREENSHOT_DEFINEFUNCTION(N, Y, Y, Y, Y, N);
    NN_DETAIL_ALBUM_SAVEANDSHARESCREENSHOT_DEFINEFUNCTION(Y, Y, Y, Y, Y, N);
    NN_DETAIL_ALBUM_SAVEANDSHARESCREENSHOT_DEFINEFUNCTION(N, Y, N, Y, Y, Y);
    NN_DETAIL_ALBUM_SAVEANDSHARESCREENSHOT_DEFINEFUNCTION(Y, Y, N, Y, Y, Y);
    NN_DETAIL_ALBUM_SAVEANDSHARESCREENSHOT_DEFINEFUNCTION(N, Y, Y, Y, Y, Y);
    NN_DETAIL_ALBUM_SAVEANDSHARESCREENSHOT_DEFINEFUNCTION(Y, Y, Y, Y, Y, Y);

}}

