﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "album_LibraryState.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#include <nn/os.h>
#include <nn/capsrv/capsrv_AlbumAccess.h>
#include <nn/capsrv/capsrv_ScreenShotForApplication.h>
#include <nn/capsrv/capsrv_AlbumAccessForApplication.h>

namespace nn{ namespace album{

    LibraryState g_LibraryState;

    //--------------------------------

    LibraryState::LibraryState() NN_NOEXCEPT
        : m_InitializeCount(0)
        , m_IsAlbumAvailable(false)
        , m_IsShareAvailable(false)
    {
    }

    bool LibraryState::IsInitialized() const NN_NOEXCEPT
    {
        return m_InitializeCount > 0;
    }

    bool LibraryState::IsAlbumAvailable() const NN_NOEXCEPT
    {
        return m_IsAlbumAvailable;
    }

    bool LibraryState::IsShareAvailable() const NN_NOEXCEPT
    {
        return m_IsShareAvailable;
    }

    //--------------------------------

    void LibraryState::Initialize() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_GREATER_EQUAL(m_InitializeCount, 0);
        if(m_InitializeCount > 0)
        {
            m_InitializeCount++;
            return;
        }

        m_LibraryMutex.Initialize();
        m_InitializeCount++;
    }

    void LibraryState::Finalize() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_GREATER(m_InitializeCount, 0);
        m_InitializeCount--;
        if(m_InitializeCount > 0)
        {
            return;
        }
        if(m_InitializeCount < 0)
        {
            // Release ビルドでは過剰な Finalize は無視する
            m_InitializeCount = 0;
            return;
        }

        if(IsAlbumAvailable())
        {
            nn::capsrv::FinalizeAlbumAccessForApplication();
            nn::capsrv::FinalizeForApplication();
            m_IsAlbumAvailable = false;
        }

        if(IsShareAvailable())
        {
            // nothing
            m_IsShareAvailable = false;
        }
    }

    //--------------------------------

    nn::os::SdkRecursiveMutexType* LibraryState::GetLibraryMutex() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        return &m_LibraryMutex;
    }

    void LibraryState::EnsureAlbumAvailable() NN_NOEXCEPT
    {
        if(!IsAlbumAvailable())
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeForApplication());
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::capsrv::InitializeAlbumAccessForApplication());
            m_IsAlbumAvailable = true;
        }
    }

    void LibraryState::EnsureShareAvailable() NN_NOEXCEPT
    {
        if(!IsShareAvailable())
        {
            // nothing
            m_IsShareAvailable = true;
        }
    }

}}

