﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <winext/cafe/os.h>
#include <cstdlib>
#include "system.h"
#include "systemi.h"
#include "exceptioni.h"
#include <stdio.h>

#include <windows.h>
#include <mmsystem.h>

// Windows Multimedia 関数を利用します。
#pragma comment(lib, "winmm.lib")

namespace
{

// 1秒間当たりのフレーム数
const int   FrameCount  = 60;

// マルチメディアタイマの分解能(ミリ秒単位)
UINT        sTimerResolution = 1;

// Vブランクシミュレーション用イベントオブジェクトのハンドラ
HANDLE      shVSyncEvent;

// Multimeida timer のID
MMRESULT    sMmTimerID;

}   // namespace


//============================================================================
//          WAIT
//============================================================================

#if _MSC_VER >= 1500
#pragma warning(push)

// 警告の無効化「例外がスローされないはずなのにスローされる」
#pragma warning(disable : 4297)
#endif

void OS_DoTerminate();

/*---------------------------------------------------------------------------*
  Name:         OS_WaitVBlankIntr

  Description:  wait till vblank interrupt occurred.
                the difference between SVC_WaitVBlankIntr and OS_WaitVBlankIntr is:
                OS_WaitVBlankIntr does switch thread,
                SVC_WaitVBlankIntr doesn't switch thread.

  Arguments:    None

  Returns:      None
 *---------------------------------------------------------------------------*/
void
GX2WaitForVsync(void)
{
    const HANDLE handles[] = { shVSyncEvent };
    const DWORD handleCnt = _countof(handles);

    DWORD waitResult = 0;
    while (
        WAIT_OBJECT_0 + handleCnt == (waitResult = MsgWaitForMultipleObjectsEx(
            handleCnt,              // ハンドルの配列に入っているハンドルの数
            handles,                // オブジェクトハンドルの配列へのポインタ
            INFINITE,               // タイムアウト時間（ミリ秒単位）
            QS_SENDMESSAGE |QS_ALLEVENTS,           // 待機する入力イベントの種類
            MWMO_INPUTAVAILABLE))   // 待機フラグ
    )
    {
        // メッセージキューにウィンドウメッセージが置かれていたら
        // 取得する
        MSG msg;
        while (PeekMessage(&msg, NULL, 0, 0, PM_REMOVE))
        {
            // 置かれていたウィンドウメッセージがWM_QUITの場合はループを脱出
            if (msg.message == WM_QUIT)
            {
                // throw nw::internal::winext::ApplicationExitException(msg.wParam);
                OS_Exit(msg.wParam);
            }

            TranslateMessage(&msg); //ウィンドウメッセージを変換
            DispatchMessage(&msg);  //ウィンドウメッセージを送出
        }
    }

    OS_DoTerminate();

    if (WAIT_FAILED == waitResult)
    {
        throw nw::internal::winext::Win32Exception(GetLastError());
    }
}

#if _MSC_VER >= 1500
#pragma warning(pop)
#endif

namespace nw {
namespace internal {
namespace winext {

/*---------------------------------------------------------------------------*
  Name:         Init_OS_System

  Description:  このソースファイルで定義される関数で必要とされる初期化処理を
                行います。

  Arguments:    なし。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
Init_OS_System()
{
    TIMECAPS timeCaps;
    MMRESULT result = timeGetDevCaps(&timeCaps, sizeof(timeCaps));
    if (TIMERR_NOERROR != result)
    {
        throw MultimediaApiException(result);
    }

    // 最小値と最大値の間に収まるようにする
    if (sTimerResolution < timeCaps.wPeriodMin)
    {
        sTimerResolution = timeCaps.wPeriodMin;
    }
    if (sTimerResolution > timeCaps.wPeriodMax)
    {
        sTimerResolution = timeCaps.wPeriodMax;
    }

    shVSyncEvent = CreateEvent(
        NULL,           // セキュリティ記述子
        false,          // 手動リセットか?
        false,          // 初期状態はシグナルか?
        NULL);          // イベントオブジェクトの名前
    if (NULL == shVSyncEvent)
    {
        throw Win32Exception(GetLastError());
    }

    result = timeBeginPeriod(sTimerResolution);
    if (TIMERR_NOERROR != result)
    {
        throw MultimediaApiException(result);
    }

    sMmTimerID = timeSetEvent(
        1000 / FrameCount,
        sTimerResolution,
        reinterpret_cast<LPTIMECALLBACK>(shVSyncEvent),
        0 /* dwUser */,
        TIME_PERIODIC | TIME_CALLBACK_EVENT_PULSE);
    if (NULL == sMmTimerID)
    {
        throw MultimediaTimerSetEventException();
    }
}

/*---------------------------------------------------------------------------*
  Name:         Finalize_OS_System

  Description:  このソースファイルで定義される関数で必要とされる終了処理を
                行います。

  Arguments:    なし。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
Finalize_OS_System()
{
    MMRESULT result = timeKillEvent(sMmTimerID);
    if (TIMERR_NOERROR != result)
    {
        throw MultimediaApiException(result);
    }

    result = timeEndPeriod(sTimerResolution);
    if (TIMERR_NOERROR != result)
    {
        throw MultimediaApiException(result);
    }

    BOOL bSuccess = CloseHandle(shVSyncEvent);
    if (! bSuccess)
    {
        throw Win32Exception(GetLastError());
    }
}

}   // namespace winext {
}   // namespace internal {
}   // namespace nw
