﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <winext/cafe/fs.h>
#include <winext/cafe/stl.h>
#include <map>
#include <string>
#include <cstdio>
#include <sys/stat.h>
#include <direct.h>
#include <errno.h>

#include <Windows.h>

namespace nw {
namespace internal {
namespace winext {

struct FSClientImpl
{
    pcsdk::string curDir;
};

struct MountInfo
{
    pcsdk::string src;
};

struct FileInfo
{
    FILE* fp;

    FileInfo():fp(NULL){}
};

bool GetOpenPath(FSClientImpl* client, const char *p_path, pcsdk::string& path);

typedef pcsdk::map<pcsdk::string,MountInfo*>::type MountList;
MountList s_MountList;

pcsdk::string s_CafeDiskFileDevice = "/dev/sd01";
pcsdk::string s_DevHFIO = "/dev/hfio01";

pcsdk::string s_CafeContentFileVolume = "/vol/content";
pcsdk::string s_CafeSaveFileVolume = "/vol/save";

pcsdk::string CAFE_CONTENT_DIR = "CAFE_CONTENT_DIR";
pcsdk::string CAFE_SAVE_DIR = "CAFE_SAVE_DIR";
pcsdk::string s_ContentDir;
pcsdk::string s_SaveDir;

pcsdk::string POSIX_MOUNT_POINT = "/cygdrive/";

pcsdk::string NW_WINEXT_FS_USE_REGISTRY = "NW_WINEXT_FS_USE_REGISTRY";

static const HKEY PARENT_KEY = HKEY_CURRENT_USER;
static const char SUB_KEY[] = "Software\\Nintendo\\NDEV";
static const char DVD_ROOT_NAME[] = "NW_DISC_ROOT";

static int s_FsInitCount = 0;
static FSError s_FsLastError = FS_ERROR_OK;

void FSInit( void )
{
    if(s_FsInitCount++ > 0)
    {
        return;
    }

    static const int cFileNameLen = 512;
    char buf[cFileNameLen] = "";

    s_ContentDir = "";

    if ( GetEnvironmentVariableA(CAFE_CONTENT_DIR.c_str(), buf, cFileNameLen) > 0 )
    {
        s_ContentDir = buf;

        // CafeSDK では CAFE_CONTENT_DIR はマウントポイント /cygdrive から始まる
        // POSIX 形式である必要があるので、PcSDK でもひとまずマウントポイント
        // /cygdrive の場合のみ変換するようにする
        if ( s_ContentDir.find(POSIX_MOUNT_POINT) == 0)
        {
            s_ContentDir = s_ContentDir.substr(POSIX_MOUNT_POINT.size() - 1, pcsdk::string::npos);
            // /c/ を c:/ に変換
            s_ContentDir[0] = s_ContentDir[1];
            s_ContentDir[1] = ':';
        }
    }

    if ( s_ContentDir.empty() &&
         GetEnvironmentVariableA(NW_WINEXT_FS_USE_REGISTRY.c_str(), buf, cFileNameLen) <= 0 )
    {
        const char CAFE_DISC_DIR_ERROR_MESSAGE[] =
          "Warning: !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!\n" \
          "Warning: !!!!!!!!!!!!!! CAFE_CONTENT_DIR not found !!!!!!!!!!!!!!\n" \
          "Warning: !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!\n";
        printf( CAFE_DISC_DIR_ERROR_MESSAGE );
        OutputDebugStringA( CAFE_DISC_DIR_ERROR_MESSAGE );
    }

    // CAFE_DISC_DIR が定義されていないか、もしくは NW_WINEXT_FS_USE_REGISTRY が定義されている場合は、
    // SetNdEnv のレジストリ設定 NW_DISC_ROOT を使用します。
    if ( s_ContentDir.empty() ||
         GetEnvironmentVariableA(NW_WINEXT_FS_USE_REGISTRY.c_str(), buf, cFileNameLen) > 0 )
    {
        HKEY hKey;
        DWORD pos;
        DWORD type = REG_SZ;
        DWORD size = cFileNameLen;

        RegCreateKeyExA( PARENT_KEY, SUB_KEY, 0, "", REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS, NULL, &hKey, &pos );
        RegQueryValueExA( hKey, DVD_ROOT_NAME, NULL, &type, reinterpret_cast<LPBYTE>( buf ), &size );
        RegCloseKey( hKey );

        char ex_path[cFileNameLen] = "";
        ExpandEnvironmentStringsA( buf, ex_path, cFileNameLen );
        s_ContentDir = ex_path;
    }

    if ( ! s_ContentDir.empty() && s_ContentDir[s_ContentDir.size()-1] != '/' ){
        s_ContentDir += "/";
    }

    if ( GetEnvironmentVariableA(CAFE_SAVE_DIR.c_str(), buf, cFileNameLen) > 0 )
    {
        s_SaveDir = buf;

        // CAFE_CONTENT_DIR と同様に処理。
        if ( s_SaveDir.find(POSIX_MOUNT_POINT) == 0)
        {
            s_SaveDir = s_SaveDir.substr(POSIX_MOUNT_POINT.size() - 1, pcsdk::string::npos);
            // /c/ を c:/ に変換
            s_SaveDir[0] = s_SaveDir[1];
            s_SaveDir[1] = ':';
        }
    }

    // PC-SDK では、hfio は PC 上のフルパスで来る事が前提の実装になってしまっているので、
    // /vol/content は /dev/sd01 に関連付けておく。
    MountInfo* mountInfo = new(malloc(sizeof(MountInfo))) MountInfo();
    mountInfo->src = s_CafeDiskFileDevice;
    s_MountList[ s_CafeContentFileVolume ] = mountInfo;
}

void FSShutdown( void )
{
    if(s_FsInitCount <= 0 || --s_FsInitCount > 0)
    {
        return;
    }

    free( s_MountList[ s_CafeContentFileVolume ] );
}

FSStatus FSAddClient( FSClient* pClient, FSRetFlag errHandling )
{
    strcpy_s( reinterpret_cast<char*>(pClient->buffer), sizeof(pClient->buffer), FS_DEVICEFILES_ROOT );
    printf(reinterpret_cast<char*>(pClient->buffer));

    return FS_STATUS_OK;
}

FSStatus FSDelClient( FSClient* /* client */, FSRetFlag /* errHandling */ )
{
    return FS_STATUS_OK;
}

FSStatus FSMount(
    FSClient                        * /* client */,
    FSCmdBlock                      * /* block */,
    FSMountSource                   *p_source,
    char                            *p_target,
    u32                             bytes,
    FSRetFlag                       /* errHandling */
)
{
    // p_source->path から /dev を取り除く
    // p_target に /vol/(トリミング p_source->path) を入れとく
    pcsdk::string path;
    path = p_source->path;
    path = "/vol" + path.substr(FS_DEVICEFILES_ROOT_NAMESIZE, pcsdk::string::npos);
    strcpy_s( p_target, bytes, path.c_str() );

    MountList::iterator p = s_MountList.find( p_target );
    if ( p != s_MountList.end() ) {
        return FS_STATUS_EXISTS;
    }

    MountInfo* mountInfo = new(malloc(sizeof(MountInfo))) MountInfo();
    mountInfo->src = p_source->path;

    s_MountList[ p_target ] = mountInfo;

    return FS_STATUS_OK;
}

FSStatus FSMakeDir(
    FSClient                *client,
    FSCmdBlock              * /* block */,
    const char              *p_path,
    FSRetFlag               /* errHandling */
)
{
    FSClientImpl* clientImpl = reinterpret_cast<FSClientImpl*>(client);

    if ( p_path == NULL ) {
        //return FS_STATUS_INVALID_PATH;
        nw::internal::winext::OSHalt("Error: Invalid path.\n");
    }

    pcsdk::string path;
    if (!GetOpenPath(clientImpl, p_path, path))
    {
        //return FS_STATUS_INVALID_PATH;
        nw::internal::winext::OSHalt("Error: Invalid path.\n");
    }

    if (-1 == _mkdir(path.c_str()))
    {
        switch (errno)
        {
        case EEXIST:
            return FS_STATUS_EXISTS;

        case ENOENT:
            //return FS_STATUS_INVALID_PATH;
            nw::internal::winext::OSHalt("Error: Invalid path.\n");

        default:
            // 予期しないエラー。
            //return FS_STATUS_SYSTEM_ERROR;
            nw::internal::winext::OSHalt("Error: Unknown error.\n");
        }
    }

    return FS_STATUS_OK;
}

FSStatus FSUnmount(
    FSClient                        * /* client */,
    FSCmdBlock                      * /* block */,
    const char                      *p_target,
    FSRetFlag                       /* flag */
)
{
    MountList::iterator p = s_MountList.find( p_target );
    if ( p == s_MountList.end() ) {
        //return FS_STATUS_INVALID_PATH;
        nw::internal::winext::OSHalt("Error: Invalid path.\n");
    }

    free( p->second );
    s_MountList.erase( p_target );

    return FS_STATUS_OK;
}

FSStatus FSOpenDir(
    FSClient                        *client,
    FSCmdBlock                      * /* block */,
    const char                      *p_path,
    FSDirHandle                     *p_returned_dir_handle,
    FSRetFlag                       /* errHandling */
    )
{
    FSClientImpl* clientImpl = reinterpret_cast<FSClientImpl*>(client);

    if ( p_path == NULL ) {
        //return FS_STATUS_INVALID_PATH;
        nw::internal::winext::OSHalt("Error: Invalid path.\n");
    }

    pcsdk::string path;
    if (!GetOpenPath(clientImpl, p_path, path))
    {
        //return FS_STATUS_INVALID_PATH;
        nw::internal::winext::OSHalt("Error: Invalid path.\n");
    }

    for (;;)
    {
        if (path.size() > 0 && path[path.size() - 1] == '/')
        {
            path = path.substr( 0, path.size() - 1 );
        }
        else
        {
            break;
        }
    }
    path += "/*";

    // "." および ".." を読み込んだ状態にします。
    WIN32_FIND_DATAA find_data;
    HANDLE win_handle = FindFirstFileA(path.c_str(), &find_data);
    if(win_handle == INVALID_HANDLE_VALUE)
    {
        //return FS_STATUS_INVALID_PATH;
        nw::internal::winext::OSHalt("Error: Invalid path.\n");
    }

    BOOL success = FindNextFileA(win_handle, &find_data);
    if(!success)
    {
        //return FS_STATUS_INVALID_PATH;
        nw::internal::winext::OSHalt("Error: Invalid path.\n");
    }

    *p_returned_dir_handle = reinterpret_cast<FSDirHandle>(win_handle);

    return FS_STATUS_OK;
}

FSStatus FSReadDir(
    FSClient                        * /* client */,
    FSCmdBlock                      * /* block */,
    FSDirHandle                     dir_handle,
    FSDirEntry                      *p_returned_dir_entry,
    FSRetFlag                       /* errHandling */
    )
{
    HANDLE win_handle = reinterpret_cast<HANDLE>(dir_handle);

    WIN32_FIND_DATAA find_data;
    BOOL success = FindNextFileA(win_handle, &find_data);

    DWORD lastRawError = GetLastError();
    if( !success )
    {
        if (lastRawError == ERROR_NO_MORE_FILES)
        {
            return FS_STATUS_END;
        }

        return FS_STATUS_UNSUPPORTED_CMD;
    }

    ZeroMemory( p_returned_dir_entry, sizeof(FSDirEntry) );
    strcpy_s( p_returned_dir_entry->name, sizeof(p_returned_dir_entry->name), find_data.cFileName );
    p_returned_dir_entry->stat.flag = (find_data.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) ? FS_STAT_FLAG_IS_DIRECTORY : 0;
    p_returned_dir_entry->stat.size = find_data.nFileSizeLow;

    return FS_STATUS_OK;
}

FSStatus FSCloseDir(
    FSClient                       * /* client */,
    FSCmdBlock                     * /* cmdBlock **/,
    FSDirHandle                    dir_handle,
    FSRetFlag                      /* errHandling */
    )
{
    BOOL success = FindClose(reinterpret_cast<HANDLE>(dir_handle));

    return FS_STATUS_OK;
}

FSStatus FSOpenFile(
    FSClient                        *client,
    FSCmdBlock                      *cmdBlock,
    const char                      *p_path,
    const char                      *p_mode,
    FSFileHandle                    *p_returned_file_handle,
    FSRetFlag                       errHandling
)
{
    FSClientImpl* clientImpl = reinterpret_cast<FSClientImpl*>(client);

    if ( p_path == NULL ) {
        return FS_STATUS_NOT_FILE;
    }

    pcsdk::string path;
    if (!GetOpenPath(clientImpl, p_path, path))
    {
        return FS_STATUS_NOT_FILE;
    }

    pcsdk::string mode = p_mode;
    mode += 'b';

    FILE* fp;
    int err = fopen_s(&fp, path.c_str(),mode.c_str());
    if ( err != 0 ) {
        switch (err)
        {
        case ENOENT:
            return FS_STATUS_NOT_FOUND;

        default:
            //return FS_STATUS_INVALID_PATH;
            nw::internal::winext::OSHalt("Error: Invalid path.\n");
        }
    }

    FileInfo* fileInfo = new(malloc(sizeof(FileInfo))) FileInfo();
    fileInfo->fp = fp;

    *p_returned_file_handle = reinterpret_cast<FSFileHandle>(fileInfo);

    return FS_STATUS_OK;
}

FSStatus FSChangeDir(
    FSClient                        *client,
    FSCmdBlock                      * /* block */,
    const char                      *p_path,
    FSRetFlag                       errHandling
)
{
    FSClientImpl* clientImpl = reinterpret_cast<FSClientImpl*>(client);

    if ( p_path == '\0' ) {
        //return FS_STATUS_INVALID_PATH;
        nw::internal::winext::OSHalt("Error: Invalid path.\n");
    }

    pcsdk::string path;
    if ( p_path[0] == '/' ) {
        path = p_path;
    }
    else {
        path = clientImpl->curDir + '/' + path;
    }

    clientImpl->curDir = path;

    return FS_STATUS_OK;
}

FSStatus FSGetCwd(
                FSClient                *client,
                FSCmdBlock              * /* block */,
                char                    *p_returned_path,
                u32                     bytes,
                FSRetFlag               errHandling
                )
{
    FSClientImpl* clientImpl = reinterpret_cast<FSClientImpl*>(client);

    _snprintf_s( p_returned_path, bytes, bytes, "%s", clientImpl->curDir.c_str() );
    return FS_STATUS_OK;
}

FSStatus FSReadFile(
    FSClient                *client,
    FSCmdBlock              * /* block */,
    void                    *dest,
    FSSize                  size,
    FSCount                 count,
    FSFileHandle            file_handle,
    FSFlag                  flag,
    FSRetFlag               errHandling
)
{
    FileInfo* fileInfo = reinterpret_cast<FileInfo*>(file_handle);

    int readSize = fread(dest, size,count,fileInfo->fp);
    return readSize;
}

FSStatus FSCloseFile(
    FSClient                *client,
    FSCmdBlock              * /* block */,
    FSFileHandle            file_handle,
    FSRetFlag               errHandling
)
{
    FileInfo* fileInfo = reinterpret_cast<FileInfo*>(file_handle);
    if ( fileInfo->fp != NULL ) {
    fclose(fileInfo->fp);
    }

    free( fileInfo );

    return FS_STATUS_OK;
}

FSStatus FSGetStatFile(
    FSClient                *client,
    FSCmdBlock              * /* block */,
    FSFileHandle            file_handle,
    FSStat                  *returnedStat,
    FSRetFlag               errHandling
)
{
    FileInfo* fileInfo = reinterpret_cast<FileInfo*>(file_handle);

    int fd = _fileno(fileInfo->fp);
    struct _stat st;
    _fstat(fd,&st);

    returnedStat->size = st.st_size;
    returnedStat->permission = 0;
    if (st.st_mode & _S_IREAD ) {
        returnedStat->permission |= FS_MODE_IRGRP;
    }
    if (st.st_mode & _S_IWRITE ) {
        returnedStat->permission |= FS_MODE_IWGRP;
    }

    return FS_STATUS_OK;
}

FSStatus FSWriteFile(
    FSClient                *client,
    FSCmdBlock              * /* block */,
    const void              *p_source,
    FSSize                  size,
    FSCount                 count,
    FSFileHandle            fileHandle,
    FSFlag                  flag,
    FSRetFlag               errHandling
)
{
    FileInfo* fileInfo = reinterpret_cast<FileInfo*>(fileHandle);

    if (p_source == NULL)
    {
        //return FS_STATUS_INVALID_BUFFER;
        nw::internal::winext::OSHalt("Error: Invalid buffer.\n");
    }

    int writeSize = fwrite(p_source, size, count, fileInfo->fp);
    return writeSize;
}

FSStatus FSWriteFileWithPos(
    FSClient                *client,
    FSCmdBlock              *block,
    const void              *p_source,
    FSSize                  size,
    FSCount                 count,
    FSFilePosition          fpos,
    FSFileHandle            file_handle,
    FSFlag                  flag,
    FSRetFlag               errHandling
)
{
    FSStatus res = FSSetPosFile(client, block, file_handle, fpos, errHandling);
    if (res != FS_STATUS_OK)
    {
        return res;
    }

    return FSWriteFile(client, block, p_source, size, count, file_handle, flag, errHandling);
}

FSStatus FSSetPosFile(
    FSClient                *client,
    FSCmdBlock              * /* block */,
    FSFileHandle            file_handle,
    FSFilePosition          fpos,
    FSRetFlag               errHandling
)
{
    FileInfo* fileInfo = reinterpret_cast<FileInfo*>(file_handle);
    fseek(fileInfo->fp, fpos, SEEK_SET);

    return FS_STATUS_OK;
}

FSStatus FSGetPosFile(
    FSClient                *client,
    FSCmdBlock              * /* block */,
    FSFileHandle            file_handle,
    FSFilePosition          *returnedFpos,
    FSRetFlag               errHandling
    )
{
    FileInfo* fileInfo = reinterpret_cast<FileInfo*>(file_handle);
    return ftell(fileInfo->fp);
}


void PCFSSetDvdRoot( const char* path )
{
    s_CafeDiskFileDevice = path;
};

bool GetOpenPath(FSClientImpl* clientImpl, const char *p_path, pcsdk::string& path)
{
    if ( p_path[0] == '/' ) {
        path = p_path;
    }
    else {
        path = clientImpl->curDir + '/' + p_path;
    }

    pcsdk::string volume;

    for( MountList::iterator p = s_MountList.begin() ; p != s_MountList.end() ; p++ ){
        volume = p->first;
        if ( path.find(volume) == 0 ) {
            path = p->second->src + path.substr( p->first.size(), pcsdk::string::npos);
            break;
        }
    }

    if ( path.find( s_CafeDiskFileDevice ) == 0 ) {
        if ( volume == s_CafeContentFileVolume )
        {
            // SDK 1.8 では /vol/content/ は $(CAFE_CONTENT_DIR) を指します。
            path = s_ContentDir + path.substr( s_CafeDiskFileDevice.size(), pcsdk::string::npos);
        }
        else
        {
            path = s_ContentDir + path.substr( s_CafeDiskFileDevice.size(), pcsdk::string::npos);
        }
    }
    else if ( path.find( s_CafeSaveFileVolume ) == 0 ) {
        path = s_SaveDir + path.substr( s_CafeSaveFileVolume.size(), pcsdk::string::npos);
    }
    else if ( path.find( s_DevHFIO ) == 0 ) {
        path = path.substr( s_DevHFIO.size(), pcsdk::string::npos);
        if (!(3 < path.size() && path[0] == '/' && isalpha(path[1]) && path[2] == '/'))
        {
            return false;
        }

        // "/c/" -> "c:/"
        path[0] = path[1];
        path[1] = ':';
    }

    return true;
};

void FSInitCmdBlock( FSCmdBlock* /* block */ )
{
}

FSStatus FSGetMountSource(
    FSClient                *client,
    FSCmdBlock              * /* block */,
    FSSourceType            type,
    FSMountSource           *source,
    FSRetFlag               errHandling
    )
{
    switch(type)
    {
    case FS_SOURCETYPE_HFIO:
      strcpy_s( source->path, sizeof(source->path), s_DevHFIO.c_str() );
      break;
    case FS_SOURCETYPE_EXTERNAL:
      strcpy_s( source->path, sizeof(source->path), s_CafeDiskFileDevice.c_str() );
      break;
    default:
        nw::internal::winext::OSHalt("Error: Unknown error.\n");
    }

    return FS_STATUS_OK;
}

FSError FSGetLastError()
{
    return s_FsLastError;
}

void SetFSLastError_()
{
}

} // namespace winext
} // namespace internal
} // namespace nw
