﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "SoundEngineSLES.h"
#include "ai_dma.h"
#include <assert.h>
#include <unistd.h>
#include <android/log.h>
#include <errno.h>

// #define ENABLE_MEASURE_RENDERING_TIME

namespace  {

#if defined(ENABLE_MEASURE_RENDERING_TIME)
static s64 GetCurrentTimeMs()
{
    int ret;
    timeval tv;

    ret = gettimeofday(&tv, NULL);

    return (s64)tv.tv_sec * 1000 + (s64)tv.tv_usec / 1000;
}
#endif

} // namespace

namespace nw {
    namespace internal {
        namespace winext {

            CSoundEngineSLES::CSoundEngineSLES()
                : m_IsActive(false), m_RequestExitThread(false), m_WaveOutProc(NULL)
{}
            CSoundEngineSLES::~CSoundEngineSLES()
{}

            CSoundEngineSLES* CSoundEngineSLES::sInstance;

bool CSoundEngineSLES::IsActive() const
{
    return m_IsActive;
}

bool CSoundEngineSLES::OpenStream(unsigned int devideID)
{
    int ret;

    AIDMA_Init(48000);

    m_RequestExitThread = false;

    ret = sem_init(&m_RenderRequestSemaphore, 0, 0);
    assert(ret >= 0);

    ret = pthread_create(&m_Thread, NULL, threadProcFunc, NULL);
    assert(ret >= 0);

    m_IsActive = true;
}

void* CSoundEngineSLES::threadProcFunc(void*)
{
    CSoundEngineSLES* engine = reinterpret_cast<CSoundEngineSLES*>(CSoundEngineSLES::Instance());
    return engine->ThreadProc(NULL);
}


void CSoundEngineSLES::RenderAudioFrames(s16 *dst, s32 numFrames)
{
#if defined(ENABLE_MEASURE_RENDERING_TIME)
    const s64 beginTimeMs = GetCurrentTimeMs();
    const s32 originalNumFrames = numFrames;
#endif
    for(; numFrames > 0; --numFrames)
    {
        WaveOutCallback(CHANNEL_COUNT, dst, WAVE_BUF_SAMPLES, SAMPLE_RATE);
        dst += (WAVE_BUF_SAMPLES * CHANNEL_COUNT);
    }
#if defined(ENABLE_MEASURE_RENDERING_TIME)
    __android_log_print(ANDROID_LOG_INFO, "CSoundEngineSLES", "[%s:%05d] renderd[ms] (%d), elapsed[ms] (%lld)", __func__, __LINE__, 3 * originalNumFrames, GetCurrentTimeMs() - beginTimeMs);
#endif
}

void* CSoundEngineSLES::ThreadProc(void*)
{
    const int numBuffers = AIDMA_GetBufferQueueCapacity();
    const int numCh = CHANNEL_COUNT;
    // 1 AudioFrame = 3 ms
    const s32 numRenderFrames = 32;
    const s32 bufferSize = WAVE_BUF_SAMPLES * numRenderFrames * numCh * sizeof(s16);

    s16* buffers = static_cast<s16*>(malloc(bufferSize * numBuffers));
    assert(buffers);

    s16* currentBuffer = buffers;
    u32 numBufferEnqueued = 0;


    // リアルタイムスレッドは権限上作成できない
    // Android のオーディオスレッド(AudioTrack) の nice 値が -19 のため
    nice(-19);

    // レンダラスレッドの処理ループ
    while(!m_RequestExitThread)
    {
        // バッファキューが空いた通知を待機
        // 備考）バッファリング中はリクエストを待たない
        if(numBufferEnqueued >= numBuffers && !WaitForRequestRender(1))
        {
            continue;
        }

        // レンダリング (ステレオ)
        RenderAudioFrames(currentBuffer, numRenderFrames);

        // OpenSL ES へ転送
        AIDMA_ExecuteDMA(reinterpret_cast<u8*>(currentBuffer), bufferSize);
        ++numBufferEnqueued;

        currentBuffer = buffers + WAVE_BUF_SAMPLES * numRenderFrames * numCh * (numBufferEnqueued % numBuffers);
    }

    free(buffers);
    buffers = NULL;
}

void CSoundEngineSLES::CloseStream()
{
    int ret;

    m_RequestExitThread = true;

    ret = pthread_join(m_Thread, NULL);
    assert(ret >= 0);

    AIDMA_Quit();

    ret = sem_destroy(&m_RenderRequestSemaphore);
    assert(ret >= 0);

    m_IsActive = false;
}

// レンダリングを要求する
void CSoundEngineSLES::RequestRender()
{
    int ret;
    ret = sem_post(&m_RenderRequestSemaphore);
    assert(ret >= 0);
}

// レンダリングを要求されるまで待つ (タイムアウト付)
bool CSoundEngineSLES::WaitForRequestRender(s32 timeoutSecond)
{
    int ret;
    timeval tv;

    ret = gettimeofday(&tv, NULL);

    timespec timeout = { tv.tv_sec + timeoutSecond, tv.tv_usec * 1000};
    ret = sem_timedwait(&m_RenderRequestSemaphore, &timeout);
    assert(ret >= 0 || errno == ETIMEDOUT);

    return ret >= 0;
}
        } // namespace winext
    } // namespace internal
} // namespace nw
