﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/types.h>
#include <nw/ut/ut_FileStream.h>
#include <nw/mcs/mcs_McsFileStream.h>

namespace nw {
namespace mcs {


#if defined(NW_MCS_ENABLE)

NW_COMPILER_ASSERT( McsFileStream::ERR_LAST == -FILEIO_ERROR_LAST );

//---------------------------------------------------------------------------
void
McsFileStream::Initialize_()
{
    m_CanRead           = false;
    m_CanWrite          = false;
    m_CloseEnableFlg    = false;
    m_CloseOnDestroyFlg = false;
    m_Available         = false;
}


//---------------------------------------------------------------------------
/* ctor */
McsFileStream::McsFileStream()
{
    this->Initialize_();
}



//---------------------------------------------------------------------------
/* ctor */
McsFileStream::McsFileStream( const char* fileName, u32 openFlag )
{
    this->Initialize_();

    (void)this->Open( fileName, openFlag );
}


//---------------------------------------------------------------------------
/* ctor */
McsFileStream::McsFileStream( const FileInfo* openedFile, u32 openFlag, bool closeEnable /* = true */  )
{
    this->Initialize_();

    (void)this->Open( openedFile, openFlag, closeEnable );
}


//---------------------------------------------------------------------------
/* dtor */
McsFileStream::~McsFileStream()
{
    // Fileが既にオープンされていてm_CloseEnableが有効な場合にファイルをクローズします。
    if ( m_CloseOnDestroyFlg )
    {
        this->Close();
    }
}


//---------------------------------------------------------------------------
bool
McsFileStream::Open( const char* fileName, u32 openFlag )
{
    // Fileが既にオープンされていてm_CloseEnableが有効な場合にファイルをクローズします。
    if ( m_CloseOnDestroyFlg )
    {
        this->Close();
    }

    m_CanRead  = ( openFlag & mcs::FILEIO_FLAG_READ  )? true : false;
    m_CanWrite = ( openFlag & mcs::FILEIO_FLAG_WRITE )? true : false;
    // クラス内部でオープンしたのでデストラクタでクローズする
    m_CloseEnableFlg    = true;
    m_CloseOnDestroyFlg = true;
    m_Available         = false;

    if ( FileIO_Open( &m_FileInfo, fileName, openFlag ) == FILEIO_ERROR_SUCCESS )
    {
        m_FilePosition.SetFileSize( mcs::FileIO_GetOpenFileSize( &m_FileInfo ) );
        m_Available = true;
        return true;
    }
    return false;
}


//---------------------------------------------------------------------------
bool
McsFileStream::Open( const FileInfo* openedFile,  u32 openFlag, bool closeEnable /* = false */ )
{
    // Fileが既にオープンされていてm_CloseEnableが有効な場合にファイルをクローズします。
    if ( m_CloseOnDestroyFlg )
    {
        this->Close();
    }

    m_CanRead =  ( openFlag & mcs::FILEIO_FLAG_READ  )? true : false;
    m_CanWrite = ( openFlag & mcs::FILEIO_FLAG_WRITE )? true : false;
    // 外部でオープンされたファイルなのでデストラクタやClose関数で
    // クローズするかどうかをユーザが指定可能です。
    m_CloseEnableFlg    = closeEnable;
    m_CloseOnDestroyFlg = true;
    m_Available         = false;

    m_FileInfo = *openedFile;

    u32 result = FileIO_Seek( &m_FileInfo, 0, FILEIO_SEEK_BEGIN, NULL );

    if ( result == mcs::FILEIO_ERROR_SUCCESS )
    {
        m_FilePosition.SetFileSize( mcs::FileIO_GetOpenFileSize( &m_FileInfo ) );
        m_Available = true;
        return true;
    }
    return false;
}


//---------------------------------------------------------------------------
void
McsFileStream::Close()
{
    if ( m_CloseEnableFlg && m_Available )
    {
        (void)FileIO_Close( &m_FileInfo );
        m_Available = false;
    }
}


//---------------------------------------------------------------------------
s32
McsFileStream::Read( void* buf, u32 length )
{
    NW_ASSERT_NOT_NULL( buf );

    u32 readBytes = 0;
    u32 result = FileIO_Read( &m_FileInfo, buf, length, &readBytes );

    if ( result == FILEIO_ERROR_SUCCESS )
    {
        (void)m_FilePosition.Skip( s32(length) );
        return s32(readBytes);
    }

    NW_ASSERT( s32(result) > 0 );
    return - s32( result );
}


//---------------------------------------------------------------------------
s32
McsFileStream::Write( const void* buf, u32 length )
{
    NW_ASSERT_NOT_NULL( buf );

    u32 result = FileIO_Write( &m_FileInfo, buf, length );

    if ( result == FILEIO_ERROR_SUCCESS )
    {
        (void)m_FilePosition.Append( s32(length) );
        return s32(length);
    }

    NW_ASSERT( s32(result) > 0 );
    return - s32( result );
}


//---------------------------------------------------------------------------
bool
McsFileStream::Seek( s32 offset, u32 origin )
{
    static const u32 ORIGIN[] =
    {
        FILEIO_SEEK_BEGIN,
        FILEIO_SEEK_CURRENT,
        FILEIO_SEEK_END
    };

    u32 result = FileIO_Seek( &m_FileInfo, offset, ORIGIN[ origin ], NULL );

    if( result != mcs::FILEIO_ERROR_SUCCESS )
    {
        NW_WARNING( result == mcs::FILEIO_ERROR_SUCCESS, "mcs::FileIO_Seek is Failed: result %d offset %d origin %d", result, offset, origin );
        return false;
    }

    m_FilePosition.Seek( offset, origin );

    return true;
}


#endif  // #if defined(NW_MCS_ENABLE)

}   /* namespace mcs  */
}   /* namespace nw */

