﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/types.h>
#include <nw/mcs/mcs_Hio2RingBuffer.h>

#if defined(NW_MCS_ENABLE)

#if defined(MCS_SERVER)

    #include <windows.h>
    #include <stdio.h>
    #include <assert.h>

    #define NW_ASSERT             assert
    #define NW_WARNING(exp, msg)  ((void) 0)

    #include <revolution/hio2.h>

  #ifndef   NW_PC_VIEWER
    #define MCS_HIO2READ            mspfHIO2Read
    #define MCS_HIO2WRITE           mspfHIO2Write
    #define MCS_HIO2GetLastError    mspfHIO2GetLastError
  #else
  //PC_COMMENT HIO関数の差し替え
    #define MCS_HIO2READ             HIO2Read
    #define MCS_HIO2WRITE            HIO2Write
    #define MCS_HIO2GetLastError     HIO2GetLastError
  #endif

#else   // #if defined(MCS_SERVER)

#include <string.h>
#include <stdarg.h>

#include <nw/mcs/hio2.h>
#include <nw/types.h>

#define MCS_HIO2READ            HIO2Read
#define MCS_HIO2WRITE           HIO2Write
#define MCS_HIO2GetLastError    HIO2GetLastError

#endif  // #if defined(MCS_SERVER)

#include <nw/mcs/mcs_Hio2RingBuffer.h>
#include <nw/mcs/mcs_Common.h>

const u32   CheckCode       = ('M' << 24) + ('C' << 16) + ('H' << 8) + 'I';     // バッファの先頭に書き込むチェックコード(起動チェックにも使用)

namespace
{

struct DataPointer
{
    u32         pointer;
    u8          reserved[24];
    u32         pointerCopy;
};


struct BufferHeader
{
    u32         signature;
    u8          reserved[28];

    DataPointer read;
    DataPointer write;
};

struct MessageHeader
{
    u32     channel;
    u32     size;
    u32     reserve1;
    u32     reserve2;
};

const u16   ReadInfoOffset  = 0x20;         // 読み込みポイントが入っている場所のOffset
const u16   WriteInfoOffset = 0x40;         // 書き込みポイントが入っている場所のOffset
const u16   HeaderSize      = sizeof(BufferHeader);     // Header : 全ヘッダのサイズ

const int   UpdateReadWritePointRetryCountMax   = 30;   // m_ReadPoint, m_WritePoint を更新する際のリトライ回数

/* ------------------------------------------------------------------------
        デバッグメッセージ出力
   ------------------------------------------------------------------------ */

#if defined(ENABLE_MCS_REPORT)

    #if defined(WIN32)

        void
        McsReport(const char* format, ...)
        {
            static HANDLE shConsole = NULL;

            va_list vargs;
            va_start(vargs, format);

            char tString[256];
            s32 tStringNum = _vsnprintf(tString, 256, format, vargs);

            if (shConsole == NULL)
            {
                AllocConsole();
                shConsole = CreateFile("CONOUT$", GENERIC_READ | GENERIC_WRITE, 0, NULL, OPEN_EXISTING, 0, NULL);
            }
            DWORD   tStringDoneNum;
            WriteConsole( shConsole , tString , tStringNum , &tStringDoneNum , NULL );

            va_end(vargs);
        }

        #define MCS_REPORT0(str)                    McsReport(str)
        #define MCS_REPORT1(fmt, arg1)              McsReport(fmt, arg1)
        #define MCS_REPORT2(fmt, arg1, arg2)        McsReport(fmt, arg1, arg2)
        #define MCS_REPORT3(fmt, arg1, arg2, arg3)  McsReport(fmt, arg1, arg2, arg3)

    #else   // #if defined(WIN32)

        #define MCS_REPORT0(str)                    OSReport(str)
        #define MCS_REPORT1(fmt, arg1)              OSReport(fmt, arg1)
        #define MCS_REPORT2(fmt, arg1, arg2)        OSReport(fmt, arg1, arg2)
        #define MCS_REPORT3(fmt, arg1, arg2, arg3)  OSReport(fmt, arg1, arg2, arg3)

    #endif  // #if defined(WIN32)

#else   // #if defined(ENABLE_MCS_REPORT)

    #define MCS_REPORT0(str)                    ((void) 0)
    #define MCS_REPORT1(fmt, arg1)              ((void) 0)
    #define MCS_REPORT2(fmt, arg1, arg2)        ((void) 0)
    #define MCS_REPORT3(fmt, arg1, arg2, arg3)  ((void) 0)

#endif  // #if defined(ENABLE_MCS_REPORT)

u32
HostToNet(u32 val)
{
    #if defined(WIN32)
        return htonl(val);
    #else
        return val;
    #endif
}

u32
NetToHost(u32 val)
{
    #if defined(WIN32)
        return ntohl(val);
    #else
        return val;
    #endif
}

void
DataPointer_Init(DataPointer* pPt, u32 pointer)
{
    pPt->pointer = HostToNet(pointer);
    (void)memset(pPt->reserved, 0, sizeof(pPt->reserved));
    pPt->pointerCopy = pPt->pointer;
}


#if !defined(WIN32)
    // const void*で通るようにする。
    void    DCFlushRange(const void* addr, u32 nBytes)
    {
        ::DCFlushRange(const_cast<void*>(addr), nBytes);
    }
#endif

}   // namespace

namespace nw
{
namespace mcs
{
namespace internal
{

#if defined(MCS_SERVER)

    HIO2ReadType            Hio2RingBuffer::mspfHIO2Read            = 0;
    HIO2WriteType           Hio2RingBuffer::mspfHIO2Write           = 0;
    HIO2GetLastErrorType    Hio2RingBuffer::mspfHIO2GetLastError    = 0;

    void
    Hio2RingBuffer::SetHIO2FuncAddress(
        HIO2ReadType            readFunc,
        HIO2WriteType           writeFunc,
        HIO2GetLastErrorType    getLastErrorFunc
    )
    {
        mspfHIO2Read            = readFunc;
        mspfHIO2Write           = writeFunc;
        mspfHIO2GetLastError    = getLastErrorFunc;
    }

#endif // #if defined(MCS_SERVER)

//---------------------------------------------------------------------------
//! @brief       Hio2RingBufferオブジェクトを初期化します。
//!
//! @param[in]   baseOffset   共有メモリの開始オフセット。
//! @param[in]   size         共有メモリのサイズ
//! @param[in]   tempBuf      Hio2RingBufferオブジェクトが使用する作業メモリ
//!
//! @return      なし。
//---------------------------------------------------------------------------
void
Hio2RingBuffer::Init(
    HIO2Handle  devHandle,
    u32         baseOffset,
    u32         size,
    void*       tempBuf
)
{
    NW_ASSERT(RoundDown(baseOffset, CopyAlignment) == baseOffset);
    NW_ASSERT(RoundDown(size, CopyAlignment) == size);

    #if !defined(WIN32)
        NW_ASSERT(RoundDown(tempBuf, CopyAlignment) == tempBuf);
    #endif

    m_DevHandle = devHandle;
    m_BaseOffset = baseOffset;
    m_BufferSize = size - HeaderSize;
    m_TempBuf = tempBuf;

    m_ReadPoint = m_WritePoint = 0;

    m_EnablePort = false;
}

//---------------------------------------------------------------------------
//! @brief       共有メモリを初期化します。
//!
//! @return      なし。
//---------------------------------------------------------------------------
void
Hio2RingBuffer::InitBuffer()
{
    BufferHeader* bufHeader = static_cast<BufferHeader*>(m_TempBuf);
    bufHeader->signature = HostToNet(CheckCode);
    (void)memset(bufHeader->reserved, 0, sizeof(bufHeader->reserved));

    DataPointer_Init(&bufHeader->read, 0);
    DataPointer_Init(&bufHeader->write, 0);

    if (!WriteSection_(m_BaseOffset + 0, bufHeader, sizeof(*bufHeader)))
    {
        return;
    }

    m_ReadPoint = m_WritePoint = 0;

    MCS_REPORT2("DEBUG: Write to Offset:%05x , Size: %d\n" , 0, 0x60 );
}


//---------------------------------------------------------------------------
//! @brief       共有メモリからの読み込みを行います。
//!              MemoryBlockSizeの境界をまたいで読み込むことは出来ません。
//!
//! @param[in]   offset   共有メモリの読み込み開始アドレス。
//! @param[in]   data     読み込むデータを格納するバッファへのポインタ。
//! @param[in]   size     読み込みデータサイズ。
//!
//! @return      関数が成功すれば true、失敗すれば falseを返します。
//---------------------------------------------------------------------------
bool
Hio2RingBuffer::ReadSection_(
    u32     offset,
    void*   data,
    u32     size
)
{
    NW_ASSERT(RoundDown(offset, 4) == offset);  // 4バイトアライメントされていること

    #if !defined(WIN32)
        NW_ASSERT(RoundDown(data, 32) == data);     // 32バイトアライメントされていること
    #endif

    NW_ASSERT(RoundUp(size, 32) == size);       // 32バイトアライメントされていること

    MCS_REPORT3("DEBUG:      Hio2RingBuffer::ReadSection_(%06x, %08x, %d);\n", offset, data, size );

    if (!IsPort())
    {
        MCS_REPORT0( "Hio2RingBuffer::ReadSection_ --- IsPort() --- returns false.\n" );
        return false;
    }

    #if !defined(WIN32)
        DCInvalidateRange(data , size);
    #endif

    for (int retry_count = 0; retry_count < HIO2FuncRetryCountMax; ++retry_count)
    {
        if (MCS_HIO2READ(m_DevHandle, offset, data, s32(size)))
        {
            return true;
        }

        if (MCS_HIO2GetLastError() != HIO2_ERROR_INTERNAL)
        {
            break;
        }

        // HIO2_ERROR_INTERNAL のときは、数回リトライしてみる。
        SleepThread(HIO2FuncRetrySleepTime);
    }

    MCS_REPORT1("Hio2RingBuffer::ReadSection_ --- HIO2Read() error. - %d\n", MCS_HIO2GetLastError());
    DisablePort();
    return false;
}

//---------------------------------------------------------------------------
//! @brief       共有メモリへの書き込みを行います。
//!              MemoryBlockSizeの境界をまたいで書き込むことは出来ません。
//!
//! @param[in]   offset   共有メモリの書き込み開始アドレス。
//! @param[in]   data     書き込むデータを格納するバッファへのポインタ。
//! @param[in]   size     書き込みデータサイズ。
//!
//! @return      関数が成功すれば true、失敗すれば falseを返します。
//---------------------------------------------------------------------------
bool
Hio2RingBuffer::WriteSection_(
    u32         offset,
    const void* data,
    u32         size
)
{
    NW_ASSERT(RoundDown(offset, 4) == offset);  // 4バイトアライメントされていること

    #if !defined(WIN32)
        NW_ASSERT(RoundDown(data, CopyAlignment) == data);  // 32バイトアライメントされていること
    #endif

    NW_ASSERT(RoundUp(size, CopyAlignment) == size);        // 32バイトアライメントされていること

    MCS_REPORT3("DEBUG:      Hio2RingBuffer::WriteSection_(0x%06x, %08x, %d);\n", offset, data, size );

    if (!IsPort())
    {
        return false;
    }

    #if !defined(WIN32)
        DCFlushRange(data, size);
    #endif

    for (int retryCount = 0; retryCount < HIO2FuncRetryCountMax; ++retryCount)
    {
        if (MCS_HIO2WRITE(m_DevHandle, offset, data, s32(size)))
        {
            return true;
        }

        if (MCS_HIO2GetLastError() != HIO2_ERROR_INTERNAL)
        {
            break;
        }

        // HIO2_ERROR_INTERNAL のときは、数回リトライしてみる。
        SleepThread(HIO2FuncRetrySleepTime);
    }

    MCS_REPORT1("HIO2Write() error. - %d\n", MCS_HIO2GetLastError());
    DisablePort();
    return false;
}

//---------------------------------------------------------------------------
//! @brief       読み込み位置/書き込み位置を最新の状態に更新します。
//!
//! @return      関数が成功すれば true、失敗すれば falseを返します。
//---------------------------------------------------------------------------
bool
Hio2RingBuffer::UpdateReadWritePoint_()
{
    if (!UpdateReadPoint_())
    {
        return false;
    }
    return UpdateWritePoint_();
}

//---------------------------------------------------------------------------
//! @brief       読み込み位置を最新の状態に更新します。
//!
//! @return      関数が成功すれば true、失敗すれば falseを返します。
//---------------------------------------------------------------------------
bool
Hio2RingBuffer::UpdateReadPoint_()
{
    if (!IsPort())
    {
        return false;
    }

    DataPointer* dataPtr = static_cast<DataPointer*>(m_TempBuf);


    int retryCount = 0;
    for (; retryCount < UpdateReadWritePointRetryCountMax; ++retryCount)
    {
        if (!ReadSection_(m_BaseOffset + ReadInfoOffset, dataPtr, sizeof(*dataPtr)))
        {
            return false;
        }
        if (dataPtr->pointer == dataPtr->pointerCopy)
        {
            break;
        }
    }
    if (retryCount >= UpdateReadWritePointRetryCountMax)
    {
        MCS_REPORT0("ReadPoint read fail.\n");
        return false;
    }

    m_ReadPoint = NetToHost(dataPtr->pointer);
    if (RoundDown(m_ReadPoint, 32) != m_ReadPoint)
    {
        MCS_REPORT1("m_ReadPoint %08X\n", m_ReadPoint);
    }

    return true;
}

//---------------------------------------------------------------------------
//! @brief       書き込み位置を最新の状態に更新します。
//!
//! @return      関数が成功すれば true、失敗すれば falseを返します。
//---------------------------------------------------------------------------
bool
Hio2RingBuffer::UpdateWritePoint_()
{
    if (!IsPort())
    {
        return false;
    }

    DataPointer* dataPtr = static_cast<DataPointer*>(m_TempBuf);

    int retryCount = 0;
    for (; retryCount < UpdateReadWritePointRetryCountMax; ++retryCount)
    {
        if (!ReadSection_(m_BaseOffset + WriteInfoOffset, dataPtr, sizeof(*dataPtr)))
        {
            MCS_REPORT0( "Hio2RingBuffer::UpdateWritePoint_ --- ReadSection_() --- returns false.\n" );
            return false;
        }
        MCS_REPORT2( "Hio2RingBuffer::UpdateWritePoint_ (ptr, ptrCopy)=[%p, %p].\n", dataPtr->pointer, dataPtr->pointerCopy );
        if (dataPtr->pointer == dataPtr->pointerCopy)
        {
            break;
        }
    }
    if (retryCount >= UpdateReadWritePointRetryCountMax)
    {
        MCS_REPORT0("Hio2RingBuffer::UpdateWritePoint_ --- WritePoint read fail.\n");
        return false;
    }

    m_WritePoint = NetToHost(dataPtr->pointer);
    if (RoundDown(m_WritePoint, 32) != m_WritePoint)
    {
        MCS_REPORT1("m_WritePoint %08X\n", m_WritePoint);
    }

    return true;
}

//---------------------------------------------------------------------------
//! @brief       共有メモリからデータを読み込みます。
//!
//!              書き込みポインターの位置を取得して、
//!              データが書き込まれているようならば、
//!              読み込みを実行し、読み込み位置を更新する。
//!
//!
//!                     なし。
//!
//! @return      関数が成功すれば true、失敗すれば falseを返します。
//---------------------------------------------------------------------------
bool
Hio2RingBuffer::Read(bool* pbAvailable)
{
    NW_ASSERT(pbAvailable != 0);

    //-----------------------------------------------------
    // 書き込みポインターの位置を取得しします。
    if (! UpdateWritePoint_())
    {
        MCS_REPORT0( "UpdateWritePoint_() --- returns false." );
        return false;
    }

    //-----------------------------------------------------
    // データが書き込まれているようならば、
    // 読み込みを実行します
    *pbAvailable = m_ReadPoint != m_WritePoint;

    if (! *pbAvailable)
    {
        return true;
    }

    //-----------------------------------------------------
    // 一時バッファ内の処理したバイト数を初期化します。
    // GetMessage() 内で更新（増加）されます。
    //
    // (m_ProcBytesの初期化は本関数の終盤にあるm_ReadPointの更新と同一の
    // 箇所で行うほうがより安全ですが、とりあえずコードは改変しないでおきます）
    m_ProcBytes = 0;
    m_TransBytes = (m_WritePoint + m_BufferSize - m_ReadPoint) % m_BufferSize;



    //-----------------------------------------------------
    // 共有メモリの内容を一時バッファ内にコピーします。
    u32 currReadPoint = m_ReadPoint;
    u8 *const msgBuf = static_cast<u8*>(m_TempBuf) + HeaderSize;
    u32 tmpOfs = 0;

    for (u32 restBytes = m_TransBytes; restBytes > 0;)
    {
        const u32 readBytes = m_WritePoint < currReadPoint ? m_BufferSize - currReadPoint: restBytes;
        if (!ReadSection_(m_BaseOffset + HeaderSize + currReadPoint, msgBuf + tmpOfs, readBytes))
        {
            MCS_REPORT0( "Hio2RingBuffer::Read() --- ReadSection_() --- returns false.\n" );
            return false;
        }

        currReadPoint = (currReadPoint + readBytes) % m_BufferSize;
        tmpOfs += readBytes;
        restBytes -= readBytes;
    }

    NW_ASSERT(currReadPoint == m_WritePoint);


    //-----------------------------------------------------
    // 読み込みが終わったので、
    // 読み込み位置の値を更新します。
    // 共有メモリ上とクラスメンバ側の両方を更新しています。
    //-----------------------------------------------------
    DataPointer* dataPtr = static_cast<DataPointer*>(m_TempBuf);
    DataPointer_Init(dataPtr, currReadPoint);
    if (!WriteSection_(m_BaseOffset + ReadInfoOffset, dataPtr, sizeof(*dataPtr)))
    {
        MCS_REPORT0( "Hio2RingBuffer::Read() --- WriteSection_() --- returns false.\n" );
        return false;
    }

    m_ReadPoint = currReadPoint;
    MCS_REPORT1("DEBUG: Read: Point:%08x\n" , m_ReadPoint);

    return true;
}

//---------------------------------------------------------------------------
//! @brief       (一時バッファ m_TempBuf から)
//!              メッセージのデータを取得します。
//!              取得するたびに次のメッセージ位置へ移動します。
//!              取得できるメッセージが無いときは NULL を返します。
//!
//! @param[out]  pChannel     メッセージのチャンネル値を格納する変数への
//!                      ポインタ。
//! @param[out]  pTotalSize   メッセージの総サイズを格納する変数へのポインタ。
//!
//! @return      取得できるメッセージが無いときは NULL、
//!               そうでない場合は、メッセージデータへのポインタを返します。
//---------------------------------------------------------------------------
void*
Hio2RingBuffer::GetMessage(
    u32*    pChannel,
    u32*    pTotalSize
)
{
    //-----------------------------------------------------
    // 一時バッファの終端まで読んでいて、
    // 取得できるメッセージが無い場合は何もしません。
    if (m_ProcBytes >= m_TransBytes)
    {
        return NULL;
    }

    //-----------------------------------------------------
    // 一時バッファ上のメッセージ読み取り位置を計算し、
    // MessageHeaderとして読み取ります。
    u8 *const msgStart = static_cast<u8*>(m_TempBuf) + HeaderSize + m_ProcBytes;
    const MessageHeader *const pMsgHeader = reinterpret_cast<MessageHeader*>(msgStart);

    *pChannel = NetToHost(pMsgHeader->channel);
    *pTotalSize = NetToHost(pMsgHeader->size);

    //-----------------------------------------------------
    // メッセージ読み取り位置を更新します。
    m_ProcBytes += RoundUp(sizeof(*pMsgHeader) + *pTotalSize, CopyAlignment);
    return msgStart + sizeof(*pMsgHeader);
}

//---------------------------------------------------------------------------
//! @brief       書き込み可能なバイト数を取得します。
//!
//! @param[out]  pBytes      書き込み可能なバイト数を取得する変数へのポインタ。
//! @param[in]   withUpdate  trueならばアップデートを行います。
//!
//! @return      関数が成功すれば true、失敗すれば falseを返します。
//---------------------------------------------------------------------------
bool
Hio2RingBuffer::GetWritableBytes(u32* pBytes, bool withUpdate)
{
    if (withUpdate)
    {
        if (!UpdateReadPoint_())
        {
            return false;
        }
    }

    u32 writeEndPoint = (m_ReadPoint + m_BufferSize - CopyAlignment) % m_BufferSize;
    u32 writableBytes = (writeEndPoint + m_BufferSize - m_WritePoint) % m_BufferSize;

    *pBytes = writableBytes >= sizeof(MessageHeader) ? writableBytes - sizeof(MessageHeader): 0;
    return true;
}

//---------------------------------------------------------------------------
//! @brief       共有メモリへデータを書き込みます。
//!
//! @param[in]   channel   チャンネル値。
//! @param[in]   buf       書き込むデータ。
//! @param[in]   size      書き込むデータのサイズ。
//!
//! @return      関数が成功すれば true、失敗すれば falseを返します。
//---------------------------------------------------------------------------
bool
Hio2RingBuffer::Write(
    u32         channel,
    const void* buf,
    u32         size
)
{
    if (!IsPort())
    {
        return false;
    }

    //-----------------------------------------------------
    // 書き込み可能バイト数を計算します。
    {
        u32 writableBytes;
        // 更新なしで書き込み可能バイト数を取得
        if (! GetWritableBytes(&writableBytes, false))
        {
            return false;
        }

        if (size > writableBytes)
        {
            // 更新ありで書き込み可能バイト数を取得
            if (! GetWritableBytes(&writableBytes, true))
            {
                return false;
            }
            if (size > writableBytes)
            {
                // 書き込めるサイズを超えていたら書き込みを拒否します。
                // この関数を呼び出す側でサイズを調整しなければなりません。
                return false;
            }
        }
    }

    //-----------------------------------------------------
    // m_TempBufを書き込み前の一時バッファとして利用し、
    // まずここでデータを構築してから、それをHIOに書き込みます。
    u8 *const tempBuf = static_cast<u8*>(m_TempBuf);
    bool bOutMessageHeader = false;
    const u8* pSrc = static_cast<const u8*>(buf);
    const u32 writeEndPoint = (m_ReadPoint + m_BufferSize - CopyAlignment) % m_BufferSize;


    //------------------------------------
    // データを書き込みます。
    u32 currWritePoint = m_WritePoint;
    for (u32 restBytes = size; restBytes > 0;)
    {
        u32 tmpOfs = 0;
        // メッセージヘッダを含まないサイズ
        // バッファ終端境界は2つのメッセージに分割しています。
        u32 writableBytes = writeEndPoint >= currWritePoint ? writeEndPoint - currWritePoint: m_BufferSize - currWritePoint;

        //------------------------------------
        // 最初のループでは、メッセージヘッダーを生成します。
        if (! bOutMessageHeader)
        {
            MessageHeader *const pHeader = reinterpret_cast<MessageHeader*>(tempBuf + tmpOfs);
            pHeader->channel = HostToNet(channel);
            pHeader->size = HostToNet(size);
            pHeader->reserve1 = 0;
            pHeader->reserve2 = 0;

            tmpOfs += sizeof(MessageHeader);
            // NW_COMPILER_ASSERT(sizeof(MessageHeader) < CopyAlignment);
            writableBytes -= sizeof(MessageHeader);

            bOutMessageHeader = true;
        }

        //--------------------------------------
        // データをテンポラリバッファにコピーします。
        const u32 msgWriteBytes = GetMin(restBytes, writableBytes);
        (void)memcpy(tempBuf + tmpOfs, pSrc, msgWriteBytes);
        tmpOfs += msgWriteBytes;
        pSrc += msgWriteBytes;
        restBytes -= msgWriteBytes;

        //--------------------------------------
        // 全てのデータの転送が完了したら...
        if (restBytes == 0)
        {
            // データサイズ・アライメント調整のためのデータを生成します。
            const u32 oldTempOfs = tmpOfs;
            tmpOfs = RoundUp(tmpOfs, CopyAlignment);

            if (tmpOfs > oldTempOfs)
            {
                (void)memset(tempBuf + oldTempOfs, 0, tmpOfs - oldTempOfs);
            }
        }

        //--------------------------------------
        // HIO2共有メモリにデータを書き込みます。
        if (!WriteSection_(m_BaseOffset + HeaderSize + currWritePoint, tempBuf, tmpOfs))
        {
            return false;
        }
        currWritePoint = (currWritePoint + tmpOfs) % m_BufferSize;
    }

    //-----------------------------------------------------
    // 書き込みが終わったので、
    // 書き込み位置の値を更新します。
    // 共有メモリ上とクラスメンバ側の両方を更新しています。
    //-----------------------------------------------------
    DataPointer* dataPtr = static_cast<DataPointer*>(m_TempBuf);
    DataPointer_Init(dataPtr, currWritePoint);

    MCS_REPORT1("DEBUG: EndWrite: Point:%08x\n" , currWritePoint);

    if (!WriteSection_(m_BaseOffset + WriteInfoOffset, dataPtr, sizeof(*dataPtr)))
    {
        return false;
    }

    m_WritePoint = currWritePoint;
    return true;
}

}   // namespace internal
}   // namespace mcs
}   // namespace nw

#endif  // #if defined(NW_MCS_ENABLE)
