﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/types.h>
#include <cafe/hio.h>

#if defined(NW_MCS_ENABLE)

#include <nw/mcs/mcs_CafeHioCommDevice.h>
#include <nw/mcs/mcs_HioNegotiate.h>
#include <nw/mcs/mcs_Common.h>
#include <nw/mcs/mcs_SimpleRingBuffer.h>
#include <nw/ut/ut_ScopedLock.h>
#include <nw/ut/ut_Inlines.h>

#include <nw/types.h>

/* ========================================================================
    定数
   ======================================================================== */

namespace {

using namespace nw::mcs::internal;

enum
{
    NEGOPHASE_NOATTACH,     //!< HIOOpen 前の未接続状態
    NEGOPHASE_ATTACHED,     //!< HIOOpen 実行後の接続待ち状態
    NEGOPHASE_REBOOT,       //!< Rebootメッセージの送信済み状態
    NEGOPHASE_CONNECT,      //!< 接続済み状態
    NEGOPHASE_ERROR         //!< Open関数成功後にエラーが発生した
};

const char CHANNEL_NAME[] = "NW_MCS";

// OSDisableInterrupts が公開されておらず、ReadAsync のコールバックが割り込みから
// 呼ばれる為、Mcs用にスレッドを用意する。
#define MCS_MAX_MESSAGE     32
#define MCS_THREAD_PRIORITY 16 // TODO: HIO よりも1つ落としている。ユーザから指定できるようにする。

static OSThread         s_McsThread;
static OSMessageQueue   s_McsMsgQ;
static OSMessage        s_Msgs[MCS_MAX_MESSAGE];
static u8               s_McsStack[4096];
static OSCond           s_McsCond;



enum
{
    HEADER_COMMAND_UNKNOWN = 0,
    HEADER_COMMAND_REGISTNAME,
    HEADER_COMMAND_REBOOT,
    HEADER_COMMAND_INITBUFFER,
    HEADER_COMMAND_ACK,
    HEADER_COMMAND_DISCONNECT,
    HEADER_COMMAND_MESSAGETOSERVER,
    HEADER_COMMAND_MESSAGETOAPPICATION,
    HEADER_COMMAND_SERVERTIME,
    HEADER_COMMAND_CHECKALIVE,
};

struct ChunkHeader
{
    u32     command;
    u32     chunkSize;
};

struct MessageHeader
{
    u32     command;
    u32     chunkSize;
    u32     channel;
    u32     size;
};

struct ServerTimeHeader
{
    u32     command;
    u32     chunkSize;
    s64     msec;
};

enum
{
    MCS_MESSAGE_READ,
    MCS_MESSAGE_WRITE = 1,
    MCS_MESSAGE_CLOSE = 2
};

//---------------------------------------------------------------------------
//! @brief        スレッドにメッセージを送信します。
//!
//! @param[in]    command コマンドを指定します。
//! @param[in]    device  HIOCommDevice へのポインタです。
//! @param[in]    status  結果のステータスです。
//---------------------------------------------------------------------------
static void
McsSendMessage(u8 command, HIOCommDevice* device, HIOStatus status)
{
    OSMessage msg;
    msg.message = device;
    msg.data0   = static_cast<u32>(command | (status << 8));

    OSSendMessage(&s_McsMsgQ, &msg,    OS_MESSAGE_BLOCK);
}

static bool
McsRecvMessage(u8* command, HIOCommDevice** device, HIOStatus* status)
{
    OSMessage msg;
    BOOL result;
    result =  OSReceiveMessage(&s_McsMsgQ, &msg, OS_MESSAGE_BLOCK);

    if (command)
    {
        *command = static_cast<u8>(msg.data0 & 0xff);
    }

    if (device)
    {
        *device = reinterpret_cast<HIOCommDevice*>(msg.message);
    }

    if (status)
    {
        *status = static_cast<HIOStatus>(msg.data0) >> 8;
    }

    return (result != FALSE);
}

}   // namespace

namespace nw  {
namespace mcs {
namespace internal {

//---------------------------------------------------------------------------
//! @brief        Mcsのデータリード/ライト処理用のスレッドです。
//---------------------------------------------------------------------------
/* static */ int
HIOCommDevice::McsThreadProc(int argc, void *ptrArg)
{
    NW_UNUSED_VARIABLE(argc);
    NW_UNUSED_VARIABLE(ptrArg);

    HIOCommDevice* device;
    u8             command;
    HIOStatus      status;

    NW_MCS_LOG("Start thread\n");

    while ( McsRecvMessage(&command, &device, &status) )
    {
        NW_MCS_LOG("Receive message %d %p %d\n", command, device, status);

        switch (command)
        {
        case MCS_MESSAGE_READ:
            device->OnRead(status);
            break;
        case MCS_MESSAGE_WRITE:
            device->OnWrite(status);
            break;
        case MCS_MESSAGE_CLOSE:
            OSExitThread(NULL);
        }
    }

    return 0;
}

//---------------------------------------------------------------------------
//! @brief        HIO ホストとの接続コールバックです。
//!
//! @param[in]    status  接続ステータスです。
//!               HIO_STATUS_NEW_CONNECTION or HIO_STATUS_NO_CONNECTIONS が帰ります。
//! @param[in]    context HIOOpen で指定したユーザパラメータです。HICommDevice のインスタンスを渡します。
//---------------------------------------------------------------------------
/* static */ void
HIOCommDevice::ConnectionCallback(HIOStatus status, void* context)
{
    HIOCommDevice* device = static_cast<HIOCommDevice*>( context );

    switch (status)
    {
    case HIO_STATUS_NEW_CONNECTION:
        NW_MCS_LOG("New Connection\n");
        device->OnConnect();
        break;

    case HIO_STATUS_NO_CONNECTIONS:
        NW_MCS_LOG("No Connection\n");
        device->OnDisConnect();
        break;

    default:
        NW_FATAL_ERROR("unknown status");
    }
}

//---------------------------------------------------------------------------
//! @brief        Mcs からのリードコールバック
//!
//! @param[in]    status
//! @param[out]   context HIOReadAsync で指定したユーザパラメータです。HICommDevice のインスタンスを渡します。
//---------------------------------------------------------------------------
/* static */ void
HIOCommDevice::ReadCallback(HIOStatus status, void* context)
{
    NW_MCS_LOG("ReadCallback %d %p\n", status, context);

    HIOCommDevice* device = static_cast<HIOCommDevice*>( context );

    McsSendMessage(MCS_MESSAGE_READ, device, status);
}



//---------------------------------------------------------------------------
//! @brief        Mcs へのライトコールバック
//!
//! @param[in]    status
//! @param[out]   context HIOWriteAsync で指定したユーザパラメータです。HICommDevice のインスタンスを渡します。
//---------------------------------------------------------------------------
/* static */ void
HIOCommDevice::WriteCallback(HIOStatus status, void* context)
{
    NW_MCS_LOG("WriteCallback\n");

    HIOCommDevice* device = static_cast<HIOCommDevice*>( context );

    McsSendMessage(MCS_MESSAGE_WRITE, device, status);
}


//---------------------------------------------------------------------------
void
HIOCommDevice::OnConnect()
{
    ut::ScopedLock<ut::Mutex> lockObj(m_Lock);

    // リブートメッセージを送信。
    ChunkHeader rebootChunk = { internal::HostToNet((u32)HEADER_COMMAND_REBOOT), 0 };
    m_SharedWriteBuffer.Write(&rebootChunk, sizeof(rebootChunk));
    this->StartWritingBuffer();

    m_NegoPhase = NEGOPHASE_REBOOT;
    m_EnableReadChain = true;

    this->StartReadingBuffer();

    // 状態の変化を通知
    OSSignalCond(&s_McsCond);
}

//---------------------------------------------------------------------------
void
HIOCommDevice::OnDisConnect()
{
    ut::ScopedLock<ut::Mutex> lockObj(m_Lock);
    m_NegoPhase = NEGOPHASE_ATTACHED;

    this->StopReadingBuffer();

    // 状態の変化を通知
    OSSignalCond(&s_McsCond);
}

//---------------------------------------------------------------------------
void
HIOCommDevice::OnRead(HIOStatus status)
{
    // 割り込み中で呼ばれるとロックができない
    ut::ScopedLock<ut::Mutex> lockObj(m_Lock);

    NW_MCS_LOG("OnRead %d\n", status);

    if (status > 0)
    {
        m_SharedReadBuffer.MoveTail(static_cast<u32>(status));

        const u8* data = static_cast<const u8*>(this->m_SharedReadBuffer.GetReadableData());
        u32 readable = m_SharedReadBuffer.GetReadableSize();

        u32 offset = 0;
        while (offset + sizeof(ChunkHeader) <= readable)
        {
            const ChunkHeader* chunk =
                reinterpret_cast<const ChunkHeader*>(data + offset);

            u32 chunkSize = internal::NetToHost(chunk->chunkSize);

            if (readable < offset + sizeof(ChunkHeader) + chunkSize)
            {
                break;
            }

            offset += sizeof(ChunkHeader) + chunkSize;

            switch (internal::NetToHost(chunk->command))
            {
            case HEADER_COMMAND_MESSAGETOAPPICATION:
                {
                    const MessageHeader* header =
                        reinterpret_cast<const MessageHeader*>(chunk);
                    const u8* body =
                        reinterpret_cast<const u8*>(header + 1);

                    u32 channel = internal::NetToHost(header->channel);
                    u32 size    = internal::NetToHost(header->size);

                    NW_MCS_LOG("channel = %08x\n", channel);
                    NW_MCS_LOG("size = %08x\n", size);
                    NW_MCS_LOG("data = %08x\n", *reinterpret_cast<const u32*>(body));

                    ChannelInfo* channelInfo = this->GetChannelInfo(channel);
                    if (channelInfo != NULL)
                    {
                        while (0 < size)
                        {
                            u32 writeSize = channelInfo->recvBuf.Write(body, size);
                            body += writeSize;
                            size -= writeSize;

                            if (writeSize == 0)
                            {
                                NW_MCS_LOG("Channel %d recv buff full.\n", channel);
                                m_Lock.WaitCond(s_McsCond);
                            }
                        }
                    }
                    else
                    {
                        // 登録されたチャンネルが無い場合は破棄する。
                    }
                }
                break;

            case HEADER_COMMAND_DISCONNECT:
                {
                    ChunkHeader rebootChunk = { internal::HostToNet((u32)HEADER_COMMAND_REBOOT), 0 };
                    m_SharedWriteBuffer.Write(&rebootChunk, sizeof(rebootChunk));

                    this->StartWritingBuffer();

                    m_NegoPhase = NEGOPHASE_REBOOT;
                }
                break;

            case HEADER_COMMAND_INITBUFFER:
                {
                    m_InitBufferPending = true;
                }
                break;

            case HEADER_COMMAND_SERVERTIME:
                {
                    const ServerTimeHeader* header =
                        reinterpret_cast<const ServerTimeHeader*>(chunk);

                    s64 msec = internal::NetToHost(header->msec);
                    this->SetServerTime(msec);
                }
                break;

            case HEADER_COMMAND_CHECKALIVE:
                {
                    m_CheckAlivePending = true;
                }
                break;


            default:
                break;
            }
        }

        m_SharedReadBuffer.MovePtr(offset);
        m_SharedReadBuffer.Compaction();
    }

    this->m_IsReading = false;

    // 状態の変化を通知
    OSSignalCond(&s_McsCond);

    if ( this->m_EnableReadChain )
    {
        this->StartReadingBuffer();
    }
}

//---------------------------------------------------------------------------
void
HIOCommDevice::OnWrite(HIOStatus status)
{
    NW_MCS_LOG("OnWrite %d\n", status);

    ut::ScopedLock<ut::Mutex> lockObj(m_Lock);

    this->m_IsWriting = false;

    // 状態の変化を通知
    OSSignalCond(&s_McsCond);

    if (this->m_SharedWriteBuffer.IsEmpty())
    {
        this->m_SharedWriteBuffer.Clear();
    }
    else
    {
        this->StartWritingBuffer();
    }
}

//---------------------------------------------------------------------------
// cafe の HIO にはデバイスの列挙が存在しないので、常に DEVICETYPE_CATDEV を返しておきます。
//---------------------------------------------------------------------------
/* static */ bool
HIOCommDevice::EnumerateDevice(IHIOCommDeviceEnumerate* pEnumerate)
{
    pEnumerate->Find( DEVICETYPE_CATDEV );

    return true;
}

//---------------------------------------------------------------------------
/* ctor */
HIOCommDevice::HIOCommDevice(
    void*           tempBuf,
    u32             tempBufSize
)
 :  m_ChannelName( CHANNEL_NAME )   // TODO: ひとまず mcs で使用するチャンネル名は固定。
 ,  m_HioHandle( HIO_HANDLE_INVALID )
 ,  m_IsReadInitCodeMail( false )
 ,  m_IsOpen( false )
 ,  m_IsWriting( false )
 ,  m_IsReading( false )
 ,  m_EnableReadChain( false )
 ,  m_InitBufferPending( false )
 ,  m_CheckAlivePending( false )
 ,  m_NegoPhase( NEGOPHASE_NOATTACH )
 ,  m_ServerTime( 0 )
 ,  m_ClientTime( 0 )
{
    NW_ASSERT(tempBufSize >= COPY_ALIGNMENT + TEMP_BUF_SIZE_MIN);

    m_TempBuf     = RoundUp(tempBuf, COPY_ALIGNMENT);
    m_TempBufSize = RoundDown(reinterpret_cast<u32>(tempBuf) + tempBufSize, COPY_ALIGNMENT) - reinterpret_cast<u32>(m_TempBuf);

    this->InitHioRingBuffer();

    m_Lock.Initialize();
}

//---------------------------------------------------------------------------
/* dtor */
HIOCommDevice::~HIOCommDevice()
{
}

//---------------------------------------------------------------------------
void
HIOCommDevice::RegisterBuffer(
    ChannelType channel,
    void*       buf,
    u32         bufSize
)
{
    ut::ScopedLock<ut::Mutex> lockObj(m_Lock);

    NW_ASSERT(NULL == GetChannelInfo(channel));   // 既に同じチャンネルを使用しているものが無いこと。

    u32 start = reinterpret_cast<u32>(buf);
    u32 end = start + bufSize;

    start = RoundUp(start, 4);
    end = RoundDown(end, 4);

    NW_ASSERT(start < end);
    NW_ASSERT(end - start >= sizeof(ChannelInfo) + sizeof(SimpleRingBufferHeader) + sizeof(u32) + 4 + 4);

    ChannelInfo* pInfo = new(reinterpret_cast<void*>(start))ChannelInfo();

    pInfo->channel  = channel;
    const u32 ringBufStart = start + sizeof(ChannelInfo);
    pInfo->recvBuf.Init(reinterpret_cast<void*>(ringBufStart), end - ringBufStart);
    pInfo->allocatedBuffer = buf;

    m_ChannelInfoList.push_back(pInfo);
}

//---------------------------------------------------------------------------
void*
HIOCommDevice::UnregisterBuffer(ChannelType channel)
{
    ut::ScopedLock<ut::Mutex> lockObj(m_Lock);

    ChannelInfo* pInfo = GetChannelInfo(channel);
    NW_ASSERT(pInfo);

    m_ChannelInfoList.erase(pInfo);

    return (pInfo) ? pInfo->allocatedBuffer : NULL;
}

//---------------------------------------------------------------------------
void*
HIOCommDevice::GetRegisteredBuffer(ChannelType channel)
{
    ut::ScopedLock<ut::Mutex> lockObj(m_Lock);

    ChannelInfo* pInfo = GetChannelInfo(channel);

    return (pInfo) ? pInfo->allocatedBuffer : NULL;
}

//---------------------------------------------------------------------------
// HIOデバイスをオープンします。
u32
HIOCommDevice::Open()
{
    ut::ScopedLock<ut::Mutex> lockObj(m_Lock);

    if (m_IsOpen)
    {
        return MCS_ERROR_SUCCESS;
    }

    m_IsOpen = true;
    m_InitBufferPending = false;
    m_CheckAlivePending = false;

    if (m_NegoPhase == NEGOPHASE_NOATTACH)
    {
        m_HioHandle = HIOOpen(m_ChannelName, ConnectionCallback, this, HIO_CHANNEL_OPTION_READ_WRITE, 0);

        NW_MCS_LOG("HIOOpen result=%d %p\n", m_HioHandle, this);

        if (m_HioHandle < 0)
        {
            return MCS_ERROR_COMERROR;
        }

        m_NegoPhase = NEGOPHASE_ATTACHED;
    }

    OSInitMessageQueue(&s_McsMsgQ, s_Msgs, MCS_MAX_MESSAGE);

    OSInitCond(&s_McsCond);

    OSCreateThread(&s_McsThread, HIOCommDevice::McsThreadProc, 0, NULL,
                   s_McsStack + sizeof (s_McsStack),
                   sizeof (s_McsStack), MCS_THREAD_PRIORITY, 0);

    OSResumeThread(&s_McsThread);

    return MCS_ERROR_SUCCESS;
}

//---------------------------------------------------------------------------
void
HIOCommDevice::Close()
{
    {
        ut::ScopedLock<ut::Mutex> lockObj(m_Lock);

        if (! m_IsOpen)
        {
            return;
        }

        HIOClose(m_HioHandle);
        m_HioHandle = HIO_HANDLE_INVALID;

        m_IsOpen = false;

        m_NegoPhase = NEGOPHASE_NOATTACH;

        this->SetServerTime(0);
    }

    // スレッドの後始末
    McsSendMessage(MCS_MESSAGE_CLOSE, this, HIO_STATUS_OK);
    OSJoinThread(&s_McsThread, NULL);

    return;
}

//---------------------------------------------------------------------------
/* private */ void
HIOCommDevice::CheckInitBuffer()
{
    ut::ScopedLock<ut::Mutex> lockObj(m_Lock);

    if (m_InitBufferPending)
    {
        m_InitBufferPending = false;

        ChunkHeader chunk = { internal::HostToNet((u32)HEADER_COMMAND_ACK), 0 };
        m_SharedWriteBuffer.Write(&chunk, sizeof(chunk));
        this->StartWritingBuffer();

        this->InitRegisteredBuffer();

        m_NegoPhase = NEGOPHASE_CONNECT;

        // 状態の変化を通知
        OSSignalCond(&s_McsCond);
    }
}

//---------------------------------------------------------------------------
/* private */ void
HIOCommDevice::CheckAlive()
{
    ut::ScopedLock<ut::Mutex> lockObj(m_Lock);

    if (m_CheckAlivePending)
    {
        m_CheckAlivePending = false;

        if (this->IsServerConnect())
        {
            // 生存確認メッセージに応答。
            ChunkHeader chunk =
            {
                internal::HostToNet((u32)HEADER_COMMAND_CHECKALIVE), 0
            };
            m_SharedWriteBuffer.Write(&chunk, sizeof(chunk));
            this->StartWritingBuffer();
        }
    }
}

//---------------------------------------------------------------------------
u32
HIOCommDevice::Polling()
{
    OSYieldThread();

    ut::ScopedLock<ut::Mutex> lockObj(m_Lock);

    if (! m_IsOpen)   // オープンしていないときは何もしない。
    {
        return MCS_ERROR_SUCCESS;
    }

    if (IsError())
    {
        return MCS_ERROR_COMERROR;
    }

    this->CheckInitBuffer();
    this->CheckAlive();

    switch (m_NegoPhase)
    {
    case NEGOPHASE_CONNECT:
        // 接続済み
        break;
    case NEGOPHASE_ATTACHED:
        // 接続待ち
        break;
    case NEGOPHASE_REBOOT:
        // バッファ初期化メッセージ待ち
        break;
    default:
        goto HioError;
    }

    return MCS_ERROR_SUCCESS;

    // HIO関数でエラーが発生
HioError:
    NW_WARNING( false, "HIOCommDevice::Polling() --- failure in HIO function.\n" );

    m_NegoPhase = NEGOPHASE_ERROR;

    return MCS_ERROR_COMERROR;
}

//---------------------------------------------------------------------------
u32
HIOCommDevice::GetReadableBytes(ChannelType channel)
{
    // 各チャンネルの持つリングバッファにどれだけリードデータが溜まっているかを返します。

    ut::ScopedLock<ut::Mutex> lockObj(m_Lock);

    NW_ASSERT(m_IsOpen);

    ChannelInfo *const pInfo = GetChannelInfo(channel);

    NW_ASSERT_NOT_NULL(pInfo);

    return pInfo->recvBuf.GetReadableBytes();
}

//---------------------------------------------------------------------------
u32
HIOCommDevice::Peek(
    ChannelType channel,
    void*       data,
    u32         size
)
{
    ut::ScopedLock<ut::Mutex> lockObj(m_Lock);

    NW_ASSERT(m_IsOpen);

    ChannelInfo *const pInfo = GetChannelInfo(channel);

    NW_ASSERT_NOT_NULL(pInfo);

    SimpleDataRange dataRange;

    pInfo->recvBuf.BeginRead(&dataRange);
    return pInfo->recvBuf.ContinueRead(&dataRange, data, size);
}

//---------------------------------------------------------------------------
u32
HIOCommDevice::Read(
    ChannelType channel,
    void*       data,
    u32         size
)
{
    ut::ScopedLock<ut::Mutex> lockObj(m_Lock);

    NW_ASSERT(m_IsOpen);

    ChannelInfo *const pInfo = GetChannelInfo(channel);

    NW_ASSERT_NOT_NULL(pInfo);

    u32 readByte = 0;
    u32 offset = 0;
    u8 *const pDst = static_cast<u8*>(data);

    while (offset < size)
    {
        if (this->IsError())
        {
            return MCS_ERROR_COMERROR;
        }

        StartReadingBuffer();

        if (pInfo->recvBuf.GetReadableBytes() != 0)
        {
            readByte = ut::Min(size - offset, pInfo->recvBuf.GetReadableBytes());
            readByte = pInfo->recvBuf.Read(pDst + offset, readByte);
            offset += readByte;
        }
        else
        {
            u32 result = this->WaitRecvData();
            if (result != MCS_ERROR_SUCCESS)
            {
                return result;
            }

            // 途中で通信が途絶した場合はエラー。
            if (0 < offset)
            {
                if (! this->IsServerConnect())
                {
                    return MCS_ERROR_NOTCONNECT;
                }
            }
        }
    }

    // 状態の変化を通知 (スレッドにバッファが空いたことを知らせる)
    OSSignalCond(&s_McsCond);

    return MCS_ERROR_SUCCESS;
}

//---------------------------------------------------------------------------
u32
HIOCommDevice::GetWritableBytes(
    ChannelType /* channel */,
    u32*        pWritableBytes
)
{
    ut::ScopedLock<ut::Mutex> lockObj(m_Lock);

    NW_ASSERT(m_IsOpen);

    if (IsError())
    {
        return MCS_ERROR_COMERROR;
    }

    if (! IsServerConnect())
    {
        return MCS_ERROR_NOTCONNECT;
    }

    u32 writableBytes = m_SharedWriteBuffer.GetWritableSize();

    if ( pWritableBytes )
    {
        *pWritableBytes = writableBytes;
        return MCS_ERROR_SUCCESS;
    }

    NW_WARNING( false, "HIOCommDevice::GetWritableBytes() --- failure in HIO function.\n" );

    m_NegoPhase = NEGOPHASE_ERROR;
    return MCS_ERROR_COMERROR;
}


//---------------------------------------------------------------------------
u32
HIOCommDevice::Write(
    ChannelType channel,
    const void* data,
    u32         size
)
{
    ut::ScopedLock<ut::Mutex> lockObj(m_Lock);

    NW_ASSERT(m_IsOpen);

    NW_MCS_LOG("HIOCommDevice::Write\n");

    if (IsError())
    {
        return MCS_ERROR_COMERROR;
    }

    if (! this->IsServerConnect())
    {
        NW_MCS_LOG("MCS_ERROR_NOTCONNECT\n");
        return MCS_ERROR_NOTCONNECT;
    }

    const u8* buff = static_cast<const u8*>(data);

    while (0 < size)
    {
        u32 writableBytes = 0;
        u32 result = this->GetWritableBytes(channel, &writableBytes);

        if (result != MCS_ERROR_SUCCESS) { goto HioError; }

        writableBytes = ut::Min(writableBytes, static_cast<u32>(sizeof(MessageHeader) + size));
        if (sizeof(MessageHeader) < writableBytes)
        {
            u32 msgBytes = writableBytes - sizeof(MessageHeader);

            MessageHeader header =
            {
                HEADER_COMMAND_MESSAGETOSERVER,
                writableBytes - sizeof(ChunkHeader),
                channel,
                msgBytes,
            };

            m_SharedWriteBuffer.Write(&header, sizeof(header));
            m_SharedWriteBuffer.Write(buff, msgBytes);

            this->StartWritingBuffer();

            buff += msgBytes;
            size -= msgBytes;
        }
        else
        {
            u32 result = this->WaitSendData();
            if (result != MCS_ERROR_SUCCESS)
            {
                return result;
            }
        }
    }

    return MCS_ERROR_SUCCESS;   // 最後まで書き込めたら成功を返す

    // HIO関数でエラーが発生
HioError:
    NW_WARNING( false, "HIOCommDevice::Write() --- failure in HIO function.\n" );
    m_NegoPhase = NEGOPHASE_ERROR;
    return MCS_ERROR_COMERROR;
}


//---------------------------------------------------------------------------
void
HIOCommDevice::StartWritingBuffer()
{
    ut::ScopedLock<ut::Mutex> lockObj(m_Lock);

    if (this->m_IsWriting)
    {
        return;
    }

    NW_MCS_LOG("StartWritingBuffer\n");

    const void* pointer = m_SharedWriteBuffer.HeadPointer();
    u32 size = m_SharedWriteBuffer.GetWrittenSize();

#if 0
    // コールバックが割り込み内で呼ばれる、かつ DisableInterrupts が提供されていない。
    HIOWriteAsync(m_HioHandle, size, const_cast<void*>(pointer), HIOCommDevice::WriteCallback, this);
#else
    HIOStatus status = HIOWrite(m_HioHandle, size, const_cast<void*>(pointer));
    NW_MCS_LOG("Write %d bytes\n", size);
    m_SharedWriteBuffer.MovePtr(size);
    this->OnWrite(status);
#endif
}

//---------------------------------------------------------------------------
void
HIOCommDevice::StartReadingBuffer()
{
    ut::ScopedLock<ut::Mutex> lockObj(m_Lock);

    if (this->m_IsReading)
    {
        return;
    }

    void* pointer = m_SharedReadBuffer.TailPointer();
    u32 size = m_SharedReadBuffer.GetRemindSize();

    if (size > 0)
    {
        this->m_IsReading = true;

        // コールバックが割り込み内で呼ばれる、かつ DisableInterrupts が提供されていない。
        HIOReadAsync(m_HioHandle, size, pointer, HIOCommDevice::ReadCallback, this);
    }
}

//---------------------------------------------------------------------------
void
HIOCommDevice::StopReadingBuffer()
{
    m_EnableReadChain = false;
}

//---------------------------------------------------------------------------
bool
HIOCommDevice::IsServerConnect()
{
    return m_NegoPhase == NEGOPHASE_CONNECT;
}

//---------------------------------------------------------------------------
s64
HIOCommDevice::GetServerTime()
{
    s64 currentServerTime;
    OSTime clientTime;

    {
        ut::ScopedLock<ut::Mutex> lockObj(m_Lock);

        currentServerTime = m_ServerTime;
        clientTime        = m_ClientTime;
    }

    OSTime currentClientTime = OSGetSystemTime();

    currentServerTime += OSTicksToMilliseconds(currentClientTime - clientTime);

    return currentServerTime;
}

//---------------------------------------------------------------------------
void
HIOCommDevice::SetServerTime(s64 serverTime)
{
    OSTime currentTime = OSGetSystemTime();

    {
        ut::ScopedLock<ut::Mutex> lockObj(m_Lock);

        m_ServerTime = serverTime;
        m_ClientTime = currentTime;
    }
}

//---------------------------------------------------------------------------
bool
HIOCommDevice::IsError()
{
    return m_NegoPhase == NEGOPHASE_ERROR;
}


//---------------------------------------------------------------------------
void
HIOCommDevice::InitHioRingBuffer()
{
    m_SharedWriteBuffer.Initialize(m_TempBuf, WRITE_TRANSBUF_SIZE);
    m_SharedReadBuffer.Initialize(ut::AddOffsetToPtr(m_TempBuf, WRITE_TRANSBUF_SIZE), READ_TRANSBUF_SIZE);
}

//---------------------------------------------------------------------------
void
HIOCommDevice::InitRegisteredBuffer()
{
    ChannelInfo* pInfo = 0;
    while (0 != (pInfo = m_ChannelInfoList.GetNext(pInfo)))
    {
        pInfo->recvBuf.Discard();
    }
}

//---------------------------------------------------------------------------
ChannelInfo*
HIOCommDevice::GetChannelInfo(ChannelType channel)
{
    ChannelInfo* pInfo = 0;

    while (0 != (pInfo = m_ChannelInfoList.GetNext(pInfo)))
    {
        if (pInfo->channel == channel)
        {
            return pInfo;
        }
    }

    return 0;
}

//---------------------------------------------------------------------------
u32
HIOCommDevice::WaitSendData()
{
    m_Lock.WaitCond(s_McsCond);

    return MCS_ERROR_SUCCESS;
}

//---------------------------------------------------------------------------
u32
HIOCommDevice::WaitRecvData()
{
    if (u32 errorCode = Polling())  // PC側が書き込み待ちでブロックしてしまわないようにしておく
    {
        return errorCode;
    }

    m_Lock.WaitCond(s_McsCond);

    return MCS_ERROR_SUCCESS;
}

}   // namespace internal
}   // namespace mcs
}   // namespace nw

#endif  // #if defined(NW_MCS_ENABLE)

