﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/types.h>
#include <nw/mcs/mcs_Base.h>

#if defined(NW_MCS_ENABLE)

#if defined(NW_PLATFORM_CAFE)
  #define HIO_ENABLE    1
#else
  #define PCIO2_ENABLE  1
#endif

#include <nw/mcs/mcs_Common.h>
#include <nw/ut/ut_Inlines.h>
#include <nw/ut/ut_ScopedLock.h>

#ifdef HIO_ENABLE
#include <nw/mcs/mcs_CafeHioCommDevice.h>
#endif

#ifdef PCIO2_ENABLE
#include <nw/mcs/mcs_Pcio2CommDevice.h>
#endif

#include <new>

#include <nw/middlewareSymbol.h>

namespace {

    using namespace nw;
    using namespace mcs;

    typedef nw::ut::Mutex LockType;

    s32                 s_IsInitialized   = 0;        // 初期化された回数の参照カウント
    s32                 s_IsOpened        = 0;        // Mcs_Open を呼ばれた回数の参照カウント
    LockType            s_Mutex;
    nw::mcs::internal::CommDevice*   s_pCommDevice;
    nw::mcs::DeviceInfo    s_DeviceInfo;

    #ifdef HIO_ENABLE

    //---------------------------------------------------------------------------
    //! @brief        HIOデバイスの列挙をおこないユーザに伝えるクラスです。
    //---------------------------------------------------------------------------
    class HIOCommDeviceEnumerate : public nw::mcs::internal::IHIOCommDeviceEnumerate
    {
    public:
       /* ctor */ HIOCommDeviceEnumerate(nw::mcs::internal::IDeviceEnumerate* pEnumerate)
                   : m_pEnumerate( pEnumerate ),
                     m_IsContinue( false )
                  {}

        virtual bool        Find(int type);

        bool                IsContinue() const  { return m_IsContinue; }

    private:
        nw::mcs::internal::IDeviceEnumerate*   m_pEnumerate; // 通知先のユーザの Enumerate クラス。
        bool                                   m_IsContinue; // 1つ列挙した後も検索を続けるかどうかのフラグです。HIOでは常に false。
    };


    //---------------------------------------------------------------------------
    //! @brief        列挙の通知をユーザの列挙関数に渡します。
    //!
    //! @param[in]    devType デバイスタイプです。
    //!
    //! @return       true を返すとデバイスの列挙を継続します。
    //---------------------------------------------------------------------------
    bool
    HIOCommDeviceEnumerate::Find(int devType)
    {
        NW_UNUSED_VARIABLE( devType );

        DeviceInfo deviceInfo;

        deviceInfo.deviceType = DEVICETYPE_CATDEV;
        deviceInfo.param1 = 0;
        deviceInfo.param2 = 0;
        deviceInfo.param3 = 0;

        m_IsContinue = m_pEnumerate->Find( deviceInfo );
        return m_IsContinue;
    }
    #endif


    // デバイスを見つけたときに呼ばれるクラス
    // CafeHIODevice, PCIO2Device で必ず１つのデバイスが見つかるものとして実装。
    class DeviceEnumerate : public nw::mcs::internal::IDeviceEnumerate
    {
    public:
        /* ctor */                      DeviceEnumerate() : m_IsFind( false ) {}

        virtual bool                    Find(const DeviceInfo& deviceInfo)
        {
            m_IsFind     = true;        // 通信デバイスを発見
            m_DeviceInfo = deviceInfo;  // 通信デバイスの情報をコピー
            return false;               // 検索を継続しません
        }

        bool                 IsFind() const          { return m_IsFind; }
        const DeviceInfo&    GetDeviceInfo() const   { return m_DeviceInfo; }

    private:
        bool                            m_IsFind;
        DeviceInfo                      m_DeviceInfo;
    };

}   // namespace

namespace nw  {
namespace mcs {

using namespace internal;

//========================================================================
//    外部関数(公開)
//========================================================================

//---------------------------------------------------------------------------
void
Mcs_Initialize()
{
    NW_PUT_MIDDLEWARE_SYMBOL(mcs);

    if (s_IsInitialized)
    {
        ut::ScopedLock<LockType> lockObj(s_Mutex);
        ++s_IsInitialized;
        return;
    }

    s_Mutex.Initialize();

    {
        ut::ScopedLock<LockType> lockObj(s_Mutex);
        ++s_IsInitialized;

    #ifdef HIO_ENABLE
        HIOInit();
    #endif

        // デバイスを設定
        DeviceEnumerate device;
        internal::Mcs_EnumDevices( &device );
        NW_ASSERT( device.IsFind() );
        if (!device.IsFind())
        {
            return;
        }

        // 通信デバイスオブジェクトの作成
        u32 deviceObjBufSize       = internal::Mcs_GetDeviceObjectMemSize( device.GetDeviceInfo() );
        void * const deviceObjBuf  = internal::Mcs_GetDeviceObjectMem( device.GetDeviceInfo() );
        NW_NULL_ASSERT( deviceObjBuf );

        bool bSuccess = internal::Mcs_CreateDeviceObject( device.GetDeviceInfo(), deviceObjBuf, deviceObjBufSize );
        NW_ASSERT( bSuccess );

        s_DeviceInfo = device.GetDeviceInfo();
    }
}

//---------------------------------------------------------------------------
void
Mcs_Finalize()
{
    NW_ASSERT( s_IsInitialized >= 0 );

    ut::ScopedLock<LockType> lockObj(s_Mutex);

    if (s_IsInitialized == 0)
    {
        return;
    }

    if (--s_IsInitialized == 0)
    {
        internal::Mcs_DestroyDeviceObject();
    }
}


//---------------------------------------------------------------------------
s32
Mcs_IsInitialized()
{
    return s_IsInitialized;
}


//---------------------------------------------------------------------------
const DeviceInfo*
Mcs_GetDeviceInfo()
{
    NW_ASSERT( s_IsInitialized );

    return &s_DeviceInfo;
}

namespace internal {

    //---------------------------------------------------------------------------
    //! @brief       通信デバイスを列挙します。
    //!
    //! @param[out]  pEnumerate   IDeviceEnumerateクラスを継承するオブジェクト。
    //!                           通信デバイスを列挙するたびにこのオブジェクトの
    //!                           Findメソッドが呼ばれます。
    //! @param[in]   flag         現時点では無視します。
    //!
    //! @return      なし。
    //---------------------------------------------------------------------------
    void
    Mcs_EnumDevices(
        IDeviceEnumerate*   pEnumerate,
        u32                 /* flag */
    )
    {
        NW_ASSERT_NOT_NULL(pEnumerate);

    #ifdef HIO_ENABLE
        HIOCommDeviceEnumerate hioCommDeviceEnumerate( pEnumerate );

        HIOCommDevice::EnumerateDevice( &hioCommDeviceEnumerate );
        if ( ! hioCommDeviceEnumerate.IsContinue() )
        {
            return;
        }
    #endif

    #ifdef PCIO2_ENABLE
        if ( ! PCIO2CommDevice::EnumerateDevice(*pEnumerate) )
        {
            return;
        }
    #endif

        // 次のデバイスの列挙へ
    }

    //---------------------------------------------------------------------------
    //! @brief       通信デバイスオブジェクトを作成するのに必要なメモリサイズを
    //!              取得します。
    //!
    //! @param[in]   deviceInfo   通信デバイスの情報を保持するオブジェクト。
    //!                           この値は、Mcs_EnumDevices関数を呼び出すと得られます。
    //!
    //! @return      通信デバイスオブジェクトを作成するのに必要なメモリサイズを
    //!               返します。
    //---------------------------------------------------------------------------
    u32
    Mcs_GetDeviceObjectMemSize(const DeviceInfo& deviceInfo)
    {
        switch (deviceInfo.deviceType)
        {
    #ifdef HIO_ENABLE
        case DEVICETYPE_CATDEV:
            return sizeof(HIOCommDevice) + HIOCommDevice::COPY_ALIGNMENT + HIOCommDevice::TEMP_BUF_SIZE_MIN;
    #endif

    #ifdef PCIO2_ENABLE
        case DEVICETYPE_PCIO2:
            return PCIO2CommDevice::GetDeviceObjectMemSize(deviceInfo);
    #endif

        // 未知なタイプの場合は0を返す
        default:
            return 0;
        }
    }

    //---------------------------------------------------------------------------
    //! @brief        通信デバイスオブジェクトを作成する為のメモリを取得します。
    //!
    //! @param[in]    deviceInfo 通信デバイスの情報を保持するオブジェクトです。
    //!
    //! @return       通信デバイスオブジェクトを作成する為のメモリを返します。
    //---------------------------------------------------------------------------
    void*
    Mcs_GetDeviceObjectMem(const DeviceInfo& deviceInfo)
    {
        switch (deviceInfo.deviceType)
        {
    #ifdef HIO_ENABLE
        case DEVICETYPE_CATDEV:
            {
                static u32 s_DeviceObject[ (sizeof(HIOCommDevice) + HIOCommDevice::COPY_ALIGNMENT + HIOCommDevice::TEMP_BUF_SIZE_MIN + 3) / 4 ];

                return s_DeviceObject;
            }
    #endif

    #ifdef PCIO2_ENABLE
        case DEVICETYPE_PCIO2:
            return malloc( PCIO2CommDevice::GetDeviceObjectMemSize(deviceInfo) );
    #endif

        // 未知なタイプの場合は NULL を返す。
        default:
            return NULL;
        }
    }

    //---------------------------------------------------------------------------
    //! @brief       通信デバイスオブジェクトを作成します。
    //!
    //! @param[in]   deviceInfo   通信デバイスの情報を保持するオブジェクト。
    //!                           この値は、Mcs_EnumDevices関数を呼び出すと得られます。
    //! @param[in]   buf          通信デバイスオブジェクトが必要とするメモリへの
    //!                           ポインタ。
    //! @param[in]   bufSize      bufが指すメモリバッファのサイズ。
    //!
    //! @return      通信デバイスオブジェクトの作成が成功したらtrue、
    //!               失敗したらfalseを返します。
    //---------------------------------------------------------------------------
    bool
    Mcs_CreateDeviceObject(
        const DeviceInfo&   deviceInfo,
        void*               buf,
        u32                 bufSize
    )
    {
        NW_ASSERT(s_IsInitialized);
        NW_ASSERT(s_pCommDevice == 0);

        CommDevice* pCommDevice = 0;

        NW_ASSERT(bufSize >= Mcs_GetDeviceObjectMemSize(deviceInfo));

        switch (deviceInfo.deviceType)
        {
    #ifdef HIO_ENABLE
        case DEVICETYPE_CATDEV:
            {
                NW_ASSERT(bufSize >= Mcs_GetDeviceObjectMemSize(deviceInfo));

                void* tempBuf = static_cast<u8*>(buf) + sizeof(HIOCommDevice);
                u32 tempBufSize = bufSize - sizeof(HIOCommDevice);

                // 配置newでコンストラクタを呼び出す
                pCommDevice = new (buf) HIOCommDevice(tempBuf, tempBufSize);
            }
            break;
    #endif

    #ifdef PCIO2_ENABLE
        case DEVICETYPE_PCIO2:
            pCommDevice = PCIO2CommDevice::Construct(buf, deviceInfo);
            (void)bufSize;
            break;
    #endif

        default:
            break;
        }

        if (! pCommDevice)
        {
            return false;
        }

        {
            ut::ScopedLock<LockType> lockObj(s_Mutex);

            s_pCommDevice = pCommDevice;
        }

        return true;
    }

    //---------------------------------------------------------------------------
    //! @brief       通信デバイスオブジェクトを破棄します。
    //!
    //! @return      なし。
    //---------------------------------------------------------------------------
    void
    Mcs_DestroyDeviceObject()
    {
        NW_ASSERT(s_pCommDevice != 0);

        ut::ScopedLock<LockType> lockObj(s_Mutex);

        s_pCommDevice->~CommDevice();

        s_pCommDevice = 0;
    }

    //---------------------------------------------------------------------------
    //! @brief       通信デバイスオブジェクトが生成されているかを取得します。
    //!
    //! @return      通信デバイスオブジェクトが生成されていたら true を返します。
    //---------------------------------------------------------------------------
    bool
    Mcs_IsDeviceObjectCreated()
    {
        NW_ASSERT(s_IsInitialized);

        ut::ScopedLock<LockType> lockObj(s_Mutex);

        return s_pCommDevice != 0;
    }

} // namespace internal

//---------------------------------------------------------------------------
void
Mcs_RegisterBuffer(
    ChannelType channel,
    void*       buf,
    u32         bufSize
)
{
    NW_ASSERT(s_IsInitialized);
    NW_ASSERT(s_pCommDevice != 0);

    ut::ScopedLock<LockType> lockObj(s_Mutex);

    s_pCommDevice->RegisterBuffer(channel, buf, bufSize);
}

//---------------------------------------------------------------------------
void*
Mcs_UnregisterBuffer(ChannelType channel)
{
    NW_ASSERT(s_IsInitialized);
    NW_ASSERT(s_pCommDevice != 0);

    ut::ScopedLock<LockType> lockObj(s_Mutex);

    return s_pCommDevice->UnregisterBuffer(channel);
}


//---------------------------------------------------------------------------
void*
Mcs_GetRegisteredBuffer(ChannelType channel)
{
    NW_ASSERT(s_IsInitialized);
    NW_ASSERT(s_pCommDevice != 0);

    return s_pCommDevice->GetRegisteredBuffer(channel);
}

//---------------------------------------------------------------------------
u32
Mcs_Open()
{
    NW_ASSERT(s_IsInitialized);
    NW_ASSERT(s_pCommDevice != 0);

    ut::ScopedLock<LockType> lockObj(s_Mutex);

    if ( s_IsOpened++ ) // 判定後にインクリメント
    {
        return MCS_ERROR_SUCCESS;
    }
    else
    {
        return s_pCommDevice->Open();
    }
}

//---------------------------------------------------------------------------
void
Mcs_Close()
{
    NW_ASSERT(s_IsInitialized);
    NW_ASSERT(s_pCommDevice != 0);

    ut::ScopedLock<LockType> lockObj(s_Mutex);

    if ( --s_IsOpened == 0 )
    {
        s_pCommDevice->Close();
    }
}

//---------------------------------------------------------------------------
u32
Mcs_Polling()
{
    NW_ASSERT(s_IsInitialized);
    NW_ASSERT(s_pCommDevice != 0);

    ut::ScopedLock<LockType> lockObj(s_Mutex);

    return s_pCommDevice->Polling();
}

//---------------------------------------------------------------------------
u32
Mcs_GetReadableBytes(ChannelType channel)
{
    NW_ASSERT(s_IsInitialized);
    NW_ASSERT(s_pCommDevice != 0);

    ut::ScopedLock<LockType> lockObj(s_Mutex);

    return s_pCommDevice->GetReadableBytes(channel);
}

//---------------------------------------------------------------------------
u32
Mcs_Peek(
    ChannelType channel,
    void*       data,
    u32         size
)
{
    NW_ASSERT(s_IsInitialized);
    NW_ASSERT(s_pCommDevice != 0);

    ut::ScopedLock<LockType> lockObj(s_Mutex);

    return s_pCommDevice->Peek(channel, data, size);
}

//---------------------------------------------------------------------------
u32
Mcs_Read(
    ChannelType channel,
    void*       data,
    u32         size
)
{
    NW_ASSERT(s_IsInitialized);
    NW_ASSERT(s_pCommDevice != 0);

    ut::ScopedLock<LockType> lockObj(s_Mutex);

    return s_pCommDevice->Read(channel, data, size);
}

//---------------------------------------------------------------------------
u32
Mcs_Skip(
    ChannelType channel,
    u32         size
)
{
    NW_ASSERT(s_IsInitialized);
    NW_ASSERT(s_pCommDevice != 0);

    ut::ScopedLock<LockType> lockObj(s_Mutex);

    u8 buf[32];

    while (size > 0)
    {
        const u32 readSize = ut::Min(u32(sizeof(buf)), size);
        if (u32 errorCode = s_pCommDevice->Read(channel, buf, readSize))
        {
            return errorCode;
        }

        size -= readSize;
    }

    return MCS_ERROR_SUCCESS;
}

//---------------------------------------------------------------------------
u32
Mcs_GetWritableBytes(
    ChannelType channel,
    u32*        pWritableBytes
)
{
    NW_ASSERT(s_IsInitialized);
    NW_ASSERT(s_pCommDevice != 0);

    ut::ScopedLock<LockType> lockObj(s_Mutex);

    return s_pCommDevice->GetWritableBytes(channel, pWritableBytes);
}

//---------------------------------------------------------------------------
u32
Mcs_Write(
    ChannelType channel,
    const void* data,
    u32         size
)
{
    NW_ASSERT(s_IsInitialized);
    NW_ASSERT(s_pCommDevice != 0);

    ut::ScopedLock<LockType> lockObj(s_Mutex);

    return s_pCommDevice->Write(channel, data, size);
}

//---------------------------------------------------------------------------
bool
Mcs_IsServerConnect()
{
    NW_ASSERT(s_IsInitialized);
    NW_ASSERT(s_pCommDevice != 0);

    ut::ScopedLock<LockType> lockObj(s_Mutex);

    return s_pCommDevice->IsServerConnect();
}

//---------------------------------------------------------------------------
s64
Mcs_GetServerTime()
{
    NW_ASSERT(s_IsInitialized);

    ut::ScopedLock<LockType> lockObj(s_Mutex);

    return s_pCommDevice->GetServerTime();
}

}   // namespace mcs
}   // namespace nw

#endif  // #if defined(NW_MCS_ENABLE)
