﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/types.h>

#if defined( NW_PLATFORM_CAFE )
#include <sdk_ver.h>
#endif

#include <nw/gfnd/gfnd_GraphicsContext.h>

namespace nw
{
namespace gfnd
{

//------------------------------------------------------------------------------
GraphicsContext::GraphicsContext()
: m_DepthTestEnable( true ),
  m_DepthWriteEnable( true ),
  m_DepthFunc( Graphics::DEPTH_FUNC_LESS_EQUAL ),
  m_CullingMode( Graphics::CULLING_MODE_BACK ),
  m_BlendEnable( true ),
  m_BlendFactorSrcRGB( Graphics::BLEND_FACTOR_SRC_ALPHA ),
  m_BlendFactorSrcA( Graphics::BLEND_FACTOR_SRC_ALPHA ),
  m_BlendFactorDstRGB( Graphics::BLEND_FACTOR_INV_SRC_ALPHA ),
  m_BlendFactorDstA( Graphics::BLEND_FACTOR_INV_SRC_ALPHA ),
  m_BlendEquationRGB( Graphics::BLEND_EQUATION_ADD ),
  m_BlendEquationA( Graphics::BLEND_EQUATION_ADD ),
  m_BlendConstantColor( nw::ut::Color4u8::WHITE ),
  m_AlphaTestEnable( false ),
  m_AlphaTestFunc( Graphics::ALPHA_FUNC_GREATER ),
  m_AlphaTestRef( 0.f ),
  m_ColorMaskR( true ),
  m_ColorMaskG( true ),
  m_ColorMaskB( true ),
  m_ColorMaskA( true ),
  m_StencilTestEnable( false ),
  m_StencilTestFunc( Graphics::STENCIL_FUNC_NEVER ),
  m_StencilTestRef( 0 ),
  m_StencilTestMask( 0xffffffff ),
  m_StencilOpFail( Graphics::STENCIL_OP_KEEP ),
  m_StencilOpZFail( Graphics::STENCIL_OP_KEEP ),
  m_StencilOpZPass( Graphics::STENCIL_OP_KEEP ),
  m_PolygonModeFront( Graphics::POLYGON_MODE_FILL ),
  m_PolygonModeBack( Graphics::POLYGON_MODE_FILL ),
  m_PolygonOffsetFrontEnable( false ),
  m_PolygonOffsetBackEnable( false ),
  m_PolygonOffsetPointLineEnable( false )
{
}

//------------------------------------------------------------------------------
void
GraphicsContext::Apply() const
{
#if defined(NW_PLATFORM_CAFE)
    Graphics* gfx = Graphics::GetInstance();

    // Depth, Stencil
    // 現状では、フロントとバックは両方有効にし、同じ値を設定する
    GX2SetDepthStencilControl(
        static_cast<GX2Boolean>(m_DepthTestEnable),   //depthTestEnable
        static_cast<GX2Boolean>(m_DepthWriteEnable),  //depthWriteEnable
        static_cast<GX2CompareFunction>(m_DepthFunc), //depthFunc
        static_cast<GX2Boolean>(m_StencilTestEnable), //stencilTestEnable
        static_cast<GX2Boolean>(m_StencilTestEnable), //backStencilEnable

        static_cast<GX2CompareFunction>(m_StencilTestFunc),   //frontStencilFunc
        static_cast<GX2StencilFunction>(m_StencilOpZPass),    //frontStencilZPass
        static_cast<GX2StencilFunction>(m_StencilOpZFail),    //frontStencilZFail
        static_cast<GX2StencilFunction>(m_StencilOpFail),     //frontStencilFail

        static_cast<GX2CompareFunction>(m_StencilTestFunc),   //backStencilFunc
        static_cast<GX2StencilFunction>(m_StencilOpZPass),    //backStencilZPass
        static_cast<GX2StencilFunction>(m_StencilOpZFail),    //backStencilZFail
        static_cast<GX2StencilFunction>(m_StencilOpFail)      //backStencilFail
    );
    GX2SetStencilMask(m_StencilTestMask, m_StencilTestMask, m_StencilTestRef,
                      m_StencilTestMask, m_StencilTestMask, m_StencilTestRef);

    // Blend
    GX2SetColorControl(GX2_LOGIC_OP_COPY, m_BlendEnable ? 0xff : 0x0, GX2_DISABLE, GX2_TRUE);
    /// TODO: 現状ではTARGET0のみを設定するようにする（本来はターゲット切り替えとの同期が必要？）
    GX2SetBlendControl(GX2_RENDER_TARGET_0,
        static_cast<GX2BlendFunction>(m_BlendFactorSrcRGB), static_cast<GX2BlendFunction>(m_BlendFactorDstRGB), static_cast<GX2BlendCombine>(m_BlendEquationRGB),
        GX2_ENABLE,
        static_cast<GX2BlendFunction>(m_BlendFactorSrcA), static_cast<GX2BlendFunction>(m_BlendFactorDstA), static_cast<GX2BlendCombine>(m_BlendEquationA));
    GX2SetBlendConstantColor(m_BlendConstantColor.r, m_BlendConstantColor.g, m_BlendConstantColor.b, m_BlendConstantColor.a);

    // Alpha
    GX2SetAlphaTest(static_cast<GX2Boolean>(m_AlphaTestEnable), static_cast<GX2CompareFunction>(m_AlphaTestFunc), m_AlphaTestRef);

    // ColorMask
    gfx->SetColorMask( m_ColorMaskR, m_ColorMaskG, m_ColorMaskB, m_ColorMaskA );

    // Culling, PolygonMode, PolygonOffset
    {
        struct CullSettings
        {
            GX2Boolean front;
            GX2Boolean back;
        };

        static const CullSettings CULL_SETTINGS[4] =
        {
            { GX2_ENABLE, GX2_DISABLE },
            { GX2_DISABLE, GX2_ENABLE },
            { GX2_DISABLE, GX2_DISABLE },
            { GX2_ENABLE, GX2_ENABLE }
        };
        const CullSettings& settings = CULL_SETTINGS[ m_CullingMode ];

        GX2SetPolygonControl(
            GX2_FRONT_FACE_CCW, // frontFaceMode
            settings.front,     // cullFront
            settings.back,      // cullBack

            static_cast<GX2Boolean>(m_PolygonModeFront != Graphics::POLYGON_MODE_FILL || m_PolygonModeBack != Graphics::POLYGON_MODE_FILL),   // enablePolygonModes
            static_cast<GX2PolygonMode>(m_PolygonModeFront), // polygonModeFront
            static_cast<GX2PolygonMode>(m_PolygonModeBack),  // polygonModeBack

            static_cast<GX2Boolean>(m_PolygonOffsetFrontEnable),    // polyOffsetFrontEnable
            static_cast<GX2Boolean>(m_PolygonOffsetBackEnable),     // polyOffsetBackEnable
            static_cast<GX2Boolean>(m_PolygonOffsetPointLineEnable) // pointLineOffsetEnable
        );
    }

#else
    Graphics* gfx = Graphics::GetInstance();

    // Depth
    gfx->SetDepthEnable( m_DepthTestEnable, m_DepthWriteEnable );
    gfx->SetDepthFunc( m_DepthFunc );

    // Blend
    gfx->SetBlendEnable( m_BlendEnable );
    gfx->SetBlendFactorSeparate( m_BlendFactorSrcRGB, m_BlendFactorDstRGB, m_BlendFactorSrcA, m_BlendFactorDstA );
    gfx->SetBlendEquationSeparate( m_BlendEquationRGB, m_BlendEquationA );
    gfx->SetBlendConstantColor( m_BlendConstantColor );

    // Alpha
    gfx->SetAlphaTestEnable( m_AlphaTestEnable );
    gfx->SetAlphaTestFunc( m_AlphaTestFunc, m_AlphaTestRef );

    // ColorMask
    gfx->SetColorMask( m_ColorMaskR, m_ColorMaskG, m_ColorMaskB, m_ColorMaskA );

    // Stencil
    gfx->SetStencilTestEnable( m_StencilTestEnable );
    gfx->SetStencilTestFunc( m_StencilTestFunc, m_StencilTestRef, m_StencilTestMask );
    gfx->SetStencilTestOp( m_StencilOpFail, m_StencilOpZFail, m_StencilOpZPass );

    // Culling
    gfx->SetCullingMode( m_CullingMode );
    // PolygonMode
    gfx->SetPolygonMode( m_PolygonModeFront, m_PolygonModeBack );
    // PolygonOffset
    gfx->SetPolygonOffsetEnable( m_PolygonOffsetFrontEnable, m_PolygonOffsetBackEnable, m_PolygonOffsetPointLineEnable );
#endif
}

} // namespace nw::gfnd
} // namespace nw

