﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdarg>
#include <cstdio>
#include <nw/types.h>
#include <nw/assert.h>
#include <nw/font/font_TextWriterBase.h>
#include <nw/font/font_TagProcessorBase.h>
#include <nw/ut/ut_Inlines.h>

namespace nw {
namespace font {


/* ------------------------------------------------------------------------
        クラス変数定義
   ------------------------------------------------------------------------ */

template <typename CharType>
TagProcessorBase<CharType> TextWriterBase<CharType>::s_DefaultTagProcessor;

namespace
{

/*!--------------------------------------------------------------------------*
  @brief        基点が中央であるときのカーソル位置補正の値を求めます。
                半分の値が浮動小数点にならないように整数値に切り上げます。

  @param[in]    value  考慮する幅/高さ

  @return       基点が中央であるときのカーソル位置補正の値を返します。
 *---------------------------------------------------------------------------*/
inline
f32
AdjustCenterValue(f32 value)
{
    return math::FCeil(value * 0.5f);
}

}   // namespace


/* =======================================================================
        public
   ======================================================================== */

/* ------------------------------------------------------------------------
        コンストラクタ/デストラクタ
   ------------------------------------------------------------------------ */

template <typename CharType>
TextWriterBase<CharType>::TextWriterBase()
: CharWriter(),
  m_WidthLimit(FLT_MAX),
  m_CharSpace(0),
  m_LineSpace(0),
  m_TabWidth (4),
  m_DrawFlag(DEFAULT_DRAWFLAG),
  m_TagProcessor(&s_DefaultTagProcessor)
{
}

template <typename CharType>
TextWriterBase<CharType>::~TextWriterBase()
{
}





/* ------------------------------------------------------------------------
        行間/文字間/タブ幅
   ------------------------------------------------------------------------ */

template <typename CharType>
void
TextWriterBase<CharType>::SetLineHeight(f32 height)
{
    const Font* font = GetFont();
    const int linefeed = font != NULL ? font->GetLineFeed(): 0;

    m_LineSpace = height - linefeed * GetScaleV();
}

template <typename CharType>
f32
TextWriterBase<CharType>::GetLineHeight() const
{
    const Font* font = GetFont();
    const int linefeed = font != NULL ? font->GetLineFeed(): 0;

    return linefeed * GetScaleV() + m_LineSpace;
}






/* ------------------------------------------------------------------------
        文字列サイズ計算
   ------------------------------------------------------------------------ */

template <typename CharType>
f32
TextWriterBase<CharType>::CalcFormatStringWidth(
    StreamType  format,
    ...
) const
{
    NW_ASSERT_VALID_POINTER(format);

    ut::Rect rect;
    std::va_list vargs;
    va_start(vargs, format);

    CalcVStringRect(&rect, format, vargs);

    va_end(vargs);
    return rect.GetWidth();
}

template <typename CharType>
f32
TextWriterBase<CharType>::CalcFormatStringHeight(
    StreamType  format,
    ...
) const
{
    NW_ASSERT_VALID_POINTER(format);

    ut::Rect rect;
    std::va_list vargs;
    va_start(vargs, format);

    CalcVStringRect(&rect, format, vargs);

    va_end(vargs);
    return rect.GetHeight();
}

template <typename CharType>
void
TextWriterBase<CharType>::CalcFormatStringRect(
    ut::Rect*   pRect,
    StreamType  format,
    ...
) const
{
    NW_ASSERT_VALID_POINTER(pRect);
    NW_ASSERT_VALID_POINTER(format);

    std::va_list vargs;
    va_start(vargs, format);

    CalcVStringRect(pRect, format, vargs);

    va_end(vargs);
}

template <typename CharType>
void
TextWriterBase<CharType>::CalcVStringRect(
    ut::Rect*       pRect,
    StreamType      format,
    std::va_list    args
) const
{
    NW_ASSERT_VALID_POINTER(pRect);
    NW_ASSERT_VALID_POINTER(format);

    CharType buffer[DEFAULT_FORMAT_BUFFER_SIZE];
    int length = this->VSNPrintf(buffer, DEFAULT_FORMAT_BUFFER_SIZE, format, args);
    length = ut::Min(length, int(DEFAULT_FORMAT_BUFFER_SIZE - 1));

    CalcStringRect(pRect, buffer, length);
}

template <typename CharType>
f32
TextWriterBase<CharType>::CalcStringWidth(
    StreamType  str,
    int         length
) const
{
    NW_ASSERT_VALID_POINTER(str);
    NW_FONT_MIN_ASSERT(length, 0);

    ut::Rect rect;

    CalcStringRect(&rect, str, length);

    return rect.GetWidth();
}

template <typename CharType>
f32
TextWriterBase<CharType>::CalcStringHeight(
    StreamType  str,
    int         length
) const
{
    NW_ASSERT_VALID_POINTER(str);
    NW_FONT_MIN_ASSERT(length, 0);

    ut::Rect rect;

    CalcStringRect(&rect, str, length);

    return rect.GetHeight();
}

template <typename CharType>
void
TextWriterBase<CharType>::CalcStringRect(
    ut::Rect*   pRect,
    StreamType  str,
    int         length
) const
{
    NW_ASSERT_VALID_POINTER(pRect);
    NW_FONT_MIN_ASSERT(length, 0);

    if (str == NULL)
    {
        pRect->left = 0.0f;
        pRect->top = 0.0f;
        pRect->right = 0.0f;
        pRect->bottom = 0.0f;
        return;
    }

    TextWriterBase<CharType> myCopy = *this;

    myCopy.CalcStringRectImpl(pRect, str, length);
}






/* ------------------------------------------------------------------------
        文字列描画
   ------------------------------------------------------------------------ */

template <typename CharType>
f32
TextWriterBase<CharType>::Printf(
    StreamType  format,
    ...
)
{
    NW_ASSERT_VALID_POINTER(format);

    std::va_list vargs;
    va_start(vargs, format);

    f32 width = VPrintf(format, vargs);

    va_end(vargs);
    return width;
}

template <typename CharType>
f32
TextWriterBase<CharType>::VPrintf(
    StreamType      format,
    std::va_list    args
)
{
    NW_ASSERT_VALID_POINTER(format);

    CharType buffer[DEFAULT_FORMAT_BUFFER_SIZE];
    int length = this->VSNPrintf(buffer, DEFAULT_FORMAT_BUFFER_SIZE, format, args);
    length = ut::Min(length, int(DEFAULT_FORMAT_BUFFER_SIZE - 1));

    f32 width = Print(buffer, length);

    return width;
}

template <typename CharType>
f32
TextWriterBase<CharType>::Print(
    StreamType  str,
    int         length
)
{
    NW_ASSERT_VALID_POINTER(str);
    NW_FONT_MIN_ASSERT(length, 0);

    TextWriterBase<CharType> myCopy = *this;

    f32 width = myCopy.PrintImpl(str, length);

    SetCursor( myCopy.GetCursorX(), myCopy.GetCursorY() );

    return width;
}

/* =======================================================================
        private
   ======================================================================== */

/* ------------------------------------------------------------------------
        文字列処理
   ------------------------------------------------------------------------ */

template <typename CharType>
f32
TextWriterBase<CharType>::CalcLineWidth(
    StreamType  str,
    int         length
)
{
    NW_ASSERT_VALID_POINTER(str);
    NW_FONT_MIN_ASSERT(length, 0);

    ut::Rect rect;
    TextWriterBase<CharType> myCopy = *this;

    myCopy.SetCursor(0, 0);
    myCopy.CalcLineRectImpl(&rect, &str, length);

    return rect.GetWidth();
}

template <typename CharType>
const CharType*
TextWriterBase<CharType>::FindPosOfWidthLimit(
    const CharType* str,
    int length) const
{
    NW_ASSERT_VALID_POINTER(str);
    NW_FONT_MIN_ASSERT(length, 0);

    ut::Rect rect;
    TextWriterBase<CharType> myCopy = *this;

    myCopy.SetCursor(0, 0);
    myCopy.CalcLineRectImpl(&rect, &str, length);

    return str;
}

template <typename CharType>
bool
TextWriterBase<CharType>::CalcLineRectImpl(
    ut::Rect*   pRect,
    StreamType* pStr,
    int         length
)
{
    NW_ASSERT_VALID_POINTER(pRect);
    NW_ASSERT_VALID_POINTER(pStr);
    NW_ASSERT_VALID_POINTER(*pStr);
    NW_FONT_MIN_ASSERT(length, 0);

    const StreamType str            = *pStr;
    const StreamType end            = str + length;
    const bool bUseLimit            = m_WidthLimit < FLT_MAX;
    PrintContext<CharType> context(this, str, end, 0, 0, GetScaleH(), GetScaleV(), 0);
    f32 limitLeft                   = 0;
    f32 limitRight                  = 0;
    bool bCharSpace                 = false;
    bool bOverLimit                 = false;
    StreamType prevStreamPos        = NULL;
    ut::Rect prevRect;

    /*
        文字列の途中でエンコーディングが変わる事は想定していない
    */
    NW_ASSERT_VALID_POINTER(GetFont());
    CharStrmReader reader = GetFont()->GetCharStrmReader(CharType(0));

    // 1行の高さにはLineSpaceを含めない。
    f32 lineFeed = GetLineHeight() - GetLineSpace();

    pRect->left     = 0;
    pRect->right    = 0;
    pRect->top      = ut::Min(0.0f, lineFeed);
    pRect->bottom   = ut::Max(0.0f, lineFeed);
    prevRect = *pRect;

    reader.Set(str);
    prevStreamPos = NULL;

    m_TagProcessor->BeginCalcRect(&context);

    for (
        CharCode code = reader.Next();
        reinterpret_cast<StreamType>(reader.GetCurrentPos()) <= end;
    )
    {
        if (code < ' ')
        {
            //---- 制御文字(タグの開始)

            typename TagProcessor::Operation operation;
            ut::Rect rect(limitRight, 0, 0, 0);

            context.str = reinterpret_cast<StreamType>(reader.GetCurrentPos());
            context.flags = 0;
            context.flags |= bCharSpace ? 0: CONTEXT_NO_CHAR_SPACE;
            SetCursorX(limitRight);

            //---- 折り返しの判定
            if ( bUseLimit
              && code != '\n'
              && prevStreamPos != NULL
            )
            {
                PrintContext<CharType> context2 = context;
                TextWriterBase<CharType> myCopy = *this;
                ut::Rect rect2;

                context2.writer = &myCopy;
                operation = m_TagProcessor->CalcRect(&rect2, code, &context2);

                if ( rect2.GetWidth() > 0.0f
                  && (myCopy.GetCursorX() - context.xOrigin > m_WidthLimit)
                )
                {
                    bOverLimit = true;
                    code       = '\n';
                    context.prevCode = 0;
                    reader.Set(prevStreamPos);
                    continue;
                }
            }

            //---- タグ処理
            operation = m_TagProcessor->CalcRect(&rect, code, &context);

            NW_ASSERT_VALID_POINTER(context.str);
            reader.Set(context.str);

            pRect->left     = ut::Min(pRect->left,      rect.left);
            pRect->top      = ut::Min(pRect->top,       rect.top);
            pRect->right    = ut::Max(pRect->right,     rect.right);
            pRect->bottom   = ut::Max(pRect->bottom,    rect.bottom);
            limitRight = GetCursorX();

            if (operation == TagProcessor::OPERATION_END_DRAW)
            {
                //---- 全部読み進んだ事にする
                *pStr += length;
                return false;
            }
            else if (operation == TagProcessor::OPERATION_NO_CHAR_SPACE)
            {
                bCharSpace = false;
            }
            else if (operation == TagProcessor::OPERATION_CHAR_SPACE)
            {
                bCharSpace = true;
            }
            else if (operation == TagProcessor::OPERATION_NEXT_LINE)
            {
                break;
            }
        }
        else
        {
            //---- 通常の文字

            f32 crntRight = limitRight;

            if (bCharSpace)
            {
                crntRight += GetCharSpace();
            }

            if (IsWidthFixed())
            {
                crntRight += GetFixedWidth();
            }
            else
            {
                crntRight += GetFont()->GetCharWidth(code) * GetScaleH();
            }
            crntRight += static_cast<float>(GetFont()->GetKerning(context.prevCode, code) * GetScaleH());

            //---- 折り返しの判定
            if (bUseLimit && prevStreamPos != NULL)
            {
                f32 width = crntRight - limitLeft;
                if (width > m_WidthLimit)
                {
                    bOverLimit = true;
                    code       = '\n';
                    context.prevCode = 0;
                    reader.Set(prevStreamPos);
                    continue;
                }
            }

            limitRight = crntRight;
            pRect->left  = ut::Min(pRect->left,  limitRight);
            pRect->right = ut::Max(pRect->right, limitRight);
            bCharSpace = true;
            context.prevCode = code;
        }

        if (bUseLimit)
        {
            prevStreamPos = reinterpret_cast<StreamType>(reader.GetCurrentPos());
        }

        code = reader.Next();
    }


    *pStr = reinterpret_cast<StreamType>(reader.GetCurrentPos());

    m_TagProcessor->EndCalcRect(&context);

    return bOverLimit;
}

template <typename CharType>
void
TextWriterBase<CharType>::CalcStringRectImpl(
    ut::Rect*   pRect,
    StreamType  str,
    int         length
)
{
    NW_ASSERT_VALID_POINTER(pRect);
    NW_ASSERT_VALID_POINTER(str);
    NW_FONT_MIN_ASSERT(length, 0);

    const StreamType end = str + length;
    int remain = length;
    StreamType pos = str;

    pRect->left     = 0;
    pRect->right    = 0;
    pRect->top      = 0;
    pRect->bottom   = 0;

    SetCursor(0, 0);

    do
    {
        ut::Rect rect;
        CalcLineRectImpl(&rect, &pos, remain);
        remain   = (end - pos);

        pRect->left     = ut::Min(pRect->left,      rect.left);
        pRect->top      = ut::Min(pRect->top,       rect.top);
        pRect->right    = ut::Max(pRect->right,     rect.right);
        pRect->bottom   = ut::Max(pRect->bottom,    rect.bottom);
    } while (remain > 0);
}

template <typename CharType>
f32
TextWriterBase<CharType>::PrintImpl(
    StreamType  str,
    int         length
)
{
    NW_ASSERT_VALID_POINTER(str);
    NW_ASSERT_VALID_POINTER(GetFont());
    NW_FONT_MIN_ASSERT(length, 0);

    f32 xOrigin    = GetCursorX();
    f32 yOrigin    = GetCursorY();
    f32 limitLeft  = 0;
    f32 limitRight = 0;
    const bool bUseLimit = m_WidthLimit < FLT_MAX;
    const f32 orgCursorY = yOrigin;
    bool bCharSpace  = false;
    StreamType prevStreamPos = str;
    StreamType prevNewLinePos = str;

    f32 textWidth  = AdjustCursor(&xOrigin, &yOrigin, str, length);
    f32 yCursorAdj = orgCursorY - GetCursorY();

    PrintContext<CharType> context(this, str, str + length, xOrigin, yOrigin, GetScaleH(), GetScaleV(), 0);
    CharStrmReader reader = GetFont()->GetCharStrmReader(CharType(0));
    reader.Set(str);

    m_TagProcessor->BeginPrint(&context);

    for (
        CharCode code = reader.Next();
        reinterpret_cast<StreamType>(reader.GetCurrentPos()) - str <= length;
    )
    {
        if (code < ' ')
        {
            //---- 制御文字(タグの開始)

            typename TagProcessor::Operation operation;
            context.str = reinterpret_cast<StreamType>(reader.GetCurrentPos());
            context.flags = 0;
            context.flags |= bCharSpace ? 0: CONTEXT_NO_CHAR_SPACE;

            //---- 折り返しの判定
            if ( bUseLimit
              && code != '\n'
              && prevStreamPos != prevNewLinePos
            )
            {
                PrintContext<CharType> context2 = context;
                TextWriterBase<CharType> myCopy = *this;
                ut::Rect rect;

                context2.writer = &myCopy;
                operation = m_TagProcessor->CalcRect(&rect, code, &context2);

                if ( rect.GetWidth() > 0.0f
                  && myCopy.GetCursorX() - context.xOrigin > m_WidthLimit
                )
                {
                    code = '\n';
                    context.prevCode = 0;
                    reader.Set(prevStreamPos);
                    continue;
                }
            }

            //---- タグ処理
            operation = m_TagProcessor->Process(code, &context);
            limitRight = GetCursorX() - xOrigin;

            if (operation == TagProcessor::OPERATION_NEXT_LINE)
            {
                NW_ASSERT_VALID_POINTER(context.str);

                //---- 次行描画開始位置Xの補正
                if (IsDrawFlagSet(HORIZONTAL_ALIGN_MASK, HORIZONTAL_ALIGN_CENTER))
                {
                    const int   remain  = length - (context.str - str);
                    const f32   width   = CalcLineWidth( context.str, remain );
                    const f32   offset  = AdjustCenterValue(textWidth) - AdjustCenterValue(width);
                    SetCursorX( context.xOrigin + offset );
                }
                else if (IsDrawFlagSet(HORIZONTAL_ALIGN_MASK, HORIZONTAL_ALIGN_RIGHT))
                {
                    const int   remain  = length - (context.str - str);
                    const f32   width   = CalcLineWidth( context.str, remain );
                    const f32   offset  = textWidth - width;
                    SetCursorX( context.xOrigin + offset );
                }
                else
                {
                    //---- 最大幅の更新
                    const f32 width = GetCursorX() - context.xOrigin;
                    textWidth = ut::Max(textWidth, width);

                    SetCursorX( context.xOrigin );
                }

                if (bUseLimit)
                {
                    prevNewLinePos = reinterpret_cast<StreamType>(reader.GetCurrentPos());
                }
                bCharSpace = false;
            }
            else if (operation == TagProcessor::OPERATION_NO_CHAR_SPACE)
            {
                bCharSpace = false;
            }
            else if (operation == TagProcessor::OPERATION_CHAR_SPACE)
            {
                bCharSpace = true;
            }
            else if (operation == TagProcessor::OPERATION_END_DRAW)
            {
                break;
            }

            NW_ASSERT_VALID_POINTER(context.str);
            reader.Set(context.str);
        }
        else
        {
            //---- 通常の文字

            const f32 baseY = GetCursorY();

            f32 crntRight = limitRight;
            f32 kerning = static_cast<float>(GetFont()->GetKerning(context.prevCode, code)) * GetScaleH();
            Glyph glyph;
            GetFont()->GetGlyph(&glyph, code);

            if (bCharSpace)
            {
                crntRight += GetCharSpace();
            }

            if (IsWidthFixed())
            {
                crntRight += GetFixedWidth();
            }
            else
            {
                crntRight += glyph.widths.charWidth * GetScaleH();
            }
            crntRight += kerning;

            //---- 折り返しの判定
            if (bUseLimit && prevStreamPos != prevNewLinePos)
            {
                f32 width = crntRight - limitLeft;
                if (width > m_WidthLimit)
                {
                    code = '\n';
                    context.prevCode = 0;
                    reader.Set(prevStreamPos);
                    continue;
                }
            }

            limitRight = crntRight;


            if (bCharSpace)
            {
                MoveCursorX( GetCharSpace() );
            }
            bCharSpace = true;

            MoveCursorX( kerning );

            //---- カーソルはベースラインにあるのでセル上端へ移動する
            {
                const Font* pFont = GetFont();
                const f32 adj = - pFont->GetBaselinePos() * GetScaleV();
                MoveCursorY( adj );
            }

            PrintGlyph(glyph);

            // 戻す
            SetCursorY( baseY );

            context.prevCode = code;
        }

        if (bUseLimit)
        {
            prevStreamPos = reinterpret_cast<StreamType>(reader.GetCurrentPos());
        }

        code = reader.Next();
    }

    //---- 最大幅の更新
    {
        const f32 width = GetCursorX() - context.xOrigin;
        textWidth = ut::Max(textWidth, width);
    }

    //---- カーソル位置Yを描画フラグに従って移動
    // AdjustCursorでベースライン上に移動したカーソル位置を元に戻す
    if ( IsDrawFlagSet(VERTICAL_ORIGIN_MASK, VERTICAL_ORIGIN_MIDDLE)
      || IsDrawFlagSet(VERTICAL_ORIGIN_MASK, VERTICAL_ORIGIN_BOTTOM)
    )
    {
        SetCursorY(orgCursorY);
    }
    else
    {
        MoveCursorY(yCursorAdj);
    }

    m_TagProcessor->EndPrint(&context);

    return textWidth;
}

template <typename CharType>
f32
TextWriterBase<CharType>::AdjustCursor(
    f32*        pXOrigin,
    f32*        pYOrigin,
    StreamType  str,
    int         length
)
{
    f32 textWidth  = 0;
    f32 textHeight = 0;

    {
        u32 flagMask  = HORIZONTAL_ALIGN_MASK | HORIZONTAL_ORIGIN_MASK | VERTICAL_ORIGIN_MASK;
        u32 blineFlag = HORIZONTAL_ALIGN_LEFT | HORIZONTAL_ORIGIN_LEFT | VERTICAL_ORIGIN_BASELINE;
        u32 topFlag   = HORIZONTAL_ALIGN_LEFT | HORIZONTAL_ORIGIN_LEFT | VERTICAL_ORIGIN_TOP;

        if ( ! IsDrawFlagSet(flagMask, blineFlag)
          && ! IsDrawFlagSet(flagMask, topFlag)
        )
        {
            ut::Rect textRect;
            CalcStringRect(&textRect, str, length);
            textWidth  = textRect.left + textRect.right;
            textHeight = textRect.top  + textRect.bottom;
        }
    }

    //---- 描画開始位置Xの補正
    // 各々の位置から文字列の左端に統一
    if (IsDrawFlagSet(HORIZONTAL_ORIGIN_MASK, HORIZONTAL_ORIGIN_CENTER))
    {
        *pXOrigin -= AdjustCenterValue(textWidth);
    }
    else if (IsDrawFlagSet(HORIZONTAL_ORIGIN_MASK, HORIZONTAL_ORIGIN_RIGHT))
    {
        *pXOrigin -= textWidth;
    }

    //---- 描画開始位置Yの補正
    // 各々の位置から 1 行目のアセンダラインに統一
    if (IsDrawFlagSet(VERTICAL_ORIGIN_MASK, VERTICAL_ORIGIN_MIDDLE))
    {
        *pYOrigin -= AdjustCenterValue(textHeight);
    }
    else if (IsDrawFlagSet(VERTICAL_ORIGIN_MASK, VERTICAL_ORIGIN_BOTTOM))
    {
        *pYOrigin -= textHeight;
    }

    //---- 1行目描画開始位置Xの補正
    // 文字列の左端からそれぞれの寄せ位置へカーソルを移動
    if (IsDrawFlagSet(HORIZONTAL_ALIGN_MASK, HORIZONTAL_ALIGN_CENTER))
    {
        const f32 width = CalcLineWidth(str, length);
        const f32 offset = AdjustCenterValue(textWidth) - AdjustCenterValue(width);
        SetCursorX( *pXOrigin + offset );
    }
    else if (IsDrawFlagSet(HORIZONTAL_ALIGN_MASK, HORIZONTAL_ALIGN_RIGHT))
    {
        const f32 width = CalcLineWidth(str, length);
        const f32 offset = textWidth - width;
        SetCursorX( *pXOrigin + offset );
    }
    else
    {
        SetCursorX( *pXOrigin );
    }

    //---- 1行目描画開始位置Yの補正
    // 1 行目のベースライン位置にカーソルを移動
    if (IsDrawFlagSet(VERTICAL_ORIGIN_MASK, VERTICAL_ORIGIN_BASELINE))
    {
        // pYOrigin はベースラインであるのでそのままセット
        SetCursorY( *pYOrigin );
    }
    else
    {
        // pYOrigin はアセンダラインであるのでベースラインにカーソルを移動
        SetCursorY( *pYOrigin + GetFontAscent() );
    }

    return textWidth;
}

/* =======================================================================
        明示的実体化
   ======================================================================== */

template class TextWriterBase<char>;
template class TextWriterBase<char16>;



}   // namespace font
}   // namespace nw

