﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/font/font_RectDrawer.h>
#include <nw/font/font_DispStringBuffer.h>
#include <nw/font/font_ResourceFormat.h>
#include <nw/ut/ut_Inlines.h>

#include <nw/dev/dev_Profile.h>
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    // TODO: NintendoSdk 対応後、このコメントを削除してください。
    #include <nn/nn_Windows.h>
#elif defined(NW_PLATFORM_CAFE)
    #include <cafe/gfd.h>
    #include <cafe/os.h>
#endif

namespace shhelp = nw::gfnd::shader_helper;

namespace nw
{
namespace font
{

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。

//----------------------------------------
u32 RectDrawer::GetWorkBufferSize(u32 charNum)
{
    return (sizeof(u16) * INDEX_NUMBER_BY_LETTER) * charNum;
}

#elif defined(NW_PLATFORM_CAFE)
//----------------------------------------
u32 RectDrawer::GetWorkBufferSize(u32 charNum, const ut::MemoryRange& shaderBinary)
{
    u32 size = (sizeof(u16) * INDEX_NUMBER_BY_LETTER) * charNum;

    for (int idx = 0; idx < SHADER_QUANTITY; ++idx)
    {
        size +=
            (GX2_SHADER_ALIGNMENT - 1) + GX2CalcFetchShaderSize(VERTEX_ATTRIBUTE_NUMBER)
            + (PPC_IO_BUFFER_ALIGN - 1) + GFDGetVertexShaderHeaderSize(idx, shaderBinary.Begin())
            + (GX2_SHADER_ALIGNMENT - 1) + GFDGetVertexShaderProgramSize(idx, shaderBinary.Begin())
            + (PPC_IO_BUFFER_ALIGN - 1) + GFDGetPixelShaderHeaderSize(idx, shaderBinary.Begin())
            + (GX2_SHADER_ALIGNMENT - 1) + GFDGetPixelShaderProgramSize(idx, shaderBinary.Begin())
#if defined(NW_DISPLAY_LIST_ENABLED)
            + (gfnd::DisplayList::ALIGNMENT - 1) + DISPLAY_LIST_BUFFER_SIZE
#endif
            ;
    }

    return size;
}

#endif

//----------------------------------------
RectDrawer::RectDrawer()
 : m_MaxCharNum(0)
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
 , m_Sampler(0)
#endif
 , m_IndexBuffer(NULL)
{
}

//----------------------------------------
RectDrawer::~RectDrawer()
{
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    // TODO: NintendoSdk 対応後、このコメントを削除してください。
    if ( m_Sampler )
    {
        glDeleteSamplers( 1, &m_Sampler );
        m_Sampler = 0;
    }
#endif
    Finalize();
}

//----------------------------------------
void RectDrawer::Finalize()
{
    for (int idx = 0; idx < SHADER_QUANTITY; ++idx) { m_Shader[idx].Finalize(); }
}

//----------------------------------------
void RectDrawer::BuildTextureUseInfos(
    DispStringBuffer* pDispStringBuffer,
    bool shadowEnabled)
{
    u32 textureUseInfoPos = 0;

    // 利用回数(dispStringBuffer->textureUseInfos[i].useNum) をカウントします。
    u32 count = pDispStringBuffer->GetCharCount();
    internal::TextureUseInfo* textureUseInfos = pDispStringBuffer->GetTextureUseInfos();
    for (u32 i = 0; i < count; ++i)
    {
        const internal::TextureObject* texObj = pDispStringBuffer->GetCharAttributes()[i].GetTexObj();
        // 既に登録されているか探す
        bool isNew = true;
        for ( u32 p = 0; p < textureUseInfoPos; ++p )
        {
            if (texObj == textureUseInfos[p].texObj)
            {
                isNew = false;
                textureUseInfos[p].useNum++;
                break;
            }
        }
        if (isNew)
        {
            internal::TextureUseInfo* info = &textureUseInfos[textureUseInfoPos];
            info->texObj = texObj;
            info->useNum = 1;
            info->bitFlags = 0;

            if (texObj->IsColorBlackWhiteInterpolationEnabled())
            {
                info->bitFlags |= COLOR_BLACK_WHITE_INTERPOLATION_ENABLED;
            }

            if (pDispStringBuffer->GetCharAttributes()[i].GetFlags() & internal::CharAttribute::BORDER_EFFECT)
            {
                info->bitFlags |= BORDER_EFFECT_ENABLED;
            }

            ++textureUseInfoPos;
            // 上限に達したら抜ける
            if (textureUseInfoPos >= internal::USE_TEXTURE_NUM_MAX)
            {
                NW_WARNING(false, "texture num exceeds limit\n");
                break;
            }
        }
    }

    // 新しい useNum で startIndex を更新します。
    if (shadowEnabled)
    {
        if (1 < textureUseInfoPos)
        {
            int lastPos = textureUseInfoPos;
            textureUseInfoPos = ut::Min(textureUseInfoPos * 2, static_cast<u32>(internal::USE_TEXTURE_NUM_MAX));

            u32 index = 0;
            for (int p = 0; p < lastPos; ++p)
            {
                textureUseInfos[p].startIndex = index;
                index += textureUseInfos[p].useNum * internal::VERTEX_NUMBER_BY_LETTER;
            }
            for (u32 p = lastPos; p < textureUseInfoPos; ++p)
            {
                textureUseInfos[p].startIndex = index;
                textureUseInfos[p].texObj = textureUseInfos[p - lastPos].texObj;
                textureUseInfos[p].useNum = textureUseInfos[p - lastPos].useNum;
                textureUseInfos[p].bitFlags = textureUseInfos[p - lastPos].bitFlags;
                index += textureUseInfos[p].useNum * internal::VERTEX_NUMBER_BY_LETTER;
            }
        }
        else
        {
            u32 index = 0;
            for (u32 p = 0; p < textureUseInfoPos; ++p)
            {
                textureUseInfos[p].useNum *= 2;
                textureUseInfos[p].startIndex = index;
                index += textureUseInfos[p].useNum * internal::VERTEX_NUMBER_BY_LETTER;
            }
        }
    }
    else
    {
        u32 index = 0;
        for (u32 p = 0; p < textureUseInfoPos; ++p)
        {
            textureUseInfos[p].startIndex = index;
            index += textureUseInfos[p].useNum * internal::VERTEX_NUMBER_BY_LETTER;
        }
    }

    pDispStringBuffer->SetTextureUseInfoPos(textureUseInfoPos);
}

//----------------------------------------
void RectDrawer::BuildVertexElements(DispStringBuffer* dispStringBuffer, const ShadowParameter* shadowParam, const PerCharacterTransformInfo* perCharacterTransformInfos, u8 perCharacterTransformCenter)
{
    NW_PROFILE("nw::font::RectDrawer::BuildVertexElements");

    // 文字数が足りなければ、頂点バッファの作成は行いません。
    const u32 charCount = dispStringBuffer->GetCharCount();
    if (charCount == 0)
    {
        // 長さ0の頂点列を生成したことにする
        dispStringBuffer->SetCommandGenerated();
        dispStringBuffer->SetTextureUseInfoPos(0);
        return;
    }
    else if (m_MaxCharNum < charCount || dispStringBuffer->GetCharCountMax() < charCount)
    {
        NW_ERR("charCount[%d] is over m_MaxCharNum[%d] or dispStringBuffer->GetCharCountMax()[%d]",
            charCount, m_MaxCharNum, dispStringBuffer->GetCharCountMax());
        return;
    }

    BuildTextureUseInfos(dispStringBuffer, shadowParam != NULL);

    //----------------------------------------
    // 頂点バッファの作成
    u32 textureUseInfoPos = dispStringBuffer->GetTextureUseInfoPos();
    internal::TextureUseInfo* textureUseInfos = dispStringBuffer->GetTextureUseInfos();
    dispStringBuffer->GetVertexBuffers().PassToNext();
#if defined(NW_PLATFORM_CAFE)
    // キャッシュミスを減らすための最適化
    //
    // キャッシュサイズ(subcore で 512kB)を超えて DCZeroRange() すると逆効果なので、
    // キャッシュサイズを超えないようにします。
    const u32 MAX_CACHE_SIZE_IN_SUB_CORE = 512 * 1024;
    const u32 MAX_ZERO_RANGE_SIZE = MAX_CACHE_SIZE_IN_SUB_CORE / 2; // キャッシュを全部使い切らないようにします。
    const u32 ZERO_RANGE_SIZE_ALIGNMENT = 32;
    const u32 charCountForVertex = charCount * (shadowParam ? 2 : 1);
    IntPtr vertex_buffer = reinterpret_cast<IntPtr>(dispStringBuffer->GetVertexBuffers().GetBuffer());
    IntPtr aligned_vertex_buffer = ut::RoundUp(vertex_buffer, ZERO_RANGE_SIZE_ALIGNMENT);
    u32 zero_range_size = nw::ut::RoundDown(
            nw::ut::Min(static_cast<u32>(charCountForVertex * internal::VERTEX_NUMBER_BY_LETTER * sizeof(internal::Vertex)), MAX_ZERO_RANGE_SIZE) - (aligned_vertex_buffer - vertex_buffer),
            ZERO_RANGE_SIZE_ALIGNMENT);
    DCZeroRange(reinterpret_cast<void*>(aligned_vertex_buffer), zero_range_size);
#endif

    // 描画用の頂点列構築前にテクスチャ初期化
    for (u32 texIdx = 0; texIdx < textureUseInfoPos; ++texIdx)
    {
        textureUseInfos[texIdx].useNum = 0; // VertexElements 生成時に更新するので一旦リセット
    }

    if (shadowParam)
    {
        for (u32 i = 0; i < charCount; ++i)
        {
            internal::CharAttribute* atr = &dispStringBuffer->GetCharAttributes()[i];
            const internal::TextureObject* texObj = atr->GetTexObj();

            internal::Vertex* vtx = dispStringBuffer->GetVertexBuffers().GetBuffer();

            for (u32 p = 0; p < textureUseInfoPos; p++)
            {
                if (texObj == textureUseInfos[p].texObj)
                {
                    // 自分が入るべき位置を求める
                    vtx += static_cast<int>(
                        textureUseInfos[p].startIndex + textureUseInfos[p].useNum * internal::VERTEX_NUMBER_BY_LETTER);
                    ++textureUseInfos[p].useNum; // 利用カウントを累積
                    break;
                }
            }

            const f32 width = atr->pos.x * shadowParam->shadowScale.x;
            const f32 height = atr->pos.y * shadowParam->shadowScale.y;
            const f32 x = atr->pos.z + shadowParam->shadowOffset.x;
            const f32 y = atr->pos.w - shadowParam->shadowOffset.y + (atr->pos.y - height);
            ut::Color4u8 upperColor = shadowParam->shadowUpperColor;
            upperColor.a = upperColor.a * atr->shadowAlpha / 255;
            ut::Color4u8 lowerColor = shadowParam->shadowLowerColor;
            lowerColor.a = lowerColor.a * atr->shadowAlpha / 255;
            const f32 italicOffset = static_cast<f32>(atr->italicOffset) + shadowParam->shadowItalicOffset * shadowParam->shadowScale.x;

            if (perCharacterTransformInfos == NULL)
            {
                vtx[0].Set(x + italicOffset, y, 0, upperColor, atr->tex.x, atr->tex.y, -static_cast<float>(atr->sheetIndex) - 1.0f);
                vtx[1].Set(x + width + italicOffset, y, 0, upperColor, atr->tex.z, atr->tex.y, -static_cast<float>(atr->sheetIndex) - 1.0f);
                vtx[2].Set(x, y + height, 0, lowerColor, atr->tex.x, atr->tex.w, -static_cast<float>(atr->sheetIndex) - 1.0f);
                vtx[3].Set(x + width, y + height, 0, lowerColor, atr->tex.z, atr->tex.w, -static_cast<float>(atr->sheetIndex) - 1.0f);
            }
            else
            {
                // 回転および平行移動を計算
                const f32 centerX = (x + x + width + italicOffset) / 2.0f;
                f32 centerY;
                switch (perCharacterTransformCenter)
                {
                case 0: // 中央を回転原点にする
                    centerY = (y + y + height) / 2.0f;
                    break;
                case 1: // 下端を回転原点にする
                    centerY = y + height;
                    break;
                default:
                    centerY = 0.0f;
                    NW_ERR("Invalid value of letterMotionCenter.");
                    break;
                }
                const PerCharacterTransformInfo& perCharacterTransformInfo = perCharacterTransformInfos[i];
                const f32 matrix00 = perCharacterTransformInfo.RotationCos[1] * perCharacterTransformInfo.RotationCos[2];
                const f32 matrix01 = -perCharacterTransformInfo.RotationCos[1] * perCharacterTransformInfo.RotationSin[2];
                const f32 matrix10 = perCharacterTransformInfo.RotationSin[0] * perCharacterTransformInfo.RotationSin[1] * perCharacterTransformInfo.RotationCos[2] + perCharacterTransformInfo.RotationCos[0] * perCharacterTransformInfo.RotationSin[2];
                const f32 matrix11 = -perCharacterTransformInfo.RotationSin[0] * perCharacterTransformInfo.RotationSin[1] * perCharacterTransformInfo.RotationSin[2] + perCharacterTransformInfo.RotationCos[0] * perCharacterTransformInfo.RotationCos[2];
                const f32 matrix20 = -perCharacterTransformInfo.RotationCos[0] * perCharacterTransformInfo.RotationSin[1] * perCharacterTransformInfo.RotationCos[2] + perCharacterTransformInfo.RotationSin[0] * perCharacterTransformInfo.RotationSin[2];
                const f32 matrix21 = perCharacterTransformInfo.RotationCos[0] * perCharacterTransformInfo.RotationSin[1] * perCharacterTransformInfo.RotationSin[2] + perCharacterTransformInfo.RotationSin[0] * perCharacterTransformInfo.RotationCos[2];
                const f32 x0 = x + italicOffset - centerX;
                const f32 y0 = y - centerY;
                vtx[0].Set
                (
                    matrix00 * x0 + matrix01 * y0 + centerX + perCharacterTransformInfo.Translation[0],
                    matrix10 * x0 + matrix11 * y0 + centerY + perCharacterTransformInfo.Translation[1],
                    matrix20 * x0 + matrix21 * y0 + perCharacterTransformInfo.Translation[2],
                    upperColor, atr->tex.x, atr->tex.y, -static_cast<float>(atr->sheetIndex) - 1.0f
                );
                const f32 x1 = x + width + italicOffset - centerX;
                const f32 y1 = y - centerY;
                vtx[1].Set
                (
                    matrix00 * x1 + matrix01 * y1 + centerX + perCharacterTransformInfo.Translation[0],
                    matrix10 * x1 + matrix11 * y1 + centerY + perCharacterTransformInfo.Translation[1],
                    matrix20 * x1 + matrix21 * y1 + perCharacterTransformInfo.Translation[2],
                    upperColor, atr->tex.z, atr->tex.y, -static_cast<float>(atr->sheetIndex) - 1.0f
                );
                const f32 x2 = x - centerX;
                const f32 y2 = y + height - centerY;
                vtx[2].Set
                (
                    matrix00 * x2 + matrix01 * y2 + centerX + perCharacterTransformInfo.Translation[0],
                    matrix10 * x2 + matrix11 * y2 + centerY + perCharacterTransformInfo.Translation[1],
                    matrix20 * x2 + matrix21 * y2 + perCharacterTransformInfo.Translation[2],
                    lowerColor, atr->tex.x, atr->tex.w, -static_cast<float>(atr->sheetIndex) - 1.0f
                );
                const f32 x3 = x + width - centerX;
                const f32 y3 = y + height - centerY;
                vtx[3].Set
                (
                    matrix00 * x3 + matrix01 * y3 + centerX + perCharacterTransformInfo.Translation[0],
                    matrix10 * x3 + matrix11 * y3 + centerY + perCharacterTransformInfo.Translation[1],
                    matrix20 * x3 + matrix21 * y3 + perCharacterTransformInfo.Translation[2],
                    lowerColor, atr->tex.z, atr->tex.w, -static_cast<float>(atr->sheetIndex) - 1.0f
                );
            }
        }
    }

    for (u32 i = 0; i < charCount; ++i)
    {
        internal::CharAttribute* atr = &dispStringBuffer->GetCharAttributes()[i];
        const internal::TextureObject* texObj = atr->GetTexObj();

        internal::Vertex* vtx = dispStringBuffer->GetVertexBuffers().GetBuffer();

        for (int p = textureUseInfoPos - 1; 0 <= p; p--)
        {
            if (texObj == textureUseInfos[p].texObj)
            {
                // 自分が入るべき位置を求める
                vtx += static_cast<int>(
                    textureUseInfos[p].startIndex + textureUseInfos[p].useNum * internal::VERTEX_NUMBER_BY_LETTER);
                ++textureUseInfos[p].useNum; // 利用カウントを累積
                break;
            }
        }

        const f32 width = atr->pos.x;
        const f32 height = atr->pos.y;
        const f32 x = atr->pos.z;
        const f32 y = atr->pos.w;
        const ut::Color4u8& upperColor = atr->color[internal::TEXTCOLOR_START];
        const ut::Color4u8& lowerColor = atr->color[internal::TEXTCOLOR_END];
        const f32 italicOffset = static_cast<f32>(atr->italicOffset);

        if (perCharacterTransformInfos == NULL)
        {
            vtx[0].Set(x + italicOffset, y, 0, upperColor, atr->tex.x, atr->tex.y, atr->sheetIndex);
            vtx[1].Set(x + width + italicOffset, y, 0, upperColor, atr->tex.z, atr->tex.y, atr->sheetIndex);
            vtx[2].Set(x, y + height, 0, lowerColor, atr->tex.x, atr->tex.w, atr->sheetIndex);
            vtx[3].Set(x + width, y + height, 0, lowerColor, atr->tex.z, atr->tex.w, atr->sheetIndex);
        }
        else
        {
            // 回転および平行移動を計算
            const f32 centerX = (x + x + width + italicOffset) / 2.0f;
            f32 centerY;
            switch (perCharacterTransformCenter)
            {
            case 0:
                centerY = (y + y + height) / 2.0f;
                break;
            case 1:
                centerY = y + height;
                break;
            default:
                centerY = 0.0f;
                NW_ERR("Invalid value of letterMotionCenter.");
                break;
            }
            const PerCharacterTransformInfo& perCharacterTransformInfo = perCharacterTransformInfos[i];
            const f32 matrix00 = perCharacterTransformInfo.RotationCos[1] * perCharacterTransformInfo.RotationCos[2];
            const f32 matrix01 = -perCharacterTransformInfo.RotationCos[1] * perCharacterTransformInfo.RotationSin[2];
            const f32 matrix10 = perCharacterTransformInfo.RotationSin[0] * perCharacterTransformInfo.RotationSin[1] * perCharacterTransformInfo.RotationCos[2] + perCharacterTransformInfo.RotationCos[0] * perCharacterTransformInfo.RotationSin[2];
            const f32 matrix11 = -perCharacterTransformInfo.RotationSin[0] * perCharacterTransformInfo.RotationSin[1] * perCharacterTransformInfo.RotationSin[2] + perCharacterTransformInfo.RotationCos[0] * perCharacterTransformInfo.RotationCos[2];
            const f32 matrix20 = -perCharacterTransformInfo.RotationCos[0] * perCharacterTransformInfo.RotationSin[1] * perCharacterTransformInfo.RotationCos[2] + perCharacterTransformInfo.RotationSin[0] * perCharacterTransformInfo.RotationSin[2];
            const f32 matrix21 = perCharacterTransformInfo.RotationCos[0] * perCharacterTransformInfo.RotationSin[1] * perCharacterTransformInfo.RotationSin[2] + perCharacterTransformInfo.RotationSin[0] * perCharacterTransformInfo.RotationCos[2];
            const f32 x0 = x + italicOffset - centerX;
            const f32 y0 = y - centerY;
            const ut::Color4u8 animUpperColor = upperColor * ut::Color4u8(static_cast<s32>(perCharacterTransformInfo.LT[0]), static_cast<s32>(perCharacterTransformInfo.LT[1]), static_cast<s32>(perCharacterTransformInfo.LT[2]), static_cast<s32>(perCharacterTransformInfo.LT[3]));
            const ut::Color4u8 animLowerColor = lowerColor * ut::Color4u8(static_cast<s32>(perCharacterTransformInfo.LB[0]), static_cast<s32>(perCharacterTransformInfo.LB[1]), static_cast<s32>(perCharacterTransformInfo.LB[2]), static_cast<s32>(perCharacterTransformInfo.LB[3]));
            vtx[0].Set
            (
                matrix00 * x0 + matrix01 * y0 + centerX + perCharacterTransformInfo.Translation[0],
                matrix10 * x0 + matrix11 * y0 + centerY + perCharacterTransformInfo.Translation[1],
                matrix20 * x0 + matrix21 * y0 + perCharacterTransformInfo.Translation[2],
                animUpperColor, atr->tex.x, atr->tex.y, atr->sheetIndex
            );
            const f32 x1 = x + width + italicOffset - centerX;
            const f32 y1 = y - centerY;
            vtx[1].Set
            (
                matrix00 * x1 + matrix01 * y1 + centerX + perCharacterTransformInfo.Translation[0],
                matrix10 * x1 + matrix11 * y1 + centerY + perCharacterTransformInfo.Translation[1],
                matrix20 * x1 + matrix21 * y1 + perCharacterTransformInfo.Translation[2],
                animUpperColor, atr->tex.z, atr->tex.y, atr->sheetIndex
            );
            const f32 x2 = x - centerX;
            const f32 y2 = y + height - centerY;
            vtx[2].Set
            (
                matrix00 * x2 + matrix01 * y2 + centerX + perCharacterTransformInfo.Translation[0],
                matrix10 * x2 + matrix11 * y2 + centerY + perCharacterTransformInfo.Translation[1],
                matrix20 * x2 + matrix21 * y2 + perCharacterTransformInfo.Translation[2],
                animLowerColor, atr->tex.x, atr->tex.w, atr->sheetIndex
            );
            const f32 x3 = x + width - centerX;
            const f32 y3 = y + height - centerY;
            vtx[3].Set
            (
                matrix00 * x3 + matrix01 * y3 + centerX + perCharacterTransformInfo.Translation[0],
                matrix10 * x3 + matrix11 * y3 + centerY + perCharacterTransformInfo.Translation[1],
                matrix20 * x3 + matrix21 * y3 + perCharacterTransformInfo.Translation[2],
                animLowerColor, atr->tex.z, atr->tex.w, atr->sheetIndex
            );
        }
    }

    // コマンド生成済みをマーク
    dispStringBuffer->SetCommandGenerated();
#if defined(NW_PLATFORM_CAFE)
    // CPUのキャッシュクリア
    GX2Invalidate(GX2_INVALIDATE_CPU, dispStringBuffer->GetVertexBuffers().GetBuffer(), charCountForVertex * sizeof(internal::Vertex) * internal::VERTEX_NUMBER_BY_LETTER);
#endif

    return;
}

//----------------------------------------
void RectDrawer::Draw(const DrawContent& content)
{
    NW_PROFILE("nw::font::RectDrawer::Draw");

    math::MTX44 wvp;
    MTX44Mult(&wvp, *content.projectionMatrix, math::MTX44(*content.viewMatrix));
    math::MTX44 user;
    MTX44Mult(&user, wvp, math::MTX44(*content.localMatrix));

    ut::Color4f shadowInterpolateWidth(
        content.shadowInterpolateWhite.r - content.shadowInterpolateBlack.r,
        content.shadowInterpolateWhite.g - content.shadowInterpolateBlack.g,
        content.shadowInterpolateWhite.b - content.shadowInterpolateBlack.b,
        content.shadowInterpolateWhite.a);

    // 黒補完のアルファにはグローバルアルファが入っているので、補完幅の計算を行うときは0と見なす。
    ut::Color4f interpolateWidth(
        content.interpolateWhite.r - content.interpolateBlack.r,
        content.interpolateWhite.g - content.interpolateBlack.g,
        content.interpolateWhite.b - content.interpolateBlack.b,
        content.interpolateWhite.a);

    Shader* lastShader = NULL;
    u32 textureUseInfoPos = content.dispStringBuffer->GetTextureUseInfoPos();
    internal::TextureUseInfo* textureUseInfos = content.dispStringBuffer->GetTextureUseInfos();
    for ( u32 p = 0; p < textureUseInfoPos; p++ )
    {
        const internal::TextureUseInfo& info = textureUseInfos[p];
        if (info.useNum == 0) { continue; }

        Shader* shader;
        if (info.bitFlags & BORDER_EFFECT_ENABLED)
        {
            shader = (content.flags & DrawContent::INVISIBLE_BORDER)
                ? &m_Shader[INVISIBLE_BORDER_SHADER]
                : &m_Shader[BORDER_SHADER];
        }
        else
        {
            shader = &m_Shader[NORMAL_SHADER];
        }
        NW_NULL_ASSERT(shader);

        if (shader != lastShader)
        {
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
            // TODO: NintendoSdk 対応後、このコメントを削除してください。
            glActiveTexture( GL_TEXTURE0 );
            glBindSampler( 0, m_Sampler );
            glEnable( GL_TEXTURE_2D );
            glBindVertexArray(0);
#endif

            shader->Activate();
            NW_GL_ASSERT();

            // ユニフォーム設定
            shhelp::SetVertexUniformReg(shader->m_ParamUser, user);
            shhelp::SetPixelUniformReg(shader->m_ParamShadowInterpolateOffset, content.shadowInterpolateBlack);
            shhelp::SetPixelUniformReg(shader->m_ParamShadowInterpolateWidth, shadowInterpolateWidth);

            lastShader = shader;
        }

        if (info.bitFlags & COLOR_BLACK_WHITE_INTERPOLATION_ENABLED)
        {
            shhelp::SetPixelUniformReg(shader->m_ParamInterpolateOffset, content.interpolateBlack);
            shhelp::SetPixelUniformReg(shader->m_ParamInterpolateWidth, interpolateWidth);
        }
        else
        {
            // 白黒補完のカラーは使用しないが、アルファ値は使用する。
            shhelp::SetPixelUniformReg(shader->m_ParamInterpolateOffset, 0.f, 0.0f, 0.0f, 0.0f);
            shhelp::SetPixelUniformReg(shader->m_ParamInterpolateWidth, 1.0f, 1.0f, 1.0f, interpolateWidth.a);
        }

        #if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
        // TODO: NintendoSdk 対応後、このコメントを削除してください。
        {
            const char* dummy = reinterpret_cast<const char*>( &content.dispStringBuffer->GetVertexBuffers().GetBuffer()[info.startIndex] );
            glVertexAttribPointer( shader->m_AttrVertex, 3, GL_FLOAT, GL_FALSE, sizeof( internal::Vertex ), dummy + internal::Vertex::POSITION_OFFSET );
            glEnableVertexAttribArray( shader->m_AttrVertex );
            glVertexAttribPointer( shader->m_AttrColor, 4, GL_UNSIGNED_BYTE, GL_TRUE, sizeof( internal::Vertex ), dummy + internal::Vertex::COLOR_OFFSET );
            glEnableVertexAttribArray( shader->m_AttrColor );
            glVertexAttribPointer( shader->m_AttrTexCoord0, 3, GL_FLOAT, GL_FALSE, sizeof( internal::Vertex ), dummy + internal::Vertex::TEXCOORD_OFFSET );
            glEnableVertexAttribArray( shader->m_AttrTexCoord0 );
        }

        glBindTexture( GL_TEXTURE_2D_ARRAY, info.texObj->GetName() );
        glDrawElements( GL_QUADS, info.useNum * 4, GL_UNSIGNED_SHORT, m_IndexBuffer );
        NW_GL_ASSERT();

        #elif defined(NW_PLATFORM_CAFE)
        GX2SetAttribBuffer(
            0,
            sizeof(internal::Vertex) * info.useNum * internal::VERTEX_NUMBER_BY_LETTER,
            sizeof(internal::Vertex),
            content.dispStringBuffer->GetVertexBuffers().GetBuffer() + info.startIndex);

        GX2SetPixelTexture(info.texObj->GetTexture(), shader->m_ParamTexture);
        GX2DrawIndexed(GX2_PRIMITIVE_QUADS, info.useNum * INDEX_NUMBER_BY_LETTER, GX2_INDEX_FORMAT_U16, m_IndexBuffer);
        #endif
    }

    NW_GL_ASSERT();
}

//----------------------------------------
void RectDrawer::CreateIndices(u16* indices, u32 charNum)
{
    for( u32 i = 0; i < charNum; i++ )
    {
        u32 indexOffset = i * INDEX_NUMBER_BY_LETTER;
        u32 vertexOffset = i * internal::VERTEX_NUMBER_BY_LETTER;
        indices[indexOffset + 0] = static_cast<u16>(vertexOffset + 0);
        indices[indexOffset + 1] = static_cast<u16>(vertexOffset + 2);
        indices[indexOffset + 2] = static_cast<u16>(vertexOffset + 3);
        indices[indexOffset + 3] = static_cast<u16>(vertexOffset + 1);
    }
}

//----------------------------------------
void RectDrawer::Shader::Activate()
{
    #if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    // TODO: NintendoSdk 対応後、このコメントを削除してください。

    // ステート初期化
    glUseProgram( m_ShaderProgram );

    for (int i = 0; i < 16; ++i)
    {
        glDisableVertexAttribArray(i);
    }
    NW_GL_ASSERT();

    glBindBuffer( GL_ARRAY_BUFFER, GL_NONE );
    glBindBuffer( GL_ELEMENT_ARRAY_BUFFER, GL_NONE );

    #elif defined(NW_PLATFORM_CAFE)
    // シェーダとサンプラーのディスプレイリストをコールします。
    m_DisplayList.Call();
    #endif
}

}   // namespace font
}   // namespace nw

//########################################
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
//########################################

namespace
{
const char cVertexShaderSource[] =
{
#include "../../../../Resources/FontShaders/font_BuildinShader_win32.vsh"
};
const char cPixelShaderSource[] =
{
#include "../../../../Resources/FontShaders/font_BuildinShader_win32.psh"
};
const char cVertexShaderBorderSource[] =
{
#include "../../../../Resources/FontShaders/font_BuildinShader-bdr_win32.vsh"
};
const char cPixelShaderBorderSource[] =
{
#include "../../../../Resources/FontShaders/font_BuildinShader-bdr_win32.psh"
};
const char cVertexShaderInvisibleBorderSource[] =
{
#include "../../../../Resources/FontShaders/font_BuildinShader-ivb_win32.vsh"
};
const char cPixelShaderInvisibleBorderSource[] =
{
#include "../../../../Resources/FontShaders/font_BuildinShader-ivb_win32.psh"
};

}   // namespace

namespace nw {
namespace font {

//----------------------------------------
void RectDrawer::Initialize(void* workBuffer, u32 charNum)
{
    m_IndexBuffer = reinterpret_cast<u16*>(workBuffer);
    m_MaxCharNum = charNum;

    void* bufferPtr = reinterpret_cast<void*>(ut::RoundUp(reinterpret_cast<uptr>(workBuffer) +
        sizeof(u16) * INDEX_NUMBER_BY_LETTER * charNum, 4));
    for (int idx = 0; idx < SHADER_QUANTITY; ++idx)
    {
        bufferPtr = m_Shader[idx].Initialize(bufferPtr, idx);
    }

    // サンプラー作成、及び設定
    {
        glGenSamplers(1, &m_Sampler);
        glSamplerParameteri(m_Sampler, GL_TEXTURE_WRAP_S, GL_CLAMP);
        glSamplerParameteri(m_Sampler, GL_TEXTURE_WRAP_T, GL_CLAMP);
        // TODO: 本来はフォントにフィルタの設定があるのだが、現状では線形補完固定にする
        glSamplerParameteri(m_Sampler, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
        glSamplerParameteri(m_Sampler, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
    }

    CreateIndices(m_IndexBuffer, charNum);
}

//----------------------------------------
RectDrawer::Shader::Shader()
 : m_VertexShader(0)
 , m_FragmentShader(0)
 , m_ShaderProgram(0)
 , m_AttrVertex(0)
 , m_AttrColor(0)
 , m_AttrTexCoord0(0)
 , m_ParamUser(0)
 , m_ParamInterpolateOffset(0)
 , m_ParamInterpolateWidth(0)
 , m_ParamShadowInterpolateOffset(0)
 , m_ParamShadowInterpolateWidth(0)
{
}

//----------------------------------------
void* RectDrawer::Shader::Initialize(void* workBuffer, int idx)
{
    // VS
    {
        NW_ASSERT( m_VertexShader == 0 );
        m_VertexShader = glCreateShader( GL_VERTEX_SHADER );
        NW_GL_ASSERT();
        static const char* vsSources[] =
        {
            cVertexShaderSource,
            cVertexShaderBorderSource,
            cVertexShaderInvisibleBorderSource
        };
        NW_ASSERT(idx < ut::GetArrayLength(vsSources));
        bool ret = CreateShader( m_VertexShader, vsSources[idx] );
        if ( !ret )
        {
            NW_ERR( "Can't create VertexShader." );
            glDeleteShader( m_VertexShader );
            m_VertexShader = 0;
            return workBuffer;
        }
    }
    // FS
    {
        NW_ASSERT( m_FragmentShader == 0 );
        m_FragmentShader = glCreateShader( GL_FRAGMENT_SHADER );
        NW_GL_ASSERT();
        static const char* psSources[] =
        {
            cPixelShaderSource,
            cPixelShaderBorderSource,
            cPixelShaderInvisibleBorderSource
        };
        NW_ASSERT(idx < ut::GetArrayLength(psSources));
        bool ret = CreateShader( m_FragmentShader, psSources[idx] );
        if ( !ret )
        {
            NW_ERR( "Can't create FragmentShader." );
            glDeleteShader( m_FragmentShader );
            m_FragmentShader = 0;
            glDeleteShader( m_VertexShader );
            m_VertexShader = 0;
            return workBuffer;
        }
    }
    // リンク.
    {
        NW_ASSERT( m_ShaderProgram == 0 );
        m_ShaderProgram = glCreateProgram();
        glAttachShader( m_ShaderProgram, m_VertexShader );
        glAttachShader( m_ShaderProgram, m_FragmentShader );
        NW_GL_ASSERT();
        glLinkProgram( m_ShaderProgram );
        GLint ret;
        glGetProgramiv( m_ShaderProgram, GL_LINK_STATUS, &ret );
        NW_GL_ASSERT();
        if ( ret == GL_FALSE )
        {
            glDeleteProgram( m_ShaderProgram );
            m_ShaderProgram = 0;
            glDeleteShader( m_FragmentShader );
            m_FragmentShader = 0;
            glDeleteShader( m_VertexShader );
            m_VertexShader = 0;
            return workBuffer;
        }
    }

    // ユニフォーム/頂点属性位置のキャッシュを取得.
    {
        m_AttrVertex = glGetAttribLocation( m_ShaderProgram, "aVertex" );
        m_AttrColor = glGetAttribLocation( m_ShaderProgram, "aColor" );
        m_AttrTexCoord0 = glGetAttribLocation( m_ShaderProgram, "aTexCoord0" );
        m_ParamUser = glGetUniformLocation( m_ShaderProgram, "uUser" );
        m_ParamInterpolateOffset = glGetUniformLocation( m_ShaderProgram, "uInterpolateOffset" );
        m_ParamInterpolateWidth = glGetUniformLocation( m_ShaderProgram, "uInterpolateWidth" );
        m_ParamShadowInterpolateOffset = glGetUniformLocation( m_ShaderProgram, "uShadowInterpolateOffset" );
        m_ParamShadowInterpolateWidth = glGetUniformLocation( m_ShaderProgram, "uShadowInterpolateWidth" );
    }

    return workBuffer;
}

//----------------------------------------
void RectDrawer::Shader::Finalize()
{
    if ( m_ShaderProgram )
    {
        glDeleteProgram( m_ShaderProgram );
        m_ShaderProgram = 0;
    }
    if ( m_FragmentShader )
    {
        glDeleteShader( m_FragmentShader );
        m_FragmentShader = 0;
    }
    if ( m_VertexShader )
    {
        glDeleteShader( m_VertexShader );
        m_VertexShader = 0;
    }
}

//----------------------------------------
bool RectDrawer::Shader::CreateShader( GLuint shader, const char* source )
{
    GLint length = strlen(source) + 1;
    // コンパイル.
    glShaderSource( shader, 1, &source, &length );
    glCompileShader( shader );

    // 結果をチェック.
    GLint result;
    glGetShaderiv( shader, GL_COMPILE_STATUS, &result );
    if ( result != GL_TRUE )
    {
        int infologLength = 0;
        glGetShaderiv(shader, GL_INFO_LOG_LENGTH, &infologLength);

        if (0 < infologLength)
        {
            const int MAX_LOG_LENGTH = 1024;
            char infoLog[MAX_LOG_LENGTH];
            int charsWritten  = 0;
            glGetShaderInfoLog(shader, std::min(infologLength, MAX_LOG_LENGTH), &charsWritten, infoLog);

            NW_LOG(infoLog);
        }

        return false;
    }
    return true;
}

}   // namespace font
}   // namespace nw

//########################################
#elif defined(NW_PLATFORM_CAFE)
//########################################

namespace nw {
namespace font {

//----------------------------------------------------------

void RectDrawer::Initialize(void* workBuffer, u32 charNum, const ut::MemoryRange& shaderBinary)
{
    m_IndexBuffer = reinterpret_cast<u16*>(workBuffer);
    m_MaxCharNum = charNum;

    GX2InitSampler(&m_Sampler, GX2_TEX_CLAMP_WRAP, GX2_TEX_XY_FILTER_BILINEAR);
    GX2InitSamplerZMFilter(&m_Sampler,GX2_TEX_Z_FILTER_POINT, GX2_TEX_MIP_FILTER_NO_MIP);

    void* bufferPtr = reinterpret_cast<void*>(ut::RoundUp(reinterpret_cast<uptr>(workBuffer) +
        sizeof(u16) * INDEX_NUMBER_BY_LETTER * charNum, GX2_SHADER_ALIGNMENT));
    for (int idx = 0; idx < SHADER_QUANTITY; ++idx)
    {
        Shader& shader = m_Shader[idx];
        bufferPtr = shader.Initialize(bufferPtr, idx, shaderBinary);

#if defined(NW_DISPLAY_LIST_ENABLED)
        void* dlMemory = ut::RoundUp(bufferPtr, gfnd::DisplayList::ALIGNMENT);
        shader.m_DisplayList.Initialize(dlMemory, DISPLAY_LIST_BUFFER_SIZE);

        NW_MAKE_DISPLAY_LIST(cache, shader.m_DisplayList)
        {
            GX2SetShaders(&shader.m_FetchShader, shader.m_VertexShader, shader.m_PixelShader);
            GX2SetPixelSampler(&m_Sampler, shader.m_ParamTexture);
        }
        bufferPtr = reinterpret_cast<void*>(ut::RoundUp(
            ut::GetIntPtr(dlMemory) + DISPLAY_LIST_BUFFER_SIZE, GX2_SHADER_ALIGNMENT));
#else
        bufferPtr = reinterpret_cast<void*>(ut::RoundUp(
            ut::GetIntPtr(bufferPtr), GX2_SHADER_ALIGNMENT));
#endif
    }
    NW_ASSERT(
        ut::GetIntPtr(bufferPtr) <= (reinterpret_cast<uptr>(workBuffer) + GetWorkBufferSize(charNum, shaderBinary)));

    CreateIndices(m_IndexBuffer, charNum);
    GX2Invalidate(GX2_INVALIDATE_CPU, m_IndexBuffer, charNum * INDEX_NUMBER_BY_LETTER * sizeof(u16));
}

//----------------------------------------------------------
RectDrawer::Shader::Shader()
 : m_VertexShader(NULL)
 , m_PixelShader(NULL)
 , m_FetchShaderBuffer(NULL)
 , m_ParamTexture(0)
 , m_AttributePosition(0)
 , m_AttributeColor(0)
 , m_AttributeTexCoord0(0)
 , m_ParamUser(0)
 , m_ParamInterpolateOffset(0)
 , m_ParamInterpolateWidth(0)
 , m_ParamShadowInterpolateOffset(0)
 , m_ParamShadowInterpolateWidth(0)
{
}

//----------------------------------------------------------
void* RectDrawer::Shader::Initialize(void* workBuffer, int idx, const ut::MemoryRange& shaderBinary)
{
    m_FetchShaderBuffer = workBuffer;
    u32 fetchShaderSize = GX2CalcFetchShaderSize(VERTEX_ATTRIBUTE_NUMBER);

    void* shaderBuffer = reinterpret_cast<void*>(
        reinterpret_cast<uptr>(m_FetchShaderBuffer) + fetchShaderSize);

    // GX2: シェーダ生成

    u32 vsCount = GFDGetVertexShaderCount(shaderBinary.Begin());
    NW_ASSERT(SHADER_QUANTITY <= vsCount);
    shaderBuffer = ReadVertexShader(&m_VertexShader, idx, shaderBinary.Begin(), shaderBuffer);
    NW_NULL_ASSERT(shaderBuffer);

    u32 psCount = GFDGetPixelShaderCount(shaderBinary.Begin());
    NW_ASSERT(SHADER_QUANTITY <= psCount);
    shaderBuffer = ReadPixelShader(&m_PixelShader, idx, shaderBinary.Begin(), shaderBuffer);
    NW_NULL_ASSERT(shaderBuffer);

    m_ParamUser = static_cast<u32>(GX2GetVertexUniformVarOffset(m_VertexShader, "uUser"));
    NW_ASSERT(m_ParamUser != GX2_UNIFORM_VAR_INVALID_OFFSET);
    m_AttributePosition = static_cast<u32>(GX2GetVertexAttribVarLocation(m_VertexShader, "aVertex"));
    NW_ASSERT(m_AttributePosition != GX2_UNIFORM_VAR_INVALID_OFFSET);
    m_AttributeColor = static_cast<u32>(GX2GetVertexAttribVarLocation(m_VertexShader, "aColor"));
    NW_ASSERT(m_AttributeColor != GX2_UNIFORM_VAR_INVALID_OFFSET);
    m_AttributeTexCoord0 = static_cast<u32>(GX2GetVertexAttribVarLocation(m_VertexShader, "aTexCoord0"));
    NW_ASSERT(m_AttributeTexCoord0 != GX2_UNIFORM_VAR_INVALID_OFFSET);

    m_ParamTexture = static_cast<u32>(GX2GetPixelSamplerVarLocation(m_PixelShader, "uTextureSrc"));
    NW_ASSERT(m_ParamTexture != GX2_UNIFORM_VAR_INVALID_OFFSET);

    m_ParamInterpolateOffset = static_cast<u32>(GX2GetPixelUniformVarOffset(m_PixelShader, "uInterpolateOffset"));
    NW_ASSERT(m_ParamInterpolateOffset != GX2_UNIFORM_VAR_INVALID_OFFSET);
    m_ParamInterpolateWidth = static_cast<u32>(GX2GetPixelUniformVarOffset(m_PixelShader, "uInterpolateWidth"));
    NW_ASSERT(m_ParamInterpolateWidth != GX2_UNIFORM_VAR_INVALID_OFFSET);

    m_ParamShadowInterpolateOffset = static_cast<u32>(GX2GetPixelUniformVarOffset(m_PixelShader, "uShadowInterpolateOffset"));
    NW_ASSERT(m_ParamShadowInterpolateOffset != GX2_UNIFORM_VAR_INVALID_OFFSET);
    m_ParamShadowInterpolateWidth = static_cast<u32>(GX2GetPixelUniformVarOffset(m_PixelShader, "uShadowInterpolateWidth"));
    NW_ASSERT(m_ParamShadowInterpolateWidth != GX2_UNIFORM_VAR_INVALID_OFFSET);

    GX2InitAttribStream(
        m_Attributes + 0, m_AttributePosition,
        0, internal::Vertex::POSITION_OFFSET,
        GX2_ATTRIB_FORMAT_32_32_32_FLOAT);
    GX2InitAttribStream(
        m_Attributes + 1, m_AttributeColor,
        0, internal::Vertex::COLOR_OFFSET,
        GX2_ATTRIB_FORMAT_8_8_8_8_UNORM);
    GX2InitAttribStream(
        m_Attributes + 2, m_AttributeTexCoord0,
        0, internal::Vertex::TEXCOORD_OFFSET,
        GX2_ATTRIB_FORMAT_32_32_32_FLOAT);

    GX2InitFetchShader(&m_FetchShader, m_FetchShaderBuffer, VERTEX_ATTRIBUTE_NUMBER, m_Attributes);
    GX2Invalidate(GX2_INVALIDATE_CPU, m_FetchShaderBuffer, fetchShaderSize);

    return shaderBuffer;
}

//----------------------------------------------------------
void RectDrawer::Shader::Finalize()
{
    // GX2: シェーダ破棄
    m_VertexShader = NULL;
    m_PixelShader = NULL;
}

//----------------------------------------
void* RectDrawer::Shader::ReadVertexShader(GX2VertexShader** ppShader, u32 index, const void* data, void* buffer)
{
    if (data == NULL || ppShader == NULL || buffer == NULL) { return NULL; }
    if (index >= GFDGetVertexShaderCount(data)) { return NULL; }

    u32 headerSize = GFDGetVertexShaderHeaderSize(index, data);
    u32 programSize = GFDGetVertexShaderProgramSize(index, data);

    if (!headerSize || !programSize) { return NULL; }

    GX2VertexShader* header = static_cast<GX2VertexShader*>(
        ut::RoundUp(buffer, PPC_IO_BUFFER_ALIGN));
    buffer = reinterpret_cast<void*>(reinterpret_cast<uptr>(header) + headerSize);

    void* program = static_cast<void*>(
        ut::RoundUp(buffer, GX2_SHADER_ALIGNMENT));
    buffer = reinterpret_cast<void*>(reinterpret_cast<uptr>(program) + programSize);

    u32 ret = GFDGetVertexShader(header, program, index, data);
    if (ret)
    {
        GX2Invalidate(GX2_INVALIDATE_CPU,
                      header->shaderPtr,
                      header->shaderSize);
        *ppShader = header;

        return buffer;
    }
    else
    {
        NW_LOG("Warning: Invalid Vertex Shader :%d", ret);

        return NULL;
    }
}

//----------------------------------------
void* RectDrawer::Shader::ReadPixelShader(GX2PixelShader** ppShader, u32 index, const void* data, void* buffer)
{
    if (data == NULL || ppShader == NULL || buffer == NULL) { return NULL; }
    if (index >= GFDGetPixelShaderCount(data)) { return NULL; }

    u32 headerSize = GFDGetPixelShaderHeaderSize(index, data);
    u32 programSize = GFDGetPixelShaderProgramSize(index, data);

    if (!headerSize || !programSize) { return NULL; }

    GX2PixelShader* header = static_cast<GX2PixelShader*>(
        ut::RoundUp(buffer, PPC_IO_BUFFER_ALIGN));
    buffer = reinterpret_cast<void*>(reinterpret_cast<uptr>(header) + headerSize);

    void* program = static_cast<void*>(
        ut::RoundUp(buffer, GX2_SHADER_ALIGNMENT));
    buffer = reinterpret_cast<void*>(reinterpret_cast<uptr>(program) + programSize);

    u32 ret = GFDGetPixelShader(header, program, index, data);
    if (ret)
    {
        GX2Invalidate(GX2_INVALIDATE_CPU,
                      header->shaderPtr,
                      header->shaderSize);
        *ppShader = header;

        return buffer;
    }
    else
    {
        NW_LOG("Warning: Invalid Pixel Shader :%d", ret);

        return NULL;
    }
}

}   // namespace font
}   // namespace nw

#endif

