﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/dw/system/dw_DrawTextArgs.h>
#include <nw/dw/system/dw_NwTextRenderer.h>
#include <nw/dw/system/dw_NwTypeUtility.h>

namespace nw {
namespace internal {
namespace dw {

NwTextRenderer::NwTextRenderer() :
m_TextWriter(32),
m_ProjectionMatrix(nw::math::Matrix44::Identity()),
m_ViewMatrix(nw::math::Matrix34::Identity()),
m_GlobalScale(1.f)
{
}

bool NwTextRenderer::Initialize(
    nw::ut::IAllocator& allocator,
    void* pFontBinary,
    u32 fontBinarySize,
    void* pShaderBinary,
    u32 shaderBinarySize)
{
#ifdef NW_PLATFORM_CAFE
    NW_NULL_ASSERT(pShaderBinary);
    NW_ASSERT(shaderBinarySize > 0);
#endif

    m_TextWriter.Initialize(pFontBinary, fontBinarySize, pShaderBinary, shaderBinarySize, &allocator);
    return true;
}

void NwTextRenderer::Finalize()
{
    m_TextWriter.Finalize();
}

void NwTextRenderer::SetProjectionMatrix(const nw::math::Matrix44& matrix)
{
    m_ProjectionMatrix = matrix;
}

void NwTextRenderer::SetViewMatrix(const nw::math::Matrix34& matrix)
{
    m_ViewMatrix = matrix;
}

void NwTextRenderer::DrawText(const DrawTextArgs& arg, const char* pText)
{
    m_TextWriter.SetCursor(arg.topLeft.x, arg.topLeft.y);
    m_TextWriter.SetScale(arg.scale * m_GlobalScale);
    m_TextWriter.SetTextColor(arg.color);
    m_TextWriter.SetMatrix(m_ProjectionMatrix, m_ViewMatrix);

    m_TextWriter.Print(pText);
}

void NwTextRenderer::DrawTextFormat(const DrawTextArgs& arg, const char* pFormat, ...)
{
    va_list list;
    va_start(list, pFormat);

    VDrawTextFormat(arg, pFormat, list);

    va_end(list);
}

void NwTextRenderer::VDrawTextFormat(const DrawTextArgs& arg, const char* pFormat, va_list valist)
{
    m_TextWriter.SetCursor(arg.topLeft.x, arg.topLeft.y);
    m_TextWriter.SetScale(arg.scale * m_GlobalScale);
    m_TextWriter.SetTextColor(arg.color);
    m_TextWriter.SetMatrix(m_ProjectionMatrix, m_ViewMatrix);

    m_TextWriter.VPrintf(pFormat, valist);
}

nw::math::Vector2 NwTextRenderer::MeasureText(const DrawTextArgs& arg, const char* pText)
{
    m_TextWriter.SetCursor(0.f, 0.f);
    m_TextWriter.SetScale(arg.scale * m_GlobalScale);
    m_TextWriter.SetMatrix(m_ProjectionMatrix, m_ViewMatrix);

    nw::ut::Rect boundBox;
    m_TextWriter.GetTextWriter().CalcStringRect(&boundBox, pText);

    return nw::math::Vector2(boundBox.GetWidth(), boundBox.GetHeight());
}

nw::math::Vector2 NwTextRenderer::MeasureTextFormat(const DrawTextArgs& arg, const char* pFormat, ...)
{
    va_list list;
    va_start(list, pFormat);

    nw::math::Vector2 result = VMeasureTextFormat(arg, pFormat, list);

    va_end(list);

    return result;
}

nw::math::Vector2 NwTextRenderer::VMeasureTextFormat(const DrawTextArgs& arg, const char* pFormat, va_list valist)
{
    m_TextWriter.SetCursor(0.f, 0.f);
    m_TextWriter.SetScale(arg.scale * m_GlobalScale);
    m_TextWriter.SetMatrix(m_ProjectionMatrix, m_ViewMatrix);

    nw::ut::Rect boundBox;
    m_TextWriter.GetTextWriter().CalcFormatStringRect(&boundBox, pFormat, valist);

    return nw::math::Vector2(boundBox.GetWidth(), boundBox.GetHeight());
}

void NwTextRenderer::Clear()
{
    m_TextWriter.Clear();
}

void NwTextRenderer::Flush()
{
    m_TextWriter.Flush();
}

} // dw
} // internal
} // nw
