﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/dw/control/dw_DockPanel.h>
#include <nw/dw/control/dw_UIElementAligner.h>

namespace nw {
namespace internal {
namespace dw {

DockPanel::DockPanel()
{
    SetIsFocusable(false);
    SetMargin(Thickness(0.f));
    SetPadding(Thickness(0.f));
}

const nw::math::Vector2 DockPanel::OnMeasure(UIRenderer& renderer) const
{
    (void)renderer;

    const UIElementList& contents = GetContents();

    nw::math::Vector2 result = nw::math::Vector2(0.f, 0.f);

    // DockPanel ではパディングを無視します。
    float currentOrientationX = 0.f;
    float currentOrientationY = 0.f;
    float fillX = 0.f;
    float fillY = 0.f;

    for(int i=0; i<contents.GetCount(); ++i)
    {
        UIElement* pContent = contents.GetItem(i);

        if(pContent == NULL ||
            pContent->GetVisibility() == COLLAPSED)
        {
            continue;
        }

        const nw::math::Vector2 contentSize = pContent->GetMeasuredSize();
        const Thickness margin = pContent->GetMargin();

        f32 width  = contentSize.x + margin.GetWidth();
        f32 height = contentSize.y + margin.GetHeight();

        switch(pContent->GetDock())
        {
        case DOCK_LEFT:
        case DOCK_RIGHT:
            {
                result.x = currentOrientationX + width;
                result.y = nw::ut::Max(result.y, currentOrientationY + height);
                currentOrientationX += width;
            }
            break;

        case DOCK_TOP:
        case DOCK_BOTTOM:
            {
                result.x = nw::ut::Max(result.x, currentOrientationX + width);
                result.y = currentOrientationY + height;
                currentOrientationY += height;
            }
            break;

        case DOCK_FILL:
            fillX = nw::ut::Max(fillX, width);
            fillY = nw::ut::Max(fillY, height);
            continue;
        }
    }

    result.x = nw::ut::Max(result.x, currentOrientationX + fillX);
    result.y = nw::ut::Max(result.y, currentOrientationY + fillY);

    return result;
}

void DockPanel::OnUpdate(const UIElementTreeContext& context)
{
    (void)context;

    // DockPanel ではパディングを無視します。
    const nw::math::Vector2 topLeft(0.f, 0.f);
    const nw::math::Vector2 size = GetFixedSize();
    const UIElementList& contents = GetContents();

    float currentX = topLeft.x;
    float currentY = topLeft.y;
    nw::math::Vector2 restSize(size);

    for(int i=0; i<contents.GetCount(); ++i)
    {
        UIElement* pContent = contents.GetItem(i);

        if(pContent == NULL ||
            pContent->GetVisibility() == COLLAPSED)
        {
            continue;
        }

        const Thickness margin = pContent->GetMargin();

        switch(pContent->GetDock())
        {
        case DOCK_LEFT:
            {
                f32 width = pContent->GetMeasuredWidth() + margin.GetWidth();
                const nw::ut::Rect alignedRect =
                    UIElementAligner::GetAlignedRect(*pContent, currentX, currentY, width, restSize.y);

                pContent->SetTopLeft(alignedRect.left, alignedRect.top);
                UIElement::SetMeasuredSize(*pContent, alignedRect.GetWidth(), alignedRect.GetHeight());

                currentX += width;
                restSize.x -= width;
            }
            break;

        case DOCK_TOP:
            {
                f32 height = pContent->GetMeasuredHeight() + margin.GetHeight();
                const nw::ut::Rect alignedRect =
                    UIElementAligner::GetAlignedRect(*pContent, currentX, currentY, restSize.x, height);

                pContent->SetTopLeft(alignedRect.left, alignedRect.top);
                UIElement::SetMeasuredSize(*pContent, alignedRect.GetWidth(), alignedRect.GetHeight());

                currentY += height;
                restSize.y -= height;
            }
            break;

        case DOCK_RIGHT:
            {
                f32 width = pContent->GetMeasuredWidth() + margin.GetWidth();
                const nw::ut::Rect alignedRect =
                    UIElementAligner::GetAlignedRect(*pContent, currentX + restSize.x - width, currentY, width, restSize.y);

                pContent->SetTopLeft(alignedRect.left, alignedRect.top);
                UIElement::SetMeasuredSize(*pContent, alignedRect.GetWidth(), alignedRect.GetHeight());

                restSize.x -= width;
            }
            break;

        case DOCK_BOTTOM:
            {
                f32 height = pContent->GetMeasuredHeight() + margin.GetHeight();
                const nw::ut::Rect alignedRect =
                    UIElementAligner::GetAlignedRect(*pContent, currentX, currentY + restSize.y - height, restSize.x, height);

                pContent->SetTopLeft(alignedRect.left, alignedRect.top);
                UIElement::SetMeasuredSize(*pContent, alignedRect.GetWidth(), alignedRect.GetHeight());

                restSize.y -= height;
            }
            break;

        case DOCK_FILL:
            {
                const nw::ut::Rect alignedRect =
                    UIElementAligner::GetAlignedRect(*pContent, currentX, currentY, restSize.x, restSize.y);

                pContent->SetTopLeft(alignedRect.left, alignedRect.top);
                UIElement::SetMeasuredSize(*pContent, alignedRect.GetWidth(), alignedRect.GetHeight());
            }
            break;

        default:
            NW_ASSERTMSG(false, "invalid Dock type.");
            break;
        }
    }
}

} // dw
} // internal
} // nw
