﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/dev/win/dev_VPadWin.h>
#include <nw/dev/dev_PadUtil.h>
#include <windows.h>
#include <mmsystem.h>

namespace nw
{
namespace dev
{

//------------------------------------------------------------------------------
const u8 VPadWin::DEFAULT_V_KEY_CONFIG[PAD_IDX_MAX] =
{
    'A', // IDX_A
    'B', // IDX_B
    0, // IDX_C
    'X', // IDX_X
    'Y', // IDX_Y
    0, // IDX_Z
    0, // IDX_2
    0, // IDX_1
    'H', // IDX_HOME
    'M', // IDX_MINUS
    'P', // IDX_PLUS
    0, // IDX_START
    0, // IDX_SELECT
    'K', // IDX_ZL
    'E', // IDX_ZR
    'L', // IDX_L
    'R', // IDX_R
    VK_LBUTTON, // IDX_TOUCH
    VK_UP,      // IDX_UP
    VK_DOWN,    // IDX_DOWN
    VK_LEFT,    // IDX_LEFT
    VK_RIGHT,   // IDX_RIGHT
    0, // IDX_LS_UP
    0, // IDX_LS_DOWN
    0, // IDX_LS_LEFT
    0, // IDX_LS_RIGHT
    0, // IDX_RS_UP
    0, // IDX_RS_DOWN
    0, // IDX_RS_LEFT
    0, // IDX_RS_RIGHT
};

const u8 VPadWin::DEFAULT_JOY_PAD_CONFIG[PAD_IDX_MAX] =
{
    4,  // IDX_A
    3,  // IDX_B
    0,  // IDX_C
    2,  // IDX_X
    1,  // IDX_Y
    0,  // IDX_Z
    10, // IDX_2
    9,  // IDX_1
    0,  // IDX_HOME
    11, // IDX_MINUS
    12, // IDX_PLUS
    0,  // IDX_START
    0,  // IDX_SELECT
    7,  // IDX_ZL
    8,  // IDX_ZR
    5,  // IDX_L
    6,  // IDX_R
    0,  // IDX_TOUCH
    0,  // IDX_UP
    0,  // IDX_DOWN
    0,  // IDX_LEFT
    0,  // IDX_RIGHT
    1,  // IDX_LS_UP  -> 上下逆転させる
    0,  // IDX_LS_DOWN
    0,  // IDX_LS_LEFT
    0,  // IDX_LS_RIGHT
    1,  // IDX_RS_UP  -> 上下逆転させる
    0,  // IDX_RS_DOWN
    0,  // IDX_RS_LEFT
    0,  // IDX_RS_RIGHT
};

//------------------------------------------------------------------------------
VPadWin::VPadWin(int joyPadNumber)
: Pad(),
  m_JoyPadNumber( joyPadNumber ),
  m_StickClampMax( STICK_CLAMP_MAX_DEFAULT ),
  m_StickClampMin( STICK_CLAMP_MIN_DEFAULT ),
  m_Flag( MASK_NORMALIZE_ANALOG_STICK_AS_CIRCLE ),
  m_KeyboardMouseDevice( NULL ),
  m_JoyPadDevice( NULL )
{
    SetVKeyConfigDefault();
    SetJoyPadConfigDefault();
}

//------------------------------------------------------------------------------
void
VPadWin::SetVKeyConfig(const u8* vkeyConfig)
{
    nw::ut::MemCpy( m_VKeyConfig, vkeyConfig, sizeof( m_VKeyConfig ) );
}

//------------------------------------------------------------------------------
void
VPadWin::SetVKeyConfig(PadIdx bit, u8 vkey)
{
    if (bit < PAD_IDX_MAX)
    {
        m_VKeyConfig[ bit ] = vkey;
    }
    else
    {
        NW_ERR( "range over[%d]\n", bit );
    }
}

//------------------------------------------------------------------------------
void
VPadWin::SetVKeyConfigDefault()
{
    nw::ut::MemCpy( m_VKeyConfig, DEFAULT_V_KEY_CONFIG, sizeof(m_VKeyConfig) );
}


//------------------------------------------------------------------------------
bool
VPadWin::IsJoyPadEnable() const
{
    if ( m_JoyPadDevice )
    {
        return ( m_JoyPadNumber < m_JoyPadDevice->GetPadNum() && m_JoyPadDevice->GetPads(m_JoyPadNumber).isEnable );
    }
    else
    {
        return false;
    }
}

//------------------------------------------------------------------------------
void
VPadWin::GetStickClamp(int* min, int* max) const
{
    NW_ASSERT_NOT_NULL( min );
    NW_ASSERT_NOT_NULL( max );

    *min = m_StickClampMin;
    *max = m_StickClampMax;
}

//------------------------------------------------------------------------------
void
VPadWin::SetStickClamp(int min, int max)
{
    if (STICK_CLAMP_VALUE_MIN <= min && min <= STICK_CLAMP_VALUE_MAX &&
        STICK_CLAMP_VALUE_MIN <= max && max <= STICK_CLAMP_VALUE_MAX &&
        min < max)
    {
        m_StickClampMin = min;
        m_StickClampMax = max;
    }
    else
    {
        NW_ERR( "illgal value[%d/%d]", min, max );
    }
}

//------------------------------------------------------------------------------
void
VPadWin::SetJoyPadConfig(const u8* joyPadConfig)
{
    nw::ut::MemCpy( m_JoyPadConfig, joyPadConfig, sizeof(m_JoyPadConfig) );
}

//------------------------------------------------------------------------------
void
VPadWin::SetJoyPadConfig(PadIdx bit, u8 no)
{
    if (bit < PAD_IDX_MAX)
    {
        m_JoyPadConfig[bit] = no;
    }
    else
    {
        NW_ERR( "range over[%d]\n", bit );
    }
}

//------------------------------------------------------------------------------
void
VPadWin::SetJoyPadConfigDefault()
{
    nw::ut::MemCpy( m_JoyPadConfig, DEFAULT_JOY_PAD_CONFIG, sizeof(m_JoyPadConfig) );
}


//------------------------------------------------------------------------------
void
VPadWin::UpdateImpl()
{
    // パッドをクリア
    m_PadHold.SetAllZero();

    // サブウインドウがアクティブな場合に、キーボード・マウスの入力を受け付けます。
    bool isActive = false;
    if ( m_KeyboardMouseDevice )
    {
        isActive = ( m_KeyboardMouseDevice->GetActiveWindow() == KeyboardMouseDeviceWin::ACTIVE_WINDOW_SUB ? true : false );
    }

    // m_PadHold の設定
    if( isActive && m_KeyboardMouseDevice->IsKeyEnable() )
    {
        for ( int i = 0; i <= IDX_RIGHT; ++i )
        {
            if( m_KeyboardMouseDevice->IsVkeyHold( m_VKeyConfig[i] ) )
            {
                m_PadHold.SetBitOn( i );
            }
        }
    }

    // ジョイパッドからの設定取得
    bool isJoypadEnable = false;
    if ( m_JoyPadDevice != NULL && isActive && m_JoyPadNumber < m_JoyPadDevice->GetPadNum() )
    {
        const JoyPadDeviceWin::PadStatus& pad = m_JoyPadDevice->GetPads( m_JoyPadNumber );
        JOYINFOEX* padInfo = reinterpret_cast<JOYINFOEX*>(const_cast<u32*>(pad.info));

        if ( pad.isEnable )
        {
            // ジョイパッドが有効
            isJoypadEnable = true;

            // パッドの値をマージする
            for ( int i = 0; i <= IDX_R; ++i )
            {
                int number = m_JoyPadConfig[i];

                if ( number > 0 && ((padInfo->dwButtons & (0x1 << (number - 1))) != 0) )
                {
                    m_PadHold.SetBitOn( i );
                }
            }

            // ハットスイッチから十字キーを取得
            if (padInfo->dwPOV < 0x8000)
            {
                int pov = (padInfo->dwPOV & 0xf000) >> 12;

                if (1 <= pov && pov <= 3)
                {
                    m_PadHold.SetMaskOn( MASK_RIGHT );
                }

                if (3 <= pov && pov <= 5)
                {
                    m_PadHold.SetMaskOn( MASK_DOWN );
                }

                if (5 <= pov && pov <= 7)
                {
                    m_PadHold.SetMaskOn( MASK_LEFT );
                }

                if (pov == 0 || pov == 1 || pov == 7)
                {
                    m_PadHold.SetMaskOn( MASK_UP );
                }
            }

            // アナログスティックの値を設定
            detail::SetStickValueWithClamp( &m_LeftStick.x, padInfo->dwXpos, m_JoyPadConfig[IDX_LS_RIGHT] != 0, m_StickClampMin, m_StickClampMax );
            detail::SetStickValueWithClamp( &m_LeftStick.y, padInfo->dwYpos, m_JoyPadConfig[IDX_LS_UP] != 0, m_StickClampMin, m_StickClampMax );
            detail::SetStickValueWithClamp( &m_RightStick.x, padInfo->dwZpos, m_JoyPadConfig[IDX_RS_RIGHT] != 0, m_StickClampMin, m_StickClampMax );
            detail::SetStickValueWithClamp( &m_RightStick.y, padInfo->dwRpos, m_JoyPadConfig[IDX_RS_UP] != 0, m_StickClampMin, m_StickClampMax );

            // 設定が有効になっていたら円形への正規化を行う
            if (IsEnableNormalizeAnalogStickAsCircle())
            {
                detail::NormalizeAnalogStickAsCircle( &m_LeftStick );
                detail::NormalizeAnalogStickAsCircle( &m_RightStick );
            }
        }
    }

    if ( !isJoypadEnable )
    {
        m_LeftStick = nw::math::VEC2::Zero();
        m_RightStick = nw::math::VEC2::Zero();
    }

    m_LeftAnalogTrigger = 0.f;
    m_RightAnalogTrigger = 0.f;

    // ポインティングデバイスの設定
    if ( isActive )
    {
        m_PointerPrev = m_Pointer;

        SetPointerWithBound(
            m_KeyboardMouseDevice->IsCursorOnClientArea(),
            m_PadHold.IsMaskOn( MASK_TOUCH ),
            m_KeyboardMouseDevice->GetCursorClientAreaPos()
        );
    }
    else
    {
        // ウィンドウがアクティブでないときは値をクリア
        m_PointerFlag.SetAllZero();
        m_PadHold.SetMaskOff( MASK_TOUCH );
    }
}

} // namespace dev
} // namespace nw

