﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/dev/dev_MeterDrawer.h>
#include <nw/gfnd/gfnd_Graphics.h>

#if defined(NW_PLATFORM_CAFE)
#include <cafe/gx2.h>
#endif

namespace nw
{
namespace dev
{

//------------------------------------------------------------------------------
void
MeterDrawer::AttachLoadMeter( LoadMeterBase* meter )
{
    m_LoadMeterList.push_back( meter );
}

//------------------------------------------------------------------------------
void
MeterDrawer::DetachLoadMeter( LoadMeterBase* meter )
{
    m_LoadMeterList.erase( meter );
}

//------------------------------------------------------------------------------
void
MeterDrawer::EndFrame()
{
    for ( LoadMeterList::iterator it = m_LoadMeterList.begin(); it != m_LoadMeterList.end(); ++it )
    {
        it->OnEndFrame();
    }

    m_FrameMeter->OnEndFrame();
}

//------------------------------------------------------------------------------
f32
MeterDrawer::GetAdjustedHeight() const
{
    if ( m_BarHeight > 0.f )
    {
        if ( m_LoadMeterList.size() > 0 )
        {
            return m_BarHeight * m_LoadMeterList.size();
        }
        else
        {
            return m_BarHeight;
        }
    }

    return m_Height;
}

//------------------------------------------------------------------------------
void
MeterDrawer::Draw()
{
    if ( ! m_Visible )
    {
        return;
    }

    nw::gfnd::Graphics* graphics = nw::gfnd::Graphics::GetInstance();
    graphics->SetBlendEnable( true );
    graphics->SetDepthTestEnable( false );
    graphics->SetDepthWriteEnable( false );
    graphics->SetBlendFactor( nw::gfnd::Graphics::BLEND_FACTOR_SRC_ALPHA, nw::gfnd::Graphics::BLEND_FACTOR_INV_SRC_ALPHA );
    graphics->SetBlendEquation( nw::gfnd::Graphics::BLEND_EQUATION_ADD );
    graphics->SetCullingMode( nw::gfnd::Graphics::CULLING_MODE_NONE );

    // フレームの区切り本数を計算します。
    if ( m_SectionNum == 0 )
    {
        u32 currentNum = CalcMaxSectionNum();

        if ( m_CurrentSectionNum <= currentNum )
        {
            m_AdjustCounter = m_AdjustTime;
            m_CurrentSectionNum = currentNum;
        }
        else
        {
            if ( m_AdjustCounter > 0 )
            {
                m_AdjustCounter--;
            }
            else
            {
                m_CurrentSectionNum = currentNum;
            }
        }
    }

    u32 sectionNum = m_SectionNum;

    if ( sectionNum == 0 )
    {
        sectionNum = m_CurrentSectionNum;
    }
    if ( sectionNum < m_MinSectionNum )
    {
        sectionNum = m_MinSectionNum;
    }

    // メーター全体の高さ、各メーターの高さを計算します。
    f32 height = GetAdjustedHeight();
    f32 barHight = ( m_BarHeight > 0.f ? m_BarHeight : height / m_LoadMeterList.size() );

    const LoadMeterBase::Section& sectionFrame = m_FrameMeter->GetLastResult( 0 );
    u64 sectionFrameTime = static_cast<u64>( sectionFrame.end - sectionFrame.begin );
    u64 referenceFrameTime = nw::ut::Tick( m_ReferenceFrame );

    nw::dev::PrimitiveRenderer* renderer = nw::dev::PrimitiveRenderer::GetInstance();
    renderer->SetProjectionMtx( m_ProjMtx );
    renderer->SetViewMtx( m_ViewMtx );
    renderer->SetModelMatrix( m_ModelMtx );

    renderer->Begin();
    {
        // フレーム全体の処理時間を描画します。
        {
            nw::dev::PrimitiveRenderer::QuadArg arg;
            arg.SetCornerAndSize(
                m_Position.x,
                m_Position.y,
                0.0f,
                CalcTimeLineWidth( m_Width, sectionNum, sectionFrameTime ),
                height
            );
            arg.SetColor( sectionFrame.color );
            renderer->DrawQuad( arg );
        }

        // 登録されている負荷メーターを描画します。
        s32 listNum = 0;
        for ( LoadMeterList::iterator it = m_LoadMeterList.begin(); it != m_LoadMeterList.end(); ++it )
        {
            for ( s32 i = 0; i < it->GetLastSectionNum(); ++i )
            {
                const LoadMeterBase::Section& section = it->GetLastResult( i );

                f32 posLeft = CalcTimeLinePos( m_Width, sectionNum, section.begin );
                f32 posRight = CalcTimeLinePos( m_Width, sectionNum, section.end );

                f32 adjustedBarHight = barHight * ( section.parent >= 0 ? 0.8f : 1.f );

                nw::dev::PrimitiveRenderer::QuadArg arg;
                arg.SetCornerAndSize(
                    m_Position.x + posLeft,
                    m_Position.y + barHight * listNum + ( barHight - adjustedBarHight ),
                    0.0f,
                    posRight - posLeft,
                    adjustedBarHight
                );
                arg.SetColor( nw::ut::Color4u8( section.color.r / 2, section.color.g / 2, section.color.b / 2, section.color.a ), section.color );
                renderer->DrawQuad( arg );
            }

            ++listNum;
        }

        // 全体の枠を描画します。
        {
            nw::dev::PrimitiveRenderer::QuadArg arg;
            arg.SetCornerAndSize(
                m_Position.x,
                m_Position.y,
                0.0f,
                m_Width,
                height
            );
            arg.SetColor( m_BorderColor );
            renderer->DrawBox( arg );
        }

        // フレームの区切り線を描画します。
        {
            u64 tick = sectionFrame.begin;
            u32 sectionLineNum = sectionNum;

            if ( m_MaxSectionSeparatorNum != 0 && sectionLineNum > m_MaxSectionSeparatorNum )
            {
                sectionLineNum = m_MaxSectionSeparatorNum;
            }

            for ( u32 i = 0; i < sectionLineNum - 1; ++i )
            {
                tick += referenceFrameTime;

                f32 posX = CalcTimeLinePos( m_Width, sectionNum, tick );
                renderer->DrawLine(
                    nw::math::VEC3( m_Position.x + posX, m_Position.y, 0.f ),
                    nw::math::VEC3( m_Position.x + posX, m_Position.y + height, 0.f ),
                    m_BorderColor
                );
            }
        }
    }
    renderer->End();

    // メーターの名前と処理率を出力します。
    if ( m_TextVisible && m_TextWriter )
    {
        m_TextWriter->SetFontSize( m_FontSize );
        f32 scale = barHight / m_TextWriter->GetTextWriter().GetLineHeight();
        m_TextWriter->SetScale( scale < 1.f ? scale : 1.f );
        m_TextWriter->SetTextColor( m_TextColor );
        m_TextWriter->SetShadowColor( nw::ut::Color4u8( 0, 0, 0, 128 ) );

        s32 listNum = 0;
        for ( LoadMeterList::iterator it = m_LoadMeterList.begin(); it != m_LoadMeterList.end(); ++it )
        {
            f32 rate = 100.f * it->GetLastTotalSpan() / referenceFrameTime;
            m_TextWriter->SetCursor( m_Position.x + 2.f, m_Position.y + barHight * listNum );
            m_TextWriter->PrintfWithShadow( "%s ( %.2f%% )", it->GetName(), rate );
            ++listNum;
        }

        m_TextWriter->Flush();
    }
}

//------------------------------------------------------------------------------
void
MeterDrawer::Dump()
{
    u64 referenceFrameTime = nw::ut::Tick( m_ReferenceFrame );

    for ( LoadMeterList::iterator it = m_LoadMeterList.begin(); it != m_LoadMeterList.end(); ++it )
    {
        nw::ut::Tick tick = it->GetLastTotalSpan();
        f32 rate = 100.f * tick / referenceFrameTime;
        NW_LOG("[ %s ] %lld usec (%f%%)\n", it->GetName(), tick.ToTimeSpan().GetMicroSeconds(), rate );
#if !defined(NW_CONSOLE_ENABLE) && (defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK))
        (void)rate;
#endif
    }
}

//------------------------------------------------------------------------------
f32
MeterDrawer::CalcTimeLinePos( f32 maxWidth, u32 sectionNum, u64 time )
{
    const LoadMeterBase::Section& sectionFrame = m_FrameMeter->GetLastResult( 0 );
    return CalcTimeLineWidth( maxWidth, sectionNum, time - static_cast<u64>( sectionFrame.begin ) );
}

//------------------------------------------------------------------------------
f32
MeterDrawer::CalcTimeLineWidth( f32 maxWidth, u32 sectionNum, u64 time )
{
    NW_ASSERT( sectionNum > 0 );

    u64 totalTime = nw::ut::Tick( m_ReferenceFrame ) * sectionNum;
    return maxWidth * time / totalTime;
}

//------------------------------------------------------------------------------
u32
MeterDrawer::CalcMaxSectionNum()
{
    const LoadMeterBase::Section& sectionFrame = m_FrameMeter->GetLastResult( 0 );
    u64 referenceFrameTime = nw::ut::Tick( m_ReferenceFrame );
    u32 sectionNum = 1;

    {
        u64 sectionFrameTime = static_cast<u64>( sectionFrame.end - sectionFrame.begin );

        sectionNum = static_cast<u32>( sectionFrameTime / referenceFrameTime ) + 1;
    }

    for ( LoadMeterList::iterator it = m_LoadMeterList.begin(); it != m_LoadMeterList.end(); ++it )
    {
        u64 sectionTime = static_cast<u64>( it->GetLastFinalEnd() - sectionFrame.begin );

        if ( sectionTime > 0 )
        {
            u32 num = static_cast<u32>( sectionTime / referenceFrameTime ) + 1;

            if ( sectionNum < num )
            {
                sectionNum = num;
            }
        }
    }

    return sectionNum;
}

} // namespace dev
} // namespace nw

