﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/dev/dev_FileDeviceManager.h>

#if defined( NW_PLATFORM_WIN32 )
  using namespace nw::internal::winext;
#endif

namespace nw
{
namespace dev
{

FileDeviceManager* FileDeviceManager::s_pInstance = NULL;
#if !defined(NW_USE_NINTENDO_SDK)
const char* FileDeviceManager::FS_SD01_MOUNT_PATH = "/vol/disc";
const char* FileDeviceManager::FS_HFIO01_MOUNT_PATH = "/vol/content";
#endif

//------------------------------------------------------------------------------
FileDeviceManager::FileDeviceManager()
: m_FileDevice( NULL ),
  m_FSInitialized( false ),
  m_Allocator( NULL ),
  m_FSMountPath( NULL )
{
    if (s_pInstance != NULL)
    {
        NW_FATAL_ERROR("nw::dev::FileDeviceManager instance already exists.");
    }
    s_pInstance = this;
}

//------------------------------------------------------------------------------
FileDeviceManager::~FileDeviceManager()
{
}

//------------------------------------------------------------------------------
void
FileDeviceManager::Initialize( const CreateArg& arg )
{
    NW_ASSERTMSG( m_Allocator == NULL, "nw::dev::FileDeviceManager is already initialized." );
    if ( m_Allocator != NULL )
    {
        return;
    }

    m_Arg = arg;

    NW_ASSERT_NOT_NULL( arg.allocator );
    m_Allocator = arg.allocator;
    if ( m_Allocator == NULL )
    {
        return;
    }

#if defined(NW_USE_NINTENDO_SDK)

    m_FileDevice = new(m_Allocator->Alloc(sizeof(FileDevice))) FileDevice();

    m_FSInitialized = false;

#else
    const char* path = NULL;

    if ( !arg.client && !arg.cmdBlock )
    {
        NW_ASSERTMSG( !m_FSInitialized, "Already initialized.");

        FSInit();

        m_FSClient   = reinterpret_cast<FSClient*>(m_Allocator->Alloc(sizeof(FSClient)));
        m_FSCmdBlock = reinterpret_cast<FSCmdBlock*>(m_Allocator->Alloc(sizeof(FSCmdBlock)));
        FSMountSource mountSrc;

        FSAddClient(m_FSClient, FS_RET_ALL_ERROR);
        FSInitCmdBlock(m_FSCmdBlock);
        NW_ASSERT( m_FSClient );
        NW_ASSERT( m_FSCmdBlock );

        if ( m_Arg.useMount )
        {
            if ( m_Arg.mountSd01 )
            {
                // sd01 デバイスをマウントして使用します。
                FSStatus status = FSGetMountSource(m_FSClient, m_FSCmdBlock, FS_SOURCETYPE_EXTERNAL, &mountSrc, FS_RET_ALL_ERROR);
                NW_ASSERTMSG( status == FS_STATUS_OK, "FSGetMountSource failed(sd01). status = %d", status );
            }
            else
            {
                FSStatus status = FSGetMountSource(m_FSClient, m_FSCmdBlock, FS_SOURCETYPE_HFIO, &mountSrc, FS_RET_ALL_ERROR);
                NW_ASSERTMSG( status == FS_STATUS_OK, "FSGetMountSource failed(hfio). status = %d", status );
            }

            m_FSMountPath = reinterpret_cast<char*>(m_Allocator->Alloc(FS_MAX_MOUNTPATH_SIZE));
            FSStatus status = FSMount(m_FSClient, m_FSCmdBlock, &mountSrc, m_FSMountPath, FS_MAX_MOUNTPATH_SIZE, FS_RET_ALL_ERROR);
            NW_ASSERTMSG( status == FS_STATUS_OK, "FSMount failed. status = %d", status );

            path = m_FSMountPath;
        }
        else
        {
            // SD でない場合、デフォルトで /vol/content にマウントされている hfio01 デバイスを使用します。
            path = m_Arg.mountSd01 ? FS_SD01_MOUNT_PATH : FS_HFIO01_MOUNT_PATH;
        }

        m_FSInitialized = true;
    }
    else
    {
        // 既に生成されているクライアントとコマンドブロックを用いて初期化を行います。
        // Finalize では FS の終了処理は行いません。
        m_FSClient = arg.client;
        m_FSCmdBlock = arg.cmdBlock;
        NW_ASSERT( m_FSClient );
        NW_ASSERT( m_FSCmdBlock );

        path = m_Arg.mountSd01 ? FS_SD01_MOUNT_PATH : FS_HFIO01_MOUNT_PATH;
    }

    m_FileDevice = new( m_Allocator->Alloc( sizeof(FileDevice) ) ) FileDevice( path );
#endif

    NW_ASSERT_NOT_NULL( m_FileDevice );
}

//------------------------------------------------------------------------------
void
FileDeviceManager::Finalize()
{
    if ( m_Allocator == NULL )
    {
        return;
    }

    nw::ut::SafeFree( m_FileDevice, m_Allocator );

    if ( m_FSInitialized )
    {
#if !defined(NW_USE_NINTENDO_SDK)
        FSStatus status;
        NW_UNUSED_VARIABLE( status );

        if ( m_Arg.useMount )
        {
            status = FSUnmount(m_FSClient, m_FSCmdBlock, m_FSMountPath, FS_RET_ALL_ERROR);
            NW_ASSERT(status == FS_STATUS_OK);
            nw::ut::SafeFree( m_FSMountPath, m_Allocator );
        }

        status = FSDelClient(m_FSClient, FS_RET_ALL_ERROR);
        NW_ASSERT(status == FS_STATUS_OK);

        nw::ut::SafeFree( m_FSClient, m_Allocator );
        nw::ut::SafeFree( m_FSCmdBlock, m_Allocator );

        FSShutdown();
#endif

        m_FSInitialized = false;
    }

    m_Allocator = NULL;
}

//------------------------------------------------------------------------------
FileDevice*
FileDeviceManager::Open(
    FileHandle* handle,
    const char* filename,
    FileDevice::FileOpenFlag flag
)
{
    NW_ASSERT_NOT_NULL( m_FileDevice );
    NW_ASSERT_NOT_NULL( filename );
    if ( m_FileDevice == NULL || filename == NULL )
    {
        return NULL;
    }

    return m_FileDevice->Open( handle, filename, flag );
}

//------------------------------------------------------------------------------
u8*
FileDeviceManager::Load(FileDevice::LoadArg& arg)
{
    NW_ASSERT_NOT_NULL( m_FileDevice );
    NW_ASSERT_NOT_NULL( arg.path );
    if ( m_FileDevice == NULL || arg.path == NULL )
    {
        return NULL;
    }

    return m_FileDevice->Load( arg );
}

//------------------------------------------------------------------------------
u8*
FileDeviceManager::TryLoad(FileDevice::LoadArg& arg)
{
    NW_ASSERT_NOT_NULL( m_FileDevice );
    NW_ASSERT_NOT_NULL( arg.path );
    if ( m_FileDevice == NULL || arg.path == NULL )
    {
        return NULL;
    }

    return m_FileDevice->TryLoad( arg );
}

//------------------------------------------------------------------------------
void
FileDeviceManager::Unload(FileDevice::LoadArg& arg, u8* data)
{
    nw::ut::SafeFree( data, arg.allocator );
}

//------------------------------------------------------------------------------
FileDevice*
FileDeviceManager::OpenDirectory(DirectoryHandle* handle, const char* dirName)
{
    NW_ASSERT_NOT_NULL( m_FileDevice );
    NW_ASSERT_NOT_NULL( dirName );
    if ( m_FileDevice == NULL || dirName == NULL )
    {
        return NULL;
    }

    return m_FileDevice->OpenDirectory( handle, dirName );
}

#if defined(NW_USE_NINTENDO_SDK)
//------------------------------------------------------------------------------
void
FileDeviceManager::SetFsAllocator()
{
    NW_ASSERTMSG(IsInitialized(), "nw::dev::FileDeviceManager is not initialized. please call Initialize() method.");
    nn::fs::SetAllocator(FileDeviceManager::FsAllocateFunction, FileDeviceManager::FsDeallocateFunction);
}

//------------------------------------------------------------------------------
void*
FileDeviceManager::FsAllocateFunction(size_t size)
{
    FileDeviceManager* pInstance = GetInstance();

    NW_ASSERTMSG(pInstance, "instance of nw::dev::FileDeviceManager does not exist.");
    NW_ASSERTMSG(pInstance->m_Allocator, "nw::dev::FileDeviceManager is not initialized. please call Initialize() method.");

    if (pInstance && pInstance->m_Allocator)
    {
        return pInstance->m_Allocator->Alloc(size);
    }
    else
    {
        return NULL;
    }
}

//------------------------------------------------------------------------------
void
FileDeviceManager::FsDeallocateFunction(void* ptr, size_t size)
{
    NW_UNUSED_VARIABLE(size);

    FileDeviceManager* pInstance = GetInstance();

    NW_ASSERTMSG(pInstance, "instance of nw::dev::FileDeviceManager does not exist.");
    NW_ASSERTMSG(pInstance->m_Allocator, "nw::dev::FileDeviceManager is not initialized. please call Initialize() method.");

    if (pInstance && pInstance->m_Allocator)
    {
        pInstance->m_Allocator->Free(ptr);
    }
}
#endif

} // namespace dev
} // namespace nw

