﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../precompiled.h"

#ifdef NW_SND_SPY_ENABLE

#include <nw/snd/spy/fnd/string/sndspyfnd_String.h>
#include <nw/snd/spy/modules/sndspy_PlotGroup.h>
#include <nw/snd/spy/modules/sndspy_PlotState.h>
#include <nw/snd/spy/modules/sndspy_PlotModule.h>

// NN_TEXT() マクロにより日本語を埋め込めるようになるまで、C4566 を無効化
#if defined(NW_PLATFORM_WIN32)
#pragma warning( disable : 4566 )
#endif

namespace nw {
namespace snd {
namespace spy {

//------------------------------------------------------------------------------
PlotState::PlotState()
    : m_Module(NULL)
    , m_Group(NULL)
    , m_Prev(NULL)
    , m_Next(NULL)
    , m_IsMetadataPushed(false)
{
    InitializePacketCommonData();

    SetName("");
    SetColor(0xF3, 0x98, 0x00); // orange
}

//------------------------------------------------------------------------------
PlotState::PlotState(const char* name)
    : m_Module(NULL)
    , m_Group(NULL)
    , m_Prev(NULL)
    , m_Next(NULL)
    , m_IsMetadataPushed(false)
{
    InitializePacketCommonData();

    SetName(name);
    SetColor(0xF3, 0x98, 0x00); // orange
}

//------------------------------------------------------------------------------
void PlotState::InitializePacketCommonData()
{
    m_MetadataPacketPayload.common.dataType = SpyPlotItemPacket::DataType_StateMetadata;
    m_MetadataPacketPayload.id = reinterpret_cast<u32>(this);
    m_MetadataPacketPayload.groupId = 0;

    m_ValuePacketPayload.common.dataType = SpyPlotItemPacket::DataType_StateValue;
    m_ValuePacketPayload.id = reinterpret_cast<u32>(this);
}

//------------------------------------------------------------------------------
PlotState::PlotState(const char* name, u8 r, u8 g, u8 b)
    : m_Module(NULL)
    , m_Group(NULL)
    , m_Prev(NULL)
    , m_Next(NULL)
    , m_IsMetadataPushed(false)
{
    InitializePacketCommonData();

    SetName(name);
    SetColor(r, g, b);
}

//------------------------------------------------------------------------------
PlotState::~PlotState()
{
    if(IsAttached())
    {
        m_Module->DetachState(*this);
    }
}

//------------------------------------------------------------------------------
void PlotState::SetName(const char* name)
{
    if(IsRequested())
    {
        NW_WARNING(false, "通信中は、spy::PlotState の名前を変更できません。");
        return;
    }

    if(name == NULL)
    {
        NW_WARNING(false, "名前が指定されていません。");
        return;
    }

    m_MetadataPacketPayload.name.Set(name);
}

//------------------------------------------------------------------------------
void PlotState::SetColor(u8 r, u8 g, u8 b)
{
    m_ValuePacketPayload.color.r = r;
    m_ValuePacketPayload.color.g = g;
    m_ValuePacketPayload.color.b = b;
    m_ValuePacketPayload.color.a = 0xFF;
}

//------------------------------------------------------------------------------
void PlotState::SetGroup(PlotGroup* pGroup)
{
    if(IsRequested())
    {
        NW_WARNING(false, "通信中は、spy::PlotState のグループを設定できません。");
        return;
    }

    m_Group = pGroup;

    if (pGroup != NULL)
    {
        m_MetadataPacketPayload.groupId = pGroup->GetGroupId();
    }
    else
    {
        m_MetadataPacketPayload.groupId = 0;
    }
}

//------------------------------------------------------------------------------
void PlotState::PlotValue(const char* state)
{
    if(!IsRequested()) { return; }

    // メタデータをプッシュしていない場合は先にプッシュする
    if(!IsMetadataPushed())
    {
        PushMetaData();
    }

    m_ValuePacketPayload.state.Set(state);

    m_Module->PushData(&m_ValuePacketPayload, offsetof(SpyPlotItemPacket::StateValuePacketPayload, state.text) + m_ValuePacketPayload.state.length + 1);
}

//------------------------------------------------------------------------------
void PlotState::Attach(PlotModule& module)
{
    NW_ASSERTMSG(m_Module == NULL || m_Module == &module, "この spy::PlotState は、すでに他の spy::PlotModule にアタッチされています。");

    m_Module = &module;
    InvalidateMetadata();
}

//------------------------------------------------------------------------------
void PlotState::Detach()
{
    m_Module = NULL;
    SetPrev(NULL);
    SetNext(NULL);
}

//------------------------------------------------------------------------------
void PlotState::PushMetaData()
{
    NW_ASSERT(IsRequested());

    if (m_Group != NULL && !m_Group->IsMetadataPushed())
    {
        m_Group->PushMetadata();
    }

    if(!m_Module->PushData(m_MetadataPacketPayload))
    {
        NW_WARNING(false, "PlotState '%s' のメタデータ送信に失敗しました。", GetName());
    }

    m_IsMetadataPushed = true;
}

} // namespace nw::snd::spy
} // namespace nw::snd
} // namespace nw

#endif // NW_SND_SPY_ENABLE
