﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/edit/sndedit_SoundArchiveEditor.h>

#ifdef NW_SND_CONFIG_ENABLE_DEV

#include <nw/snd/edit/sndedit_SoundEditConnection.h>
#include <nw/snd/edit/res/sndedit_CacheManager.h>
#include <nw/snd/edit/res/sndedit_ResItemInfo.h>

#if !defined(NW_RELEASE)
//#define NW_ENABLE_ERROR_DEBUG
#endif

namespace nw {
namespace snd {
namespace edit {

//----------------------------------------------------------
SoundArchiveEditor::SoundArchiveEditor() :
m_MaxItemName(0),
m_TotalBufferLength(0),
m_EditConnection(NULL)
{
}

//----------------------------------------------------------
SoundArchiveEditor::~SoundArchiveEditor()
{
    // Finalize() 済みであることを確認します。
    NW_ASSERTMSG(!IsInitialized(), "SoundArchiveEditor is not finalized.\n");
}

//----------------------------------------------------------
Result
SoundArchiveEditor::Initialize(
    void* buffer,
    u32 bufferLength,
    SoundArchive* soundArchive,
    SoundArchivePlayer* soundArchivePlayer,
    const Configs& configs)
{
    if(buffer == NULL ||
        bufferLength == 0 ||
        soundArchive == NULL ||
        soundArchivePlayer == NULL ||
        (configs.maxEditableItems == 0 && !configs.isMaxEditableItemsExpandable) ||
        (configs.maxEditableFiles == 0 && !configs.isMaxEditableFilesExpandable) ||
        configs.maxItemName == 0)
    {
        NW_FATAL_ERROR("invalid arguments.\n");
        return Result(SNDEDIT_RESULT_FAILED);
    }

    Result result = Result(SNDEDIT_RESULT_TRUE);

#if defined(NW_PLATFORM_CTR)
    m_ConnectionLockObject.Initialize();
#endif

    snd::internal::fnd::FrameHeap frameHeap;
    frameHeap.Initialize(buffer, bufferLength);

    m_TotalBufferLength = frameHeap.GetFreeLength();

    {
        u32 editControllerBufferSize = m_EditController.GetRequiredMemorySize(configs.maxItemName);
        void* editControllerBuffer = frameHeap.Alloc(editControllerBufferSize);

        if(editControllerBuffer == NULL)
        {
            SetLastError(SNDEDIT_RESULT_OUT_OF_MEMORY);
            Finalize();
            return Result(SNDEDIT_RESULT_OUT_OF_MEMORY);
        }

        internal::SoundArchiveEditController::InitializeArgs args;
        args.errorProvider = this;
        args.buffer = editControllerBuffer;
        args.bufferLength = editControllerBufferSize;
        args.maxItemName = configs.maxItemName;
        args.resourceManager = &m_ResourceManager;
        args.soundArchive = soundArchive;
        args.soundArchivePlayer = soundArchivePlayer;

        result = m_EditController.Initialize(args);

        if(!result.IsTrue())
        {
            SetLastError(result);
            Finalize();
            return result;
        }
    }

    s32 soundCount = 0;
    if (soundArchive->IsAvailable())
    {
        soundCount = soundArchive->GetSoundCount();
    }

    result = m_ResourceManager.Initialize(
        frameHeap,
        soundArchive->GetSoundIdFromIndex(soundCount),
        configs.maxEditableItems,
        configs.maxEditableFiles,
        configs.maxItemName,
        configs.isMaxEditableItemsExpandable,
        configs.isMaxEditableFilesExpandable);

    if(result.IsFailed())
    {
        SetLastError(result);
        Finalize();
        return result;
    }

    result = m_ParameterEditor.Initialize(&m_EditController, &m_ResourceManager, soundArchive);

    if(!result.IsTrue())
    {
        SetLastError(result);
        Finalize();
        return result;
    }

    result = m_FileEditor.Initialize(&m_EditController, soundArchivePlayer);

    if(!result.IsTrue())
    {
        SetLastError(result);
        Finalize();
        return result;
    }

    m_MaxItemName = configs.maxItemName;

    return Result(SNDEDIT_RESULT_TRUE);
}

//----------------------------------------------------------
void
SoundArchiveEditor::Finalize()
{
    detail_Stop();

    m_ParameterEditor.Finalize();
    m_FileEditor.Finalize();

    m_EditController.Finalize();
    m_ResourceManager.Finalize();

    m_MaxItemName = 0;
    m_TotalBufferLength = 0;

#if defined(NW_PLATFORM_CTR)
    m_ConnectionLockObject.Finalize();
#endif
}

//----------------------------------------------------------
u32
SoundArchiveEditor::GetMemoryUsage() const
{
    return m_EditController.GetTotalMemoryUsage();
}

//----------------------------------------------------------
u32
SoundArchiveEditor::GetMemoryMax() const
{
    return m_TotalBufferLength;
}

//----------------------------------------------------------
Result
SoundArchiveEditor::detail_Start(internal::SoundEditConnection& connection)
{
    snd::internal::fnd::FndCriticalSectionScopedLock lock(m_ConnectionLockObject);

    detail_Stop();

    if(connection.IsOpened())
    {
        return Result(SNDEDIT_RESULT_FALSE);
    }

    // インゲーム編集の準備を行います。
    if(m_ParameterEditor.IsInitialized())
    {
        m_ParameterEditor.Start();
    }

    if(m_FileEditor.IsInitialized())
    {
        m_FileEditor.Start();
    }

    Result result = m_EditController.Start(&connection);

    if(!result.IsTrue())
    {
        detail_Stop();
        return result;
    }

    m_EditConnection = &connection;

    return Result(SNDEDIT_RESULT_TRUE);
}

//----------------------------------------------------------
void
SoundArchiveEditor::detail_Stop()
{
    snd::internal::fnd::FndCriticalSectionScopedLock lock(m_ConnectionLockObject);

    m_EditConnection = NULL;

    if(m_ParameterEditor.IsInitialized())
    {
        m_ParameterEditor.Stop();
    }

    if(m_FileEditor.IsInitialized())
    {
        m_FileEditor.Stop();
    }

    m_EditController.Stop();
}

//----------------------------------------------------------
Result
SoundArchiveEditor::UpdateCaches(ICanceller* canceller /*= NULL*/, void* cancelerParam /*= NULL*/)
{
    NW_ASSERTMSG(IsInitialized(), "SoundArchiveEditor is not initialized.\n");

    snd::internal::fnd::FndCriticalSectionScopedLock lock(m_ConnectionLockObject);

    if(!IsStarted())
    {
        return Result(SNDEDIT_RESULT_FALSE);
    }

    return m_EditController.UpdateCaches(canceller, cancelerParam);
}

//----------------------------------------------------------
void
SoundArchiveEditor::detail_UpdateAllItemCaches()
{
    NW_ASSERTMSG(IsInitialized(), "SoundArchiveEditor is not initialized.\n");
    NW_ASSERTMSG(IsStarted(), "SoundArchiveEditor is not started.\n");

    m_EditController.InvokeUpdateAllItemInfos();
}

//----------------------------------------------------------
Result
SoundArchiveEditor::detail_UpdateItemCache(const char* name)
{
    NW_ASSERTMSG(IsInitialized(), "SoundArchiveEditor is not initialized.\n");
    return m_EditController.MakeItemInfoEditable(name);
}

//----------------------------------------------------------
void
SoundArchiveEditor::detail_RemoveAllItemCaches()
{
    NW_ASSERTMSG(IsInitialized(), "SoundArchiveEditor is not initialized.\n");
    NW_ASSERTMSG(IsStarted(), "SoundArchiveEditor is not started.\n");

    m_EditController.InvokeRemoveAllItemInfos();
}

//----------------------------------------------------------
Result
SoundArchiveEditor::detail_RemoveItemCache(const char* name)
{
    NW_ASSERTMSG(IsInitialized(), "SoundArchiveEditor is not initialized.\n");
    NW_ASSERTMSG(IsStarted(), "SoundArchiveEditor is not started.\n");

    return m_EditController.InvokeRemoveItemInfo(name);
}

//----------------------------------------------------------
CacheState
SoundArchiveEditor::GetItemCacheState(const char* name) const
{
    NW_ASSERTMSG(IsInitialized(), "SoundArchiveEditor is not initialized.\n");

    if(!IsStarted())
    {
        return CACHE_STATE_NONE;
    }

    return m_EditController.GetItemCacheState(name);
}

//----------------------------------------------------------
void
SoundArchiveEditor::SetLastError(ResultType value)
{
    m_LastError = Result(value);

#if defined(NW_ENABLE_ERROR_DEBUG)
    NW_LOG("[sndedit] SoundArchiveEditor::SetLastError '%s'.\n", Result(value).ToString());
#endif
}

} // namespace nw::snd::edit
} // namespace nw::snd
} // namespace nw

#endif // NW_SND_CONFIG_ENABLE_DEV
