﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/edit/res/sndedit_CacheManager.h>

#ifdef NW_SND_CONFIG_ENABLE_DEV

#include <nw/snd/fnd/basis/sndfnd_Memory.h>
#include <nw/snd/edit/res/sndedit_ResItemInfo.h>

#if !defined(NW_RELEASE)
//#define NW_ENABLE_CACHE_DEBUG
//#define NW_ENABLE_CACHE_DUMP_DEBUG
//#define NW_ENABLE_CACHE_DUMP_DEBUG_HASHCODE
#endif

namespace nw {
namespace snd {
namespace edit {
namespace internal {

u32 CacheManager::INVALID_ID = std::numeric_limits<u32>::max();

//----------------------------------------------------------
CacheManager::InitializeArgs::InitializeArgs() :
buffer(NULL),
bufferLength(0),
dataBufferAllocator(NULL),
baseID(INVALID_ID),
maxEntries(0),
maxName(0)
{
}

//----------------------------------------------------------
CacheManager::Item::Item(
    CacheManager& owner,
    u32 id,
    ResName& name,
    bool isOverrideOriginal,
    u32 bufferSize,
    u32 dataSize,
    const void* data) :
m_Owner(owner),
m_IsFreezed(false),
m_IsDisposed(false),
m_IsOverrideOriginal(isOverrideOriginal),
m_ID(id),
m_ReferenceCount(1),
m_DataType(RES_DATA_TYPE_UNKNOWN),
m_DataSize(dataSize),
m_BufferSize(bufferSize),
m_UserParameter(0)
{
    NW_ASSERT_NOT_NULL(data);

    m_Reference.value = this;
    m_UnfreezedReference.value = this;
    m_UnusedReference.value = this;

    m_Name.set_ptr(name.GetName());
    m_DataOffset.set_ptr(data);
    m_HashCode.SetEmpty();
}

//----------------------------------------------------------
void
CacheManager::Item::Freeze()
{
    if(m_IsFreezed)
    {
        return;
    }

    m_IsFreezed = true;

    m_Owner.m_UnfreezedItems.Erase(&GetUnfreezedReference());

#if defined(NW_ENABLE_CACHE_DUMP_DEBUG)
    m_Owner.DumpItems("Item::Freeze");
#endif
}

//----------------------------------------------------------
void
CacheManager::Item::Unfreeze()
{
    if(!m_IsFreezed)
    {
        return;
    }

    m_IsFreezed = false;

    m_Owner.m_UnfreezedItems.PushBack(&GetUnfreezedReference());

#if defined(NW_ENABLE_CACHE_DUMP_DEBUG)
    m_Owner.DumpItems("Item::Unfreeze");
#endif
}

//----------------------------------------------------------
void
CacheManager::Item::IncrementReferenceCount()
{
    ++m_ReferenceCount;

    if(m_ReferenceCount == 1)
    {
        m_Owner.m_UnusedItems.Erase(&GetUnusedReference());
    }
}

//----------------------------------------------------------
void
CacheManager::Item::DecrementReferenceCount()
{
    if(m_ReferenceCount == 0)
    {
        NW_FATAL_ERROR("invalid CacheManager::Item reference count.\n");
        return;
    }

    --m_ReferenceCount;

    if(m_ReferenceCount == 0)
    {
        m_Owner.m_UnusedItems.PushBack(&GetUnusedReference());
    }
}

//----------------------------------------------------------
void
CacheManager::Item::ClearReferenceCount()
{
    if(m_ReferenceCount == 0)
    {
        return;
    }

    m_ReferenceCount = 0;
    m_Owner.m_UnusedItems.PushBack(&GetUnusedReference());
}

//----------------------------------------------------------
void
CacheManager::Item::Dispose()
{
    if(m_IsDisposed)
    {
        return;
    }

    if(m_ReferenceCount == 0)
    {
        m_Owner.m_UnusedItems.Erase(&GetUnusedReference());
    }

    m_IsDisposed = true;

    m_Owner.m_MemoryUsage -= GetBufferSize();
    m_Owner.m_DataBufferAllocator->Free(this);

    if (!m_IsOverrideOriginal)
    {
        if(m_Owner.m_IDTable != NULL && m_Owner.m_BaseID != INVALID_ID)
        {
            s32 index = m_ID - m_Owner.m_BaseID;
            NW_ASSERT(index < m_Owner.m_MaxItemCount);

            m_Owner.m_IDTable[index] = NULL;
        }
    }
}

//----------------------------------------------------------
CacheManager::CacheManager() :
m_DataBufferAllocator(NULL),
m_ItemDictinoaryData(NULL),
m_MaxItemCount(0),
m_ValidItemCount(0),
m_MaxName(0),
m_MemoryUsage(0),
m_BaseID(INVALID_ID),
m_IDTable(NULL)
{
}

//----------------------------------------------------------
Result
CacheManager::Initialize(const InitializeArgs& args)
{
    if((args.buffer == NULL && args.bufferLength > 0) ||
        args.bufferLength < GetRequiredMemorySize(args.maxEntries) ||
        args.dataBufferAllocator == NULL ||
        args.maxEntries < 0 ||
        args.maxName == 0)
    {
        NW_FATAL_ERROR("invalid arguments.\n");
        return Result(SNDEDIT_RESULT_FAILED);
    }

    NW_ASSERT(m_MemoryUsage == 0);

    snd::internal::fnd::FrameHeap heap;
    heap.Initialize(args.buffer, args.bufferLength);

    if(args.buffer == NULL)
    {
        m_ItemDictinoaryData = NULL;
        m_IDTable = NULL;
    }
    else
    {
        // IDテーブルを初期化します。
        if(args.baseID < INVALID_ID)
        {
            m_IDTable = reinterpret_cast<Item**>(heap.Alloc(GetIDTableSize(args.maxEntries)));
            m_BaseID = args.baseID;

            for(s32 index = 0; index < args.maxEntries; ++index)
            {
                m_IDTable[index] = NULL;
            }
        }

        // 辞書を初期化します。
        u32 itemDictinaryBufferLength = heap.GetFreeLength();
        void* itemDictinaryBuffer = heap.Alloc(itemDictinaryBufferLength);

        m_ItemDictinoaryData = PatriciaDictionary::CreateData(
            itemDictinaryBuffer,
            itemDictinaryBufferLength,
            args.maxEntries);
    }

    m_DataBufferAllocator = args.dataBufferAllocator;
    m_MaxItemCount = args.maxEntries;
    m_ValidItemCount = 0;
    m_MaxName = args.maxName;

    return Result(SNDEDIT_RESULT_TRUE);
}

//----------------------------------------------------------
void
CacheManager::Finalize()
{
    RemoveAllItems();
    m_ItemDictinoaryData = NULL;
    m_IDTable = NULL;
    m_BaseID = INVALID_ID;
    m_MemoryUsage = 0;

    m_DataBufferAllocator = NULL;
    m_MaxItemCount = 0;
    m_ValidItemCount = 0;
    m_MaxName = 0;
}

//----------------------------------------------------------
u32
CacheManager::GetRequiredMemorySize(u32 maxEntries) const
{
    if(maxEntries == 0)
    {
        return 0;
    }

    return ut::RoundUp(
        PatriciaDictionary::GetRequiredSize(maxEntries) + GetIDTableSize(maxEntries),
        snd::internal::fnd::MemoryTraits::DEFAULT_ALIGNMENT);
}

//----------------------------------------------------------
u32
CacheManager::GetRequiredMemorySizeForItem(u32 maxName, u32 maxDataLength) const
{
    return ut::RoundUp(
        sizeof(Item) + ResName::GetRequiredSize(maxName) + maxDataLength,
        snd::internal::fnd::MemoryTraits::DEFAULT_ALIGNMENT);
}

//----------------------------------------------------------
bool
CacheManager::TryGetFreeID(u32* id) const
{
    if(m_IDTable == NULL || m_BaseID == INVALID_ID)
    {
        return false;
    }

    for(s32 index = 0; index < m_MaxItemCount; ++index)
    {
        if(m_IDTable[index] == NULL)
        {
            if(id != NULL)
            {
                *id = m_BaseID + index;
            }

            return true;
        }
    }

    return false;
}

//----------------------------------------------------------
const void*
CacheManager::GetData(
    const char* name,
    u32* dataType /*= NULL*/,
    bool allowUnfreezedData /*= false*/) const
{
    NW_ASSERTMSG(IsInitialized(), "CacheManager is not initialized.\n");

    if(dataType != NULL)
    {
        *dataType = RES_DATA_TYPE_UNKNOWN;
    }

    Item* item = GetItemImpl(name);

    if(item == NULL ||
        (!allowUnfreezedData && !item->IsFreezed()))
    {
        return NULL;
    }

    u32 cachedDataType = item->GetDataType();

    if(dataType != NULL)
    {
        *dataType = cachedDataType;
    }

    return const_cast<const Item*>(item)->GetData();
}

//----------------------------------------------------------
CacheManager::Item*
CacheManager::GetItem(const char* name)
{
    return GetItemImpl(name);
}

//----------------------------------------------------------
const CacheManager::Item*
CacheManager::GetItem(const char* name) const
{
    return GetItemImpl(name);
}

//----------------------------------------------------------
const CacheManager::Item*
CacheManager::GetItem(u32 index) const
{
    if(index >= static_cast<u32>(m_ValidItemCount))
    {
        return NULL;
    }

    return m_ItemDictinoaryData->node[index + 1].ofsData.to_ptr<Item>();
}

//----------------------------------------------------------
const CacheManager::Item*
CacheManager::GetItemFromID(u32 id) const
{
    NW_ASSERT(m_IDTable != NULL && m_BaseID != INVALID_ID);

    if(id < m_BaseID)
    {
        return NULL;
    }

    s32 index = id - m_BaseID;

    if(index < 0 || m_MaxItemCount <= index)
    {
        return NULL;
    }

    return m_IDTable[index];
}

//----------------------------------------------------------
Result
CacheManager::AddItem(
    u32 id,
    const char* name,
    bool isOverrideOriginal,
    u32 dataType,
    const void* data,
    u32 dataSize)
{
    NW_ASSERTMSG(IsInitialized(), "CacheManager is not initialized.\n");

    if(name == NULL || name[0] == '\0' || data == NULL || dataSize == 0)
    {
        return Result(SNDEDIT_RESULT_FAILED);
    }

    Result result = CanAddItem();

    if(result.IsFailed())
    {
        return result;
    }

    Item* newItem = NULL;
    result = CreateItem(
        newItem,
        id,
        name,
        isOverrideOriginal,
        data,
        dataSize);

    if(result.IsFailed())
    {
        return result;
    }

    NW_NULL_ASSERT(newItem);
    newItem->SetDataType(dataType);

    return newItem != NULL ? AddItem(*newItem) : Result(SNDEDIT_RESULT_FAILED);
}

//----------------------------------------------------------
Result
CacheManager::AddItem(Item& item)
{
    Result result = CanAddItem();
    if(result.IsFailed())
    {
        return result;
    }

    s32 index = FindEmptyNodeIndex();

    if(index < 0)
    {
        return Result(SNDEDIT_RESULT_CACHE_OVER_FLOW);
    }

    PatriciaDictionaryData::Node& node = m_ItemDictinoaryData->node[index];

    node.ofsName.set_ptr(item.GetName().GetName());
    node.ofsData.set_ptr(&item);

    NW_ASSERTMSG(node.ofsName.to_ptr() != NULL, "invalid name.\n");

    // 有効インデックスを更新します。
    m_ItemDictinoaryData->numData = ut::Max(index, m_ItemDictinoaryData->numData);
    ++m_ValidItemCount;

    // （最も最近使用されたアイテムとして）最後尾に追加します。
    m_Items.PushBack(&item.GetReference());

    // 非フリーズアイテムは別途リストに追加します。
    if(!item.IsFreezed())
    {
        m_UnfreezedItems.PushBack(&item.GetUnfreezedReference());
    }

#if defined(NW_ENABLE_CACHE_DEBUG)
    NW_LOG(
        "[sndedit] CacheManager::AddItem() : %d/%d : index=%3d, Name='%s'\n",
        m_ValidItemCount,
        m_MaxItemCount,
        index,
        node.ofsName.to_ptr()
        );
#endif

#if defined(NW_ENABLE_CACHE_DUMP_DEBUG)
    DumpItems("CacheManager::AddItem");
#endif

    return Result(SNDEDIT_RESULT_TRUE);
}

//----------------------------------------------------------
Result
CacheManager::RemoveItem(const char* name, ACTION action /*= NULL*/)
{
    NW_ASSERTMSG(IsInitialized(), "CacheManager is not initialized.\n");

    if(name == NULL || name[0] == '0')
    {
        return Result(SNDEDIT_RESULT_FAILED);
    }

    PatriciaDictionary* dictionary = GetPatriciaDictionary();

    if(dictionary == NULL)
    {
        return Result(SNDEDIT_RESULT_FAILED);
    }

    PatriciaDictionary::Node* node = dictionary->FindNode(name, std::strlen(name));

    if(node == NULL)
    {
        return Result(SNDEDIT_RESULT_FALSE);
    }

    RemoveItem(*node, action);
    return Result(SNDEDIT_RESULT_TRUE);
}

//----------------------------------------------------------
void
CacheManager::RemoveAllItems(ACTION action /*= NULL*/)
{
    if(!IsInitialized())
    {
        return;
    }

    if(m_ItemDictinoaryData == NULL)
    {
        return;
    }

    // 辞書を一括リセットします。
    for(s32 index = 1; index <= m_ItemDictinoaryData->numData; ++index)
    {
        PatriciaDictionary::ResetNode(m_ItemDictinoaryData->node[index]);
    }

    m_ItemDictinoaryData->numData = 1;

    // アイテムデータを削除します。
    for(Item::ReferenceList::iterator it = m_Items.begin(); it != m_Items.end(); ++it)
    {
        NW_ASSERT_NOT_NULL(it->value);

        // 辞書は一括リセットしたので、DisposeItem() します。
        DisposeItem(*(it->value), action);
    }

    m_Items.clear();
    m_UnfreezedItems.clear();

    m_ValidItemCount = 0;

#if defined(NW_ENABLE_CACHE_DEBUG)
    NW_LOG(
        "[sndedit] CacheManager::RemoveAllItems() : %d/%d\n",
        m_ValidItemCount,
        m_MaxItemCount
        );
#endif

#if defined(NW_ENABLE_CACHE_DUMP_DEBUG)
    DumpItems("CacheManager::RemoveAllItems");
#endif
}

//----------------------------------------------------------
void
CacheManager::RemoveGarbages()
{
    while(!m_UnusedItems.IsEmpty())
    {
        NW_ASSERT_NOT_NULL(m_UnusedItems.begin()->value);
        RemoveItem(m_UnusedItems.begin()->value->GetName().GetName());
    }

#if defined(NW_ENABLE_CACHE_DEBUG)
    NW_LOG(
        "[sndedit] CacheManager::RemoveGarbages() : %d/%d\n",
        m_ValidItemCount,
        m_MaxItemCount
        );
#endif
}

//----------------------------------------------------------
void
CacheManager::ClearAllItemReferenceCounts()
{
    for(Item::ReferenceList::const_iterator it = Item::ReferenceList::const_iterator(m_Items.begin()); it != Item::ReferenceList::const_iterator(m_Items.end()); ++it)
    {
        NW_ASSERT_NOT_NULL(it->value);
        it->value->ClearReferenceCount();
    }
}

//----------------------------------------------------------
void
CacheManager::UpdateDictionary()
{
    if(GetPatriciaDictionary() == NULL)
    {
        return;
    }

    GetPatriciaDictionary()->Build();

#if defined(NW_ENABLE_CACHE_DUMP_DEBUG)
    DumpItems("CacheManager::UpdateDictionary");
#endif
}

//----------------------------------------------------------
Result
CacheManager::RebuildDictionary(void* buffer, u32 bufferLength, s32 maxEntries, void** oldBuffer)
{
    NW_ASSERT_NOT_NULL(buffer);
    NW_ASSERTMSG(IsInitialized(), "CacheManager is not initialized.\n");

    s32 currentNumData = 0;

    if(m_ItemDictinoaryData != NULL)
    {
        currentNumData = m_ItemDictinoaryData->numData;
    }

    // 削除後の欠番が考えられるので、
    // 有効アイテムのインデックスまでの領域を保持する必要があります。
    NW_ASSERT(maxEntries >= currentNumData);

    // 現在、利用可能なバッファ長
    u32 currentUsableLength = GetRequiredMemorySize(currentNumData);

    if(currentUsableLength > bufferLength)
    {
        return Result(SNDEDIT_RESULT_OUT_OF_MEMORY);
    }

    PatriciaDictionaryData* newData =
        PatriciaDictionary::CreateData(buffer, bufferLength, maxEntries);
    NW_NULL_ASSERT(newData);

    if(m_ItemDictinoaryData != NULL)
    {
        newData->size = m_ItemDictinoaryData->size;
        newData->numData = m_ItemDictinoaryData->numData;

        for(s32 index = 0; index <= m_ItemDictinoaryData->numData; ++index)
        {
            PatriciaDictionaryData::Node& srcNode = m_ItemDictinoaryData->node[index];
            PatriciaDictionaryData::Node& destNode = newData->node[index];

            destNode.refBit = srcNode.refBit;
            destNode.idxLeft = srcNode.idxLeft;
            destNode.idxRight = srcNode.idxRight;

            destNode.ofsName.set_ptr(srcNode.ofsName.to_ptr());
            NW_ASSERTMSG(srcNode.ofsName.to_ptr() == destNode.ofsName.to_ptr(), "failed to offset.\n");

#if !defined(NW_RELEASE)
            if(index == 0)
            {
                NW_ASSERTMSG(srcNode.ofsName.to_ptr() == NULL, "failed to offset.\n");
                NW_ASSERTMSG(destNode.ofsName.to_ptr() == NULL, "failed to offset.\n");
            }
            else
            {
                NW_ASSERTMSG(srcNode.ofsName.to_ptr() != NULL, "failed to offset.\n");
                NW_ASSERTMSG(destNode.ofsName.to_ptr() != NULL, "failed to offset.\n");
            }
#endif

            destNode.ofsData.set_ptr(srcNode.ofsData.to_ptr());
            NW_ASSERTMSG(srcNode.ofsData.to_ptr() == destNode.ofsData.to_ptr(), "failed to offset.\n");
        }
    }

#if defined(NW_ENABLE_CACHE_DEBUG)
    NW_LOG(
        "[sndedit] CacheManager::RebuildDictionary() : %d/%d : OldMaxItemCount=%d\n",
        m_ValidItemCount,
        maxEntries,
        m_MaxItemCount
        );
#endif

    if(oldBuffer != NULL)
    {
        *oldBuffer = m_ItemDictinoaryData;
    }

    m_ItemDictinoaryData = newData;
    m_MaxItemCount = maxEntries;

#if defined(NW_ENABLE_CACHE_DUMP_DEBUG)
    DumpItems("CacheManager::RemoveItem");
#endif

    return Result(SNDEDIT_RESULT_TRUE);
}

//----------------------------------------------------------
Result
CacheManager::CreateItem(
    Item*& item,
    u32 id,
    const char* name,
    bool isOverrideOriginal,
    const void* data,
    u32 dataSize)
{
    if(m_ItemDictinoaryData == NULL)
    {
        NW_FATAL_ERROR("CacheManager is not initialized.\n");
        return Result(SNDEDIT_RESULT_NOT_INITIALIZED);
    }

    // アイテムバッファを確保します。
    // バッファの内訳は以下の通りです。
    //  - Item
    //  - ResNameData (可変長)
    //  - アイテムデータ (可変長)
    u32 nameLength = std::strlen(name);

    if(nameLength >= m_MaxName)
    {
        return Result(SNDEDIT_RESULT_NAME_TOO_LONG);
    }

    u32 resNameLength = ResName::GetRequiredSize(nameLength);
    u32 itemBufferSize = sizeof(Item) + resNameLength + dataSize;

    void* itemBuffer = m_DataBufferAllocator->Alloc(itemBufferSize);
    if(itemBuffer == NULL)
    {
        return Result(SNDEDIT_RESULT_OUT_OF_MEMORY);
    }

    // 名前をコピーします。
    ResNameData* resNameData = reinterpret_cast<ResNameData*>(
        ut::AddOffsetToPtr(itemBuffer, sizeof(Item)));

    resNameData->len = nameLength;
    ut::strncpy(resNameData->str, nameLength + 1, name, nameLength);

    // データをコピーします。
    // HACK : ※Item の直後に data が参照する ResName 等の実体が続いていることが前提です。
    void* resData = ut::AddOffsetToPtr(resNameData, resNameLength);

    if(data != NULL)
    {
        std::memcpy(resData, data, dataSize);
    }

    // Item（名前とデータのコンテナ）を生成します。
    item = new(itemBuffer) Item(
        *this,
        id,
        *reinterpret_cast<ResName*>(resNameData),
        isOverrideOriginal,
        itemBufferSize,
        dataSize,
        resData);

    m_MemoryUsage += item->GetBufferSize();

    if (!isOverrideOriginal)
    {
        if(m_IDTable != NULL && m_BaseID != INVALID_ID)
        {
            s32 index = id - m_BaseID;
            NW_ASSERT(index < m_MaxItemCount);

            m_IDTable[index] = item;
        }
    }

    return Result(SNDEDIT_RESULT_TRUE);
}

//----------------------------------------------------------
CacheManager::Item*
CacheManager::GetFirstUnfreezedItem(FILTER_FUNC filter /*= NULL*/) const
{
    if(m_UnfreezedItems.empty())
    {
        return NULL;
    }

    if(filter == NULL)
    {
        NW_ASSERT_NOT_NULL(m_UnfreezedItems.rbegin()->value);
        return m_UnfreezedItems.rbegin()->value;
    }

    for(Item::ReferenceList::const_reverse_iterator it = m_UnfreezedItems.rbegin(); it != m_UnfreezedItems.rend(); ++it)
    {
        Item* item = it->value;
        NW_ASSERT_NOT_NULL(item);

        if(filter(*item))
        {
            return item;
        }
    }

    return NULL;
}

//----------------------------------------------------------
CacheManager::Item*
CacheManager::GetFirstUnusedItem(FILTER_FUNC filter /*= NULL*/) const
{
    if(m_UnusedItems.empty())
    {
        return NULL;
    }

    if(filter == NULL)
    {
        return m_UnusedItems.begin()->value;
    }

    for(Item::ReferenceList::const_iterator it = m_UnusedItems.begin(); it != m_UnusedItems.end(); ++it)
    {
        Item* item = it->value;
        NW_ASSERT_NOT_NULL(item);

        if(filter(*item))
        {
            return item;
        }
    }

    return NULL;
}

//----------------------------------------------------------
Result
CacheManager::CanAddItem() const
{
    if(m_ItemDictinoaryData == NULL)
    {
        return Result(SNDEDIT_RESULT_CACHE_OVER_FLOW);
    }

    return m_ValidItemCount <= m_MaxItemCount ?
        Result(SNDEDIT_RESULT_TRUE) : Result(SNDEDIT_RESULT_CACHE_OVER_FLOW);
}

//----------------------------------------------------------
s32
CacheManager::FindEmptyNodeIndex()
{
    if(m_ItemDictinoaryData == NULL)
    {
        return -1;
    }

    for(s32 index = 1; index <= m_MaxItemCount; ++index)
    {
        if( m_ItemDictinoaryData->node[index].ofsName.to_ptr() == NULL)
        {
            return index;
        }
    }

    return -1;
}

//----------------------------------------------------------
CacheManager::Item*
CacheManager::GetItemImpl(const char* name) const
{
    NW_ASSERTMSG(IsInitialized(), "CacheManager is not initialized.\n");

    if(name == NULL || name[0] == '\0')
    {
        return NULL;
    }

    if(m_ItemDictinoaryData == NULL)
    {
        return NULL;
    }

    PatriciaDictionary* dictionary = GetPatriciaDictionary();

    if(dictionary == NULL)
    {
        return NULL;
    }

    Item* result = reinterpret_cast<Item*>(dictionary->Find(name));

    if(result == NULL)
    {
        return NULL;
    }

    return result;
}

//----------------------------------------------------------
void
CacheManager::RemoveItem(PatriciaDictionary::Node& node, ACTION action /*= NULL*/)
{
    Item* item = node.ofsData.to_ptr<Item>();

    if(item == NULL)
    {
        NW_FATAL_ERROR("unexpected error.\n");
        return;
    }

    --m_ValidItemCount;
    PatriciaDictionary::ResetNode(node);

    // 正しく索引できない可能性があるので、ノードを削除したら必ず更新します。
    UpdateDictionary();

    if(!item->IsFreezed())
    {
        m_UnfreezedItems.Erase(&item->GetUnfreezedReference());
    }

    m_Items.Erase(&item->GetReference());

#if defined(NW_ENABLE_CACHE_DEBUG)
    NW_LOG(
        "[sndedit] CacheManager::RemoveItem() : %d/%d : Name='%s'\n",
        m_ValidItemCount,
        m_MaxItemCount,
        item->GetName().GetName()
        );
#endif

#if defined(NW_ENABLE_CACHE_DUMP_DEBUG)
    DumpItems("CacheManager::RemoveItem");
#endif

    DisposeItem(*item, action);
}

//----------------------------------------------------------
void
CacheManager::DisposeItem(Item& item, ACTION action /*= NULL*/)
{
    if(action != NULL)
    {
        action(item);
    }

    // HACK : Dispose() 内部で解放され、m_MemoryUsage から使用メモリ量が差し引かれます。
    item.Dispose();
}

//----------------------------------------------------------
void
CacheManager::DumpItems(const char* tag) const
{
    (void)tag;

#if defined(NW_ENABLE_CACHE_DUMP_DEBUG)
    NW_LOG("[%s] ----------------\n", tag);

    NW_LOG("Items:\n");
    NW_LOG("   0 : Left=%3d, Right=%3d\n", m_ItemDictinoaryData->node[0].idxLeft, m_ItemDictinoaryData->node[0].idxRight);

    static const u32 hashStringBufferLength = Hash32::Size * 4 + 1;
    char hashString[hashStringBufferLength];

    for(s32 index = 1; index <= m_MaxItemCount; ++index)
    {
        PatriciaDictionaryData::Node& node = m_ItemDictinoaryData->node[index];

        if( node.ofsName.to_ptr() == NULL)
        {
            continue;
        }

        Item* item = node.ofsData.to_ptr<Item>();
        NW_ASSERT_NOT_NULL(item);

#if defined(NW_ENABLE_CACHE_DUMP_DEBUG_HASHCODE)
        item->GetHashCode().ToString(hashString, hashStringBufferLength);
#else
        hashString[0] = '\0';
#endif

        NW_LOG(
            " %3d : Left=%3d, Right=%3d, HashCode=%s, RefCount=%3d, Key=%s\n",
            index,
            node.idxLeft,
            node.idxRight,
            hashString,
            item->GetReferenceCount(),
            node.ofsName.to_ptr()
            );
    }

    NW_LOG("UnfreezedItems:\n");

    u32 index = 0;

    for(Item::ReferenceList::const_iterator it = m_UnfreezedItems.begin(); it != m_UnfreezedItems.end(); ++it)
    {
        NW_LOG(" %3d : %s\n", index, it->value->GetName().GetName());
        ++index;
    }

    NW_LOG("-------------------------------\n");
#endif
}

} // namespace nw::snd::edit::internal
} // namespace nw::snd::edit
} // namespace nw::snd
} // namespace nw

#endif // NW_SND_CONFIG_ENABLE_DEV
