﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/snd_WaveSoundFileReader.h>

namespace nw {
namespace snd {
namespace internal {

namespace
{

const u32 SIGNATURE_INFO_BLOCK_WSD = NW_UT_MAKE_SIGWORD( 'I', 'N', 'F', 'O' );

const u32 SUPPORTED_FILE_VERSION_WSD   = NW_UT_MAKE_VERSION( 0, 1, 0, 0 );   // ライブラリがサポートする最低バージョン
const u32 CURRENT_FILE_VERSION_WSD     = NW_UT_MAKE_VERSION( 0, 1, 1, 0 );   // ライブラリがサポートする最新バージョン
const u32 FILTER_SUPPORTED_VERSION_WSD = NW_UT_MAKE_VERSION( 0, 1, 1, 0 );

bool IsValidFileHeaderWsd( const void* waveSoundFile )
{
#if defined(NW_PLATFORM_CAFE)
    const ut::BinaryFileHeader& header =
        *reinterpret_cast<const ut::BinaryFileHeader*>( waveSoundFile );
#else
    const BinaryFileHeader& header =
        *reinterpret_cast<const BinaryFileHeader*>( waveSoundFile );
#endif

    // シグニチャ確認
    NW_ASSERTMSG( header.signature == WaveSoundFileReader::SIGNATURE_FILE,
            "invalid file signature." );
    if ( header.signature != WaveSoundFileReader::SIGNATURE_FILE )
    {
        return false;
    }

    // バージョン確認
    bool isSupportedVersion = false;
    if ( (SUPPORTED_FILE_VERSION_WSD <= header.version) &&
                                   (header.version <= CURRENT_FILE_VERSION_WSD) )
    {
        isSupportedVersion = true;

    }
    NW_ASSERTMSG( isSupportedVersion,
            "bfwsd file is not supported version.\n"
            "please reconvert file using new version tools.\n"
            "(version condition: 0x%08x <= ... <= 0x%08x, but your version[0x%08x])\n",
            SUPPORTED_FILE_VERSION_WSD, CURRENT_FILE_VERSION_WSD, header.version
    );
    return isSupportedVersion;
}

} // anonymous namespace

WaveSoundFileReader::WaveSoundFileReader( const void* waveSoundFile )
: m_pHeader( NULL ),
  m_pInfoBlockBody( NULL )
{
    NW_NULL_ASSERT( waveSoundFile );

    if ( ! IsValidFileHeaderWsd( waveSoundFile ) ) return;

    m_pHeader = reinterpret_cast<const WaveSoundFile::FileHeader*>( waveSoundFile );

    // INFO ブロックチェック
    const WaveSoundFile::InfoBlock* infoBlock = m_pHeader->GetInfoBlock();

    NW_NULL_ASSERT( infoBlock );
    if ( infoBlock == NULL ) return;

    // シグニチャチェック
    NW_ASSERTMSG( infoBlock->header.kind == SIGNATURE_INFO_BLOCK_WSD, "invalid block signature." );
    if ( infoBlock->header.kind != SIGNATURE_INFO_BLOCK_WSD )
    {
        return;
    }
    m_pInfoBlockBody = &infoBlock->body;
}

u32 WaveSoundFileReader::GetWaveSoundCount() const
{
    NW_NULL_ASSERT( m_pInfoBlockBody );
    return m_pInfoBlockBody->GetWaveSoundCount();
}

u32 WaveSoundFileReader::GetNoteInfoCount( u32 index ) const
{
    NW_NULL_ASSERT( m_pInfoBlockBody );
    NW_ASSERT( index < GetWaveSoundCount() );

    const WaveSoundFile::WaveSoundData& wsdData =
        m_pInfoBlockBody->GetWaveSoundData( index );
    return wsdData.GetNoteCount();
}

u32 WaveSoundFileReader::GetTrackInfoCount( u32 index ) const
{
    NW_NULL_ASSERT( m_pInfoBlockBody );
    NW_ASSERT( index < GetWaveSoundCount() );

    const WaveSoundFile::WaveSoundData& wsdData =
        m_pInfoBlockBody->GetWaveSoundData( index );
    return wsdData.GetTrackCount();
}

bool WaveSoundFileReader::ReadWaveSoundInfo( WaveSoundInfo* dst, u32 index ) const
{
    NW_NULL_ASSERT( m_pInfoBlockBody );
    NW_NULL_ASSERT( dst );

    const WaveSoundFile::WaveSoundInfo& src =
        m_pInfoBlockBody->GetWaveSoundData( index ).GetWaveSoundInfo();

    dst->pitch = src.GetPitch();
    dst->pan = src.GetPan();
    dst->surroundPan = src.GetSurroundPan();
    src.GetSendValue( &dst->mainSend, dst->fxSend, AUX_BUS_NUM );
    dst->adshr = src.GetAdshrCurve();
    if ( IsFilterSupportedVersion() )
    {
        dst->lpfFreq = src.GetLpfFreq();
        dst->biquadType = src.GetBiquadType();
        dst->biquadValue = src.GetBiquadValue();
    }
    else
    {
        dst->lpfFreq = 64;
        dst->biquadType = 0;
        dst->biquadValue = 0;
    }

    return true;
}

bool WaveSoundFileReader::ReadNoteInfo(
        WaveSoundNoteInfo* dst,
        u32 index,
        u32 noteIndex ) const
{
    NW_NULL_ASSERT( m_pInfoBlockBody );
    NW_NULL_ASSERT( dst );

    const WaveSoundFile::NoteInfo& src =
        m_pInfoBlockBody->GetWaveSoundData( index ).GetNoteInfo( noteIndex );

    const Util::WaveId* pWaveId =
        m_pInfoBlockBody->GetWaveIdTable().GetWaveId( src.waveIdTableIndex );

    if ( pWaveId == NULL )
    {
        return false;
    }

    dst->waveArchiveId = pWaveId->waveArchiveId;
    dst->waveIndex = pWaveId->waveIndex;
    dst->pitch = src.GetPitch();
    dst->adshr = src.GetAdshrCurve();
    dst->originalKey = src.GetOriginalKey();
    dst->pan = src.GetPan();
    dst->surroundPan = src.GetSurroundPan();
    dst->volume = src.GetVolume();
    return true;
}

bool WaveSoundFileReader::IsFilterSupportedVersion() const
{
#if defined(NW_PLATFORM_CAFE)
    const ut::BinaryFileHeader& header =
        *reinterpret_cast<const ut::BinaryFileHeader*>( m_pHeader );
#else
    const BinaryFileHeader& header =
        *reinterpret_cast<const BinaryFileHeader*>( m_pHeader );
#endif
    if (header.version >= FILTER_SUPPORTED_VERSION_WSD)
    {
        return true;
    }
    return false;
}

} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw
