﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/snd_StreamSoundFileLoader.h>

namespace nw {
namespace snd {
namespace internal {

/* ========================================================================
        StreamSoundFileLoader
   ======================================================================== */

/*---------------------------------------------------------------------------*
  Name:         LoadFileHeader

  Description:  ストリームデータのファイルヘッダとHEADブロックをロードし、
                StreamSoundFileReaderに渡す

  Arguments:    buffer - ロードに使用するバッファ
                size - バッファサイズ

  Returns:      AdpcmInfo構造体
 *---------------------------------------------------------------------------*/
bool StreamSoundFileLoader::LoadFileHeader(
        StreamSoundFileReader* reader, void* buffer, unsigned long size )
{
    const size_t HEADER_SIZE = NW_SND_ROUND_UP_32B( sizeof( StreamSoundFile::FileHeader ) );
    const size_t ALIGN_SIZE = 64;
        // NOTE: FSAGetVolumeInfo → FSAVolumeInfo.buf_align_r が 32 になっていたが、
        //       FSAReadFile で 32 バイト境界のバッファを渡すと FSA_STATUS_INVALID_ALIGNMENT
        //       を返すため、ALIGN_SIZE を 64 とする。(CafeSDK-1.3)

    u8 buffer2[ HEADER_SIZE+ALIGN_SIZE ];

    // ファイルヘッダロード
    m_pStream->Seek( 0, ut::FILE_STREAM_SEEK_BEGIN );
    s32 readSize = m_pStream->Read( ut::RoundUp( buffer2, ALIGN_SIZE ), HEADER_SIZE );
    if ( readSize != HEADER_SIZE )
    {
        return false;
    }

    StreamSoundFile::FileHeader* header =
        reinterpret_cast<StreamSoundFile::FileHeader*>( ut::RoundUp( buffer2, ALIGN_SIZE ) );

    if ( ! reader->IsValidFileHeader( header ) )
    {
        return false;
    }

    // バッファサイズがファイルヘッダーと INFO ブロックをロードできるサイズかチェック
    // (必ず、ヘッダーと INFO ブロックが並んで格納されているものとする)
    u32 loadSize = header->GetInfoBlockOffset() + header->GetInfoBlockSize();
    if ( loadSize > size )
    {
        return false;
    }

    // ファイルヘッダーと INFO ブロックをロード
    m_pStream->Seek( 0, ut::FILE_STREAM_SEEK_BEGIN );
    readSize = m_pStream->Read( buffer, loadSize );
    if ( readSize != static_cast<s32>(loadSize) )
    {
        return false;
    }

    reader->Initialize( buffer );

    m_SeekBlockOffset = reader->GetSeekBlockOffset();
    m_RegionDataOffset = reader->GetRegionDataOffset();
    m_RegionInfoBytes = static_cast<u16>(reader->GetRegionInfoBytes());

    return true;
}

/*---------------------------------------------------------------------------*
  Name:         ReadSeekBlockData

  Description:  SEEK ブロックの指定されたデータを読み取る

  Arguments:    yn1, yn2 - 読み取ったデータを格納する配列
                blockIndex - 読み取るデータのブロック番号
                channelCount - チャンネル数(yn1,yn2の配列サイズ)

  Returns:      読み取れたら true, 読み取れなかったら false
 *---------------------------------------------------------------------------*/
bool StreamSoundFileLoader::ReadSeekBlockData(
    u16* yn1, u16* yn2, int blockIndex, int channelCount )
{
    static const int SEEK_INFO_MAX_SIZE = STRM_CHANNEL_NUM_PER_TRACK * STRM_TRACK_NUM * sizeof(u16) * 2;  // 2ch * 8 Track * 2byte [yn1, yn2 のサイズ] * 2個 [ yn1, yn2 ]
    s32 readDataSize = sizeof(u16) * 2 * channelCount;

    s32 readOffset = static_cast<s32>(
        m_SeekBlockOffset +
        sizeof(ut::BinaryBlockHeader) +
        readDataSize * blockIndex
    );
    m_pStream->Seek( readOffset, ut::FILE_STREAM_SEEK_BEGIN );

    // 以下は 16 チャンネル以上で動作できないコードなのでASSERT
    NW_ASSERT( readDataSize <= SEEK_INFO_MAX_SIZE );
    if ( readDataSize > SEEK_INFO_MAX_SIZE )
    {
        return false;
    }

    const int ALIGN = PPC_IO_BUFFER_ALIGN;
    u16 bufferBase[SEEK_INFO_MAX_SIZE + ALIGN];
    u16* buffer = reinterpret_cast<u16*>(ut::RoundUp( bufferBase, ALIGN ));
    int readSize = m_pStream->Read( buffer, readDataSize );
    if ( readSize != readDataSize )
    {
        return false;
    }

    for ( int i = 0; i < channelCount; i++ )
    {
        yn1[i] = buffer[i*2];
        yn2[i] = buffer[i*2+1];
        // NW_LOG("[%d] yn1(%6d) yn2(%6d)\n",
        //         i, static_cast<s16>(yn1[i]), static_cast<s16>(yn2[i]));
    }

    return true;
}

bool StreamSoundFileLoader::ReadRegionInfo(
        StreamSoundFile::RegionInfo* info, u32 regionIndex ) const
{
    if ( m_RegionDataOffset == 0 || m_RegionInfoBytes == 0 )
    {
        return false;
    }

    s32 offset = m_RegionDataOffset + (m_RegionInfoBytes * regionIndex);
    m_pStream->Seek( offset, ut::FILE_STREAM_SEEK_BEGIN );

    const int ALIGN = PPC_IO_BUFFER_ALIGN;
    const int READ_SIZE = sizeof(StreamSoundFile::RegionInfo);
    u8 bufferBase[READ_SIZE + (ALIGN/2)*2];
    u8* buffer = reinterpret_cast<u8*>(ut::RoundUp( bufferBase, ALIGN ));
    int readSize = m_pStream->Read( buffer, READ_SIZE );
    if ( readSize != READ_SIZE )
    {
        return false;
    }

    *info = *reinterpret_cast<StreamSoundFile::RegionInfo*>(buffer);
    return true;
}

} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw

