﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/snd_Sound3DEngine.h>

#include <nw/snd/snd_BasicSound.h>      // nw::snd::SoundAmbientParam

namespace nw {
namespace snd {

/*---------------------------------------------------------------------------*
  Name:         Sound3DEngine

  Description:  コンストラクタ

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
Sound3DEngine::Sound3DEngine()
{
}

void Sound3DEngine::UpdateAmbientParam(
    const Sound3DManager* sound3DManager,
    const Sound3DParam* sound3DParam,
    u32 soundId,
    u32 updateFlag,
    SoundAmbientParam* ambientParam
)
{
    NW_NULL_ASSERT( sound3DManager );
    NW_NULL_ASSERT( sound3DParam );
    NW_NULL_ASSERT( ambientParam );

    (void)soundId;

    const Sound3DManager::ListenerList& listenerList = sound3DManager->GetListenerList();

    if ( updateFlag & UPDATE_PRIORITY )
    {
        ambientParam->priority = - sound3DManager->GetMaxPriorityReduction();
    }
    if ( updateFlag & UPDATE_FILTER )
    {
        ambientParam->biquadFilterValue = 0.0f;
    }
    if ( updateFlag & UPDATE_VOLUME )
    {
        ambientParam->tvParam.volume = 0.0f;

        for ( int i = 0; i < nw::snd::DRC_OUT_COUNT; i++ )
        {
            ambientParam->drcParam[i].volume = 0.0f;
        }
    }

    //  ▼デフォルト実装の挙動
    //
    //  ※ 本デフォルト実装では、volume, lpf, userData, outputLineFlag, {drc}FxSend
    //     は変更しない。
    //
    //  ----------------------------------------------------------
    //  SoundAmbientParam       Single      Multi(1)    Multi(*)
    //  ----------------------------------------------------------
    //      volume              -           -           -
    //      pitch               o           -           -
    //      lpf                 -           -           -
    //      biquadFilterValue   o           max         max
    //      biquadFilterType    o           o           o       (プレイヤーの値を参照)
    //      priority            o           max         max
    //      userData            -           -           -
    //      outputLineFlag      -           -           -
    //
    //      mainOutVolume       a1          b1          max
    //      pan                 a2          b2          -
    //      span                a3          b3          -
    //      fxSend              -           -           -
    //
    //      drcOutVolume        a1          b1          max
    //      drcPan              a2          b2          -
    //      drcSpan             a3          b3          -
    //      drcFxSend           -           -           -
    //  ----------------------------------------------------------
    //
    //  ※ 凡例
    //      Multi(1) : 当該出力先 (TV/DRC0/DRC1) にひとつしかリスナーが割り当てられていない
    //      Multi(*) :          〃                 複数のリスナーが割り当てられている
    //      - : 計算しない
    //      o : 計算する
    //      max : 計算して得られた値のうち、最大の値を採用する
    //      a1  : 同じ a1 のうちどちらか or どちらも計算する


    // マルチリスナー場合、ピッチ計算を行わない
    if ( listenerList.GetSize() > 1 )
    {
        updateFlag &= ~UPDATE_PITCH;
    }

    int destTv = 0;
    int destDrc[DRC_OUT_COUNT] = {0};
    for( Sound3DManager::ListenerList::ConstIterator itr = listenerList.GetBeginIter() ;
         itr != listenerList.GetEndIter() ; itr++ )
    {
        u32 flag = itr->GetOutputTypeFlag();
        if ( flag & Sound3DListener::OUTPUT_TYPE_TV )
        {
            destTv += 1;
        }
        for ( int i = 0; i < DRC_OUT_COUNT; i++ )
        {
            if ( flag & (Sound3DListener::OUTPUT_TYPE_DRC0<<i) )
            {
                destDrc[i] += 1;
            }
        }
    }

    for( Sound3DManager::ListenerList::ConstIterator itr = listenerList.GetBeginIter() ;
         itr != listenerList.GetEndIter() ; itr++ )
    {
        const Sound3DListener& listener = *itr;

        bool outTv = false;
        bool outDrc[DRC_OUT_COUNT] = {false};
        {
            u32 flag = listener.GetOutputTypeFlag();
            if ( flag & Sound3DListener::OUTPUT_TYPE_TV )
            {
                outTv = true;
            }
            for ( int i = 0; i < DRC_OUT_COUNT; i++ )
            {
                if ( flag & (Sound3DListener::OUTPUT_TYPE_DRC0<<i) )
                {
                    outDrc[i] = true;
                }
            }
        }

        // 各 Calc で繰り返し計算される値を、先に計算しておく
        nw::math::VEC3 actorPosition;
        f32 actorDistance = 0.0f;
        if ( updateFlag & ( UPDATE_VOLUME |
                            UPDATE_PRIORITY |
                            UPDATE_PITCH |
                            UPDATE_FILTER ) )
        {
            nw::math::VEC3Sub( &actorPosition,
                               &sound3DParam->position,
                               &listener.GetPosition() );
            actorDistance = nw::math::VEC3Len( &actorPosition );
        }

        //------------------------------------------------------------------
        // 音量/プライオリティ計算
        if ( updateFlag & ( UPDATE_VOLUME | UPDATE_PRIORITY ) )
        {
            f32 volume;
            int priority;
            Sound3DCalculator::CalcVolumeAndPriority(
                *sound3DManager,
                listener,
                *sound3DParam,
                actorDistance,
                &volume,
                &priority
            );

            if ( updateFlag & UPDATE_VOLUME )
            {
                if ( outTv )
                {
                    // マルチリスナーの場合は、一番大きい値を採用する
                    ambientParam->tvParam.volume =
                        ut::Max( volume, ambientParam->tvParam.volume );
                }
                for ( int i = 0; i < DRC_OUT_COUNT; i++ )
                {
                    if ( outDrc[i] )
                    {
                        // マルチリスナーの場合は、一番大きい値を採用する
                        ambientParam->drcParam[i].volume =
                            ut::Max( volume, ambientParam->drcParam[i].volume );
                    }
                }
            }
            if ( updateFlag & UPDATE_PRIORITY )
            {
                // マルチリスナーの場合は、一番大きい値を採用する
                ambientParam->priority = ut::Max( priority, ambientParam->priority );
            }
        }

        //------------------------------------------------------------------
        // パン/サラウンドパン計算
        if ( updateFlag & ( UPDATE_PAN | UPDATE_SPAN ))
        {
            // 当該リスナーの出力先に、ひとつのリスナーだけ割り当てられている場合に計算する
            bool isCalc = false;
            if ( outTv && destTv == 1 )
            {
                isCalc = true;
            }
            else
            {
                for ( int i = 0; i < DRC_OUT_COUNT; i++ )
                {
                    if ( outDrc[i] && destDrc[i] == 1 )
                    {
                        isCalc = true;
                        break;
                    }
                }
            }

            // 実際の計算
            if ( isCalc )
            {
                f32 pan;
                f32 span;

                Sound3DCalculator::CalcPan(
                        *sound3DManager,
                        listener,
                        *sound3DParam,
                        m_CalcPanParam,
                        &pan,
                        &span
                        );

                if ( updateFlag & UPDATE_PAN )
                {
                    if ( outTv )
                    {
                        ambientParam->tvParam.pan = pan;
                    }
                    for ( int i = 0; i < DRC_OUT_COUNT; i++ )
                    {
                        if ( outDrc[i] )
                        {
                            ambientParam->drcParam[i].pan = pan;
                        }
                    }
                }
                if ( updateFlag & UPDATE_SPAN )
                {
                    if ( outTv )
                    {
                        ambientParam->tvParam.span = span;
                    }
                    for ( int i = 0; i < DRC_OUT_COUNT; i++ )
                    {
                        if ( outDrc[i] )
                        {
                            ambientParam->drcParam[i].span = span;
                        }
                    }
                }
            }
        }

        //------------------------------------------------------------------
        // ピッチ計算
        if ( updateFlag & UPDATE_PITCH )
        {
            float pitch;

            Sound3DCalculator::CalcPitch(
                *sound3DManager,
                listener,
                *sound3DParam,
                actorPosition,
                actorDistance,
                &pitch
            );

            ambientParam->pitch = pitch;
        }

        //------------------------------------------------------------------
        // Biquadフィルタ計算
        if ( updateFlag & UPDATE_FILTER )
        {
            float biquadFilterValue;

            Sound3DCalculator::CalcBiquadFilterValue(
                *sound3DManager,
                listener,
                *sound3DParam,  // 使わないが、今後の拡張性を考慮しておく
                actorDistance,
                &biquadFilterValue
            );

            ambientParam->biquadFilterType = sound3DManager->GetBiquadFilterType();
            // NW_LOG("(%.2f %.2f) ", biquadFilterValue, ambientParam->biquadFilterValue);

            // マルチリスナーの場合は、一番大きい値を採用する
            ambientParam->biquadFilterValue =
                    ut::Max( biquadFilterValue, ambientParam->biquadFilterValue );
        }

        //------------------------------------------------------------------
        // NOTE:
        // 本関数をオーバーライドすることで、他のパラメータ (volume, lpf, userData,
        // outputLineFlag, fxSend, drcFxSend) を上書きすることができます。
    }
    // NW_LOG("=> %.2f\n", ambientParam->biquadFilterValue);
}

void Sound3DEngine::detail_UpdateAmbientParam(
    const Sound3DManager* sound3DManager,
    const Sound3DParam* sound3DParam,
    u32 soundId,
    SoundAmbientParam* ambientParam
)
{
    NW_NULL_ASSERT( sound3DManager );
    NW_NULL_ASSERT( sound3DParam );
    NW_NULL_ASSERT( ambientParam );

    u32 updateFlag = 0;
    if ( sound3DParam->ctrlFlag & SoundArchive::Sound3DInfo::FLAG_CTRL_VOLUME )
    {
        updateFlag |= UPDATE_VOLUME;
    }
    if ( sound3DParam->ctrlFlag & SoundArchive::Sound3DInfo::FLAG_CTRL_PRIORITY )
    {
        updateFlag |= UPDATE_PRIORITY;
    }
    if ( sound3DParam->ctrlFlag & SoundArchive::Sound3DInfo::FLAG_CTRL_PAN )
    {
        updateFlag |= UPDATE_PAN;
    }
    if ( sound3DParam->ctrlFlag & SoundArchive::Sound3DInfo::FLAG_CTRL_SPAN )
    {
        updateFlag |= UPDATE_SPAN;
    }
    if ( sound3DParam->ctrlFlag & SoundArchive::Sound3DInfo::FLAG_CTRL_FILTER )
    {
        updateFlag |= UPDATE_FILTER;
    }
    if ( sound3DParam->dopplerFactor > 0 )
    {
        updateFlag |= UPDATE_PITCH;
    }

    UpdateAmbientParam(
        sound3DManager,
        sound3DParam,
        soundId,
        updateFlag,
        ambientParam
    );
}

int Sound3DEngine::GetAmbientPriority(
    const Sound3DManager* sound3DManager,
    const Sound3DParam* sound3DParam,
    u32 soundId
)
{
    NW_NULL_ASSERT( sound3DManager );
    NW_NULL_ASSERT( sound3DParam );

    u32 updateFlag = UPDATE_START_PRIORITY ;
    if ( sound3DParam->ctrlFlag & SoundArchive::Sound3DInfo::FLAG_CTRL_PRIORITY )
    {
        updateFlag |= UPDATE_PRIORITY;
    }

    SoundAmbientParam ambientParam;
    UpdateAmbientParam(
        sound3DManager,
        sound3DParam,
        soundId,
        updateFlag,
        &ambientParam
    );

    return ambientParam.priority;
}

} // namespace nw::snd
} // namespace nw

