﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/snd_SequenceTrack.h>

#include <nw/snd/snd_Bank.h>
#include <nw/snd/snd_SequenceSound.h>
#include <nw/snd/snd_SequenceSoundPlayer.h>
#include <nw/snd/snd_HardwareManager.h>

namespace nw {
namespace snd {
namespace internal {
namespace driver {

/* ========================================================================
        memnber function
   ======================================================================== */

void SequenceTrack::SetPlayerTrackNo( int playerTrackNo )
{
    NW_MINMAX_ASSERT( playerTrackNo, 0, SequenceSoundPlayer::TRACK_NUM_PER_PLAYER );
    m_PlayerTrackNo  = static_cast<u8>( playerTrackNo );
}

SequenceTrack::SequenceTrack()
: m_OpenFlag( false ),
  m_pSequenceSoundPlayer( NULL ),
  m_pChannelList( NULL )

{
    InitParam();
}

SequenceTrack::~SequenceTrack()
{
    Close();
}

/*---------------------------------------------------------------------------*
  Name:         InitParam

  Description:  トラック構造体を初期化します

  Arguments:    track - トラックポインタ

  Returns:      None.
 *---------------------------------------------------------------------------*/
void SequenceTrack::InitParam()
{
    m_ExtVolume                  = 1.0f;
    m_ExtPitch                   = 1.0f;
    m_PanRange                   = 1.0f;

    m_TvParam.Initialize();
    for ( int i = 0; i < DRC_OUT_COUNT; i++ )
    {
        m_DrcParam[i].Initialize();
    }

    // Parser Param
    m_ParserTrackParam.baseAddr       = NULL;
    m_ParserTrackParam.currentAddr    = NULL;

    m_ParserTrackParam.cmpFlag        = true;
    m_ParserTrackParam.noteWaitFlag   = true;
    m_ParserTrackParam.tieFlag        = false;
    m_ParserTrackParam.monophonicFlag = false;

    m_ParserTrackParam.callStackDepth = 0;

    m_ParserTrackParam.wait           = 0;

    m_ParserTrackParam.muteFlag       = false;
    m_ParserTrackParam.silenceFlag    = false;
    m_ParserTrackParam.noteFinishWait = false;
    m_ParserTrackParam.portaFlag      = false;
    m_ParserTrackParam.damperFlag     = false;

    m_ParserTrackParam.bankIndex      = 0;
    m_ParserTrackParam.prgNo          = 0;

    for (int i = 0; i < Channel::MOD_COUNT; i++)
    {
        m_ParserTrackParam.lfoParam[i].Initialize();
        m_ParserTrackParam.lfoTarget[i] = Channel::LFO_TARGET_PITCH;
    }
    m_ParserTrackParam.sweepPitch     = 0.0f;

    m_ParserTrackParam.volume.InitValue( 127 );
    m_ParserTrackParam.volume2.InitValue( 127 );
    m_ParserTrackParam.pan.InitValue( 0 );
    m_ParserTrackParam.surroundPan.InitValue( 0 );
    m_ParserTrackParam.velocityRange  = 127;
    m_ParserTrackParam.pitchBend.InitValue( 0 );
    m_ParserTrackParam.bendRange      = DEFAULT_BENDRANGE;
    m_ParserTrackParam.initPan        = 0;
    m_ParserTrackParam.transpose      = 0;
    m_ParserTrackParam.priority       = DEFAULT_PRIORITY;
    m_ParserTrackParam.portaKey       = DEFAULT_PORTA_KEY;
    m_ParserTrackParam.portaTime      = 0;

    m_ParserTrackParam.attack         = INVALID_ENVELOPE;
    m_ParserTrackParam.decay          = INVALID_ENVELOPE;
    m_ParserTrackParam.sustain        = INVALID_ENVELOPE;
    m_ParserTrackParam.release        = INVALID_ENVELOPE;
    m_ParserTrackParam.envHold        = INVALID_ENVELOPE;

    m_ParserTrackParam.mainSend       = 127;
    for ( int i=0; i<AUX_BUS_NUM; i++ )
    {
        m_ParserTrackParam.fxSend[i] = 0;
    }

    m_ParserTrackParam.lpfFreq        = 0.0f;
    m_ParserTrackParam.biquadType     = 0;
    m_ParserTrackParam.biquadValue    = 0;
    m_ParserTrackParam.outputLine     = -1;

    for( int varNo = 0; varNo < TRACK_VARIABLE_NUM ; varNo++ )
    {
        m_TrackVariable[ varNo ] = SequenceSoundPlayer::VARIABLE_DEFAULT_VALUE;
    }
}

/*---------------------------------------------------------------------------*
  Name:         SetSeqData

  Description:  シーケンスデータをセットします

  Arguments:    seqBase - シーケンスデータベースアドレス
                seqOffset - シーケンスデータオフセット

  Returns:      None.
 *---------------------------------------------------------------------------*/
void SequenceTrack::SetSeqData( const void* seqBase, s32 seqOffset )
{
    m_ParserTrackParam.baseAddr    = static_cast<const u8*>( seqBase );
    m_ParserTrackParam.currentAddr = m_ParserTrackParam.baseAddr + seqOffset;
}

void SequenceTrack::Open()
{
    m_ParserTrackParam.noteFinishWait = false;
    m_ParserTrackParam.callStackDepth = 0;
    m_ParserTrackParam.wait           = 0;

    m_OpenFlag = true;
}

/*---------------------------------------------------------------------------*
  Name:         Close

  Description:  トラックを閉じます

  Arguments:    player - プレイヤーポインタ

  Returns:      None.
 *---------------------------------------------------------------------------*/
void SequenceTrack::Close()
{
    ReleaseAllChannel( -1 );
    FreeAllChannel();

    m_OpenFlag = false;
}

/*---------------------------------------------------------------------------*
  Name:         UpdateChannelLength

  Description:  チャンネル長を更新します

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
void SequenceTrack::UpdateChannelLength()
{
    if ( ! m_OpenFlag ) return;

    Channel* channel = m_pChannelList;
    while( channel != NULL )
    {
        if ( channel->GetLength() > 0 )
        {
            channel->SetLength( channel->GetLength() - 1 );
        }

        UpdateChannelRelease( channel );

        if ( ! channel->IsAutoUpdateSweep() )
        {
            channel->UpdateSweep( 1 );
        }

        channel = channel->GetNextTrackChannel();
    }
}

/*---------------------------------------------------------------------------*
  Name:         UpdateChannelRelease

  Description:  チャンネルをリリース状態を更新する

  Arguments:    channel - チャンネル

  Returns:      なし
 *---------------------------------------------------------------------------*/
void SequenceTrack::UpdateChannelRelease( Channel* channel )
{
    if ( ( channel->GetLength() == 0 ) && ( ! channel->IsRelease() ) )
    {
        if ( ! m_ParserTrackParam.damperFlag )
        {
            channel->NoteOff();
        }
    }
}

/*---------------------------------------------------------------------------*
  Name:         ParseNextTick

  Description:  トラックのシーケンス処理を進めます

  Arguments:    doNoteOn - ノートオンするかどうか

  Returns:      シーケンス継続時には正を、終了時には負を、未Open時には0を返します
 *---------------------------------------------------------------------------*/
int SequenceTrack::ParseNextTick( bool doNoteOn )
{
    if ( ! m_OpenFlag ) {
        return 0;
    }

    // MoveValueの更新
    m_ParserTrackParam.volume.Update();
    m_ParserTrackParam.volume2.Update();
    m_ParserTrackParam.pan.Update();
    m_ParserTrackParam.surroundPan.Update();
    m_ParserTrackParam.pitchBend.Update();

    // ウェイトチェック
    if ( m_ParserTrackParam.noteFinishWait ) {
        if ( m_pChannelList != NULL ) {
            return 1;
        }
        m_ParserTrackParam.noteFinishWait = false;
    }
    if ( m_ParserTrackParam.wait > 0 ) {
        m_ParserTrackParam.wait--;
        if ( m_ParserTrackParam.wait > 0 ) {
            return 1;
        }
    }

    // シーケンス処理
    if ( m_ParserTrackParam.currentAddr != NULL )
    {
        int counter = 0;
        const int COUNTER_MAX = 10000;
        while( m_ParserTrackParam.wait == 0 && ! m_ParserTrackParam.noteFinishWait )
        {
            counter++;
            if ( counter > COUNTER_MAX )
            {
                NW_WARNING( false, "cannot process %d SEQ command!\n", COUNTER_MAX);
                return 1;
            }

            ParseResult result = Parse( doNoteOn );
            if ( result == PARSE_RESULT_FINISH ) {
                return -1;
            }
        }
    }

    return 1;
}

/*---------------------------------------------------------------------------*
  Name:         StopAllChannel

  Description:  トラックのチャンネルを全て停止します

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
void SequenceTrack::StopAllChannel()
{
    Channel* channel = m_pChannelList;

    while( channel != NULL )
    {
        Channel* nextChannel = channel->GetNextTrackChannel();

        channel->Stop();
        channel->CallChannelCallback( Channel::CALLBACK_STATUS_STOPPED );
        Channel::FreeChannel(channel);

        channel = nextChannel;
    }

    m_pChannelList = NULL;
}

/*---------------------------------------------------------------------------*
  Name:         ReleaseAllChannel

  Description:  トラックのチャンネルを全てリリースします

  Arguments:    release - リリース値（負値の場合は、既に設定されているリリース値を使用する）

  Returns:      None.
 *---------------------------------------------------------------------------*/
void SequenceTrack::ReleaseAllChannel( int release )
{
    UpdateChannelParam();

    Channel* channel = m_pChannelList;
    while( channel != NULL )
    {
        if ( channel->IsActive() )
        {
            if ( release >= 0 )
            {
                NW_MINMAX_ASSERT( release, 0, 127 );
                channel->SetRelease( static_cast<u8>( release ) );
            }
            channel->Release();
        }
        channel = channel->GetNextTrackChannel();
    }
}

/*---------------------------------------------------------------------------*
  Name:         PauseAllChannel

  Description:  トラックのチャンネルを全てポーズします

  Arguments:    track - トラックポインタ
                player - プレイヤーポインタ
                release - リリース値（負値の場合は、既に設定されているリリース値を使用する）

  Returns:      None.
 *---------------------------------------------------------------------------*/
void SequenceTrack::PauseAllChannel( bool flag )
{
    Channel* channel = m_pChannelList;

    while( channel != NULL )
    {
        if ( channel->IsActive() && ( channel->IsPause() != flag ) )
        {
            channel->Pause( flag );
        }
        channel = channel->GetNextTrackChannel();
    }
}

/*---------------------------------------------------------------------------*
  Name:         AddChannel

  Description:  トラックへチャンネルを追加します

  Arguments:    channel - チャンネルポインタ

  Returns:      None.
 *---------------------------------------------------------------------------*/
void SequenceTrack::AddChannel( Channel* channel )
{
    channel->SetNextTrackChannel( m_pChannelList );
    m_pChannelList = channel;
}

/*---------------------------------------------------------------------------*
  Name:         UpdateChannelParam

  Description:  トラックが保持しているチャンネルのパラメータを更新します

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
void SequenceTrack::UpdateChannelParam()
{
    if ( ! m_OpenFlag ) return;

    if ( m_pChannelList == NULL ) {
        // 更新不要
        return;
    }

    // volume
    f32 volume
        = m_ParserTrackParam.volume.GetValue()
        * m_ParserTrackParam.volume2.GetValue()
        * m_pSequenceSoundPlayer->GetParserPlayerParam().volume.GetValue()
        / (127.0f * 127.0f * 127.0f);
    volume = volume * volume * m_ExtVolume * m_pSequenceSoundPlayer->GetVolume();

    f32 pitch = ( m_ParserTrackParam.pitchBend.GetValue() / 128.0f ) * m_ParserTrackParam.bendRange;

    f32 pitchRatio = m_pSequenceSoundPlayer->GetPitch() * m_ExtPitch;

    // lpf freq
    f32 lpfFreq
        = m_ParserTrackParam.lpfFreq
        + m_pSequenceSoundPlayer->GetLpfFreq();

    // biquad filter
    int biquadType = m_ParserTrackParam.biquadType;
    f32 biquadValue = m_ParserTrackParam.biquadValue;
    if ( m_pSequenceSoundPlayer->GetBiquadFilterType() != BIQUAD_FILTER_TYPE_INHERIT )
    {
        biquadType = m_pSequenceSoundPlayer->GetBiquadFilterType();
        biquadValue = m_pSequenceSoundPlayer->GetBiquadFilterValue();
    }

    // outputLine
    u32 outputLine = m_pSequenceSoundPlayer->GetOutputLine();
    if ( m_ParserTrackParam.outputLine != -1 )
    {
        outputLine = m_ParserTrackParam.outputLine;
    }



    // ベースデータ
    f32 panBase =
        ut::Clamp( static_cast<f32>( m_ParserTrackParam.pan.GetValue() ) / 63.0f,
                   -1.0f, 1.0f )
        * m_PanRange
        * m_pSequenceSoundPlayer->GetPanRange();
    f32 spanBase =
        ut::Clamp( static_cast<f32>( m_ParserTrackParam.surroundPan.GetValue() ) / 63.0f,
                   0.0f, 2.0f );
    f32 mainSendBase =
        static_cast<f32>( m_ParserTrackParam.mainSend ) / 127.0f - 1.0f; // -1.0 ～ 0.0 に
    f32 fxSendBase[AUX_BUS_NUM];
    for ( int i = 0; i < AUX_BUS_NUM; i++ )
    {
        fxSendBase[i] = static_cast<f32>( m_ParserTrackParam.fxSend[i] ) / 127.0f;
    }

    // TV 出力向けパラメータ
    OutputParam tvParam = m_pSequenceSoundPlayer->GetTvParam();
    {
        tvParam.volume *= m_TvParam.volume;
        for ( int i = 0; i < WAVE_CHANNEL_MAX; i++ )
        {
            for ( int j = 0; j < CHANNEL_INDEX_NUM; j++ )
            {
                tvParam.mixParameter[i].ch[j] *= m_TvParam.mixParameter[i].ch[j];
            }
        }
        tvParam.pan += panBase;
        tvParam.pan += m_TvParam.pan;
        tvParam.span += spanBase;
        tvParam.span += m_TvParam.span;
        tvParam.mainSend += mainSendBase;
        tvParam.mainSend += m_TvParam.mainSend;
        for ( int i = 0; i < AUX_BUS_NUM; i++ )
        {
            tvParam.fxSend[i] += fxSendBase[i];
            tvParam.fxSend[i] += m_TvParam.fxSend[i];
        }
    }

    // DRC 出力向けパラメータ
    OutputParam drcParam[DRC_OUT_COUNT];
    for ( u32 drc = 0; drc < DRC_OUT_COUNT; drc++ )
    {
        OutputParam& param = drcParam[drc];
        param = m_pSequenceSoundPlayer->GetDrcParam( drc );

        param.volume *= m_DrcParam[drc].volume;
        for ( int i = 0; i < WAVE_CHANNEL_MAX; i++ )
        {
            for ( int j = 0; j < CHANNEL_INDEX_NUM; j++ )
            {
                param.mixParameter[i].ch[j] *= m_DrcParam[drc].mixParameter[i].ch[j];
            }
        }
        param.pan += panBase;
        param.pan += m_DrcParam[drc].pan;
        param.span += spanBase;
        param.span += m_DrcParam[drc].span;
        param.mainSend += mainSendBase;
        param.mainSend += m_DrcParam[drc].mainSend;
        for ( int i = 0; i < AUX_BUS_NUM; i++ )
        {
            param.fxSend[i] += fxSendBase[i];
            param.fxSend[i] += m_DrcParam[drc].fxSend[i];
        }
    }

    // set channel params of track
    Channel* channel = m_pChannelList;
    while( channel != NULL )
    {
        channel->SetUserVolume( volume );
        channel->SetUserPitch( pitch );
        channel->SetUserPitchRatio( pitchRatio );
        channel->SetUserLpfFreq( lpfFreq );
        channel->SetBiquadFilter( biquadType, biquadValue );
        channel->SetOutputLine( outputLine );
        for (int i = 0; i < Channel::MOD_COUNT; i++)
        {
            channel->SetLfoParam( m_ParserTrackParam.lfoParam[i], i );
            channel->SetLfoTarget( static_cast<Channel::LfoTarget>( m_ParserTrackParam.lfoTarget[i] ), i );
        }

        channel->SetTvParam( tvParam );
        for ( int i = 0; i < DRC_OUT_COUNT; i++ )
        {
            channel->SetDrcParam( i, drcParam[i] );
        }
        for ( int i = 0; i < REMOTE_OUT_COUNT; i++ )
        {
            channel->SetRemoteParam(
                    i, m_pSequenceSoundPlayer->GetRemoteParam(i) );
        }

        channel = channel->GetNextTrackChannel();
    }
}

/*---------------------------------------------------------------------------*
  Name:         FreeAllChannel

  Description:  トラックのチャンネルを全て解放します

  Arguments:    track - トラックポインタ

  Returns:      None.
 *---------------------------------------------------------------------------*/
void SequenceTrack::FreeAllChannel( void )
{
    Channel* channel = m_pChannelList;

    while( channel != NULL ) {
        Channel::DetachChannel( channel );
        channel = channel->GetNextTrackChannel();
    }
    m_pChannelList = NULL;
}


/*---------------------------------------------------------------------------*
  Name:         GetChannelCount

  Description:  トラックが保持しているチャンネル数を取得する

  Arguments:    なし

  Returns:      チャンネル数
 *---------------------------------------------------------------------------*/
int SequenceTrack::GetChannelCount() const
{
    Channel* channel = m_pChannelList;
    int count = 0;
    while( channel != NULL ) {
        ++count;
        channel = channel->GetNextTrackChannel();
    }

    return count;
}

/*---------------------------------------------------------------------------*
  Name:         ChannelCallbackFunc

  Description:  チャンネルから呼びだされるコールバック関数

  Arguments:    dropChannel - チャンネルポインタ
                status - チャンネルコールバックステータス
                userData - トラックポインタを格納したコールバックユーザーデータ

  Returns:      None.
 *---------------------------------------------------------------------------*/
void SequenceTrack::ChannelCallbackFunc(
    Channel* dropChannel,
    Channel::ChannelCallbackStatus /* status */,
    void* userData
)
{
    SequenceTrack* track = reinterpret_cast<SequenceTrack*>( userData );

    NW_NULL_ASSERT( dropChannel );
    NW_NULL_ASSERT( track );

    if ( track->m_pSequenceSoundPlayer != NULL ) {
        track->m_pSequenceSoundPlayer->ChannelCallback( dropChannel );
    }

    // チャンネル参照の切断
    if ( track->m_pChannelList == dropChannel ) {
        track->m_pChannelList = dropChannel->GetNextTrackChannel();
        return;
    }

    Channel* channel = track->m_pChannelList;
    NW_NULL_ASSERT( channel );
    while( channel->GetNextTrackChannel() != NULL ) {
        if ( channel->GetNextTrackChannel() == dropChannel ) {
            channel->SetNextTrackChannel( dropChannel->GetNextTrackChannel() );
            return;
        }
        channel = channel->GetNextTrackChannel();
    }

    NW_ASSERT( false );
}

void SequenceTrack::SetMute( SeqMute mute )
{
    switch ( mute )
    {
    case SEQ_MUTE_OFF:
        m_ParserTrackParam.muteFlag = false;
        break;
    case SEQ_MUTE_STOP:
        StopAllChannel();
        m_ParserTrackParam.muteFlag = true;
        break;
    case SEQ_MUTE_RELEASE:
        ReleaseAllChannel( -1 );
        FreeAllChannel();
        m_ParserTrackParam.muteFlag = true;
        break;
    case SEQ_MUTE_NO_STOP:
        m_ParserTrackParam.muteFlag = true;
        break;
    }
}

void SequenceTrack::SetSilence( bool silenceFlag, int fadeTimes )
{
    m_ParserTrackParam.silenceFlag = silenceFlag;

    Channel* channel = m_pChannelList;
    while( channel != NULL )
    {
        channel->SetSilence(
            silenceFlag,
            ( fadeTimes + HardwareManager::SOUND_FRAME_INTERVAL_MSEC - 1 ) /
              HardwareManager::SOUND_FRAME_INTERVAL_MSEC );
        channel = channel->GetNextTrackChannel();
    }
}

void SequenceTrack::SetBiquadFilter( int type, f32 value )
{
    m_ParserTrackParam.biquadType = static_cast<s8>( type );
    m_ParserTrackParam.biquadValue = value;
}

void SequenceTrack::SetBankIndex( int bankIndex )
{
    // SequenceSoundHandle::SetTrackBankIndex でも範囲チェックしているが、念のため。
    NW_ASSERT_MINMAXLT( bankIndex, SequenceSound::BANK_INDEX_MIN, SequenceSound::BANK_INDEX_MAX );
    m_ParserTrackParam.bankIndex = static_cast<u8>( bankIndex );
}

void SequenceTrack::SetTranspose( s8 transpose )
{
    NW_ASSERT_MINMAXLT( transpose, SequenceSound::TRANSPOSE_MIN, SequenceSound::TRANSPOSE_MAX );
    m_ParserTrackParam.transpose = transpose;
}

void SequenceTrack::SetVelocityRange( u8 range )
{
    NW_ASSERT_MAX( range, SequenceSound::VELOCITY_RANGE_MAX );
    m_ParserTrackParam.velocityRange = range;
}

void SequenceTrack::SetOutputLine( s32 outputLine )
{
    m_ParserTrackParam.outputLine = outputLine;
}

/*---------------------------------------------------------------------------*
  Name:         GetTrackVariable

  Description:  シーケンストラック変数を取得します。

  Arguments:    varNo - 変数番号

  Returns:      変数の値
 *---------------------------------------------------------------------------*/
s16 SequenceTrack::GetTrackVariable( int varNo ) const
{
    NW_MINMAXLT_ASSERT( varNo, 0, TRACK_VARIABLE_NUM );

    return m_TrackVariable[ varNo ];
}

/*---------------------------------------------------------------------------*
  Name:         SetTrackVariable

  Description:  シーケンスローカル変数を設定します。

  Arguments:    player - プレイヤー
                varNo - 変数番号
                var - 変数値

  Returns:      変数の値
 *---------------------------------------------------------------------------*/
void SequenceTrack::SetTrackVariable( int varNo, s16 var )
{
    NW_MINMAXLT_ASSERT( varNo, 0, TRACK_VARIABLE_NUM );

    m_TrackVariable[ varNo ] = var;
}

/*---------------------------------------------------------------------------*
  Name:         GetVariablePtr

  Description:  変数のポインタを取得します

  Arguments:    player - プレイヤーポインタ
                varNo    - 変数番号

  Returns:      変数のポインタ
 *---------------------------------------------------------------------------*/
vs16* SequenceTrack::GetVariablePtr( int varNo )
{
    NW_MINMAX_ASSERT( varNo, 0, TRACK_VARIABLE_NUM );

    if ( varNo < TRACK_VARIABLE_NUM )
    {
        return & m_TrackVariable[ varNo ];
    }
    else
    {
        return NULL;
    }
}

/*---------------------------------------------------------------------------*
  Name:         NoteOn

  Description:  ノートオン

  Arguments:    key      - キー番号
                velocity - ベロシティ
                length   - ノート長 ( -1 なら無限長 )
                tieFlag  - タイフラグ

  Returns:      発音したチャンネルのポインタ
 *---------------------------------------------------------------------------*/
Channel* SequenceTrack::NoteOn(
    int key,
    int velocity,
    s32 length,
    bool tieFlag
)
{
    NW_MINMAX_ASSERT( key, 0, 127 );
    NW_MINMAX_ASSERT( velocity, 0, 127 );

    const SequenceSoundPlayer::ParserPlayerParam& playerParam = m_pSequenceSoundPlayer->GetParserPlayerParam();

    Channel* channel = NULL;

    // ベロシティレンジ適用
    velocity = velocity * m_ParserTrackParam.velocityRange / 127;

    if ( tieFlag )
    {
        // タイ
        channel = GetLastChannel();
        if ( channel != NULL )
        {
            channel->SetKey( static_cast<u8>( key ) );
            channel->SetVelocity( Bank::CalcChannelVelocityVolume(static_cast<u8>(velocity)) );
            // TODO: パンも更新するか？
        }
    }

    if ( GetParserTrackParam().monophonicFlag )
    {
        // モノフォニック
        channel = GetLastChannel();
        if ( channel != NULL )
        {
            if ( channel->IsRelease() )
            {
                channel->Stop();
                channel->CallChannelCallback( Channel::CALLBACK_STATUS_STOPPED );
                Channel::FreeChannel(channel);
                channel = NULL;
            }
            else
            {
                channel->SetKey( static_cast<u8>( key ) );
                channel->SetVelocity( Bank::CalcChannelVelocityVolume(static_cast<u8>(velocity)) );
                channel->SetLength( length );
            }
        }
    }

    if ( channel == NULL )
    {
        NoteOnInfo noteOnInfo = {
            m_ParserTrackParam.prgNo,
            key,
            velocity,
            tieFlag ? -1 : length,
            m_ParserTrackParam.initPan,
            playerParam.priority + m_ParserTrackParam.priority,
            SequenceTrack::ChannelCallbackFunc,
            this,
            m_pSequenceSoundPlayer->GetVoiceRendererType()
        };
        channel = m_pSequenceSoundPlayer->NoteOn(
            m_ParserTrackParam.bankIndex,
            noteOnInfo
        );
        if ( channel == NULL ) return NULL;

        // オルタネートアサイン処理
        if ( channel->GetKeyGroupId() > 0 )
        {
            Channel* itr = m_pChannelList;
            while( itr != NULL )
            {
                if ( itr->GetKeyGroupId() == channel->GetKeyGroupId() )
                {
                    itr->SetRelease(126);
                    itr->Release();
                }
                itr = itr->GetNextTrackChannel();
            }
        }

        // チャンネルリストへの結合
        AddChannel( channel );
    }

    // 初期パラメータ設定

    // エンベロープ更新
    if ( m_ParserTrackParam.attack <= SequenceTrack::MAX_ENVELOPE_VALUE )
    {
        channel->SetAttack( m_ParserTrackParam.attack );
    }

    if ( m_ParserTrackParam.decay <= SequenceTrack::MAX_ENVELOPE_VALUE )
    {
        channel->SetDecay( m_ParserTrackParam.decay );
    }

    if ( m_ParserTrackParam.sustain <= SequenceTrack::MAX_ENVELOPE_VALUE )
    {
        channel->SetSustain( m_ParserTrackParam.sustain );
    }

    if ( m_ParserTrackParam.release <= SequenceTrack::MAX_ENVELOPE_VALUE )
    {
        channel->SetRelease( m_ParserTrackParam.release );
    }
    if ( m_ParserTrackParam.envHold <= SequenceTrack::MAX_ENVELOPE_VALUE )
    {
        channel->SetHold( m_ParserTrackParam.envHold );
    }

    // スイープ＆ポルタメント更新
    f32 sweepPitch = m_ParserTrackParam.sweepPitch;

    if ( m_ParserTrackParam.portaFlag )
    {
        sweepPitch += m_ParserTrackParam.portaKey - key;
    }

    if ( m_ParserTrackParam.portaTime == 0 )
    {
        NW_WARNING( length != 0, "portatime zero is invalid." );
        channel->SetSweepParam( sweepPitch, length, false );
    }
    else
    {
        int sweepTime = m_ParserTrackParam.portaTime;
        sweepTime *= sweepTime;
        sweepTime = static_cast<int>( sweepTime * ( sweepPitch >= 0 ? sweepPitch : -sweepPitch ) );
        sweepTime >>= 5;
        sweepTime *= 5; // 単位はmsec

        channel->SetSweepParam( sweepPitch, sweepTime, true );
    }

    // ポルタメント開始キーを更新
    m_ParserTrackParam.portaKey = static_cast<u8>( key );

    // その他
    channel->SetSilence( m_ParserTrackParam.silenceFlag != 0, 0 );
    channel->SetReleasePriorityFix( m_pSequenceSoundPlayer->IsReleasePriorityFix() );
    channel->SetPanMode( m_pSequenceSoundPlayer->GetPanMode() );
    channel->SetPanCurve( m_pSequenceSoundPlayer->GetPanCurve() );
    channel->SetFrontBypass( m_ParserTrackParam.frontBypassFlag );
    channel->SetRemoteFilter( m_pSequenceSoundPlayer->GetRemoteFilter() );
    channel->SetVoiceRendererType( m_pSequenceSoundPlayer->GetVoiceRendererType() );

    return channel;
}

} // namespace nw::snd::internal::driver
} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw

