﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/snd_SequenceSound.h>

#include <nw/snd/snd_SoundInstanceManager.h>
#include <nw/snd/snd_SequenceTrack.h>
#include <nw/snd/snd_SequenceSoundHandle.h>
#include <nw/snd/snd_TaskManager.h>
#include <nw/snd/snd_DriverCommand.h>
#include <nw/snd/snd_PlayerHeap.h>
#include <nw/snd/snd_Util.h>
#include <nw/snd/snd_SoundDataManager.h>
#include <nw/snd/snd_SoundPlayer.h>

namespace nw {
namespace snd {
namespace internal {

/* ========================================================================
        member function
   ======================================================================== */

/*---------------------------------------------------------------------------*
  Name:         SequenceSound

  Description:  コンストラクタ

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
SequenceSound::SequenceSound( SequenceSoundInstanceManager& manager )
: m_Manager( manager ),
  m_InitializeFlag(false),
  m_IsCalledPrepare(false)
{
}

void SequenceSound::Initialize()
{
    BasicSound::Initialize();
    m_pTempSpecialHandle = NULL;
    m_IsCalledPrepare = false;
    m_InitializeFlag = true;
}

/*---------------------------------------------------------------------------*
  Name:         Shutdown

  Description:  サウンドの終了

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
void SequenceSound::Finalize()
{
    if ( ! m_InitializeFlag )
    {
        return;
    }

    m_InitializeFlag = false;
    m_IsCalledPrepare = false;
    BasicSound::Finalize();
    m_Manager.Free( this );
}

void SequenceSound::Setup(
    driver::SequenceTrackAllocator* trackAllocator,
    u32 allocTracks,
    driver::NoteOnCallback* noteOnCallback,
    int channelPriority,
    bool isReleasePriorityFix,
    SequenceUserprocCallback userproc,
    void* userprocArg
)
{
    NW_NULL_ASSERT( noteOnCallback );
    NW_MINMAX_ASSERT( channelPriority, 0, 127 );

    {
        DriverCommand& cmdmgr = DriverCommand::GetInstance();
        DriverCommandSequenceSoundSetup* command =
            cmdmgr.AllocCommand<DriverCommandSequenceSoundSetup>();
        command->id = DRIVER_COMMAND_SEQ_SETUP;
        command->player = &m_PlayerInstance;

        driver::SequenceSoundPlayer::SetupArg arg;
        {
            arg.trackAllocator = trackAllocator;
            arg.allocTracks = allocTracks;
            arg.callback = noteOnCallback;
        }
        command->arg = arg;  // 構造体コピー
        command->channelPriority = static_cast<u8>(channelPriority);
        command->isReleasePriorityFix = isReleasePriorityFix;
        command->userproc = reinterpret_cast<uptr>(userproc);
        command->userprocArg = userprocArg;

        cmdmgr.PushCommand(command);
    }
}

void SequenceSound::Prepare(
        const Resource& res,
        const driver::SequenceSoundPlayer::StartInfo& startInfo )
{
    NW_NULL_ASSERT( res.seq );
    {
        DriverCommand& cmdmgr = DriverCommand::GetInstance();
        DriverCommandSequenceSoundPrepare* command =
            cmdmgr.AllocCommand<DriverCommandSequenceSoundPrepare>();
        command->id = DRIVER_COMMAND_SEQ_PREPARE;
        command->player = &m_PlayerInstance;

        driver::SequenceSoundPlayer::PrepareArg arg;
        {
            arg.seqFile = res.seq;
            for (int i = 0; i < SEQ_BANK_MAX; i++)
            {
                arg.bankFiles[i] = res.banks[i];
                arg.warcFiles[i] = res.warcs[i];
                arg.warcIsIndividuals[i] = res.warcIsIndividuals[i];
            }
            arg.seqOffset = startInfo.seqOffset;
            arg.delayTime = startInfo.delayTime;
        }
        command->arg = arg;

        cmdmgr.PushCommand(command);
    }

    if ( startInfo.startOffset > 0 )
    {
        Skip( startInfo.startOffsetType, startInfo.startOffset );
    }

    m_IsCalledPrepare = true;
}



/*---------------------------------------------------------------------------*
  Name:         Skip

  Description:  スキップ

  Arguments:    tick - スキップするTick数

  Returns:      None.
 *---------------------------------------------------------------------------*/
void SequenceSound::Skip( driver::SequenceSoundPlayer::StartOffsetType offsetType, int offset )
{
    {
        DriverCommand& cmdmgr = DriverCommand::GetInstance();
        DriverCommandSequenceSoundSkip* command =
            cmdmgr.AllocCommand<DriverCommandSequenceSoundSkip>();
        command->id = DRIVER_COMMAND_SEQ_SKIP;
        command->player = &m_PlayerInstance;
        command->offsetType = offsetType;
        command->offset = offset;
        cmdmgr.PushCommand(command);
    }
}

/*---------------------------------------------------------------------------*
  Name:         SetTempoRatio

  Description:  テンポ比率の変更

  Arguments:    tempoRatio  - テンポの比率

  Returns:      None.
 *---------------------------------------------------------------------------*/
void SequenceSound::SetTempoRatio( f32 tempoRatio )
{
    NW_ASSERT( tempoRatio >= 0.0f );

    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandSequenceSoundTempoRatio* command =
        cmdmgr.AllocCommand<DriverCommandSequenceSoundTempoRatio>();
    command->id = DRIVER_COMMAND_SEQ_TEMPORATIO;
    command->player = &m_PlayerInstance;
    command->tempoRatio = tempoRatio;
    cmdmgr.PushCommand(command);
}

/*---------------------------------------------------------------------------*
  Name:         SetChannelPriority

  Description:  発音プライオリティを変更

  Arguments:    priority - 発音プライオリティ

  Returns:      None.
 *---------------------------------------------------------------------------*/
void SequenceSound::SetChannelPriority( int priority )
{
    NW_MINMAX_ASSERT( priority, 0, 127 );

    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandSequenceSoundChannelPrio* command =
        cmdmgr.AllocCommand<DriverCommandSequenceSoundChannelPrio>();
    command->id = DRIVER_COMMAND_SEQ_CHANNELPRIO;
    command->player = &m_PlayerInstance;
    command->priority = static_cast<u8>(priority);
    cmdmgr.PushCommand(command);
}

/*---------------------------------------------------------------------------*
  Name:         OnUpdatePlayerPriority

  Description:  プレイヤープライオリティを変更

  Arguments:

  Returns:      None.
 *---------------------------------------------------------------------------*/
void SequenceSound::OnUpdatePlayerPriority()
{
    int priority = CalcCurrentPlayerPriority();
    m_Manager.UpdatePriority( this, priority );
}

/*---------------------------------------------------------------------------*
  Name:         SetTrackMute

  Description:  トラックミュート

  Arguments:    trackBitFlag - トラックビットフラグ
                flag         - ミュート

  Returns:      None.
 *---------------------------------------------------------------------------*/
void SequenceSound::SetTrackMute( u32 trackBitFlag, SeqMute mute )
{
    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandSequenceSoundTrackMute* command =
        cmdmgr.AllocCommand<DriverCommandSequenceSoundTrackMute>();
    command->id = DRIVER_COMMAND_SEQTRACK_MUTE;
    command->player = &m_PlayerInstance;
    command->trackBitFlag = trackBitFlag;
    command->mute = mute;
    cmdmgr.PushCommand(command);
}

/*---------------------------------------------------------------------------*
  Name:         SetTrackMute

  Description:  トラックミュート

  Arguments:    trackBitFlag - トラックビットフラグ
                muteFlag     - ミュートフラグ

  Returns:      None.
 *---------------------------------------------------------------------------*/
void SequenceSound::SetTrackMute( u32 trackBitFlag, bool muteFlag )
{
    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandSequenceSoundTrackMute* command =
        cmdmgr.AllocCommand<DriverCommandSequenceSoundTrackMute>();
    command->id = DRIVER_COMMAND_SEQTRACK_MUTE;
    command->player = &m_PlayerInstance;
    command->trackBitFlag = trackBitFlag;
    command->mute = muteFlag ? SEQ_MUTE_STOP: SEQ_MUTE_OFF;
    cmdmgr.PushCommand(command);
}

/*---------------------------------------------------------------------------*
  Name:         SetTrackSilence

  Description:  トラックサイレンス

  Arguments:    trackBitFlag - トラックビットフラグ
                silenceFlag  - ミュートフラグ
                fadeFrames   - フェードフレーム数

  Returns:      None.
 *---------------------------------------------------------------------------*/
void SequenceSound::SetTrackSilence( u32 trackBitFlag, bool silenceFlag, int fadeFrames )
{
    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandSequenceSoundTrackSilence* command =
        cmdmgr.AllocCommand<DriverCommandSequenceSoundTrackSilence>();
    command->id = DRIVER_COMMAND_SEQTRACK_SILENCE;
    command->player = &m_PlayerInstance;
    command->trackBitFlag = trackBitFlag;
    command->silenceFlag = silenceFlag;
    command->fadeFrames = fadeFrames;
    cmdmgr.PushCommand(command);
}

/*---------------------------------------------------------------------------*
  Name:         SetTrackVolume

  Description:  トラックボリュームの変更

  Arguments:    trackBitFlag - トラックビットフラグ
                volume       - ボリューム

  Returns:      None.
 *---------------------------------------------------------------------------*/
void SequenceSound::SetTrackVolume( u32 trackBitFlag, f32 volume )
{
    NW_ASSERT( volume >= 0.0f );

    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandSequenceSoundTrackParam* command =
        cmdmgr.AllocCommand<DriverCommandSequenceSoundTrackParam>();
    command->id = DRIVER_COMMAND_SEQTRACK_VOLUME;
    command->player = &m_PlayerInstance;
    command->trackBitFlag = trackBitFlag;
    command->value = volume;
    cmdmgr.PushCommand(command);
}

/*---------------------------------------------------------------------------*
  Name:         SetTrackPitch

  Description:  トラックピッチの変更

  Arguments:    trackBitFlag - トラックビットフラグ
                pitch        - 音程

  Returns:      None.
 *---------------------------------------------------------------------------*/
void SequenceSound::SetTrackPitch( u32 trackBitFlag, f32 pitch )
{
    NW_ASSERT( pitch >= 0.0f );

    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandSequenceSoundTrackParam* command =
        cmdmgr.AllocCommand<DriverCommandSequenceSoundTrackParam>();
    command->id = DRIVER_COMMAND_SEQTRACK_PITCH;
    command->player = &m_PlayerInstance;
    command->trackBitFlag = trackBitFlag;
    command->value = pitch;
    cmdmgr.PushCommand(command);
}





void SequenceSound::SetTrackMainOutVolume( u32 trackBitFlag, f32 volume )
{
    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandSequenceSoundTrackParam* command =
        cmdmgr.AllocCommand<DriverCommandSequenceSoundTrackParam>();
    command->id = DRIVER_COMMAND_SEQTRACK_TV_VOLUME;
    command->player = &m_PlayerInstance;
    command->trackBitFlag = trackBitFlag;
    command->value = volume;
    cmdmgr.PushCommand(command);
}

void SequenceSound::SetTrackChannelMixParameter( u32 trackBitFlag, u32 srcChNo, const MixParameter& mixParam )
{
    NW_ASSERT_MAXLT( srcChNo, WAVE_CHANNEL_MAX );

    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandSequenceSoundTrackMixParameter* command =
        cmdmgr.AllocCommand<DriverCommandSequenceSoundTrackMixParameter>();
    command->id = DRIVER_COMMAND_SEQTRACK_TV_MIXPARAMETER;
    command->player = &m_PlayerInstance;
    command->trackBitFlag = trackBitFlag;
    command->srcChNo = srcChNo;
    for ( int channel = 0; channel < CHANNEL_INDEX_NUM; channel++ )
    {
        command->param.ch[channel] = mixParam.ch[channel];
    }
    cmdmgr.PushCommand(command);
}

/*---------------------------------------------------------------------------*
  Name:         SetTrackPan

  Description:  トラックパンの変更

  Arguments:    trackBitFlag - トラックビットフラグ
                pan          - パン

  Returns:      None.
 *---------------------------------------------------------------------------*/
void SequenceSound::SetTrackPan( u32 trackBitFlag, f32 pan )
{
    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandSequenceSoundTrackParam* command =
        cmdmgr.AllocCommand<DriverCommandSequenceSoundTrackParam>();
    command->id = DRIVER_COMMAND_SEQTRACK_TV_PAN;
    command->player = &m_PlayerInstance;
    command->trackBitFlag = trackBitFlag;
    command->value = pan;
    cmdmgr.PushCommand(command);
}

/*---------------------------------------------------------------------------*
  Name:         SetTrackSurroundPan

  Description:  トラックサラウンドパンの変更

  Arguments:    trackBitFlag - トラックビットフラグ
                surroundPan  - サラウンドパン

  Returns:      None.
 *---------------------------------------------------------------------------*/
void SequenceSound::SetTrackSurroundPan( u32 trackBitFlag, f32 surroundPan )
{
    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandSequenceSoundTrackParam* command =
        cmdmgr.AllocCommand<DriverCommandSequenceSoundTrackParam>();
    command->id = DRIVER_COMMAND_SEQTRACK_TV_SPAN;
    command->player = &m_PlayerInstance;
    command->trackBitFlag = trackBitFlag;
    command->value = surroundPan;
    cmdmgr.PushCommand(command);
}

void SequenceSound::SetTrackMainSend( u32 trackBitFlag, f32 send )
{
    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandSequenceSoundTrackParam* command =
        cmdmgr.AllocCommand<DriverCommandSequenceSoundTrackParam>();
    command->id = DRIVER_COMMAND_SEQTRACK_TV_MAINSEND;
    command->player = &m_PlayerInstance;
    command->trackBitFlag = trackBitFlag;
    command->value = send;
    cmdmgr.PushCommand(command);
}

void SequenceSound::SetTrackFxSend( u32 trackBitFlag, AuxBus bus, f32 send )
{
    NW_ASSERT_MINMAXLT( bus, 0, AUX_BUS_NUM );

    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandSequenceSoundTrackParam* command =
        cmdmgr.AllocCommand<DriverCommandSequenceSoundTrackParam>();
    command->id = DRIVER_COMMAND_SEQTRACK_TV_FXSEND;
    command->player = &m_PlayerInstance;
    command->trackBitFlag = trackBitFlag;
    command->value = send;
    command->u32Value = bus;
    cmdmgr.PushCommand(command);
}



void SequenceSound::SetTrackDrcOutVolume( u32 drcIndex, u32 trackBitFlag, f32 volume )
{
    NW_ASSERT_MAXLT( drcIndex, DRC_OUT_COUNT );

    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandSequenceSoundTrackDrcParam* command =
        cmdmgr.AllocCommand<DriverCommandSequenceSoundTrackDrcParam>();
    command->id = DRIVER_COMMAND_SEQTRACK_DRC_VOLUME;
    command->player = &m_PlayerInstance;
    command->trackBitFlag = trackBitFlag;
    command->value = volume;
    command->drcIndex = drcIndex;
    cmdmgr.PushCommand(command);
}

void SequenceSound::SetTrackDrcChannelMixParameter(  u32 drcIndex, u32 trackBitFlag, u32 srcChNo, const MixParameter& mixParam )
{
    NW_ASSERT_MAXLT( srcChNo, WAVE_CHANNEL_MAX );

    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandSequenceSoundTrackMixParameter* command =
        cmdmgr.AllocCommand<DriverCommandSequenceSoundTrackMixParameter>();
    command->id = DRIVER_COMMAND_SEQTRACK_DRC_MIXPARAMETER;
    command->player = &m_PlayerInstance;
    command->trackBitFlag = trackBitFlag;
    command->srcChNo = srcChNo;
    command->drcIndex = drcIndex;
    for ( int channel = 0; channel < CHANNEL_INDEX_NUM; channel++ )
    {
        command->param.ch[channel] = mixParam.ch[channel];
    }
    cmdmgr.PushCommand(command);
}

void SequenceSound::SetTrackDrcPan( u32 drcIndex, u32 trackBitFlag, f32 pan )
{
    NW_ASSERT_MAXLT( drcIndex, DRC_OUT_COUNT );

    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandSequenceSoundTrackDrcParam* command =
        cmdmgr.AllocCommand<DriverCommandSequenceSoundTrackDrcParam>();
    command->id = DRIVER_COMMAND_SEQTRACK_DRC_PAN;
    command->player = &m_PlayerInstance;
    command->trackBitFlag = trackBitFlag;
    command->value = pan;
    command->drcIndex = drcIndex;
    cmdmgr.PushCommand(command);
}

void SequenceSound::SetTrackDrcSurroundPan( u32 drcIndex, u32 trackBitFlag, f32 span )
{
    NW_ASSERT_MAXLT( drcIndex, DRC_OUT_COUNT );

    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandSequenceSoundTrackDrcParam* command =
        cmdmgr.AllocCommand<DriverCommandSequenceSoundTrackDrcParam>();
    command->id = DRIVER_COMMAND_SEQTRACK_DRC_SPAN;
    command->player = &m_PlayerInstance;
    command->trackBitFlag = trackBitFlag;
    command->value = span;
    command->drcIndex = drcIndex;
    cmdmgr.PushCommand(command);
}

void SequenceSound::SetTrackDrcMainSend( u32 drcIndex, u32 trackBitFlag, f32 send )
{
    NW_ASSERT_MAXLT( drcIndex, DRC_OUT_COUNT );

    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandSequenceSoundTrackDrcParam* command =
        cmdmgr.AllocCommand<DriverCommandSequenceSoundTrackDrcParam>();
    command->id = DRIVER_COMMAND_SEQTRACK_DRC_MAINSEND;
    command->player = &m_PlayerInstance;
    command->trackBitFlag = trackBitFlag;
    command->value = send;
    command->drcIndex = drcIndex;
    cmdmgr.PushCommand(command);
}

void SequenceSound::SetTrackDrcFxSend(
        u32 drcIndex, u32 trackBitFlag, AuxBus bus, f32 send )
{
    NW_ASSERT_MAXLT( drcIndex, DRC_OUT_COUNT );
    NW_ASSERT_MINMAXLT( bus, 0, AUX_BUS_NUM );

    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandSequenceSoundTrackDrcParam* command =
        cmdmgr.AllocCommand<DriverCommandSequenceSoundTrackDrcParam>();
    command->id = DRIVER_COMMAND_SEQTRACK_DRC_FXSEND;
    command->player = &m_PlayerInstance;
    command->trackBitFlag = trackBitFlag;
    command->value = send;
    command->u32Value = bus;
    command->drcIndex = drcIndex;
    cmdmgr.PushCommand(command);
}

/*---------------------------------------------------------------------------*
  Name:         SetTrackLpfFreq

  Description:  ローパスフィルタのカットオフの変更

  Arguments:    trackBitFlag - トラックビットフラグ
                lpfFreq  - フィルタのカットオフ値

  Returns:      None.
 *---------------------------------------------------------------------------*/
void SequenceSound::SetTrackLpfFreq( u32 trackBitFlag, f32 lpfFreq )
{
    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandSequenceSoundTrackParam* command =
        cmdmgr.AllocCommand<DriverCommandSequenceSoundTrackParam>();
    command->id = DRIVER_COMMAND_SEQTRACK_LPF;
    command->player = &m_PlayerInstance;
    command->trackBitFlag = trackBitFlag;
    command->value = lpfFreq;
    cmdmgr.PushCommand(command);
}

/*---------------------------------------------------------------------------*
  Name:         SetTrackBiquadFilter

  Description:  Biquadフィルタの設定

  Arguments:    trackBitFlag - トラックビットフラグ
                type  - フィルタの種類 (0 - 127)
                value - フィルタの値 (0.0f - 1.0f)

  Returns:      None.
 *---------------------------------------------------------------------------*/
void SequenceSound::SetTrackBiquadFilter( u32 trackBitFlag, int type, f32 value )
{
    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandSequenceSoundTrackBiquad* command =
        cmdmgr.AllocCommand<DriverCommandSequenceSoundTrackBiquad>();
    command->id = DRIVER_COMMAND_SEQTRACK_BIQUAD;
    command->player = &m_PlayerInstance;
    command->trackBitFlag = trackBitFlag;
    command->type = type;
    command->value = value;
    cmdmgr.PushCommand(command);
}

void SequenceSound::SetTrackBankIndex( u32 trackBitFlag, int bankIndex )
{
    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandSequenceSoundTrackBankIndex* command =
        cmdmgr.AllocCommand<DriverCommandSequenceSoundTrackBankIndex>();
    command->id = DRIVER_COMMAND_SEQTRACK_BANKINDEX;
    command->player = &m_PlayerInstance;
    command->trackBitFlag = trackBitFlag;
    command->bankIndex = bankIndex;
    cmdmgr.PushCommand(command);
}

void SequenceSound::SetTrackTranspose( u32 trackBitFlag, s8 transpose )
{
    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandSequenceSoundTrackTranspose* command =
        cmdmgr.AllocCommand<DriverCommandSequenceSoundTrackTranspose>();
    command->id = DRIVER_COMMAND_SEQTRACK_TRANSPOSE;
    command->player = &m_PlayerInstance;
    command->trackBitFlag = trackBitFlag;
    command->transpose = transpose;
    cmdmgr.PushCommand(command);
}

void SequenceSound::SetTrackVelocityRange( u32 trackBitFlag, u8 velocityRange )
{
    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandSequenceSoundTrackVelocityRange* command =
        cmdmgr.AllocCommand<DriverCommandSequenceSoundTrackVelocityRange>();
    command->id = DRIVER_COMMAND_SEQTRACK_VELOCITYRANGE;
    command->player = &m_PlayerInstance;
    command->trackBitFlag = trackBitFlag;
    command->range = velocityRange;
    cmdmgr.PushCommand(command);
}

void SequenceSound::SetTrackOutputLine( u32 trackBitFlag, u32 outputLine )
{
    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandSequenceSoundTrackOutputLine* command =
        cmdmgr.AllocCommand<DriverCommandSequenceSoundTrackOutputLine>();
    command->id = DRIVER_COMMAND_SEQTRACK_OUTPUTLINE;
    command->player = &m_PlayerInstance;
    command->trackBitFlag = trackBitFlag;
    command->outputLine = outputLine;
    cmdmgr.PushCommand(command);
}

void SequenceSound::ResetTrackOutputLine( u32 trackBitFlag )
{
    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandSequenceSoundTrackOutputLine* command =
        cmdmgr.AllocCommand<DriverCommandSequenceSoundTrackOutputLine>();
    command->id = DRIVER_COMMAND_SEQTRACK_OUTPUTLINE_RESET;
    command->player = &m_PlayerInstance;
    command->trackBitFlag = trackBitFlag;
    cmdmgr.PushCommand(command);
}

#if 0   // トラック操作コマンドのひな形
void SequenceSound::SetTrackXXX( u32 trackBitFlag, yyy zzz )
{
    DriverCommandManager& cmdmgr = DriverCommandManager::GetInstance();
    DriverCommandSequenceSoundTrackXXX* command =
        cmdmgr.AllocCommand<DriverCommandSequenceSoundTrackXXX>();
    command->id = DRIVER_COMMAND_SEQTRACK_AAA;
    command->player = &m_PlayerInstance;
    command->trackBitFlag = trackBitFlag;
    command->zzz = zzz;
    cmdmgr.PushCommand(command);
}
#endif

/*---------------------------------------------------------------------------*
  Name:         ReadVariable

  Description:  シーケンスローカル変数の読みとり

  Arguments:    varNo  - 変数番号
                var    - 読み込み先アドレス

  Returns:      成功したかどうか
 *---------------------------------------------------------------------------*/
bool SequenceSound::ReadVariable( int varNo, s16* var ) const
{
    NW_NULL_ASSERT( var );
    NW_MINMAXLT_ASSERT( varNo, 0, driver::SequenceSoundPlayer::PLAYER_VARIABLE_NUM );

    if ( ! IsPlayerAvailable() )
    {
        *var = driver::SequenceSoundPlayer::VARIABLE_DEFAULT_VALUE;
    }
    else
    {
        *var = m_PlayerInstance.GetLocalVariable( varNo );
    }
    return true;
}

/*---------------------------------------------------------------------------*
  Name:         ReadGlobalVariable

  Description:  シーケンスグローバル変数の読みとり

  Arguments:    varNo - 変数番号
                var   - 読み込み先アドレス

  Returns:      成功したかどうか
 *---------------------------------------------------------------------------*/
bool SequenceSound::ReadGlobalVariable( int varNo, s16* var )
{
    NW_NULL_ASSERT( var );
    NW_MINMAXLT_ASSERT( varNo, 0, driver::SequenceSoundPlayer::GLOBAL_VARIABLE_NUM );

    *var = driver::SequenceSoundPlayer::GetGlobalVariable( varNo );
    return true;
}

/*---------------------------------------------------------------------------*
  Name:         ReadTrackVariable

  Description:  シーケンストラック変数の読みとり

  Arguments:    trackNo - トラック番号
                varNo  - 変数番号
                var    - 読み込み先アドレス

  Returns:      成功したかどうか
 *---------------------------------------------------------------------------*/
bool SequenceSound::ReadTrackVariable( int trackNo, int varNo, s16* var ) const
{
    NW_NULL_ASSERT( var );
    NW_MINMAXLT_ASSERT( trackNo, 0, driver::SequenceSoundPlayer::TRACK_NUM_PER_PLAYER );
    NW_MINMAXLT_ASSERT( varNo, 0, driver::SequenceTrack::TRACK_VARIABLE_NUM );

    if ( ! IsPlayerAvailable() )
    {
        *var = driver::SequenceSoundPlayer::VARIABLE_DEFAULT_VALUE;
    }
    else
    {
        const driver::SequenceTrack* track = m_PlayerInstance.GetPlayerTrack( trackNo );
        if ( track == NULL ) *var = driver::SequenceSoundPlayer::VARIABLE_DEFAULT_VALUE;
        else *var = track->GetTrackVariable( varNo );
    }
    return true;
}

/*---------------------------------------------------------------------------*
  Name:         WriteVariable

  Description:  シーケンスローカル変数の書き込み

  Arguments:    varNo  - 変数番号
                var    - 書き込む値

  Returns:      None.
 *---------------------------------------------------------------------------*/
void SequenceSound::WriteVariable( int varNo, s16 var )
{
    NW_MINMAXLT_ASSERT( varNo, 0, driver::SequenceSoundPlayer::PLAYER_VARIABLE_NUM );

    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandSequenceSoundSetVar* command =
        cmdmgr.AllocCommand<DriverCommandSequenceSoundSetVar>();
    command->id = DRIVER_COMMAND_SEQ_SETVAR;
    command->player = &m_PlayerInstance;
    command->varNo = varNo;
    command->var = var;
    cmdmgr.PushCommand(command);
}

/*---------------------------------------------------------------------------*
  Name:         WriteGlobalVariable

  Description:  シーケンスグローバル変数の書き込み

  Arguments:    varNo - 変数番号
                var   - 書き込む値

  Returns:      None.
 *---------------------------------------------------------------------------*/
void SequenceSound::WriteGlobalVariable( int varNo, s16 var )
{
    NW_MINMAXLT_ASSERT( varNo, 0, driver::SequenceSoundPlayer::GLOBAL_VARIABLE_NUM );

    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandSequenceSoundSetVar* command =
        cmdmgr.AllocCommand<DriverCommandSequenceSoundSetVar>();
    command->id = DRIVER_COMMAND_SEQ_SETGVAR;
    command->varNo = varNo;
    command->var = var;
    cmdmgr.PushCommand(command);
}

/*---------------------------------------------------------------------------*
  Name:         WriteTrackVariable

  Description:  シーケンストラック変数の書き込み

  Arguments:    trackNo - トラック番号
                varNo  - 変数番号
                var    - 書き込む値

  Returns:      None.
 *---------------------------------------------------------------------------*/
void SequenceSound::WriteTrackVariable( int trackNo, int varNo, s16 var )
{
    NW_MINMAXLT_ASSERT( trackNo, 0, driver::SequenceSoundPlayer::TRACK_NUM_PER_PLAYER );
    NW_MINMAXLT_ASSERT( varNo, 0, driver::SequenceTrack::TRACK_VARIABLE_NUM );

    DriverCommand& cmdmgr = DriverCommand::GetInstance();
    DriverCommandSequenceSoundSetVar* command =
        cmdmgr.AllocCommand<DriverCommandSequenceSoundSetVar>();
    command->id = DRIVER_COMMAND_SEQ_SETTVAR;
    command->player = &m_PlayerInstance;
    command->trackNo = trackNo;
    command->varNo = varNo;
    command->var = var;
    cmdmgr.PushCommand(command);
}

/*---------------------------------------------------------------------------*
  Name:         GetTick

  Description:  再生開始からのティック数の取得

  Arguments:    None.

  Returns:      ティック数
 *---------------------------------------------------------------------------*/
u32 SequenceSound::GetTick() const
{
    if ( ! IsPlayerAvailable() )
    {
        // pre start
        return 0;
    }

    return m_PlayerInstance.GetTickCounter();
}

bool SequenceSound::IsAttachedTempSpecialHandle()
{
    return m_pTempSpecialHandle != NULL;
}

void SequenceSound::DetachTempSpecialHandle()
{
    m_pTempSpecialHandle->DetachSound();
}

void SequenceSound::RegisterDataLoadTask(
    const driver::SequenceSoundLoader::LoadInfo& loadInfo,
    const driver::SequenceSoundPlayer::StartInfo& startInfo )
{
    {
        DriverCommand& cmdmgr = DriverCommand::GetInstance();
        DriverCommandSequenceSoundLoad* command =
            cmdmgr.AllocCommand<DriverCommandSequenceSoundLoad>();
        command->id = DRIVER_COMMAND_SEQ_LOAD;
        command->player = &m_PlayerInstance;
        command->startInfo = startInfo; // 構造体コピー
        command->arg.soundArchive = loadInfo.soundArchive;
        command->arg.soundDataManager = loadInfo.soundDataManager;
        command->arg.soundPlayer = loadInfo.soundPlayer;
        command->arg.loadInfoSeq = *(loadInfo.loadInfoSeq); // 構造体コピー
        for (int i = 0; i < SoundArchive::SEQ_BANK_MAX; i++)
        {
            command->arg.loadInfoBanks[i] = *(loadInfo.loadInfoBanks[i]);
            // command->arg.loadInfoWarcss[i] = *(loadInfo.loadInfoWarcs[i]);
        }

        cmdmgr.PushCommand(command);
    }
}


} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw

