﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/snd_PlayerHeap.h>

#include <nw/ut/ut_Inlines.h>
#include <nw/snd/snd_SoundThread.h>
#include <nw/snd/snd_DisposeCallbackManager.h>
#include <nw/snd/snd_DriverCommand.h>
#include <nw/snd/snd_Util.h>

namespace {
    const size_t ALIGN_SIZE = PPC_IO_BUFFER_ALIGN;
}

namespace nw {
namespace snd {
namespace internal {

/*---------------------------------------------------------------------------*
  Name:         SoundHeap

  Description:  コンストラクタ

  Arguments:    無し

  Returns:      無し
 *---------------------------------------------------------------------------*/
PlayerHeap::PlayerHeap()
: m_pPlayer( NULL ),
  m_pStartAddress( NULL ),
  m_pEndAddress( NULL ),
  m_pAllocAddress( NULL ),
  m_State(STATE_CONSTRUCTED)
{
}

/*---------------------------------------------------------------------------*
  Name:         ~PlayerHeap

  Description:  デストラクタ

  Arguments:    無し

  Returns:      無し
 *---------------------------------------------------------------------------*/
PlayerHeap::~PlayerHeap()
{
    Destroy();
    m_State = STATE_DESTRUCTED;
}

/*---------------------------------------------------------------------------*
  Name:         Create

  Description:

  Arguments:    startAddress -
                size -

  Returns:
 *---------------------------------------------------------------------------*/
bool PlayerHeap::Create( void* startAddress, size_t size )
{
    NW_WARNING(
        snd::internal::Util::IsValidMemoryForDsp( startAddress, size ),
        "the memory area (0x%08x - 0x%08x %dbyte) provided cross a 512 MB segment.",
        startAddress,
        ut::AddOffsetToPtr( startAddress, size ),
        size );

    void* endAddress = ut::AddOffsetToPtr( startAddress, size );
    startAddress = ut::RoundUp( startAddress, ALIGN_SIZE );
    if ( startAddress > endAddress )
    {
        return false;
    }

    m_pStartAddress = startAddress;
    m_pEndAddress = endAddress;
    m_pAllocAddress = m_pStartAddress;

    return true;
}

/*---------------------------------------------------------------------------*
  Name:         Destroy

  Description:

  Arguments:    なし

  Returns:      なし
 *---------------------------------------------------------------------------*/
void PlayerHeap::Destroy()
{
    Clear();
    m_pAllocAddress = NULL;
}

/*---------------------------------------------------------------------------*
  Name:         Alloc

  Description:

  Arguments:    size -

  Returns:
 *---------------------------------------------------------------------------*/
void* PlayerHeap::Alloc( size_t size )
{
    NW_ALIGN32_ASSERT( m_pAllocAddress );

    void* endp = ut::AddOffsetToPtr( m_pAllocAddress, size );
    if ( endp > m_pEndAddress )
    {
        return NULL;
    }

    void* allocAddress = m_pAllocAddress;
    m_pAllocAddress = ut::RoundUp( endp, ALIGN_SIZE );
    return allocAddress;
}

/*---------------------------------------------------------------------------*
  Name:         Clear

  Description:

  Arguments:    なし

  Returns:      なし
 *---------------------------------------------------------------------------*/
void PlayerHeap::Clear()
{
#if 0   // サウンドスレッドから呼ばれ、かつ、だれもこのヒープを使っていないことが保障されている
    internal::DriverCommand& cmdmgr = internal::DriverCommand::GetInstance();

    internal::DriverCommandInvalidateData* command =
        cmdmgr.AllocCommand<internal::DriverCommandInvalidateData>();
    command->id = internal::DRIVER_COMMAND_INVALIDATE_DATA;
    command->mem = m_pStartAddress;
    command->size = static_cast<unsigned long>( ut::GetOffsetFromPtr( m_pStartAddress, m_pAllocAddress ) );
    cmdmgr.PushCommand(command);
#endif
    m_pAllocAddress = m_pStartAddress;
}

/*---------------------------------------------------------------------------*
  Name:         GetFreeSize

  Description:

  Arguments:    なし

  Returns:
 *---------------------------------------------------------------------------*/
size_t PlayerHeap::GetFreeSize() const
{
    size_t offset = ut::GetOffsetFromPtr( m_pAllocAddress, m_pEndAddress );
    return offset;
}


} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw

