﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/config.h>
#include <nw/snd/snd_DriverCommand.h>
#include <nw/snd/snd_MultiVoiceManager.h>
#include <nw/snd/snd_BasicSoundPlayer.h>
#include <nw/snd/snd_SequenceSoundPlayer.h>
#include <nw/snd/snd_StreamSoundPlayer.h>
#include <nw/snd/snd_WaveSoundPlayer.h>
#include <nw/snd/snd_SequenceTrack.h>
#include <nw/snd/snd_DisposeCallbackManager.h>
#include <nw/snd/snd_MultiVoice.h>
#include <nw/snd/snd_TaskManager.h>
#include <nw/snd/snd_SoundThread.h>

namespace nw {
namespace snd {
namespace internal {

void DriverCommand::ProcessCommandList( Command* commandList )
{
    Command* command = commandList;

    while ( command != NULL )
    {
        switch( command->id )
        {
        case DRIVER_COMMAND_DUMMY:
            // do nothing
            break;

        case DRIVER_COMMAND_REPLY:
        {
            DriverCommandReply* c = reinterpret_cast<DriverCommandReply*>(command);
            *(c->ptr) = true;
            break;
        }

        case DRIVER_COMMAND_PLAYER_INIT:
        {
            DriverCommandPlayerInit* c =
                reinterpret_cast<DriverCommandPlayerInit*>(command);
            c->player->Initialize();
            *(c->availableFlagPtr) = true;
            break;
        }

        case DRIVER_COMMAND_PLAYER_RENDERER_TYPE:
        {
            DriverCommandPlayerRendererType* c =
                reinterpret_cast<DriverCommandPlayerRendererType*>(command);
            c->player->SetVoiceRendererType(
                    static_cast<VoiceRendererType>(c->rendererType) );
            break;
        }

        case DRIVER_COMMAND_PLAYER_PANMODE:
        {
            DriverCommandPlayerPanParam* c =
                reinterpret_cast<DriverCommandPlayerPanParam*>(command);
            c->player->SetPanMode( static_cast<PanMode>(c->panMode) );
            break;
        }
        case DRIVER_COMMAND_PLAYER_PANCURVE:
        {
            DriverCommandPlayerPanParam* c =
                reinterpret_cast<DriverCommandPlayerPanParam*>(command);
            c->player->SetPanCurve( static_cast<PanCurve>(c->panCurve) );
            break;
        }

        case DRIVER_COMMAND_PLAYER_REMOTE_FILTER:
        {
            DriverCommandPlayerRemoteFilter* c =
                reinterpret_cast<DriverCommandPlayerRemoteFilter*>(command);
            c->player->SetRemoteFilter( c->remoteFilter );
            break;
        }

        case DRIVER_COMMAND_PLAYER_FRONT_BYPASS:
        {
            DriverCommandPlayerFrontBypass* c =
                reinterpret_cast<DriverCommandPlayerFrontBypass*>(command);
            c->player->SetFrontBypass( c->isFrontBypass );
            break;
        }

        case DRIVER_COMMAND_PLAYER_FINALIZE:
        {
            DriverCommandPlayer* c = reinterpret_cast<DriverCommandPlayer*>(command);
            c->player->Finalize();
            break;
        }

        case DRIVER_COMMAND_PLAYER_START:
        {
            DriverCommandPlayer* c = reinterpret_cast<DriverCommandPlayer*>(command);
            c->player->Start();
            break;
        }

        case DRIVER_COMMAND_PLAYER_STOP:
        {
            DriverCommandPlayer* c = reinterpret_cast<DriverCommandPlayer*>(command);
            if ( c->flag ) {
                c->player->SetVolume( 0.0f );
            }
            c->player->Stop();
            break;
        }

        case DRIVER_COMMAND_PLAYER_PAUSE:
        {
            DriverCommandPlayer* c = reinterpret_cast<DriverCommandPlayer*>(command);
            c->player->Pause( c->flag );
            break;
        }

        case DRIVER_COMMAND_PLAYER_PARAM:
        {
            DriverCommandPlayerParam* c = reinterpret_cast<DriverCommandPlayerParam*>(command);
            c->player->SetVolume( c->volume );
            c->player->SetPitch( c->pitch );
            c->player->SetLpfFreq( c->lpfFreq );
            c->player->SetBiquadFilter( c->biquadFilterType, c->biquadFilterValue );
            c->player->SetOutputLine( c->outputLineFlag );
            c->player->SetTvParam( c->tvParam );
            for ( u32 i = 0; i < DRC_OUT_COUNT; i++ )
            {
                c->player->SetDrcParam( i, c->drcParam[i] );
            }
            for ( u32 i = 0; i < REMOTE_OUT_COUNT; i++ )
            {
                c->player->SetRemoteParam( i, c->remoteParam[i] );
            }
            break;
        }

        case DRIVER_COMMAND_PLAYER_CLEAR_RESOURCEFLAG:
        {
            DriverCommandPlayerClearResourceFlag* c =
                reinterpret_cast<DriverCommandPlayerClearResourceFlag*>(command);
            c->player->ClearIsFinalizedForCannotAllocatedResourceFlag();
            break;
        }

        case DRIVER_COMMAND_SEQ_SETUP:
        {
            DriverCommandSequenceSoundSetup* c =
                reinterpret_cast<DriverCommandSequenceSoundSetup*>(command);
            c->player->Setup(c->arg);
            c->player->SetChannelPriority(c->channelPriority);
            c->player->SetReleasePriorityFix(c->isReleasePriorityFix);
            c->player->SetSequenceUserprocCallback(
                reinterpret_cast<SequenceUserprocCallback>(c->userproc),
                c->userprocArg
            );
            break;
        }

        case DRIVER_COMMAND_SEQ_LOAD:
        {
            DriverCommandSequenceSoundLoad* c =
                reinterpret_cast<DriverCommandSequenceSoundLoad*>(command);
            c->player->RequestLoad(c->startInfo, c->arg);
            break;
        }

        case DRIVER_COMMAND_SEQ_PREPARE:
        {
            DriverCommandSequenceSoundPrepare* c =
                reinterpret_cast<DriverCommandSequenceSoundPrepare*>(command);
            c->player->Prepare(c->arg);
            break;
        }

        case DRIVER_COMMAND_SEQ_SKIP:
        {
            DriverCommandSequenceSoundSkip* c =
                reinterpret_cast<DriverCommandSequenceSoundSkip*>(command);
            c->player->Skip(
                driver::SequenceSoundPlayer::StartOffsetType(c->offsetType),
                c->offset
            );
            break;
        }

        case DRIVER_COMMAND_SEQ_TEMPORATIO:
        {
            DriverCommandSequenceSoundTempoRatio* c =
                reinterpret_cast<DriverCommandSequenceSoundTempoRatio*>(command);
            c->player->SetTempoRatio(
                c->tempoRatio
            );
            break;
        }

        case DRIVER_COMMAND_SEQ_CHANNELPRIO:
        {
            DriverCommandSequenceSoundChannelPrio* c =
                reinterpret_cast<DriverCommandSequenceSoundChannelPrio*>(command);
            c->player->SetChannelPriority(
                c->priority
            );
            break;
        }

        case DRIVER_COMMAND_SEQ_SETVAR:
        {
            DriverCommandSequenceSoundSetVar* c =
                reinterpret_cast<DriverCommandSequenceSoundSetVar*>(command);
            c->player->SetLocalVariable(
                c->varNo,
                static_cast<s16>(c->var)
            );
            break;
        }

        case DRIVER_COMMAND_SEQ_SETGVAR:
        {
            DriverCommandSequenceSoundSetVar* c =
                reinterpret_cast<DriverCommandSequenceSoundSetVar*>(command);
            driver::SequenceSoundPlayer::SetGlobalVariable(
                c->varNo,
                static_cast<s16>(c->var)
            );
            break;
        }

        case DRIVER_COMMAND_SEQ_SETTVAR:
        {
            DriverCommandSequenceSoundSetVar* c =
                reinterpret_cast<DriverCommandSequenceSoundSetVar*>(command);
            driver::SequenceTrack* track = c->player->GetPlayerTrack( c->trackNo );
            if ( track != NULL ) {
                track->SetTrackVariable( c->varNo, static_cast<s16>(c->var) );
            }
            break;
        }

        case DRIVER_COMMAND_SEQTRACK_MUTE:
        {
            DriverCommandSequenceSoundTrackMute* c = reinterpret_cast<DriverCommandSequenceSoundTrackMute*>(command);
            c->player->SetTrackMute(
                c->trackBitFlag,
                SeqMute(c->mute)
            );
            break;
        }

        case DRIVER_COMMAND_SEQTRACK_SILENCE:
        {
            DriverCommandSequenceSoundTrackSilence* c =
                reinterpret_cast<DriverCommandSequenceSoundTrackSilence*>(command);
            c->player->SetTrackSilence(
                c->trackBitFlag,
                c->silenceFlag,
                c->fadeFrames
            );
            break;
        }

        case DRIVER_COMMAND_SEQTRACK_VOLUME:
        {
            DriverCommandSequenceSoundTrackParam* c =
                reinterpret_cast<DriverCommandSequenceSoundTrackParam*>(command);
            c->player->SetTrackVolume(
                c->trackBitFlag,
                c->value
            );
            break;
        }

        case DRIVER_COMMAND_SEQTRACK_PITCH:
        {
            DriverCommandSequenceSoundTrackParam* c =
                reinterpret_cast<DriverCommandSequenceSoundTrackParam*>(command);
            c->player->SetTrackPitch(
                c->trackBitFlag,
                c->value
            );
            break;
        }

        case DRIVER_COMMAND_SEQTRACK_LPF:
        {
            DriverCommandSequenceSoundTrackParam* c =
                reinterpret_cast<DriverCommandSequenceSoundTrackParam*>(command);
            c->player->SetTrackLpfFreq(
                c->trackBitFlag,
                c->value
            );
            break;
        }

        case DRIVER_COMMAND_SEQTRACK_BIQUAD:
        {
            DriverCommandSequenceSoundTrackBiquad* c =
                reinterpret_cast<DriverCommandSequenceSoundTrackBiquad*>(command);
            c->player->SetTrackBiquadFilter(
                c->trackBitFlag,
                c->type,
                c->value
            );
            break;
        }

        case DRIVER_COMMAND_SEQTRACK_BANKINDEX:
        {
            DriverCommandSequenceSoundTrackBankIndex* c =
                reinterpret_cast<DriverCommandSequenceSoundTrackBankIndex*>(command);
            c->player->SetTrackBankIndex(
                c->trackBitFlag,
                c->bankIndex
            );
            break;
        }

        case DRIVER_COMMAND_SEQTRACK_TRANSPOSE:
        {
            DriverCommandSequenceSoundTrackTranspose* c =
                reinterpret_cast<DriverCommandSequenceSoundTrackTranspose*>(command);
            c->player->SetTrackTranspose(
                    c->trackBitFlag,
                    c->transpose
            );
            break;
        }

        case DRIVER_COMMAND_SEQTRACK_VELOCITYRANGE:
        {
            DriverCommandSequenceSoundTrackVelocityRange* c =
                reinterpret_cast<DriverCommandSequenceSoundTrackVelocityRange*>(command);
            c->player->SetTrackVelocityRange(
                    c->trackBitFlag,
                    c->range
            );
            break;
        }

        case DRIVER_COMMAND_SEQTRACK_OUTPUTLINE:
        {
            DriverCommandSequenceSoundTrackOutputLine* c =
                reinterpret_cast<DriverCommandSequenceSoundTrackOutputLine*>(command);
            c->player->SetTrackOutputLine(
                    c->trackBitFlag,
                    c->outputLine
            );
            break;
        }

        case DRIVER_COMMAND_SEQTRACK_OUTPUTLINE_RESET:
        {
            DriverCommandSequenceSoundTrackOutputLine* c =
                reinterpret_cast<DriverCommandSequenceSoundTrackOutputLine*>(command);
            c->player->ResetTrackOutputLine(
                    c->trackBitFlag
            );
            break;
        }

        case DRIVER_COMMAND_SEQTRACK_TV_VOLUME:
        {
            DriverCommandSequenceSoundTrackParam* c =
                reinterpret_cast<DriverCommandSequenceSoundTrackParam*>(command);
            c->player->SetTrackTvVolume(
                c->trackBitFlag,
                c->value
            );
            break;
        }

        case DRIVER_COMMAND_SEQTRACK_TV_MIXPARAMETER:
        {
            DriverCommandSequenceSoundTrackMixParameter* c =
                reinterpret_cast<DriverCommandSequenceSoundTrackMixParameter*>(command);
            c->player->SetTrackChannelTvMixParameter(
                c->trackBitFlag,
                c->srcChNo,
                c->param
                );
            break;
        }

        case DRIVER_COMMAND_SEQTRACK_TV_PAN:
        {
            DriverCommandSequenceSoundTrackParam* c =
                reinterpret_cast<DriverCommandSequenceSoundTrackParam*>(command);
            c->player->SetTrackTvPan(
                c->trackBitFlag,
                c->value
            );
            break;
        }

        case DRIVER_COMMAND_SEQTRACK_TV_SPAN:
        {
            DriverCommandSequenceSoundTrackParam* c =
                reinterpret_cast<DriverCommandSequenceSoundTrackParam*>(command);
            c->player->SetTrackTvSurroundPan(
                c->trackBitFlag,
                c->value
            );
            break;
        }

        case DRIVER_COMMAND_SEQTRACK_TV_MAINSEND:
        {
            DriverCommandSequenceSoundTrackParam* c =
                reinterpret_cast<DriverCommandSequenceSoundTrackParam*>(command);
            c->player->SetTrackTvMainSend(
                c->trackBitFlag,
                c->value
            );
            break;
        }

        case DRIVER_COMMAND_SEQTRACK_TV_FXSEND:
        {
            DriverCommandSequenceSoundTrackParam* c =
                reinterpret_cast<DriverCommandSequenceSoundTrackParam*>(command);
            c->player->SetTrackTvFxSend(
                c->trackBitFlag,
                static_cast<AuxBus>(c->u32Value),
                c->value
            );
            break;
        }

        case DRIVER_COMMAND_SEQTRACK_DRC_VOLUME:
        {
            DriverCommandSequenceSoundTrackDrcParam* c =
                reinterpret_cast<DriverCommandSequenceSoundTrackDrcParam*>(command);
            c->player->SetTrackDrcVolume(
                c->drcIndex,
                c->trackBitFlag,
                c->value
            );
            break;
        }

        case DRIVER_COMMAND_SEQTRACK_DRC_MIXPARAMETER:
        {
            DriverCommandSequenceSoundTrackMixParameter* c =
                reinterpret_cast<DriverCommandSequenceSoundTrackMixParameter*>(command);
            c->player->SetTrackChannelDrcMixParameter(
                c->drcIndex,
                c->trackBitFlag,
                c->srcChNo,
                c->param
            );
            break;
        }

        case DRIVER_COMMAND_SEQTRACK_DRC_PAN:
        {
            DriverCommandSequenceSoundTrackDrcParam* c =
                reinterpret_cast<DriverCommandSequenceSoundTrackDrcParam*>(command);
            c->player->SetTrackDrcPan(
                c->drcIndex,
                c->trackBitFlag,
                c->value
            );
            break;
        }

        case DRIVER_COMMAND_SEQTRACK_DRC_SPAN:
        {
            DriverCommandSequenceSoundTrackDrcParam* c =
                reinterpret_cast<DriverCommandSequenceSoundTrackDrcParam*>(command);
            c->player->SetTrackDrcSurroundPan(
                c->drcIndex,
                c->trackBitFlag,
                c->value
            );
            break;
        }

        case DRIVER_COMMAND_SEQTRACK_DRC_MAINSEND:
        {
            DriverCommandSequenceSoundTrackDrcParam* c =
                reinterpret_cast<DriverCommandSequenceSoundTrackDrcParam*>(command);
            c->player->SetTrackDrcMainSend(
                c->drcIndex,
                c->trackBitFlag,
                c->value
            );
            break;
        }

        case DRIVER_COMMAND_SEQTRACK_DRC_FXSEND:
        {
            DriverCommandSequenceSoundTrackDrcParam* c =
                reinterpret_cast<DriverCommandSequenceSoundTrackDrcParam*>(command);
            c->player->SetTrackDrcFxSend(
                c->drcIndex,
                c->trackBitFlag,
                static_cast<AuxBus>(c->u32Value),
                c->value
            );
            break;
        }

        case DRIVER_COMMAND_WSD_PREPARE:
        {
            DriverCommandWaveSoundPrepare* c =
                reinterpret_cast<DriverCommandWaveSoundPrepare*>(command);
            c->player->Prepare(c->startInfo, c->arg);
            break;
        }

        case DRIVER_COMMAND_WSD_LOAD:
        {
            DriverCommandWaveSoundLoad* c =
                reinterpret_cast<DriverCommandWaveSoundLoad*>(command);
            c->player->RequestLoad(c->startInfo, c->arg);
            break;
        }

        case DRIVER_COMMAND_WSD_CHANNEL_PRIO:
        {
            DriverCommandWaveSoundChannelPrio* c =
                reinterpret_cast<DriverCommandWaveSoundChannelPrio*>(command);
            c->player->SetChannelPriority(c->priority);
            break;
        }

        case DRIVER_COMMAND_WSD_CHANNEL_PARAM:
        {
            DriverCommandWaveSoundChannelParam* c =
                reinterpret_cast<DriverCommandWaveSoundChannelParam*>(command);
            c->player->SetChannelPriority(c->priority);
            c->player->SetReleasePriorityFix(c->isReleasePriorityFix);
            break;
        }

        case DRIVER_COMMAND_STRM_SETUP:
        {
            DriverCommandStreamSoundSetup* c =
                reinterpret_cast<DriverCommandStreamSoundSetup*>(command);
            c->player->Setup(c->arg);
            break;
        }

        case DRIVER_COMMAND_STRM_PREPARE:
        {
            DriverCommandStreamSoundPrepare* c =
                reinterpret_cast<DriverCommandStreamSoundPrepare*>(command);
            c->player->Prepare(c->arg);
            break;
        }

        case DRIVER_COMMAND_STRM_PREPARE_PREFETCH:
        {
            DriverCommandStreamSoundPreparePrefetch* c =
                reinterpret_cast<DriverCommandStreamSoundPreparePrefetch*>(command);
            c->player->PreparePrefetch(c->arg);
            break;
        }

        case DRIVER_COMMAND_STRM_LOADHEADER:
        {
            DriverCommandStreamSoundLoadHeader* c = reinterpret_cast<DriverCommandStreamSoundLoadHeader*>(command);
            c->player->LoadHeader(
                c->result,
                c->adpcmParam,
                c->requestIndex
            );
            break;
        }

        case DRIVER_COMMAND_STRM_LOADDATA:
        {
            DriverCommandStreamSoundLoadData* c = reinterpret_cast<DriverCommandStreamSoundLoadData*>(command);
            c->player->LoadStreamData(
                c->result,
                c->loadDataParam
            );
            break;
        }

        case DRIVER_COMMAND_STRM_FORCE_FINISH:
        {
            DriverCommandStreamSoundForceFinish* c = reinterpret_cast<DriverCommandStreamSoundForceFinish*>(command);
            c->player->ForceFinish();
            break;
        }


        case DRIVER_COMMAND_STRM_TRACK_VOLUME:
        {
            DriverCommandStreamSoundTrackParam* c =
                reinterpret_cast<DriverCommandStreamSoundTrackParam*>(command);
            c->player->SetTrackVolume(
                c->trackBitFlag,
                c->value
            );
            break;
        }

        case DRIVER_COMMAND_STRM_TRACK_INITIAL_VOLUME:
        {
            DriverCommandStreamSoundTrackInitialVolume* c =
                reinterpret_cast<DriverCommandStreamSoundTrackInitialVolume*>(command);
            c->player->SetTrackInitialVolume(
                c->trackBitFlag,
                c->value
            );
            break;
        }

        case DRIVER_COMMAND_STRM_TRACK_OUTPUTLINE:
        {
            DriverCommandStreamSoundTrackParam* c =
                reinterpret_cast<DriverCommandStreamSoundTrackParam*>(command);
            c->player->SetTrackOutputLine(
                c->trackBitFlag,
                c->u32Value
            );
            break;
        }

        case DRIVER_COMMAND_STRM_TRACK_OUTPUTLINE_RESET:
        {
            DriverCommandStreamSoundTrackParam* c =
                reinterpret_cast<DriverCommandStreamSoundTrackParam*>(command);
            c->player->ResetTrackOutputLine(
                c->trackBitFlag
            );
            break;
        }

        case DRIVER_COMMAND_STRM_TRACK_TV_VOLUME:
        {
            DriverCommandStreamSoundTrackParam* c =
                reinterpret_cast<DriverCommandStreamSoundTrackParam*>(command);
            c->player->SetTrackTvVolume(
                c->trackBitFlag,
                c->value
            );
            break;
        }

        case DRIVER_COMMAND_STRM_TRACK_TV_MIXPARAMETER:
        {
            DriverCommandStreamSoundTrackMixParameter* c =
                reinterpret_cast<DriverCommandStreamSoundTrackMixParameter*>(command);
            c->player->SetTrackChannelTvMixParameter(
                c->trackBitFlag,
                c->srcChNo,
                c->param
                );
            break;
        }

        case DRIVER_COMMAND_STRM_TRACK_TV_PAN:
        {
            DriverCommandStreamSoundTrackParam* c =
                reinterpret_cast<DriverCommandStreamSoundTrackParam*>(command);
            c->player->SetTrackTvPan(
                c->trackBitFlag,
                c->value
            );
            break;
        }

        case DRIVER_COMMAND_STRM_TRACK_TV_SPAN:
        {
            DriverCommandStreamSoundTrackParam* c =
                reinterpret_cast<DriverCommandStreamSoundTrackParam*>(command);
            c->player->SetTrackTvSurroundPan(
                c->trackBitFlag,
                c->value
            );
            break;
        }

        case DRIVER_COMMAND_STRM_TRACK_TV_MAINSEND:
        {
            DriverCommandStreamSoundTrackParam* c =
                reinterpret_cast<DriverCommandStreamSoundTrackParam*>(command);
            c->player->SetTrackTvMainSend(
                c->trackBitFlag,
                c->value
            );
            break;
        }

        case DRIVER_COMMAND_STRM_TRACK_TV_FXSEND:
        {
            DriverCommandStreamSoundTrackParam* c =
                reinterpret_cast<DriverCommandStreamSoundTrackParam*>(command);
            c->player->SetTrackTvFxSend(
                c->trackBitFlag,
                static_cast<AuxBus>(c->u32Value),
                c->value
            );
            break;
        }

        case DRIVER_COMMAND_STRM_TRACK_DRC_VOLUME:
        {
            DriverCommandStreamSoundTrackParam* c =
                reinterpret_cast<DriverCommandStreamSoundTrackParam*>(command);
            c->player->SetTrackDrcVolume(
                c->drcIndex,
                c->trackBitFlag,
                c->value
            );
            break;
        }

        case DRIVER_COMMAND_STRM_TRACK_DRC_MIXPARAMETER:
        {
            DriverCommandStreamSoundTrackMixParameter* c =
                reinterpret_cast<DriverCommandStreamSoundTrackMixParameter*>(command);
            c->player->SetTrackChannelDrcMixParameter(
                c->drcIndex,
                c->trackBitFlag,
                c->srcChNo,
                c->param
                );
            break;
        }

        case DRIVER_COMMAND_STRM_TRACK_DRC_PAN:
        {
            DriverCommandStreamSoundTrackParam* c =
                reinterpret_cast<DriverCommandStreamSoundTrackParam*>(command);
            c->player->SetTrackDrcPan(
                c->drcIndex,
                c->trackBitFlag,
                c->value
            );
            break;
        }

        case DRIVER_COMMAND_STRM_TRACK_DRC_SPAN:
        {
            DriverCommandStreamSoundTrackParam* c =
                reinterpret_cast<DriverCommandStreamSoundTrackParam*>(command);
            c->player->SetTrackDrcSurroundPan(
                c->drcIndex,
                c->trackBitFlag,
                c->value
            );
            break;
        }

        case DRIVER_COMMAND_STRM_TRACK_DRC_MAINSEND:
        {
            DriverCommandStreamSoundTrackParam* c =
                reinterpret_cast<DriverCommandStreamSoundTrackParam*>(command);
            c->player->SetTrackDrcMainSend(
                c->drcIndex,
                c->trackBitFlag,
                c->value
            );
            break;
        }

        case DRIVER_COMMAND_STRM_TRACK_DRC_FXSEND:
        {
            DriverCommandStreamSoundTrackParam* c =
                reinterpret_cast<DriverCommandStreamSoundTrackParam*>(command);
            c->player->SetTrackDrcFxSend(
                c->drcIndex,
                c->trackBitFlag,
                static_cast<AuxBus>(c->u32Value),
                c->value
            );
            break;
        }


        case DRIVER_COMMAND_INVALIDATE_DATA:
        {
            DriverCommandInvalidateData* c =
                reinterpret_cast<DriverCommandInvalidateData*>(command);
            driver::DisposeCallbackManager::GetInstance().Dispose(
                c->mem,
                c->size
            );
            break;
        }

        case DRIVER_COMMAND_REGIST_DISPOSE_CALLBACK:
        {
            DriverCommandDisposeCallback* c =
                reinterpret_cast<DriverCommandDisposeCallback*>(command);
            driver::DisposeCallbackManager::GetInstance().RegisterDisposeCallback(
                c->callback
            );
            break;
        }

        case DRIVER_COMMAND_UNREGIST_DISPOSE_CALLBACK:
        {
            DriverCommandDisposeCallback* c =
                reinterpret_cast<DriverCommandDisposeCallback*>(command);
            driver::DisposeCallbackManager::GetInstance().UnregisterDisposeCallback(
                c->callback
            );
            break;
        }

        case DRIVER_COMMAND_APPEND_EFFECT:
        {
            DriverCommandEffect* c = reinterpret_cast<DriverCommandEffect*>(command);
            internal::driver::HardwareManager::GetInstance().AppendEffect(
                c->bus,
                c->effect,
                c->device
            );
            break;
        }

        case DRIVER_COMMAND_CLEAR_EFFECT:
        {
            DriverCommandEffect* c = reinterpret_cast<DriverCommandEffect*>(command);
            internal::driver::HardwareManager::GetInstance().ClearEffect(
                c->bus,
                c->fadeTimes,
                c->device
            );
            break;
        }

        case DRIVER_COMMAND_DEBUG:
        {
        #ifdef NW_CONSOLE_ENABLE
            DriverCommandDebug* c = reinterpret_cast<DriverCommandDebug*>(command);
            NW_LOG("[CMD] %s\n", c->str );
        #endif // NW_CONSOLE_ENABLE
            break;
        }

        case DRIVER_COMMAND_ALLVOICES_SYNC:
        {
            DriverCommandAllVoicesSync* c =
                reinterpret_cast<DriverCommandAllVoicesSync*>(command);
            driver::MultiVoiceManager::GetInstance().UpdateAllVoicesSync( c->syncFlag );
            break;
        }

        case DRIVER_COMMAND_VOICE_PLAY:
        {
            DriverCommandVoicePlay* c = reinterpret_cast<DriverCommandVoicePlay*>(command);
            switch ( c->state )
            {
            case DriverCommandVoicePlay::STATE_START:
                c->voice->Start();
                break;
            case DriverCommandVoicePlay::STATE_STOP:
                c->voice->Stop();
                c->voice->Free();
                break;
            case DriverCommandVoicePlay::STATE_PAUSE_ON:
                c->voice->Pause( true );
                break;
            case DriverCommandVoicePlay::STATE_PAUSE_OFF:
                c->voice->Pause( false );
                break;
            }
            break;
        }
        case DRIVER_COMMAND_VOICE_WAVEINFO:
        {
            DriverCommandVoiceWaveInfo* c =
                reinterpret_cast<DriverCommandVoiceWaveInfo*>(command);
            driver::MultiVoice* voice = c->voice;
            voice->SetSampleFormat( c->format );
            voice->SetSampleRate( c->sampleRate );
            voice->SetInterpolationType( static_cast<u8>(c->interpolationType) );
            break;
        }
        case DRIVER_COMMAND_VOICE_ADPCM_PARAM:
        {
            DriverCommandVoiceAdpcmParam* c =
                reinterpret_cast<DriverCommandVoiceAdpcmParam*>(command);
            c->voice->SetAdpcmParam( c->channel, *c->param );
            break;
        }
#if 0
        case DRIVER_COMMAND_VOICE_APPEND_WAVEBUFFER:
        {
            DriverCommandVoiceAppendWaveBuffer* c =
                reinterpret_cast<DriverCommandVoiceAppendWaveBuffer*>(command);
            c->voice->AppendWaveBuffer(
                c->channel,
                c->buffer,
                c->lastFlag );
            break;
        }
#endif

#if 0
        case DRIVER_COMMAND_VOICE_PRIORITY:
        {
            DriverCommandVoicePriority* c =
                reinterpret_cast<DriverCommandVoicePriority*>(command);
            c->voice->SetPriority( c->priority );
            break;
        }
        case DRIVER_COMMAND_VOICE_VOLUME:
        {
            DriverCommandVoiceVolume* c = reinterpret_cast<DriverCommandVoiceVolume*>(command);
            c->voice->SetVolume( c->volume );
            break;
        }
        case DRIVER_COMMAND_VOICE_PITCH:
        {
            DriverCommandVoicePitch* c = reinterpret_cast<DriverCommandVoicePitch*>(command);
            c->voice->SetPitch( c->pitch );
            break;
        }
        case DRIVER_COMMAND_VOICE_PAN_MODE:
        {
            DriverCommandVoicePanMode* c =
                reinterpret_cast<DriverCommandVoicePanMode*>(command);
            c->voice->SetPanMode( c->mode );
            break;
        }
        case DRIVER_COMMAND_VOICE_PAN_CURVE:
        {
            DriverCommandVoicePanCurve* c =
                reinterpret_cast<DriverCommandVoicePanCurve*>(command);
            c->voice->SetPanCurve( c->curve );
            break;
        }
        case DRIVER_COMMAND_VOICE_PAN:
        {
            DriverCommandVoicePan* c = reinterpret_cast<DriverCommandVoicePan*>(command);
            c->voice->SetPan( c->pan );
            break;
        }
        case DRIVER_COMMAND_VOICE_SPAN:
        {
            DriverCommandVoiceSurroundPan* c =
                reinterpret_cast<DriverCommandVoiceSurroundPan*>(command);
            c->voice->SetSurroundPan( c->span );
            break;
        }
        case DRIVER_COMMAND_VOICE_LPF:
        {
            DriverCommandVoiceLpfFreq* c =
                reinterpret_cast<DriverCommandVoiceLpfFreq*>(command);
            c->voice->SetLpfFreq( c->lpfFreq );
            break;
        }
        case DRIVER_COMMAND_VOICE_BIQUAD:
        {
            DriverCommandVoiceBiquadFilter* c =
                reinterpret_cast<DriverCommandVoiceBiquadFilter*>(command);
            c->voice->SetBiquadFilter( c->type, c->value );
            break;
        }
        case DRIVER_COMMAND_VOICE_OUTPUT_LINE:
        {
            DriverCommandVoiceOutputLine* c =
                reinterpret_cast<DriverCommandVoiceOutputLine*>(command);
            c->voice->SetOutputLine( c->lineFlag );
            break;
        }
        case DRIVER_COMMAND_VOICE_MAIN_OUT_VOLUME:
        {
            DriverCommandVoiceMainOutVolume* c =
                reinterpret_cast<DriverCommandVoiceMainOutVolume*>(command);
            c->voice->SetMainOutVolume( c->volume );
            break;
        }
        case DRIVER_COMMAND_VOICE_REMOTE_OUT_VOLUME:
        {
            DriverCommandVoiceRemoteOutVolume* c =
                reinterpret_cast<DriverCommandVoiceRemoteOutVolume*>(command);
            c->voice->SetRemoteOutVolume( c->remoteIndex, c->volume );
            break;
        }
        case DRIVER_COMMAND_VOICE_DRC_OUT_VOLUME:
        {
            DriverCommandVoiceDrcOutVolume* c =
                reinterpret_cast<DriverCommandVoiceDrcOutVolume*>(command);
            c->voice->SetDrcOutVolume( c->drcIndex, c->volume );
            break;
        }
        case DRIVER_COMMAND_VOICE_MAIN_SEND:
        {
            DriverCommandVoiceMainSend* c =
                reinterpret_cast<DriverCommandVoiceMainSend*>(command);
            c->voice->SetMainSend( c->send );
            break;
        }
        case DRIVER_COMMAND_VOICE_FX_SEND:
        {
            DriverCommandVoiceFxSend* c = reinterpret_cast<DriverCommandVoiceFxSend*>(command);
            c->voice->SetFxSend( c->bus, c->send );
            break;
        }
#endif

        }   // switch

        command = command->next;

    } // while
}

DriverCommand& DriverCommand::GetInstance()
{
    static DriverCommand instance;
    return instance;
}

DriverCommand& DriverCommand::GetInstanceForTaskThread()
{
    static DriverCommand instance;
    return instance;
}

DriverCommand::DriverCommand()
{
}

void DriverCommand::Initialize( void* commandBuffer, u32 commandBufferSize )
{
    CommandManager::Initialize( commandBuffer, commandBufferSize, ProcessCommandList );
    CommandManager::SetRequestProcessCommandFunc( RequestProcessCommand );
}

void DriverCommand::RequestProcessCommand()
{
    driver::SoundThread::GetInstance().ForceWakeup();
}

} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw
