﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/snd_CommandManager.h>

#if defined( NW_PLATFORM_WIN32 )
#include <windows.h>
#elif defined(NW_PLATFORM_ANDROID) || defined(NW_PLATFORM_IOS)
#include <atomic>
#endif

namespace nw {
namespace snd {
namespace internal {

CommandBuffer::CommandBuffer()
    : m_CommandMemoryArea(NULL)
    , m_CommandMemoryAreaSize(0)
    , m_CommandMemoryAreaBegin(0)
    , m_CommandMemoryAreaEnd(0)
    , m_CommandMemoryAreaZeroFlag(true)
{
}

CommandBuffer::~CommandBuffer()
{
    Finalize();
}

void CommandBuffer::Initialize( void* commandBuffer, u32 commandBufferSize )
{
    m_CommandMemoryAreaBegin = 0;
    m_CommandMemoryAreaEnd = 0;
    m_CommandMemoryAreaZeroFlag = false;

    m_CommandMemoryArea = static_cast<u32*>(commandBuffer);
    m_CommandMemoryAreaSize = commandBufferSize / sizeof(m_CommandMemoryArea[0]);
}

void CommandBuffer::Finalize()
{
}

void* CommandBuffer::AllocMemory( u32 size )
{
    NW_ASSERT( size >= sizeof(Command) );

    void* ptr = NULL;

    if ( m_CommandMemoryAreaZeroFlag ) return NULL;

    const u32 count = ( size + sizeof(m_CommandMemoryArea[0]) - 1 ) / sizeof(m_CommandMemoryArea[0]);
    NW_ASSERT( count <= m_CommandMemoryAreaSize );

    // 処理途中で FreeMemory() が呼ばれて End の位置が変わる可能性があるため退避する
    volatile u32 curEnd = m_CommandMemoryAreaEnd;

    if ( curEnd > m_CommandMemoryAreaBegin ) {
        // 連続領域の場合
        if ( m_CommandMemoryAreaBegin + count <= curEnd ) {
            ptr = &m_CommandMemoryArea[m_CommandMemoryAreaBegin];
            m_CommandMemoryAreaBegin += count;
        }
    }
    else {
        // 非連続領域の場合
        if ( m_CommandMemoryAreaBegin + count <= m_CommandMemoryAreaSize ) {
            ptr = &m_CommandMemoryArea[m_CommandMemoryAreaBegin];
            m_CommandMemoryAreaBegin += count;
        }
        else if ( count <= curEnd ) {
            ptr = &m_CommandMemoryArea[0];
            m_CommandMemoryAreaBegin = count;
        }
    }

    if ( ptr == NULL )
    {
        return NULL;
    }

    if ( m_CommandMemoryAreaBegin == curEnd ) {
        m_CommandMemoryAreaZeroFlag = true;
    }

    Command* command = reinterpret_cast<Command*>(ptr);
    NW_NULL_ASSERT(command);
    command->memory_next = m_CommandMemoryAreaBegin;

    return command;
}

//---------------------------------------------------------------------------
//! @brief    コマンドバッファのメモリを解放します。
//!
//! @param[in]    lastCommand  開放するメモリの最後のコマンド
//!
//---------------------------------------------------------------------------
void CommandBuffer::FreeMemory( Command* lastCommand )
{
    m_CommandMemoryAreaEnd = lastCommand->memory_next;
    m_CommandMemoryAreaZeroFlag = false;
}

u32 CommandBuffer::GetAllocatableCommandSize() const
{
    if ( m_CommandMemoryAreaZeroFlag ) return 0;

    u32 count;

    if ( m_CommandMemoryAreaEnd > m_CommandMemoryAreaBegin ) {
        // 連続領域の場合
        count =  m_CommandMemoryAreaEnd - m_CommandMemoryAreaBegin;
    }
    else {
        // 非連続領域の場合
        count = ut::Max(
            m_CommandMemoryAreaSize - m_CommandMemoryAreaBegin,
            m_CommandMemoryAreaEnd
        );
    }
    return count * sizeof( m_CommandMemoryArea[0] );
}





CommandManager::CommandManager()
: m_Available( false )
, m_pProcessCommandListFunc( NULL )
, m_pRequestProcessCommandFunc( NULL )
{
}

CommandManager::~CommandManager()
{
    Finalize();
}

void CommandManager::Initialize( void* commandBuffer, u32 commandBufferSize, ProcessCommandListFunc func )
{
    NW_NULL_ASSERT( func );

    m_pProcessCommandListFunc = func;
    m_pRequestProcessCommandFunc = NULL;

    m_CommandBuffer.Initialize( commandBuffer, commandBufferSize );

    m_CommandListBegin = NULL;
    m_CommandListEnd = NULL;

    m_CommandTag = 0;

    m_SendCommandQueue.Initialize(m_SendCommandQueueBuffer, SEND_COMMAND_QUEUE_SIZE);
    m_RecvCommandQueue.Initialize(m_RecvCommandQueueBuffer, RECV_COMMAND_QUEUE_SIZE);

#ifdef NW_PLATFORM_CAFE
    m_CommandListCount.u.u32 = 0;
#else
    m_CommandListCount = 0;
#endif

    m_Available = true;
}

void CommandManager::Finalize()
{
    m_SendCommandQueue.Finalize();
    m_RecvCommandQueue.Finalize();

    m_CommandBuffer.Finalize();

    m_Available = false;
}

void* CommandManager::AllocMemory( u32 size )
{
    void* ptr = TryAllocMemory( size );
    if ( ptr != NULL ) return ptr;

    RecvCommandReply();
    ptr = TryAllocMemory( size );
    if ( ptr != NULL ) return ptr;

    // 強制フラッシュ
#ifndef NW_SND_CONFIG_ENABLE_VOICE_COMMAND
    while( ptr == NULL )
    {
        if ( m_CommandListBegin != NULL )
        {
            u32 tag = FlushCommand( true );
            WaitCommandReply( tag );
        }
        else
        {
            if ( m_pRequestProcessCommandFunc )
            {
                m_pRequestProcessCommandFunc();
            }
            RecvCommandReply();
        }
        ptr = TryAllocMemory( size );
    }
#endif

    return ptr;
}

void* CommandManager::TryAllocMemory( u32 size )
{
    return m_CommandBuffer.AllocMemory( size );
}

u32 CommandManager::PushCommand( Command* command )
{
    if ( m_CommandListEnd == NULL )
    {
        m_CommandListBegin = command;
        m_CommandListEnd = command;
    }
    else
    {
        m_CommandListEnd->next = command;
        m_CommandListEnd = command;
    }

    command->next = NULL;

    return m_CommandTag;
}

//---------------------------------------------------------------------------
//! @brief    コマンドをフラッシュします。
//!
//!           forceFlagがtrueの時は、必ず処理を行います。
//!           コマンドが１つも登録されていない場合は、ダミーのコマンドを発行します。
//!           受信側の処理が滞っている場合は、空くまで待ちます。
//!
//!           forceFlagがfalseの時は、必要以上の処理は行いません。
//!           コマンドが１つも登録されていない場合は、何もせずに返ります。
//!           受信側の処理が滞っている場合も、何もせずに返ります。
//!           何もせずに返った場合、返り値は0（無効値）になることに注意してください。
//!
//! @param[in]    forceFlag  フラッシュ処理を強制するかどうかのフラグです。
//!
//! @return       コマンドタグを返します。
//!               ただし、forceFlagがfalseの時に、コマンドフラッシュが
//!               行われない場合は、0（無効値）を返します。
//!
//---------------------------------------------------------------------------
u32 CommandManager::FlushCommand( bool forceFlag )
{
    if ( m_CommandListBegin == NULL ) {
        if ( ! forceFlag ) return 0;

        // ダミーコマンド発行
        Command* command = AllocCommand<Command>();
        command->id = INVALID_COMMAND;
        PushCommand(command);
    }

    ut::MessageQueue::MessageType msg = reinterpret_cast<ut::MessageQueue::MessageType>(m_CommandListBegin);
    u32 tag = m_CommandTag;

#if defined( NW_PLATFORM_CAFE )
    OSMemoryBarrier();
#endif // defined( NW_PLATFORM_CAFE )

#if defined( NW_PLATFORM_CAFE )
    OSIncAtomic(&m_CommandListCount);
#elif defined(NW_PLATFORM_ANDROID) || defined(NW_PLATFORM_IOS)
    ++m_CommandListCount;
#elif defined( NW_PLATFORM_WIN32 )
    InterlockedIncrement(&m_CommandListCount);
#elif defined( NW_USE_NINTENDO_SDK )
    m_CommandListCount.fetch_add(1);
#endif

    bool result = m_SendCommandQueue.Send( msg, false );
    if ( result == false )
    {
        if ( forceFlag )
        {
            if ( m_pRequestProcessCommandFunc )
            {
                m_pRequestProcessCommandFunc();
            }
            m_SendCommandQueue.Send(msg,true);
        }
        else
        {
            return 0;
        }
    }

    m_CommandListBegin->tag = tag;
    m_CommandTag++;

    m_CommandListBegin = NULL;
    m_CommandListEnd = NULL;

    return tag;
}

void CommandManager::FinalizeCommandList( Command* commandList )
{
    m_FinishCommandTag = commandList->tag;

    Command* command = commandList;

    while( command != NULL )
    {
        if ( command->next == NULL ) {
            m_CommandBuffer.FreeMemory( command );
            break;
        }
        command = command->next;
    }
}

bool CommandManager::IsFinishCommand( u32 tag ) const
{
    static const u32 limit = (1 << ((sizeof(tag) * 8)-2) );

    if ( tag > m_FinishCommandTag )
    {
        return ( tag - m_FinishCommandTag >= limit );
    }
    else
    {
        return ( m_FinishCommandTag - tag < limit );
    }
}

void CommandManager::RecvCommandReply()
{
    ut::MessageQueue::MessageType msg;
    while( m_RecvCommandQueue.Recv( &msg, false ) )
    {
        Command* commandList = reinterpret_cast<Command*>(msg);
        FinalizeCommandList( commandList );
    }
}

void CommandManager::WaitCommandReply( u32 tag )
{
#ifdef NW_SND_CONFIG_ENABLE_VOICE_COMMAND
    NW_UNUSED_VARIABLE(tag);
    while( ProcessCommand() ) {}
#else
    ut::MessageQueue::MessageType msg;
    for ( ;; )
    {
        bool result = m_RecvCommandQueue.Recv(&msg,false);
        if ( ! result )
        {
            if ( m_pRequestProcessCommandFunc )
            {
                m_pRequestProcessCommandFunc();
            }
            m_RecvCommandQueue.Recv(&msg,true);
        }

        Command* commandList = reinterpret_cast<Command*>(msg);
        FinalizeCommandList( commandList );

        if ( tag == commandList->tag ) break;
    }
#endif
}

bool CommandManager::ProcessCommand()
{
    ut::MessageQueue::MessageType msg;
    bool result = m_SendCommandQueue.Recv( &msg, false );
    if ( result == false )
    {
        return false;
    }

#ifdef NW_PLATFORM_CAFE
    OSDecAtomic(&m_CommandListCount);
#elif defined(NW_PLATFORM_ANDROID) || defined(NW_PLATFORM_IOS)
    --m_CommandListCount;
#elif defined( NW_PLATFORM_WIN32 )
    InterlockedDecrement(&m_CommandListCount);
#elif defined( NW_USE_NINTENDO_SDK )
    m_CommandListCount.fetch_sub(1);
#else
    #error
#endif

    Command* commandList = reinterpret_cast<Command*>(msg);

    if ( commandList->id != INVALID_COMMAND )
    {
        m_pProcessCommandListFunc( commandList );

#if defined( NW_PLATFORM_CAFE )
        OSMemoryBarrier();
#endif // defined( NW_PLATFORM_CAFE )
    }

    m_RecvCommandQueue.Send(msg,true);

#ifdef NW_SND_CONFIG_ENABLE_VOICE_COMMAND
    RecvCommandReply();
#endif

    return true;
}

} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw
