﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/snd_Channel.h>
#include <nw/snd/snd_Util.h>
#include <nw/snd/snd_MultiVoiceManager.h>
#include <nw/snd/snd_ChannelManager.h>
#include <nw/snd/snd_Global.h>
#include <nw/snd/snd_DisposeCallbackManager.h>

namespace nw {
namespace snd {
namespace internal {
namespace driver {

// SILENCE_VOLUME_MAX の逆数です。割り算の掛け算化による高速化が狙いです。
const f32 Channel::SILENCE_VOLUME_MAX_R = 1.0f / static_cast<f32>(SILENCE_VOLUME_MAX);

namespace
{

u8 GetNwInterpolationTypeFromHardwareManager()
{
    u8 result = 0;  // 0: ４点補間, 1: 線形補間, 2: 補間なし
                    // (BankFile::VelocityRegion に格納されている値と同じ)
    switch ( HardwareManager::GetInstance().GetSrcType() )
    {
    case SRC_TYPE_NONE:
        result = 2;
        break;
    case SRC_TYPE_LINEAR:
        result = 1;
        break;
    case SRC_TYPE_4TAP:
        // 0 のまま
        break;
    }

    return result;
}
} // anonymous namespace

/* ========================================================================
        member function
   ======================================================================== */

/*---------------------------------------------------------------------------*
  Name:         Channel

  Description:  コンストラクタ

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
Channel::Channel()
: m_PauseFlag( 0 ),
  m_ActiveFlag( false ),
  m_AllocFlag( false ),
  m_pVoice( NULL )
{
    m_Disposer.Initialize(this);
    DisposeCallbackManager::GetInstance().RegisterDisposeCallback( &m_Disposer );
}

/*---------------------------------------------------------------------------*
  Name:         Channel

  Description:  デストラクタ

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
Channel::~Channel()
{
    DisposeCallbackManager::GetInstance().UnregisterDisposeCallback( &m_Disposer );
}

/*---------------------------------------------------------------------------*
  Name:         InitParam

  Description:  パラメータの初期化

  Arguments:

  Returns:      None.
 *---------------------------------------------------------------------------*/
void Channel::InitParam( ChannelCallback callback, void* callbackData )
{
    m_pNextLink              = NULL; // ユーザー領域

    m_Callback               = callback;
    m_CallbackData           = callbackData;

    //  msync_flag           = 0; // NOTE: クリアしてはならない！
    m_PauseFlag              = false;
    m_AutoSweep              = true;
    m_ReleasePriorityFixFlag = false;
    m_IsIgnoreNoteOff        = false;

    m_LoopFlag               = false;
    m_LoopStartFrame         = 0;
    m_OriginalLoopStartFrame = 0;
    m_StartOffsetSamples     = 0;

    m_Length                 = 0;
    m_Key                    = KEY_INIT;
    m_OriginalKey            = ORIGINAL_KEY_INIT;
    m_InitPan                = 0.0f;
    m_InitSurroundPan        = 0.0f;
    m_Tune                   = 1.0f;

    m_Cent                   = 0.0f;
    m_CentPitch              = 1.0f;

    m_UserVolume             = 1.0f;
    m_UserPitch              = 0.0f;
    m_UserPitchRatio         = 1.0f;
    m_UserLpfFreq            = 0.0f;
    m_BiquadType             = BIQUAD_FILTER_TYPE_NONE;
    m_BiquadValue            = 0.0f;

    m_OutputLineFlag         = OUTPUT_LINE_MAIN;

    m_TvParam.Initialize();
    for ( int i = 0; i < DRC_OUT_COUNT; i++ )
    {
        m_DrcParam[i].Initialize();
    }
    for ( int i = 0; i < REMOTE_OUT_COUNT; i++ )
    {
        m_RemoteParam[i].Initialize();
    }

    m_SilenceVolume.InitValue( SILENCE_VOLUME_MAX );

    m_SweepPitch   = 0.0f;
    m_SweepLength  = 0;
    m_SweepCounter = 0;

    m_CurveAdshr.Initialize();
    for (int i = 0; i < MOD_COUNT; i++)
    {
        m_Lfo[i].GetParam().Initialize();
        m_LfoTarget[i] = LFO_TARGET_PITCH;
    }

    m_PanMode = PAN_MODE_DUAL;
    m_PanCurve = PAN_CURVE_SQRT;
    m_RemoteFilter = 0;

    m_KeyGroupId = 0;
    m_InterpolationType = GetNwInterpolationTypeFromHardwareManager();
        // WSD はこの値が反映される。
        // SEQ は Channel::SetInterpolationType によって設定された値
        // (インストのベロシティリージョンに書かれた値) が反映される。
    m_InstrumentVolume = 1.0f;
    m_Velocity         = 1.0f;

    m_UseContextInfo = false;
}

// #define NW_SND_CHANNEL_PROFILE
#ifdef NW_PLATFORM_CTR
#ifdef NW_SND_CHANNEL_PROFILE
namespace
{
nn::os::Tick s_Tick;
int s_Count;
}
nn::os::Tick Channel::GetTick()
{
    nn::os::Tick ret = static_cast<nn::os::Tick>( s_Tick / s_Count );
    s_Tick = nn::os::Tick( 0 );
    s_Count = 0;
    return ret;
}
#else
nn::os::Tick Channel::GetTick()
{
    return nn::os::Tick( 0 );
}
#endif
#endif

/*---------------------------------------------------------------------------*
  Name:         Update

  Description:  チャンネルのフレームワーク

  Arguments:    doPeriodicProc - 周期処理を行うかどうかのフラグ

  Returns:      None.
 *---------------------------------------------------------------------------*/
void Channel::Update( bool doPeriodicProc )
{
    if ( ! m_ActiveFlag ) return;

#ifdef NW_SND_CHANNEL_PROFILE
    nn::os::Tick tick = nn::os::Tick::GetSystemCurrent();
#endif

    if ( m_PauseFlag ) doPeriodicProc = false;

    // volume
    m_SilenceVolume.Update();
    register f32 volume = m_Velocity * m_Velocity * m_InstrumentVolume * m_UserVolume * m_SilenceVolume.GetValue() * SILENCE_VOLUME_MAX_R;

    // 減衰完了チェック
    if ( m_CurveAdshr.GetStatus() == CurveAdshr::STATUS_RELEASE &&
         m_CurveAdshr.GetValue() < -90.4f )
    {
        Stop();
        return;
    }

    // pitch
    f32 cent = m_Key - m_OriginalKey + m_UserPitch + GetSweepValue();
    for (int i = 0; i < MOD_COUNT; i++)
    {
        if ( m_LfoTarget[i] == LFO_TARGET_PITCH )
        {
            cent += m_Lfo[i].GetValue();
        }
    }

    register f32 pitch;
    if ( cent == m_Cent )
    {
        pitch = m_CentPitch;
    }
    else
    {
        pitch = Util::CalcPitchRatio( static_cast<s32>( cent * ( 1 << Util::PITCH_DIVISION_BIT ) ) );
        m_Cent = cent;
        m_CentPitch = pitch;
    }

    pitch = pitch * m_Tune * m_UserPitchRatio;


    OutputParam tvParam = m_TvParam;
    {
        tvParam.pan += m_InitPan;
        for (int i = 0; i < MOD_COUNT; i++)
        {
            if ( m_LfoTarget[i] == LFO_TARGET_PAN )
            {
                tvParam.pan += m_Lfo[i].GetValue();
            }
        }
        tvParam.span += m_InitSurroundPan;
    }

    OutputParam drcParam[DRC_OUT_COUNT];
    for ( int i = 0; i < DRC_OUT_COUNT; i++ )
    {
        OutputParam& param = drcParam[i];
        param = m_DrcParam[i];
        param.pan += m_InitPan;
        for (int j = 0; j < MOD_COUNT; j++)
        {
            if ( m_LfoTarget[j] == LFO_TARGET_PAN )
            {
                param.pan += m_Lfo[j].GetValue();
            }
        }
        param.span += m_InitSurroundPan;
    }

    // カウンタ更新
    if ( doPeriodicProc )
    {
        if ( m_AutoSweep )
        {
            UpdateSweep( HardwareManager::SOUND_FRAME_INTERVAL_MSEC );
        }
        for (int i = 0; i < MOD_COUNT; i++)
        {
            m_Lfo[i].Update( HardwareManager::SOUND_FRAME_INTERVAL_MSEC );
        }
        m_CurveAdshr.Update( HardwareManager::SOUND_FRAME_INTERVAL_MSEC );
    }

    // lpf
    f32 lpfFreq = 1.0f + m_UserLpfFreq;

    // ボリューム値更新
    volume *= Util::CalcVolumeRatio( m_CurveAdshr.GetValue() );
    for (int i = 0; i < MOD_COUNT; i++)
    {
        if ( m_LfoTarget[i] == LFO_TARGET_VOLUME )
        {
            volume *= Util::CalcVolumeRatio( m_Lfo[i].GetValue() * 6.0f );
        }
    }

    if ( m_pVoice != NULL )
    {
        m_pVoice->SetVolume( volume );
        m_pVoice->SetPitch( pitch );

        m_pVoice->SetPanMode( m_PanMode );
        m_pVoice->SetPanCurve( m_PanCurve );
        m_pVoice->SetRemoteFilter( m_RemoteFilter );

        m_pVoice->SetLpfFreq( lpfFreq );
        m_pVoice->SetBiquadFilter( m_BiquadType, m_BiquadValue );

        m_pVoice->SetOutputLine( m_OutputLineFlag );
        m_pVoice->SetTvParam( tvParam );
        for ( int i = 0; i < DRC_OUT_COUNT; i++ )
        {
            m_pVoice->SetDrcParam( i, drcParam[i] );
        }
        for ( int i = 0; i < REMOTE_OUT_COUNT; i++ )
        {
            m_pVoice->SetRemoteParam( i, m_RemoteParam[i] );
        }
    }

#ifdef NW_SND_CHANNEL_PROFILE
    s_Tick += nn::os::Tick::GetSystemCurrent() - tick;
    s_Count += 1;
#endif
}

/*---------------------------------------------------------------------------*
  Name:         Start

  Description:  チャンネルでPCMを再生します

  Arguments:    wave - 波形情報構造体のポインタ
                data - 波形データアドレス
                length - 発音長

  Returns:      成功したかどうか
 *---------------------------------------------------------------------------*/
void Channel::Start( const WaveInfo& waveInfo, int length, u32 startOffsetSamples )
{
    // Channel Param
    m_Length = length;
    for (int i = 0; i < MOD_COUNT; i++)
    {
        m_Lfo[i].Reset();
    }
    m_CurveAdshr.Reset();
    m_SweepCounter = 0;

    m_pVoice->SetSampleFormat( waveInfo.sampleFormat );
    m_pVoice->SetSampleRate( waveInfo.sampleRate );
    m_pVoice->SetInterpolationType( m_InterpolationType );

    AppendWaveBuffer( waveInfo, startOffsetSamples );

    m_pVoice->Start();
    m_ActiveFlag = true;
}

void Channel::SetOffsetContext(const WaveFileReader::OffsetContextInfo& contextInfo)
{
    m_ContextInfo = contextInfo;
    m_UseContextInfo = true;
}

void Channel::AppendWaveBuffer( const WaveInfo& waveInfo, u32 startOffsetSamples )
{
    m_LoopFlag = waveInfo.loopFlag;
    m_LoopStartFrame = waveInfo.loopStartFrame;
    m_OriginalLoopStartFrame = waveInfo.originalLoopStartFrame;

    // 8 バイト境界に切り下げる (DSP ADPCM は 8 バイト 14 サンプル単位のエンコードのため)
    if ( startOffsetSamples > 0 &&
         waveInfo.sampleFormat == nw::snd::SAMPLE_FORMAT_DSP_ADPCM )
    {
        startOffsetSamples = ( startOffsetSamples / 14 ) * 14;
    }

    m_StartOffsetSamples = startOffsetSamples;
    NW_ASSERT( waveInfo.loopEndFrame > startOffsetSamples );


    u32 loopStartByte = MultiVoice::FrameToByte( m_LoopStartFrame, waveInfo.sampleFormat );

    const int sdkVoiceCount = m_pVoice->GetSdkVoiceCount();
    for ( int ch = 0; ch < sdkVoiceCount; ch++ )
    {
        const void* originalDataAddress = waveInfo.channelParam[ch].dataAddress;
        const void* adjustDataAddress = originalDataAddress;
        if ( startOffsetSamples > 0 )
        {
            adjustDataAddress = ut::AddOffsetToPtr(
                originalDataAddress,
                MultiVoice::FrameToByte( startOffsetSamples, waveInfo.sampleFormat ) );
        }

        AdpcmContext& adpcmContext = m_AdpcmContext[ch];
        AdpcmContext& adpcmLoopContext = m_AdpcmLoopContext[ch];
        AdpcmContext& offsetAdpcmContext = m_ContextInfo.contexts[ch];

        // ADPCM の場合、コンテキストを生成
        if ( waveInfo.sampleFormat == SAMPLE_FORMAT_DSP_ADPCM )
        {
            const DspAdpcmParam* pParam = &waveInfo.channelParam[ch].adpcmParam;

            // 係数設定 (u16 coef[16])
            nw::snd::AdpcmParam param;
            for ( int i = 0; i < 8; i++ )
            {
                param.coef[i][0] = pParam->coef[i][0];
                param.coef[i][1] = pParam->coef[i][1];
            }

            // コンテキスト設定 (nw::snd::DspVoice が使用するまで SDK ユーザー側で保持)
            if ( startOffsetSamples == 0 )
            {
                adpcmContext.pred_scale = pParam->predScale;
                adpcmContext.yn1 = pParam->yn1;
                adpcmContext.yn2 = pParam->yn2;
            }
            else    // startOffsetSamples > 0
            {
                if (m_UseContextInfo)
                {
                    adpcmContext = offsetAdpcmContext;  // 構造体コピー
                }
                else
                {
                    // 8 バイト境界の先頭 1 バイトを pred_scale に入れる。
                    // pred_scale は u16 型だが、上位 8 bit は参照されないので、
                    // *reinterpret_cast<const u16*>( adjustDataAddress ) を pred_scale に代入でよいはず。
                    adpcmContext.pred_scale = pParam->predScale;
                    adpcmContext.yn1 = pParam->yn1;
                    adpcmContext.yn2 = pParam->yn2;

                    // 途中まで自前でデコード (通常は DSP でデコードするため、その分 CPU 負荷がかかる)
                    MultiVoice::CalcOffsetAdpcmParam(
                        &adpcmContext,
                        param,
                        startOffsetSamples,
                        originalDataAddress
                    );
                }
            }

            if ( waveInfo.loopFlag )
            {
                const DspAdpcmLoopParam* pLoopParam = &waveInfo.channelParam[ch].adpcmLoopParam;
                adpcmLoopContext.pred_scale = pLoopParam->loopPredScale;
                adpcmLoopContext.yn1 = pLoopParam->loopYn1;
                adpcmLoopContext.yn2 = pLoopParam->loopYn2;
            }

            m_pVoice->SetAdpcmParam( ch, param );
        }

        {
            // |------------------|<---------- loop --------->| のような波形を、
            //
            // |<-------------- pBuffer0 -------------------->|
            //                    |<--------- pBuffer1 ------>| として利用する。

            // WaveBuffer の登録
            WaveBuffer* pBuffer0 = &m_WaveBuffer[ch][0];
            WaveBuffer* pBuffer1 = &m_WaveBuffer[ch][1];

            pBuffer0->bufferAddress = adjustDataAddress;
            pBuffer0->sampleLength = waveInfo.loopEndFrame - startOffsetSamples;
            pBuffer0->loopFlag = false;
            if ( waveInfo.sampleFormat == SAMPLE_FORMAT_DSP_ADPCM )
            {
                pBuffer0->pAdpcmContext = &adpcmContext;
            }
            else
            {
                pBuffer0->pAdpcmContext = NULL;
            }

            if ( waveInfo.loopFlag )
            {
                pBuffer1->bufferAddress = ut::AddOffsetToPtr( originalDataAddress, loopStartByte );
                pBuffer1->sampleLength = waveInfo.loopEndFrame - m_LoopStartFrame;
                pBuffer1->loopFlag = true;
                if ( waveInfo.sampleFormat == SAMPLE_FORMAT_DSP_ADPCM )
                {
                    pBuffer1->pAdpcmContext = &adpcmLoopContext;
                }
                else
                {
                    pBuffer1->pAdpcmContext = NULL;
                }
                m_pVoice->AppendWaveBuffer( ch, pBuffer0, false );
                m_pVoice->AppendWaveBuffer( ch, pBuffer1, true );
            }
            else
            {
                m_pVoice->AppendWaveBuffer( ch, pBuffer0, true );
            }

            // NN_LOG("fmt(%d) [0] len(%d) lp(%d) a(%d,%d,%d) [1] len(%d) lp(%d) a(%d,%d,%d)\n",
            //         waveInfo.sampleFormat,
            //         pBuffer0->sampleLength, pBuffer0->loopFlag,
            //         pBuffer0->pAdpcmContext.pred_scale,
            //         pBuffer0->pAdpcmContext.yn1,
            //         pBuffer0->pAdpcmContext.yn2,
            //         pBuffer1->sampleLength, pBuffer1->loopFlag,
            //         pBuffer1->pAdpcmContext.pred_scale,
            //         pBuffer1->pAdpcmContext.yn1,
            //         pBuffer1->pAdpcmContext.yn2 );
        }
    }
}

/*---------------------------------------------------------------------------*
  Name:         Release

  Description:  チャンネルをリリース状態にします

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
void Channel::Release()
{
    if ( m_CurveAdshr.GetStatus() != CurveAdshr::STATUS_RELEASE )
    {
        if ( m_pVoice != NULL )
        {
            if ( ! m_ReleasePriorityFixFlag )
            {
                m_pVoice->SetPriority( Channel::PRIORITY_RELEASE );
            }
        }
        m_CurveAdshr.SetStatus( CurveAdshr::STATUS_RELEASE );
    }
    m_PauseFlag = false;
}

/*---------------------------------------------------------------------------*
  Name:         NoteOff

  Description:  チャンネルをノートオフします
                通常はリリース状態になります

  Arguments:    なし

  Returns:      なし
 *---------------------------------------------------------------------------*/
void Channel::NoteOff()
{
    if ( m_IsIgnoreNoteOff ) return;

    Release();
}

/*---------------------------------------------------------------------------*
    Name:           Stop

    Description:    チャンネルを停止します

    Arguments:      無し

    Returns:        無し
 *---------------------------------------------------------------------------*/
void Channel::Stop()
{
    if ( m_pVoice == NULL ) return;

    m_pVoice->Stop();
    m_pVoice->Free();
    m_pVoice = NULL;

    m_PauseFlag = false;
    m_ActiveFlag = false;

#if 0
    CallChannelCallback( CALLBACK_STATUS_STOPPED );
    if ( m_AllocFlag )
    {
        m_AllocFlag = false;
        ChannelManager::GetInstance().Free( this );
    }
#endif
}

void Channel::UpdateSweep( int count )
{
    m_SweepCounter += count;
    if ( m_SweepCounter > m_SweepLength )
    {
        m_SweepCounter = m_SweepLength;
    }
}

void Channel::SetSweepParam( f32 sweepPitch, int sweepTime, bool autoUpdate )
{
    m_SweepPitch = sweepPitch;
    m_SweepLength = sweepTime;
    m_AutoSweep = autoUpdate;
    m_SweepCounter = 0;
}

f32 Channel::GetSweepValue() const
{
    if ( m_SweepPitch == 0.0f ) return 0.0f;
    if ( m_SweepCounter >= m_SweepLength ) return 0.0f;

    f32 sweep = m_SweepPitch;
    sweep *= m_SweepLength - m_SweepCounter;

    NW_ASSERT( m_SweepLength != 0 );
    sweep /= m_SweepLength;

    return sweep;
}

void Channel::SetBiquadFilter( int type, f32 value )
{
    // ここに来るまでに、INHERITの処理は済んでいるはずなので、BIQUAD_FILTER_TYPE_DATA_MINが最小になるはずです。
    NW_MINMAX_ASSERT( type, BIQUAD_FILTER_TYPE_DATA_MIN, BIQUAD_FILTER_TYPE_MAX );
    m_BiquadType = static_cast<u8>( type );
    m_BiquadValue = value;
}

u32 Channel::GetCurrentPlayingSample(bool isOriginalSamplePosition) const
{
    if ( ! m_ActiveFlag ) return 0;

    NW_NULL_ASSERT( m_pVoice );

    u32 playSamplePosition = 0;
    if ( m_LoopFlag && m_WaveBuffer[0][0].status == WaveBuffer::STATUS_DONE )
    {
        playSamplePosition = m_LoopStartFrame + m_pVoice->GetCurrentPlayingSample();
    }
    else
    {
        playSamplePosition = m_StartOffsetSamples + m_pVoice->GetCurrentPlayingSample();
    }

    // ループ補正前のサンプル位置を計算
    if ( isOriginalSamplePosition && m_LoopFlag )
    {
        u32 loopEnd = m_StartOffsetSamples + m_WaveBuffer[0][0].sampleLength;
        u32 originalLoopEnd = loopEnd - (m_LoopStartFrame - m_OriginalLoopStartFrame);

        if ( playSamplePosition > originalLoopEnd )
        {
            playSamplePosition = m_OriginalLoopStartFrame + ( playSamplePosition - originalLoopEnd );
        }
    }

    return playSamplePosition;
}

/*---------------------------------------------------------------------------*
    Name:           VoiceCallback

    Description:    ボイスがドロップされたときに呼ばれる

    Arguments:      param: ボイスへのポインタ

    Returns:        none
 *---------------------------------------------------------------------------*/
void Channel::VoiceCallbackFunc(
    MultiVoice* voice,
    MultiVoice::VoiceCallbackStatus status,
    void* arg
)
{
    NW_NULL_ASSERT( arg );

    ChannelCallbackStatus chStatus = CALLBACK_STATUS_FINISH;    // 適当な初期化
    switch ( status )
    {
    case MultiVoice::CALLBACK_STATUS_FINISH_WAVE:
        chStatus = CALLBACK_STATUS_FINISH;
        voice->Free();
        break;
    case MultiVoice::CALLBACK_STATUS_CANCEL:
        chStatus = CALLBACK_STATUS_CANCEL;
        voice->Free();
        break;
    case MultiVoice::CALLBACK_STATUS_DROP_VOICE:
        chStatus = CALLBACK_STATUS_DROP;
        break;
    case MultiVoice::CALLBACK_STATUS_DROP_DSP:
        chStatus = CALLBACK_STATUS_DROP;
        break;
    }

    Channel* channel = static_cast<Channel*>( arg );

#if 0
    int sdkVoiceCount = channel->m_pVoice->GetSdkVoiceCount();
    for ( int channelIndex = 0; channelIndex < sdkVoiceCount; channelIndex++ )
    {
        for ( int waveBufferIndex = 0; waveBufferIndex < WAVE_BUFFER_MAX; waveBufferIndex++ )
        {
            const WaveBuffer& waveBuffer =
                channel->m_WaveBuffer[ channelIndex ][ waveBufferIndex ];
            NW_ASSERT( waveBuffer.status == WaveBuffer::STATUS_FREE || waveBuffer.status == WaveBuffer::STATUS_DONE );
        }
    }
#endif

    channel->CallChannelCallback( chStatus );

    channel->m_pVoice = NULL;
    channel->m_PauseFlag = false;
    channel->m_ActiveFlag = false;

    channel->m_AllocFlag = false;
}

/*---------------------------------------------------------------------------*
  Name:         AllocChannel

  Description:  チャンネルを確保します

  Arguments:    priority - プライオリティ
                callback - チャンネルコールバック関数
                callbackData - チャンネルコールバック関数のユーザー引数

  Returns:      確保したチャンネルのポインタ
                確保できなかった場合はNULL
 *---------------------------------------------------------------------------*/
Channel* Channel::AllocChannel(
    int voiceChannelCount,
    int priority,
    Channel::ChannelCallback callback,
    void* callbackData,
    VoiceRendererType mode
)
{
    NW_MINMAX_ASSERT( priority, Voice::PRIORITY_MIN, Voice::PRIORITY_MAX );

    Channel* channel = ChannelManager::GetInstance().Alloc();
    if ( channel == NULL )
    {
        Util::WarningLogger::GetInstance().Log(
            Util::WarningLogger::LOG_ID_CHANNEL_ALLOCATION_FAILED );
        return NULL;
    }
    channel->m_AllocFlag = true;

    // ボイス取得
    MultiVoice* voice = MultiVoiceManager::GetInstance().AllocVoice(
        voiceChannelCount,
        priority,
        VoiceCallbackFunc,
        channel,
        mode
    );
    if ( voice == NULL )
    {
        ChannelManager::GetInstance().Free( channel );
        return NULL;
    }

    channel->m_pVoice = voice;
    channel->InitParam( callback, callbackData );

    return channel;
}

void Channel::FreeChannel( Channel* channel )
{
    NW_NULL_ASSERT( channel );

    ChannelManager::GetInstance().Free( channel );
}

/*---------------------------------------------------------------------------*
  Name:         DetachChannel

  Description:  チャンネルを解放します

  Arguments:    channel - チャンネルポインタ

  Returns:      None.
 *---------------------------------------------------------------------------*/
void Channel::DetachChannel( Channel* channel )
{
    NW_NULL_ASSERT( channel );

    channel->m_Callback     = NULL;
    channel->m_CallbackData = NULL;
}

void Channel::CallChannelCallback( ChannelCallbackStatus status )
{
    if ( m_Callback != NULL )
    {
        m_Callback( this, status, m_CallbackData );

        m_Callback = NULL;
        m_CallbackData = NULL;
    }
}

void Channel::Disposer::InvalidateData( const void* start, const void* end )
{
    if ( m_pChannel->m_pVoice == NULL ) return;

    // 一つでも start, end 内に波形データを含んでいたらボイス停止
    bool disposeFlag = false;
    int sdkVoiceCount = m_pChannel->m_pVoice->GetSdkVoiceCount();

    for ( int channelIndex = 0; channelIndex < sdkVoiceCount; channelIndex++ )
    {
        for ( int waveBufferIndex = 0; waveBufferIndex < WAVE_BUFFER_MAX; waveBufferIndex++ )
        {
            const WaveBuffer& waveBuffer =
                m_pChannel->m_WaveBuffer[ channelIndex ][ waveBufferIndex ];
            if ( waveBuffer.status == WaveBuffer::STATUS_DONE )
            {
                continue;
            }

            const void* bufferEnd = ut::AddOffsetToPtr(
                    waveBuffer.bufferAddress,
                    Util::GetByteBySample(
                        waveBuffer.sampleLength,
                        m_pChannel->m_pVoice->GetFormat() ) );

            if ( start <= bufferEnd && end >= waveBuffer.bufferAddress )
            {
                disposeFlag = true;
                break;
            }
        }
    }

    if ( disposeFlag )
    {
        m_pChannel->CallChannelCallback( CALLBACK_STATUS_CANCEL );

        m_pChannel->Stop();

        FreeChannel( m_pChannel  );
    }
}


} // namespace nw::snd::internal::driver
} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw

