﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/snd_BiquadFilterPresets.h>
#include <nw/ut/ut_Inlines.h>

#if defined( NW_PLATFORM_WIN32 ) || defined( NW_USE_NINTENDO_SDK )
    #pragma warning(disable:4309)
    // ↑ 係数値 (0x3ab3) が s16 に格納されることで下記警告が出るが、これを抑制する。
    // 「warning C4309:  '初期化中' : 定数値が切り捨てられました。」
#endif

namespace nw {
namespace snd {
namespace internal {

// LPF係数
// （1/16オクターブ間隔、チェビシェフ近似）
const BiquadFilterCallback::Coefficients
BiquadFilterLpf::COEFFICIENTS_TABLE[ BiquadFilterLpf::COEFFICIENTS_TABLE_SIZE ] =
{
    //    n0      n1      n2      d1      d2
    { 0x3ab3, 0x7566, 0x3ab3, 0x83b8, 0xc391 },     // ( 1568.9372Hz )
    { 0x371b, 0x6e36, 0x371b, 0x8bf8, 0xca8c },
    { 0x33c7, 0x678d, 0x33c7, 0x9428, 0xd078 },
    { 0x30b0, 0x6161, 0x30b0, 0x9c32, 0xd582 },
    { 0x2dd2, 0x5ba5, 0x2dd2, 0xa40c, 0xd9cf },
    { 0x2b27, 0x564e, 0x2b27, 0xabae, 0xdd7d },
    { 0x28a9, 0x5152, 0x28a9, 0xb316, 0xe0a3 },
    { 0x2654, 0x4ca8, 0x2654, 0xba45, 0xe355 },
    { 0x2425, 0x4849, 0x2425, 0xc13a, 0xe5a2 },
    { 0x2217, 0x442f, 0x2217, 0xc7f8, 0xe797 },
    { 0x2029, 0x4051, 0x2029, 0xce7f, 0xe93e },
    { 0x1e57, 0x3cad, 0x1e57, 0xd4d3, 0xeaa2 },
    { 0x1c9f, 0x393d, 0x1c9f, 0xdaf5, 0xebc8 },
    { 0x1aff, 0x35fd, 0x1aff, 0xe0e6, 0xecb9 },
    { 0x1975, 0x32eb, 0x1975, 0xe6aa, 0xed78 },
    { 0x1801, 0x3002, 0x1801, 0xec40, 0xee0b },     // 8192Hz
    { 0x16a0, 0x2d40, 0x16a0, 0xf1ab, 0xee77 },
    { 0x1552, 0x2aa4, 0x1552, 0xf6ed, 0xeebe },
    { 0x1415, 0x282a, 0x1415, 0xfc05, 0xeee5 },
    { 0x12e8, 0x25d0, 0x12e8, 0x00f6, 0xeeee },
    { 0x11cb, 0x2396, 0x11cb, 0x05c1, 0xeedb },
    { 0x10bd, 0x217a, 0x10bd, 0x0a65, 0xeeb0 },
    { 0x0fbd, 0x1f79, 0x0fbd, 0x0ee5, 0xee6d },
    { 0x0eca, 0x1d93, 0x0eca, 0x1341, 0xee17 },
    { 0x0de3, 0x1bc7, 0x0de3, 0x177a, 0xedad },
    { 0x0d09, 0x1a13, 0x0d09, 0x1b90, 0xed33 },
    { 0x0c3b, 0x1876, 0x0c3b, 0x1f85, 0xeca9 },
    { 0x0b78, 0x16ef, 0x0b78, 0x2358, 0xec12 },
    { 0x0abf, 0x157e, 0x0abf, 0x270b, 0xeb6e },
    { 0x0a10, 0x1421, 0x0a10, 0x2a9d, 0xeabf },
    { 0x096b, 0x12d7, 0x096b, 0x2e11, 0xea06 },
    { 0x08d0, 0x119f, 0x08d0, 0x3166, 0xe945 },     // 4096Hz
    { 0x083d, 0x107a, 0x083d, 0x349d, 0xe87c },
    { 0x07b2, 0x0f65, 0x07b2, 0x37b7, 0xe7ad },
    { 0x0730, 0x0e60, 0x0730, 0x3ab4, 0xe6d8 },
    { 0x06b5, 0x0d6a, 0x06b5, 0x3d95, 0xe5ff },
    { 0x0642, 0x0c83, 0x0642, 0x405b, 0xe522 },
    { 0x05d5, 0x0baa, 0x05d5, 0x4306, 0xe443 },
    { 0x056f, 0x0adf, 0x056f, 0x4598, 0xe361 },
    { 0x0510, 0x0a1f, 0x0510, 0x4810, 0xe27e },
    { 0x04b6, 0x096c, 0x04b6, 0x4a70, 0xe19b },
    { 0x0462, 0x08c4, 0x0462, 0x4cb8, 0xe0b7 },
    { 0x0413, 0x0826, 0x0413, 0x4ee9, 0xdfd3 },
    { 0x03c9, 0x0793, 0x03c9, 0x5104, 0xdef1 },
    { 0x0384, 0x0709, 0x0384, 0x5309, 0xde10 },
    { 0x0344, 0x0688, 0x0344, 0x54f8, 0xdd31 },
    { 0x0308, 0x0610, 0x0308, 0x56d4, 0xdc54 },
    { 0x02d0, 0x059f, 0x02d0, 0x589c, 0xdb7a },     // 2048Hz
    { 0x029b, 0x0537, 0x029b, 0x5a51, 0xdaa3 },
    { 0x026b, 0x04d5, 0x026b, 0x5bf4, 0xd9cf },
    { 0x023d, 0x047a, 0x023d, 0x5d85, 0xd8fe },
    { 0x0213, 0x0426, 0x0213, 0x5f06, 0xd831 },
    { 0x01eb, 0x03d7, 0x01eb, 0x6076, 0xd768 },
    { 0x01c7, 0x038d, 0x01c7, 0x61d6, 0xd6a3 },
    { 0x01a5, 0x0349, 0x01a5, 0x6327, 0xd5e2 },
    { 0x0185, 0x030a, 0x0185, 0x646a, 0xd526 },
    { 0x0168, 0x02cf, 0x0168, 0x659f, 0xd46d },
    { 0x014c, 0x0299, 0x014c, 0x66c6, 0xd3ba },
    { 0x0133, 0x0266, 0x0133, 0x67e1, 0xd30a },
    { 0x011c, 0x0237, 0x011c, 0x68ef, 0xd25f },
    { 0x0106, 0x020c, 0x0106, 0x69f1, 0xd1b9 },
    { 0x00f2, 0x01e4, 0x00f2, 0x6ae9, 0xd117 },
    { 0x00df, 0x01be, 0x00df, 0x6bd5, 0xd07a },
    { 0x00ce, 0x019c, 0x00ce, 0x6cb7, 0xcfe1 },     // 1024Hz
    { 0x00be, 0x017c, 0x00be, 0x6d8f, 0xcf4d },
    { 0x00af, 0x015e, 0x00af, 0x6e5d, 0xcebd },
    { 0x00a1, 0x0143, 0x00a1, 0x6f23, 0xce31 },
    { 0x0095, 0x012a, 0x0095, 0x6fe0, 0xcdaa },
    { 0x0089, 0x0112, 0x0089, 0x7094, 0xcd27 },
    { 0x007e, 0x00fd, 0x007e, 0x7140, 0xcca8 },
    { 0x0074, 0x00e9, 0x0074, 0x71e5, 0xcc2e },
    { 0x006b, 0x00d6, 0x006b, 0x7283, 0xcbb7 },
    { 0x0063, 0x00c5, 0x0063, 0x7319, 0xcb44 },
    { 0x005b, 0x00b6, 0x005b, 0x73a9, 0xcad5 },
    { 0x0054, 0x00a7, 0x0054, 0x7433, 0xca6a },
    { 0x004d, 0x009a, 0x004d, 0x74b7, 0xca03 },
    { 0x0047, 0x008e, 0x0047, 0x7534, 0xc99f },
    { 0x0041, 0x0082, 0x0041, 0x75ac, 0xc93f },
    { 0x003c, 0x0078, 0x003c, 0x761f, 0xc8e2 },
    { 0x0037, 0x006e, 0x0037, 0x768d, 0xc889 },     // 512Hz
    { 0x0033, 0x0066, 0x0033, 0x76f6, 0xc833 },
    { 0x002f, 0x005d, 0x002f, 0x775a, 0xc7e0 },
    { 0x002b, 0x0056, 0x002b, 0x77ba, 0xc790 },
    { 0x0027, 0x004f, 0x0027, 0x7816, 0xc743 },
    { 0x0024, 0x0049, 0x0024, 0x786e, 0xc6f8 },
    { 0x0021, 0x0043, 0x0021, 0x78c2, 0xc6b1 },
    { 0x001f, 0x003d, 0x001f, 0x7912, 0xc66c },
    { 0x001c, 0x0038, 0x001c, 0x795f, 0xc62a },
    { 0x001a, 0x0034, 0x001a, 0x79a8, 0xc5ea },
    { 0x0018, 0x0030, 0x0018, 0x79ee, 0xc5ad },
    { 0x0016, 0x002c, 0x0016, 0x7a31, 0xc572 },
    { 0x0014, 0x0028, 0x0014, 0x7a71, 0xc53a },
    { 0x0012, 0x0025, 0x0012, 0x7aae, 0xc503 },
    { 0x0011, 0x0022, 0x0011, 0x7ae9, 0xc4cf },
    { 0x0010, 0x001f, 0x0010, 0x7b21, 0xc49d },
    { 0x000e, 0x001d, 0x000e, 0x7b57, 0xc46d },     // 256Hz
    { 0x000d, 0x001a, 0x000d, 0x7b8a, 0xc43e },
    { 0x000c, 0x0018, 0x000c, 0x7bbb, 0xc412 },
    { 0x000b, 0x0016, 0x000b, 0x7bea, 0xc3e7 },
    { 0x000a, 0x0014, 0x000a, 0x7c17, 0xc3be },
    { 0x0009, 0x0013, 0x0009, 0x7c42, 0xc396 },
    { 0x0009, 0x0011, 0x0009, 0x7c6b, 0xc371 },
    { 0x0008, 0x0010, 0x0008, 0x7c92, 0xc34c },
    { 0x0007, 0x000e, 0x0007, 0x7cb8, 0xc329 },
    { 0x0007, 0x000d, 0x0007, 0x7cdc, 0xc308 },
    { 0x0006, 0x000c, 0x0006, 0x7cfe, 0xc2e8 },
    { 0x0006, 0x000b, 0x0006, 0x7d1f, 0xc2c9 },
    { 0x0005, 0x000a, 0x0005, 0x7d3f, 0xc2ab },
    { 0x0005, 0x0009, 0x0005, 0x7d5d, 0xc28f },
    { 0x0004, 0x0009, 0x0004, 0x7d7a, 0xc274 },
    { 0x0004, 0x0008, 0x0004, 0x7d96, 0xc25a },
    { 0x0004, 0x0007, 0x0004, 0x7db0, 0xc241 }      // 128Hz
};


// HPF係数
// （1/16オクターブ間隔、チェビシェフ近似）
const BiquadFilterCallback::Coefficients
BiquadFilterHpf::COEFFICIENTS_TABLE[ BiquadFilterHpf::COEFFICIENTS_TABLE_SIZE ] =
{
    //    n0      n1      n2      d1      d2
    { 0x3bb5, 0x8895, 0x3bb5, 0x7e7b, 0xc17f },     // 128Hz
    { 0x3bad, 0x88a6, 0x3bad, 0x7e69, 0xc18f },
    { 0x3ba5, 0x88b6, 0x3ba5, 0x7e57, 0xc1a1 },
    { 0x3b9c, 0x88c8, 0x3b9c, 0x7e44, 0xc1b3 },
    { 0x3b93, 0x88da, 0x3b93, 0x7e30, 0xc1c6 },
    { 0x3b89, 0x88ed, 0x3b89, 0x7e1c, 0xc1da },
    { 0x3b7f, 0x8901, 0x3b7f, 0x7e06, 0xc1ee },
    { 0x3b75, 0x8916, 0x3b75, 0x7df0, 0xc204 },
    { 0x3b6a, 0x892c, 0x3b6a, 0x7dd8, 0xc21a },
    { 0x3b5f, 0x8942, 0x3b5f, 0x7dbf, 0xc232 },
    { 0x3b53, 0x895a, 0x3b53, 0x7da6, 0xc24a },
    { 0x3b47, 0x8973, 0x3b47, 0x7d8b, 0xc264 },
    { 0x3b3a, 0x898d, 0x3b3a, 0x7d6f, 0xc27e },
    { 0x3b2c, 0x89a8, 0x3b2c, 0x7d51, 0xc29a },
    { 0x3b1e, 0x89c4, 0x3b1e, 0x7d33, 0xc2b7 },
    { 0x3b10, 0x89e1, 0x3b10, 0x7d13, 0xc2d5 },
    { 0x3b00, 0x8a00, 0x3b00, 0x7cf1, 0xc2f4 },     // 256Hz
    { 0x3af0, 0x8a20, 0x3af0, 0x7cce, 0xc315 },
    { 0x3ae0, 0x8a41, 0x3ae0, 0x7ca9, 0xc337 },
    { 0x3ace, 0x8a64, 0x3ace, 0x7c83, 0xc35a },
    { 0x3abc, 0x8a88, 0x3abc, 0x7c5b, 0xc37f },
    { 0x3aa9, 0x8aae, 0x3aa9, 0x7c31, 0xc3a6 },
    { 0x3a95, 0x8ad6, 0x3a95, 0x7c05, 0xc3ce },
    { 0x3a81, 0x8aff, 0x3a81, 0x7bd8, 0xc3f8 },
    { 0x3a6b, 0x8b2a, 0x3a6b, 0x7ba8, 0xc423 },
    { 0x3a55, 0x8b57, 0x3a55, 0x7b76, 0xc451 },
    { 0x3a3d, 0x8b86, 0x3a3d, 0x7b42, 0xc480 },
    { 0x3a25, 0x8bb7, 0x3a25, 0x7b0b, 0xc4b1 },
    { 0x3a0b, 0x8bea, 0x3a0b, 0x7ad2, 0xc4e4 },
    { 0x39f0, 0x8c1f, 0x39f0, 0x7a96, 0xc519 },
    { 0x39d5, 0x8c57, 0x39d5, 0x7a58, 0xc550 },
    { 0x39b8, 0x8c91, 0x39b8, 0x7a16, 0xc58a },
    { 0x3999, 0x8ccd, 0x3999, 0x79d2, 0xc5c6 },     // 512Hz
    { 0x397a, 0x8d0c, 0x397a, 0x798b, 0xc604 },
    { 0x3959, 0x8d4e, 0x3959, 0x7940, 0xc645 },
    { 0x3937, 0x8d93, 0x3937, 0x78f2, 0xc688 },
    { 0x3913, 0x8dda, 0x3913, 0x78a0, 0xc6ce },
    { 0x38ed, 0x8e25, 0x38ed, 0x784a, 0xc716 },
    { 0x38c7, 0x8e73, 0x38c7, 0x77f1, 0xc762 },
    { 0x389e, 0x8ec4, 0x389e, 0x7793, 0xc7b0 },
    { 0x3874, 0x8f19, 0x3874, 0x7731, 0xc802 },
    { 0x3847, 0x8f71, 0x3847, 0x76cb, 0xc856 },
    { 0x3819, 0x8fcd, 0x3819, 0x7660, 0xc8ae },
    { 0x37e9, 0x902d, 0x37e9, 0x75f0, 0xc909 },
    { 0x37b7, 0x9091, 0x37b7, 0x757a, 0xc968 },
    { 0x3783, 0x90f9, 0x3783, 0x74ff, 0xc9ca },
    { 0x374d, 0x9166, 0x374d, 0x747f, 0xca2f },
    { 0x3714, 0x91d7, 0x3714, 0x73f8, 0xca98 },
    { 0x36d9, 0x924d, 0x36d9, 0x736b, 0xcb05 },     // 1024Hz
    { 0x369c, 0x92c8, 0x369c, 0x72d8, 0xcb76 },
    { 0x365c, 0x9348, 0x365c, 0x723e, 0xcbeb },
    { 0x3619, 0x93cd, 0x3619, 0x719c, 0xcc64 },
    { 0x35d4, 0x9458, 0x35d4, 0x70f3, 0xcce1 },
    { 0x358c, 0x94e9, 0x358c, 0x7042, 0xcd63 },
    { 0x3540, 0x957f, 0x3540, 0x6f89, 0xcde8 },
    { 0x34f2, 0x961c, 0x34f2, 0x6ec7, 0xce73 },
    { 0x34a0, 0x96bf, 0x34a0, 0x6dfb, 0xcf01 },
    { 0x344b, 0x9769, 0x344b, 0x6d27, 0xcf94 },
    { 0x33f3, 0x981a, 0x33f3, 0x6c48, 0xd02c },
    { 0x3397, 0x98d2, 0x3397, 0x6b5f, 0xd0c9 },
    { 0x3337, 0x9992, 0x3337, 0x6a6a, 0xd16a },
    { 0x32d4, 0x9a59, 0x32d4, 0x696a, 0xd210 },
    { 0x326c, 0x9b28, 0x326c, 0x685e, 0xd2bb },
    { 0x3200, 0x9bff, 0x3200, 0x6746, 0xd36b },
    { 0x3190, 0x9cdf, 0x3190, 0x6620, 0xd41f },     // 2048Hz
    { 0x311c, 0x9dc8, 0x311c, 0x64ec, 0xd4d9 },
    { 0x30a3, 0x9eba, 0x30a3, 0x63a9, 0xd597 },
    { 0x3025, 0x9fb6, 0x3025, 0x6258, 0xd65a },
    { 0x2fa3, 0xa0bb, 0x2fa3, 0x60f6, 0xd721 },
    { 0x2f1b, 0xa1ca, 0x2f1b, 0x5f83, 0xd7ee },
    { 0x2e8e, 0xa2e4, 0x2e8e, 0x5dff, 0xd8be },
    { 0x2dfc, 0xa409, 0x2dfc, 0x5c68, 0xd993 },
    { 0x2d64, 0xa539, 0x2d64, 0x5abe, 0xda6d },
    { 0x2cc6, 0xa674, 0x2cc6, 0x58ff, 0xdb4a },
    { 0x2c22, 0xa7bb, 0x2c22, 0x572b, 0xdc2b },
    { 0x2b79, 0xa90e, 0x2b79, 0x5541, 0xdd10 },
    { 0x2ac9, 0xaa6e, 0x2ac9, 0x533f, 0xddf8 },
    { 0x2a12, 0xabdc, 0x2a12, 0x5125, 0xdee3 },
    { 0x2955, 0xad56, 0x2955, 0x4ef0, 0xdfd1 },
    { 0x2891, 0xaedf, 0x2891, 0x4ca1, 0xe0c0 },
    { 0x27c5, 0xb075, 0x27c5, 0x4a34, 0xe1b1 },     // 4096Hz
    { 0x26f3, 0xb21b, 0x26f3, 0x47aa, 0xe2a4 },
    { 0x2618, 0xb3cf, 0x2618, 0x4500, 0xe396 },
    { 0x2536, 0xb593, 0x2536, 0x4234, 0xe489 },
    { 0x244c, 0xb768, 0x244c, 0x3f45, 0xe57a },
    { 0x235a, 0xb94d, 0x235a, 0x3c31, 0xe669 },
    { 0x225f, 0xbb43, 0x225f, 0x38f6, 0xe756 },
    { 0x215b, 0xbd4a, 0x215b, 0x3591, 0xe83e },
    { 0x204e, 0xbf64, 0x204e, 0x3200, 0xe920 },
    { 0x1f38, 0xc191, 0x1f38, 0x2e40, 0xe9fc },
    { 0x1e17, 0xc3d1, 0x1e17, 0x2a4e, 0xeacf },
    { 0x1ced, 0xc625, 0x1ced, 0x2628, 0xeb97 },
    { 0x1bb9, 0xc88e, 0x1bb9, 0x21c8, 0xec52 },
    { 0x1a7a, 0xcb0b, 0x1a7a, 0x1d2d, 0xecfd },
    { 0x1930, 0xcd9f, 0x1930, 0x1850, 0xed96 },
    { 0x17dc, 0xd049, 0x17dc, 0x132e, 0xee18 },
    { 0x167b, 0xd30a, 0x167b, 0x0dc1, 0xee80 }      // 8192Hz
};


// BPF係数その１
// （中心周波数 512Hz、チェビシェフ近似）
const BiquadFilterCallback::Coefficients
BiquadFilterBpf512::COEFFICIENTS_TABLE[ BiquadFilterBpf512::COEFFICIENTS_TABLE_SIZE ] =
{
    //    n0      n1      n2      d1      d2
    { 0x2efa, 0x0000, 0xd106, 0x21d7, 0x1df4 },     // 32 - 78446.86 Hz
    { 0x2e8f, 0x0000, 0xd171, 0x22ad, 0x1d1e },
    { 0x2e24, 0x0000, 0xd1dc, 0x2381, 0x1c49 },
    { 0x2dba, 0x0000, 0xd246, 0x2455, 0x1b74 },
    { 0x2d50, 0x0000, 0xd2b0, 0x2527, 0x1aa1 },
    { 0x2ce7, 0x0000, 0xd319, 0x25f9, 0x19ce },
    { 0x2c7e, 0x0000, 0xd382, 0x26cb, 0x18fc },
    { 0x2c15, 0x0000, 0xd3eb, 0x279c, 0x182a },
    { 0x2bad, 0x0000, 0xd453, 0x286c, 0x1759 },
    { 0x2b44, 0x0000, 0xd4bc, 0x293c, 0x1689 },
    { 0x2add, 0x0000, 0xd523, 0x2a0b, 0x15b9 },
    { 0x2a75, 0x0000, 0xd58b, 0x2ada, 0x14ea },
    { 0x2a0d, 0x0000, 0xd5f3, 0x2ba8, 0x141a },
    { 0x29a6, 0x0000, 0xd65a, 0x2c76, 0x134c },
    { 0x293f, 0x0000, 0xd6c1, 0x2d43, 0x127d },
    { 0x28d8, 0x0000, 0xd728, 0x2e11, 0x11af },
    { 0x2871, 0x0000, 0xd78f, 0x2ede, 0x10e2 },
    { 0x280a, 0x0000, 0xd7f6, 0x2faa, 0x1014 },
    { 0x27a4, 0x0000, 0xd85c, 0x3076, 0x0f47 },
    { 0x273d, 0x0000, 0xd8c3, 0x3142, 0x0e7a },
    { 0x26d7, 0x0000, 0xd929, 0x320e, 0x0dae },
    { 0x2671, 0x0000, 0xd98f, 0x32da, 0x0ce2 },
    { 0x260b, 0x0000, 0xd9f5, 0x33a5, 0x0c16 },
    { 0x25a5, 0x0000, 0xda5b, 0x3470, 0x0b4a },
    { 0x253f, 0x0000, 0xdac1, 0x353a, 0x0a7e },
    { 0x24da, 0x0000, 0xdb26, 0x3605, 0x09b3 },
    { 0x2474, 0x0000, 0xdb8c, 0x36cf, 0x08e8 },
    { 0x240f, 0x0000, 0xdbf1, 0x3799, 0x081d },
    { 0x23a9, 0x0000, 0xdc57, 0x3863, 0x0753 },
    { 0x2344, 0x0000, 0xdcbc, 0x392c, 0x0689 },
    { 0x22df, 0x0000, 0xdd21, 0x39f5, 0x05bf },
    { 0x227b, 0x0000, 0xdd85, 0x3abe, 0x04f5 },
    { 0x2216, 0x0000, 0xddea, 0x3b86, 0x042c },     // 64 - 39223.43 Hz
    { 0x21b1, 0x0000, 0xde4f, 0x3c4f, 0x0363 },
    { 0x214d, 0x0000, 0xdeb3, 0x3d17, 0x029a },
    { 0x20e9, 0x0000, 0xdf17, 0x3dde, 0x01d1 },
    { 0x2085, 0x0000, 0xdf7b, 0x3ea6, 0x0109 },
    { 0x2021, 0x0000, 0xdfdf, 0x3f6d, 0x0041 },
    { 0x1fbd, 0x0000, 0xe043, 0x4033, 0xff7a },
    { 0x1f59, 0x0000, 0xe0a7, 0x40fa, 0xfeb3 },
    { 0x1ef6, 0x0000, 0xe10a, 0x41c0, 0xfdec },
    { 0x1e93, 0x0000, 0xe16d, 0x4285, 0xfd25 },
    { 0x1e30, 0x0000, 0xe1d0, 0x434a, 0xfc5f },
    { 0x1dcd, 0x0000, 0xe233, 0x440f, 0xfb9a },
    { 0x1d6a, 0x0000, 0xe296, 0x44d4, 0xfad4 },
    { 0x1d08, 0x0000, 0xe2f8, 0x4597, 0xfa10 },
    { 0x1ca6, 0x0000, 0xe35a, 0x465b, 0xf94b },
    { 0x1c44, 0x0000, 0xe3bc, 0x471e, 0xf887 },
    { 0x1be2, 0x0000, 0xe41e, 0x47e1, 0xf7c4 },
    { 0x1b80, 0x0000, 0xe480, 0x48a3, 0xf701 },
    { 0x1b1f, 0x0000, 0xe4e1, 0x4964, 0xf63f },
    { 0x1abe, 0x0000, 0xe542, 0x4a25, 0xf57d },
    { 0x1a5e, 0x0000, 0xe5a2, 0x4ae6, 0xf4bb },
    { 0x19fd, 0x0000, 0xe603, 0x4ba6, 0xf3fa },
    { 0x199d, 0x0000, 0xe663, 0x4c65, 0xf33a },
    { 0x193d, 0x0000, 0xe6c3, 0x4d24, 0xf27a },
    { 0x18dd, 0x0000, 0xe723, 0x4de3, 0xf1bb },
    { 0x187e, 0x0000, 0xe782, 0x4ea0, 0xf0fc },
    { 0x181f, 0x0000, 0xe7e1, 0x4f5e, 0xf03e },
    { 0x17c0, 0x0000, 0xe840, 0x501a, 0xef81 },
    { 0x1762, 0x0000, 0xe89e, 0x50d6, 0xeec4 },
    { 0x1704, 0x0000, 0xe8fc, 0x5192, 0xee07 },
    { 0x16a6, 0x0000, 0xe95a, 0x524d, 0xed4c },
    { 0x1648, 0x0000, 0xe9b8, 0x5307, 0xec91 },
    { 0x15eb, 0x0000, 0xea15, 0x53c0, 0xebd6 },     // 128 - 19611.72 Hz
    { 0x158e, 0x0000, 0xea72, 0x5479, 0xeb1c },
    { 0x1532, 0x0000, 0xeace, 0x5532, 0xea63 },
    { 0x14d5, 0x0000, 0xeb2b, 0x55ea, 0xe9ab },
    { 0x1479, 0x0000, 0xeb87, 0x56a1, 0xe8f3 },
    { 0x141e, 0x0000, 0xebe2, 0x5757, 0xe83b },
    { 0x13c2, 0x0000, 0xec3e, 0x580d, 0xe785 },
    { 0x1367, 0x0000, 0xec99, 0x58c2, 0xe6ce },
    { 0x130d, 0x0000, 0xecf3, 0x5977, 0xe619 },
    { 0x12b2, 0x0000, 0xed4e, 0x5a2b, 0xe564 },
    { 0x1258, 0x0000, 0xeda8, 0x5ade, 0xe4b0 },
    { 0x11fe, 0x0000, 0xee02, 0x5b91, 0xe3fc },
    { 0x11a5, 0x0000, 0xee5b, 0x5c43, 0xe349 },
    { 0x114b, 0x0000, 0xeeb5, 0x5cf5, 0xe297 },
    { 0x10f2, 0x0000, 0xef0e, 0x5da6, 0xe1e5 },
    { 0x109a, 0x0000, 0xef66, 0x5e56, 0xe134 },
    { 0x1041, 0x0000, 0xefbf, 0x5f06, 0xe083 },
    { 0x0fe9, 0x0000, 0xf017, 0x5fb6, 0xdfd3 },
    { 0x0f91, 0x0000, 0xf06f, 0x6065, 0xdf23 },
    { 0x0f3a, 0x0000, 0xf0c6, 0x6113, 0xde74 },
    { 0x0ee2, 0x0000, 0xf11e, 0x61c1, 0xddc5 },
    { 0x0e8b, 0x0000, 0xf175, 0x626f, 0xdd16 },
    { 0x0e34, 0x0000, 0xf1cc, 0x631c, 0xdc69 },
    { 0x0dde, 0x0000, 0xf222, 0x63c8, 0xdbbb },
    { 0x0d87, 0x0000, 0xf279, 0x6475, 0xdb0e },
    { 0x0d31, 0x0000, 0xf2cf, 0x6520, 0xda61 },
    { 0x0cda, 0x0000, 0xf326, 0x65cc, 0xd9b5 },
    { 0x0c84, 0x0000, 0xf37c, 0x6677, 0xd909 },
    { 0x0c2f, 0x0000, 0xf3d1, 0x6722, 0xd85d },
    { 0x0bd9, 0x0000, 0xf427, 0x67cd, 0xd7b2 },
    { 0x0b83, 0x0000, 0xf47d, 0x6878, 0xd706 },
    { 0x0b2d, 0x0000, 0xf4d3, 0x6922, 0xd65b },
    { 0x0ad8, 0x0000, 0xf528, 0x69cc, 0xd5b0 },     // 256 - 9805.86 Hz
    { 0x0a82, 0x0000, 0xf57e, 0x6a77, 0xd505 },
    { 0x0a2d, 0x0000, 0xf5d3, 0x6b21, 0xd45a },
    { 0x09d7, 0x0000, 0xf629, 0x6bcb, 0xd3af },
    { 0x0982, 0x0000, 0xf67e, 0x6c75, 0xd304 },
    { 0x092c, 0x0000, 0xf6d4, 0x6d1f, 0xd259 },
    { 0x08d7, 0x0000, 0xf729, 0x6dca, 0xd1ae },
    { 0x0881, 0x0000, 0xf77f, 0x6e74, 0xd102 },
    { 0x082b, 0x0000, 0xf7d5, 0x6f1f, 0xd056 },
    { 0x07d5, 0x0000, 0xf82b, 0x6fca, 0xcfaa },
    { 0x077f, 0x0000, 0xf881, 0x7076, 0xcefe },
    { 0x0729, 0x0000, 0xf8d7, 0x7122, 0xce51 },
    { 0x06d2, 0x0000, 0xf92e, 0x71ce, 0xcda4 },
    { 0x067b, 0x0000, 0xf985, 0x727b, 0xccf6 },
    { 0x0624, 0x0000, 0xf9dc, 0x7329, 0xcc48 },
    { 0x05cc, 0x0000, 0xfa34, 0x73d7, 0xcb98 },
    { 0x0574, 0x0000, 0xfa8c, 0x7487, 0xcae8 },
    { 0x051c, 0x0000, 0xfae4, 0x7537, 0xca38 },
    { 0x04c3, 0x0000, 0xfb3d, 0x75e7, 0xc986 },
    { 0x046a, 0x0000, 0xfb96, 0x7699, 0xc8d3 },
    { 0x0410, 0x0000, 0xfbf0, 0x774c, 0xc81f },
    { 0x03b5, 0x0000, 0xfc4b, 0x7800, 0xc76a },
    { 0x035a, 0x0000, 0xfca6, 0x78b6, 0xc6b4 },
    { 0x02fe, 0x0000, 0xfd02, 0x796d, 0xc5fc },
    { 0x02a2, 0x0000, 0xfd5e, 0x7a25, 0xc543 },
    { 0x0244, 0x0000, 0xfdbc, 0x7adf, 0xc488 }
};


// BPF係数その２
// （中心周波数 1024Hz、チェビシェフ近似）
const BiquadFilterCallback::Coefficients
BiquadFilterBpf1024::COEFFICIENTS_TABLE[ BiquadFilterBpf1024::COEFFICIENTS_TABLE_SIZE ] =
{
    //    n0      n1      n2      d1      d2
    { 0x2f04, 0x0000, 0xd0fc, 0x21de, 0x1e08 },     // 16 - 7844.686 Hz
    { 0x2e99, 0x0000, 0xd167, 0x22b3, 0x1d32 },
    { 0x2e2f, 0x0000, 0xd1d1, 0x2386, 0x1c5d },
    { 0x2dc5, 0x0000, 0xd23b, 0x2459, 0x1b89 },
    { 0x2d5b, 0x0000, 0xd2a5, 0x252b, 0x1ab6 },
    { 0x2cf2, 0x0000, 0xd30e, 0x25fd, 0x19e4 },
    { 0x2c89, 0x0000, 0xd377, 0x26ce, 0x1912 },
    { 0x2c20, 0x0000, 0xd3e0, 0x279e, 0x1841 },
    { 0x2bb8, 0x0000, 0xd448, 0x286d, 0x1770 },
    { 0x2b50, 0x0000, 0xd4b0, 0x293c, 0x16a0 },
    { 0x2ae8, 0x0000, 0xd518, 0x2a0b, 0x15d0 },
    { 0x2a80, 0x0000, 0xd580, 0x2ad9, 0x1500 },
    { 0x2a19, 0x0000, 0xd5e7, 0x2ba7, 0x1431 },
    { 0x29b1, 0x0000, 0xd64f, 0x2c74, 0x1363 },
    { 0x294a, 0x0000, 0xd6b6, 0x2d41, 0x1294 },
    { 0x28e3, 0x0000, 0xd71d, 0x2e0e, 0x11c6 },
    { 0x287c, 0x0000, 0xd784, 0x2eda, 0x10f8 },     // 32 - 5547.031 Hz
    { 0x2815, 0x0000, 0xd7eb, 0x2fa6, 0x102a },
    { 0x27af, 0x0000, 0xd851, 0x3072, 0x0f5d },
    { 0x2748, 0x0000, 0xd8b8, 0x313e, 0x0e90 },
    { 0x26e1, 0x0000, 0xd91f, 0x3209, 0x0dc3 },
    { 0x267b, 0x0000, 0xd985, 0x32d4, 0x0cf6 },
    { 0x2614, 0x0000, 0xd9ec, 0x339f, 0x0c29 },
    { 0x25ae, 0x0000, 0xda52, 0x346a, 0x0b5c },
    { 0x2548, 0x0000, 0xdab8, 0x3535, 0x0a8f },
    { 0x24e1, 0x0000, 0xdb1f, 0x35ff, 0x09c3 },
    { 0x247b, 0x0000, 0xdb85, 0x36ca, 0x08f6 },
    { 0x2415, 0x0000, 0xdbeb, 0x3794, 0x082a },
    { 0x23af, 0x0000, 0xdc51, 0x385e, 0x075d },
    { 0x2349, 0x0000, 0xdcb7, 0x3928, 0x0691 },
    { 0x22e2, 0x0000, 0xdd1e, 0x39f3, 0x05c5 },
    { 0x227c, 0x0000, 0xdd84, 0x3abd, 0x04f8 },
    { 0x2216, 0x0000, 0xddea, 0x3b86, 0x042c },     // 64 - 3922.343 Hz
    { 0x21b0, 0x0000, 0xde50, 0x3c50, 0x035f },
    { 0x2149, 0x0000, 0xdeb7, 0x3d1a, 0x0293 },
    { 0x20e3, 0x0000, 0xdf1d, 0x3de4, 0x01c6 },
    { 0x207d, 0x0000, 0xdf83, 0x3eae, 0x00f9 },
    { 0x2016, 0x0000, 0xdfea, 0x3f78, 0x002c },
    { 0x1fb0, 0x0000, 0xe050, 0x4042, 0xff5f },
    { 0x1f49, 0x0000, 0xe0b7, 0x410c, 0xfe92 },
    { 0x1ee2, 0x0000, 0xe11e, 0x41d7, 0xfdc4 },
    { 0x1e7b, 0x0000, 0xe185, 0x42a1, 0xfcf7 },
    { 0x1e14, 0x0000, 0xe1ec, 0x436c, 0xfc29 },
    { 0x1dad, 0x0000, 0xe253, 0x4436, 0xfb5b },
    { 0x1d46, 0x0000, 0xe2ba, 0x4501, 0xfa8c },
    { 0x1cdf, 0x0000, 0xe321, 0x45cd, 0xf9bd },
    { 0x1c77, 0x0000, 0xe389, 0x4698, 0xf8ee },
    { 0x1c0f, 0x0000, 0xe3f1, 0x4764, 0xf81e },
    { 0x1ba7, 0x0000, 0xe459, 0x4831, 0xf74d },     // 128 - 2773.515 Hz
    { 0x1b3e, 0x0000, 0xe4c2, 0x48fd, 0xf67c },
    { 0x1ad5, 0x0000, 0xe52b, 0x49cb, 0xf5ab },
    { 0x1a6c, 0x0000, 0xe594, 0x4a99, 0xf4d8 },
    { 0x1a03, 0x0000, 0xe5fd, 0x4b67, 0xf405 },
    { 0x1998, 0x0000, 0xe668, 0x4c37, 0xf331 },
    { 0x192e, 0x0000, 0xe6d2, 0x4d07, 0xf25c },
    { 0x18c3, 0x0000, 0xe73d, 0x4dd8, 0xf186 },
    { 0x1857, 0x0000, 0xe7a9, 0x4eaa, 0xf0af },
    { 0x17eb, 0x0000, 0xe815, 0x4f7d, 0xefd6 },
    { 0x177e, 0x0000, 0xe882, 0x5051, 0xeefc },
    { 0x1710, 0x0000, 0xe8f0, 0x5127, 0xee21 },
    { 0x16a2, 0x0000, 0xe95e, 0x51fe, 0xed44 },
    { 0x1632, 0x0000, 0xe9ce, 0x52d7, 0xec65 },
    { 0x15c2, 0x0000, 0xea3e, 0x53b2, 0xeb84 },
    { 0x1550, 0x0000, 0xeab0, 0x548e, 0xeaa1 },
    { 0x14de, 0x0000, 0xeb22, 0x556d, 0xe9bc },     // 256 - 1961.172 Hz
    { 0x146a, 0x0000, 0xeb96, 0x564e, 0xe8d4 },
    { 0x13f5, 0x0000, 0xec0b, 0x5731, 0xe7e9 },
    { 0x137e, 0x0000, 0xec82, 0x5817, 0xe6fc },
    { 0x1306, 0x0000, 0xecfa, 0x5900, 0xe60b },
    { 0x128c, 0x0000, 0xed74, 0x59ed, 0xe517 },
    { 0x1210, 0x0000, 0xedf0, 0x5adc, 0xe41f },
    { 0x1192, 0x0000, 0xee6e, 0x5bd0, 0xe323 },
    { 0x1111, 0x0000, 0xeeef, 0x5cc8, 0xe223 },
    { 0x108f, 0x0000, 0xef71, 0x5dc4, 0xe11e },
    { 0x100a, 0x0000, 0xeff6, 0x5ec5, 0xe013 },
    { 0x0f82, 0x0000, 0xf07e, 0x5fcb, 0xdf04 },
    { 0x0ef7, 0x0000, 0xf109, 0x60d7, 0xddee },
    { 0x0e69, 0x0000, 0xf197, 0x61e9, 0xdcd1 },
    { 0x0dd7, 0x0000, 0xf229, 0x6301, 0xdbae },
    { 0x0d41, 0x0000, 0xf2bf, 0x6421, 0xda83 },
    { 0x0ca8, 0x0000, 0xf358, 0x6549, 0xd94f },     // 512 - 1386.758 Hz
    { 0x0c09, 0x0000, 0xf3f7, 0x6679, 0xd813 },
    { 0x0b66, 0x0000, 0xf49a, 0x67b3, 0xd6cd },
    { 0x0abe, 0x0000, 0xf542, 0x68f6, 0xd57c },
    { 0x0a10, 0x0000, 0xf5f0, 0x6a45, 0xd420 },
    { 0x095b, 0x0000, 0xf6a5, 0x6b9f, 0xd2b7 },
    { 0x08a0, 0x0000, 0xf760, 0x6d06, 0xd141 },
    { 0x07de, 0x0000, 0xf822, 0x6e7b, 0xcfbc },
    { 0x0714, 0x0000, 0xf8ec, 0x6fff, 0xce27 },
    { 0x0640, 0x0000, 0xf9c0, 0x7195, 0xcc81 },
    { 0x0564, 0x0000, 0xfa9c, 0x733c, 0xcac7 },
    { 0x047d, 0x0000, 0xfb83, 0x74f7, 0xc8f9 },
    { 0x038a, 0x0000, 0xfc76, 0x76c8, 0xc714 }
};


// BPF係数その３
// （中心周波数 2048Hz、チェビシェフ近似）
const BiquadFilterCallback::Coefficients
BiquadFilterBpf2048::COEFFICIENTS_TABLE[ BiquadFilterBpf2048::COEFFICIENTS_TABLE_SIZE ] =
{
    //    n0      n1      n2      d1      d2
    { 0x3f42, 0x0000, 0xc0be, 0x0136, 0x3e83 },     // 32 - 15689.372 Hz
    { 0x3e8a, 0x0000, 0xc176, 0x02a3, 0x3d14 },
    { 0x3dd9, 0x0000, 0xc227, 0x0401, 0x3bb3 },
    { 0x3d2f, 0x0000, 0xc2d1, 0x0553, 0x3a5e },
    { 0x3c8b, 0x0000, 0xc375, 0x0699, 0x3916 },
    { 0x3bec, 0x0000, 0xc414, 0x07d5, 0x37d7 },
    { 0x3b51, 0x0000, 0xc4af, 0x0907, 0x36a2 },
    { 0x3abb, 0x0000, 0xc545, 0x0a31, 0x3576 },
    { 0x3a28, 0x0000, 0xc5d8, 0x0b53, 0x3451 },
    { 0x399a, 0x0000, 0xc666, 0x0c6d, 0x3333 },
    { 0x390e, 0x0000, 0xc6f2, 0x0d81, 0x321c },
    { 0x3885, 0x0000, 0xc77b, 0x0e8f, 0x310b },
    { 0x37ff, 0x0000, 0xc801, 0x0f97, 0x2fff },
    { 0x377c, 0x0000, 0xc884, 0x109b, 0x2ef7 },
    { 0x36fa, 0x0000, 0xc906, 0x119a, 0x2df5 },
    { 0x367b, 0x0000, 0xc985, 0x1294, 0x2cf6 },
    { 0x35fe, 0x0000, 0xca02, 0x138b, 0x2bfb },     // 64 - 11094.062 Hz
    { 0x3582, 0x0000, 0xca7e, 0x147e, 0x2b04 },
    { 0x3508, 0x0000, 0xcaf8, 0x156e, 0x2a0f },
    { 0x348f, 0x0000, 0xcb71, 0x165b, 0x291e },
    { 0x3417, 0x0000, 0xcbe9, 0x1745, 0x282f },
    { 0x33a1, 0x0000, 0xcc5f, 0x182d, 0x2742 },
    { 0x332c, 0x0000, 0xccd4, 0x1913, 0x2657 },
    { 0x32b7, 0x0000, 0xcd49, 0x19f6, 0x256f },
    { 0x3244, 0x0000, 0xcdbc, 0x1ad7, 0x2488 },
    { 0x31d1, 0x0000, 0xce2f, 0x1bb7, 0x23a2 },
    { 0x315f, 0x0000, 0xcea1, 0x1c95, 0x22be },
    { 0x30ee, 0x0000, 0xcf12, 0x1d72, 0x21dc },
    { 0x307d, 0x0000, 0xcf83, 0x1e4d, 0x20fa },
    { 0x300c, 0x0000, 0xcff4, 0x1f28, 0x2019 },
    { 0x2f9c, 0x0000, 0xd064, 0x2001, 0x1f39 },
    { 0x2f2d, 0x0000, 0xd0d3, 0x20d9, 0x1e59 },
    { 0x2ebd, 0x0000, 0xd143, 0x21b1, 0x1d7a },     // 128 - 7844.686 Hz
    { 0x2e4e, 0x0000, 0xd1b2, 0x2288, 0x1c9c },
    { 0x2ddf, 0x0000, 0xd221, 0x235e, 0x1bbe },
    { 0x2d70, 0x0000, 0xd290, 0x2434, 0x1ae0 },
    { 0x2d01, 0x0000, 0xd2ff, 0x250a, 0x1a02 },
    { 0x2c92, 0x0000, 0xd36e, 0x25df, 0x1923 },
    { 0x2c23, 0x0000, 0xd3dd, 0x26b4, 0x1845 },
    { 0x2bb3, 0x0000, 0xd44d, 0x2789, 0x1767 },
    { 0x2b44, 0x0000, 0xd4bc, 0x285e, 0x1688 },
    { 0x2ad4, 0x0000, 0xd52c, 0x2934, 0x15a8 },
    { 0x2a64, 0x0000, 0xd59c, 0x2a09, 0x14c8 },
    { 0x29f4, 0x0000, 0xd60c, 0x2adf, 0x13e8 },
    { 0x2983, 0x0000, 0xd67d, 0x2bb6, 0x1306 },
    { 0x2912, 0x0000, 0xd6ee, 0x2c8d, 0x1224 },
    { 0x28a0, 0x0000, 0xd760, 0x2d64, 0x1140 },
    { 0x282e, 0x0000, 0xd7d2, 0x2e3c, 0x105c },
    { 0x27bb, 0x0000, 0xd845, 0x2f15, 0x0f76 },     // 256 - 5547.031 Hz
    { 0x2747, 0x0000, 0xd8b9, 0x2ff0, 0x0e8e },
    { 0x26d3, 0x0000, 0xd92d, 0x30cb, 0x0da5 },
    { 0x265d, 0x0000, 0xd9a3, 0x31a7, 0x0cbb },
    { 0x25e7, 0x0000, 0xda19, 0x3285, 0x0bce },
    { 0x2570, 0x0000, 0xda90, 0x3364, 0x0ae0 },
    { 0x24f7, 0x0000, 0xdb09, 0x3445, 0x09ef },
    { 0x247e, 0x0000, 0xdb82, 0x3528, 0x08fc },
    { 0x2403, 0x0000, 0xdbfd, 0x360c, 0x0806 },
    { 0x2387, 0x0000, 0xdc79, 0x36f3, 0x070d },
    { 0x2309, 0x0000, 0xdcf7, 0x37dc, 0x0612 },
    { 0x228a, 0x0000, 0xdd76, 0x38c8, 0x0513 },
    { 0x2208, 0x0000, 0xddf8, 0x39b6, 0x0411 },
    { 0x2185, 0x0000, 0xde7b, 0x3aa8, 0x030b },
    { 0x2100, 0x0000, 0xdf00, 0x3b9d, 0x0201 },
    { 0x2079, 0x0000, 0xdf87, 0x3c95, 0x00f2 },
    { 0x1ff0, 0x0000, 0xe010, 0x3d91, 0xffdf },     // 512 - 3922.343 Hz
    { 0x1f63, 0x0000, 0xe09d, 0x3e91, 0xfec7 },
    { 0x1ed5, 0x0000, 0xe12b, 0x3f96, 0xfda9 },
    { 0x1e43, 0x0000, 0xe1bd, 0x409f, 0xfc85 },
    { 0x1dae, 0x0000, 0xe252, 0x41ae, 0xfb5b },
    { 0x1d15, 0x0000, 0xe2eb, 0x42c3, 0xfa2a },
    { 0x1c79, 0x0000, 0xe387, 0x43de, 0xf8f2 },
    { 0x1bd9, 0x0000, 0xe427, 0x44ff, 0xf7b1 },
    { 0x1b34, 0x0000, 0xe4cc, 0x4628, 0xf668 },
    { 0x1a8b, 0x0000, 0xe575, 0x4759, 0xf515 },
    { 0x19dc, 0x0000, 0xe624, 0x4893, 0xf3b8 },
    { 0x1928, 0x0000, 0xe6d8, 0x49d6, 0xf250 },
    { 0x186e, 0x0000, 0xe792, 0x4b24, 0xf0dc },
    { 0x17ad, 0x0000, 0xe853, 0x4c7d, 0xef5a },
    { 0x16e5, 0x0000, 0xe91b, 0x4de3, 0xedca },
    { 0x1615, 0x0000, 0xe9eb, 0x4f56, 0xec2a },
    { 0x153d, 0x0000, 0xeac3, 0x50d7, 0xea79 },     // 1024 - 2773.515 Hz
    { 0x145a, 0x0000, 0xeba6, 0x526a, 0xe8b5 },
    { 0x136e, 0x0000, 0xec92, 0x540e, 0xe6dc },
    { 0x1276, 0x0000, 0xed8a, 0x55c6, 0xe4ec },
    { 0x1172, 0x0000, 0xee8e, 0x5794, 0xe2e3 },
    { 0x105f, 0x0000, 0xefa1, 0x597a, 0xe0be },
    { 0x0f3d, 0x0000, 0xf0c3, 0x5b7b, 0xde7a },
    { 0x0e0a, 0x0000, 0xf1f6, 0x5d9b, 0xdc14 },
    { 0x0cc3, 0x0000, 0xf33d, 0x5fdc, 0xd987 },
    { 0x0b67, 0x0000, 0xf499, 0x6242, 0xd6cf },
    { 0x09f3, 0x0000, 0xf60d, 0x64d3, 0xd3e7 },
    { 0x0864, 0x0000, 0xf79c, 0x6793, 0xd0c8 },
    { 0x06b6, 0x0000, 0xf94a, 0x6a89, 0xcd6c }
};


/*---------------------------------------------------------------------------*
  Name:         GetCoefficients

  Description:

  Arguments:    type -
                value -
                coef -

  Returns:      なし
 *---------------------------------------------------------------------------*/
void BiquadFilterLpf::GetCoefficients(
    int type,
    f32 value,
    BiquadFilterCallback::Coefficients* coef
) const
{
    (void)type;
    int coefSize = COEFFICIENTS_TABLE_SIZE;
    int coefIndex = static_cast<int>( static_cast<f32>( coefSize - 1 ) * value );

    coefIndex = ut::Clamp( coefIndex, 0, COEFFICIENTS_TABLE_SIZE-1 );
    *coef = COEFFICIENTS_TABLE[ coefIndex ];
}

/*---------------------------------------------------------------------------*
  Name:         GetCoefficients

  Description:

  Arguments:    type -
                value -
                coef -

  Returns:      なし
 *---------------------------------------------------------------------------*/
void BiquadFilterHpf::GetCoefficients(
    int type,
    f32 value,
    BiquadFilterCallback::Coefficients* coef
) const
{
    (void)type;
    int coefSize = COEFFICIENTS_TABLE_SIZE;
    int coefIndex = static_cast<int>( static_cast<f32>( coefSize - 1 ) * value );

    coefIndex = ut::Clamp( coefIndex, 0, COEFFICIENTS_TABLE_SIZE-1 );
    *coef = COEFFICIENTS_TABLE[ coefIndex ];
}

/*---------------------------------------------------------------------------*
  Name:         GetCoefficients

  Description:

  Arguments:    type -
                value -
                coef -

  Returns:      なし
 *---------------------------------------------------------------------------*/
void BiquadFilterBpf512::GetCoefficients(
    int type,
    f32 value,
    BiquadFilterCallback::Coefficients* coef
) const
{
    (void)type;

    value *= ( 2.0f - value );  // 係数の取り方に重み付け

    int coefSize = COEFFICIENTS_TABLE_SIZE;
    int coefIndex = static_cast<int>( static_cast<f32>( coefSize - 1 ) * value );

    coefIndex = ut::Clamp( coefIndex, 0, COEFFICIENTS_TABLE_SIZE-1 );
    *coef = COEFFICIENTS_TABLE[ coefIndex ];
}

/*---------------------------------------------------------------------------*
  Name:         GetCoefficients

  Description:

  Arguments:    type -
                value -
                coef -

  Returns:      なし
 *---------------------------------------------------------------------------*/
void BiquadFilterBpf1024::GetCoefficients(
    int type,
    f32 value,
    BiquadFilterCallback::Coefficients* coef
) const
{
    (void)type;

    value *= ( 2.0f - value );   // 係数の取り方に重み付け

    int coefSize = COEFFICIENTS_TABLE_SIZE;
    int coefIndex = static_cast<int>( static_cast<f32>( coefSize - 1 ) * value );

    coefIndex = ut::Clamp( coefIndex, 0, COEFFICIENTS_TABLE_SIZE-1 );
    *coef = COEFFICIENTS_TABLE[ coefIndex ];
}

/*---------------------------------------------------------------------------*
  Name:         GetCoefficients

  Description:

  Arguments:    type -
                value -
                coef -

  Returns:      なし
 *---------------------------------------------------------------------------*/
void BiquadFilterBpf2048::GetCoefficients(
    int type,
    f32 value,
    BiquadFilterCallback::Coefficients* coef
) const
{
    (void)type;

    value *= ( 2.0f - value );   // 係数の取り方に重み付け

    int coefSize = COEFFICIENTS_TABLE_SIZE;
    int coefIndex = static_cast<int>( static_cast<f32>( coefSize - 1 ) * value );

    coefIndex = ut::Clamp( coefIndex, 0, COEFFICIENTS_TABLE_SIZE-1 );
    *coef = COEFFICIENTS_TABLE[ coefIndex ];
}

} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw

