﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/snd_AxfxImpl.h>
#include <nw/assert.h>
#include <nw/ut/ut_Inlines.h>

#if defined( NW_PLATFORM_WIN32 ) || defined ( NW_USE_NINTENDO_SDK )
using namespace nw::internal::winext;
#endif

namespace nw {
namespace snd {
namespace internal {

/* ========================================================================
        static variable
   ======================================================================== */

AxfxImpl* AxfxImpl::m_pCurrentFx;
u32 AxfxImpl::m_AllocatedSize;

/* ========================================================================
        member function
   ======================================================================== */

/*---------------------------------------------------------------------------*
  Name:         AssignWorkBuffer

  Description:  エフェクトのワークバッファを割り当てる

  Arguments:    buffer - バッファアドレス
                size - バッファサイズ

  Returns:      成功したら true
 *---------------------------------------------------------------------------*/
bool AxfxImpl::CreateHeap( void *buffer, u32 size )
{
    if ( ( buffer == NULL ) || ( size == 0 ) )
    {
        m_Handle = MEM_HEAP_INVALID_HANDLE;
        return false;
    }

    m_Handle = MEMCreateFrmHeap( buffer, size );
    return m_Handle != MEM_HEAP_INVALID_HANDLE;
}

/*---------------------------------------------------------------------------*
  Name:         ReleaseWorkBuffer

  Description:  エフェクトのワークバッファを開放する

  Arguments:    なし

  Returns:      なし
 *---------------------------------------------------------------------------*/
void AxfxImpl::DestroyHeap()
{
    if ( m_Handle != MEM_HEAP_INVALID_HANDLE )
    {
        MEMDestroyFrmHeap( m_Handle );
        m_Handle = MEM_HEAP_INVALID_HANDLE;
    }
}

/*---------------------------------------------------------------------------*
  Name:         HookAlloc

  Description:  AxfxImpl用のアロケート関数を設定し、
                これまで設定されていたアロケート関数を返す

  Arguments:    alloc - これまで設定されていたalloc関数
                free  - これまで設定されていたfree関数

  Returns:      なし
 *---------------------------------------------------------------------------*/
void AxfxImpl::HookAlloc( AXFXAlloc* alloc, AXFXFree* free )
{
    // 設定されているアロケート関数を保存する
    AXFXGetHooks( alloc, free );

    // AxfxImplのアロケート関数を保存する
    AXFXSetHooks( Alloc, Free );

    // Alloc, Free関数内で使用するAxfxImplのインスタンスを設定する
    m_pCurrentFx = this;
    m_AllocatedSize = 0;
}
#if defined(NW_SND_CONFIG_ENABLE_MULTICHEFT) && !defined(NW_SND_CONFIG_ENABLE_SOUND2)
void AxfxImpl::HookMultichAlloc( AXFXAlloc* alloc, AXFXFree* free )
{
    // 設定されているアロケート関数を保存する
    AXFX2GetMemAllocFns( alloc, free );

    // AxfxImplのアロケート関数を保存する
    AXFX2SetMemAllocFns( Alloc, Free );

    // Alloc, Free関数内で使用するAxfxImplのインスタンスを設定する
    m_pCurrentFx = this;
    m_AllocatedSize = 0;
}
#endif
/*---------------------------------------------------------------------------*
  Name:         RestoreAlloc

  Description:  引数で渡されたアロケート関数に戻す

  Arguments:    alloc - HookAllocで覚えていたalloc関数
                free  - HookAllocで覚えていたfree関数

  Returns:      なし
 *---------------------------------------------------------------------------*/
u32 AxfxImpl::RestoreAlloc( AXFXAlloc alloc, AXFXFree free )
{
    // アロケート関数を元に戻す
    AXFXSetHooks( alloc, free );

    // Alloc, Free関数内で使用するAxfxImplのインスタンス設定を削除する
    m_pCurrentFx = NULL;

    return m_AllocatedSize;
}
#if defined(NW_SND_CONFIG_ENABLE_MULTICHEFT) && !defined(NW_SND_CONFIG_ENABLE_SOUND2)
u32 AxfxImpl::RestoreMultichAlloc( AXFXAlloc alloc, AXFXFree free )
{
    // アロケート関数を元に戻す
    AXFX2SetMemAllocFns( alloc, free );

    // Alloc, Free関数内で使用するAxfxImplのインスタンス設定を削除する
    m_pCurrentFx = NULL;

    return m_AllocatedSize;
}
#endif
/*---------------------------------------------------------------------------*
  Name:         Alloc

  Description:  AxfxImpl用のAlloc関数

  Arguments:

  Returns:
 *---------------------------------------------------------------------------*/
void* AxfxImpl::Alloc( u32 size )
{
    NW_ASSERT_NOT_NULL( m_pCurrentFx );
    NW_ASSERT( m_pCurrentFx->m_Handle != MEM_HEAP_INVALID_HANDLE );
    void* addr = MEMAllocFromFrmHeap( m_pCurrentFx->m_Handle, size );
    NW_ASSERT_NOT_NULL( addr );
    ++m_pCurrentFx->m_AllocCount;
    m_pCurrentFx->m_AllocatedSize += ut::RoundUp( size, 4 );
    return addr;
}

/*---------------------------------------------------------------------------*
  Name:         Free

  Description:  AxfxImpl用のFree関数

  Arguments:

  Returns:      なし
 *---------------------------------------------------------------------------*/
void AxfxImpl::Free( void* ptr )
{
    (void)ptr;

    NW_ASSERT_NOT_NULL( m_pCurrentFx );
    if ( m_pCurrentFx->m_AllocCount > 0 ) --m_pCurrentFx->m_AllocCount;
    if ( m_pCurrentFx->m_AllocCount == 0 )
    {
        MEMFreeToFrmHeap( m_pCurrentFx->m_Handle, MEM_FRMHEAP_FREE_ALL );
    }

}

} // namespace nw::snd::detail

} // namespace nw::snd
} // namespace nw
