﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/fnd/os/sndfnd_Thread.h>
#include <nn/nn_TimeSpan.h>

namespace nw {
namespace snd {
namespace internal {
namespace fnd {

//---------------------------------------------------------------------------
//! @brief  スレッドのメイン関数オブジェクトです。
//---------------------------------------------------------------------------
class Thread::ThreadMain
{
public:
    static void Run(void* ptrArg)
    {
        Thread* owner = reinterpret_cast<Thread*>(ptrArg);

        NW_ASSERT_NOT_NULL(owner);
        NW_ASSERT_NOT_NULL(owner->m_Handler);

        owner->m_IsTerminated = false;
        owner->OnRun();

        owner->m_Handler->Run(owner->m_Param);

        owner->OnExit();
        owner->m_IsTerminated = true;
    }
};

//---------------------------------------------------------------------------
Thread::Thread() :
m_State(STATE_NOT_RUN),
m_ID(INVALID_ID),
m_Priority(DEFAULT_THREAD_PRIORITY),
m_Name(""),
m_Handler(NULL),
m_IsTerminated(false)
{
}

//---------------------------------------------------------------------------
void
Thread::SetPriority(s32 value)
{
    nn::os::ChangeThreadPriority(&m_Handle, value);
}

//---------------------------------------------------------------------------
void
Thread::Sleep(const TimeSpan& timeSpan)
{
    nn::os::SleepThread(nn::TimeSpan::FromNanoSeconds(timeSpan.ToNanoSeconds()));
}

//---------------------------------------------------------------------------
bool
Thread::Create(Handle& handle, u32& id, const RunArgs& args)
{
    NW_UNUSED_VARIABLE(id);

    if (!nn::os::CreateThread(
        &handle,
        ThreadMain::Run,
        this,
        args.stack,
        args.stackSize,
        args.priority).IsSuccess())
    {
        m_ID = INVALID_ID;
        return false;
    }

    nn::os::StartThread(&handle);

    // OSThread へのポインタを ID として利用します。
    m_ID = reinterpret_cast<u32>(&handle);

    return true;
}

//---------------------------------------------------------------------------
void
Thread::Detach()
{
    if(IsTerminated())
    {
        return;
    }

    nn::os::DestroyThread(&m_Handle);
}

//---------------------------------------------------------------------------
void
Thread::SetName(const char* name)
{
    nn::os::SetThreadNamePointer(&m_Handle, name == NULL ? "" : name);
}

//---------------------------------------------------------------------------
void
Thread::SetAffinityMask(AFFINITY_MASK value)
{
    // 対応APIが存在しないため未対応
    NW_UNUSED_VARIABLE(value);
}

//---------------------------------------------------------------------------
void
Thread::Resume()
{
}

//---------------------------------------------------------------------------
void
Thread::Join()
{
    nn::os::WaitThread(&m_Handle);
}

//---------------------------------------------------------------------------
bool
Thread::IsTerminated() const
{
    return m_IsTerminated;
}

} // namespace nw::snd::internal::fnd
} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw


